"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Mpa = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [mpa](https://docs.aws.amazon.com/service-authorization/latest/reference/list_multi-partyapproval.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Mpa extends shared_1.PolicyStatement {
    /**
     * Grants permission to cancel an approval session
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifRequestedOperation()
     * - .ifProtectedResourceAccount()
     *
     * https://docs.aws.amazon.com/mpa/latest/APIReference/API_CancelSession.html
     */
    toCancelSession() {
        return this.to('CancelSession');
    }
    /**
     * Grants permission to create an approval team
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mpa/latest/APIReference/API_CreateApprovalTeam.html
     */
    toCreateApprovalTeam() {
        return this.to('CreateApprovalTeam');
    }
    /**
     * Grants permission to create an identity source
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mpa/latest/APIReference/API_CreateIdentitySource.html
     */
    toCreateIdentitySource() {
        return this.to('CreateIdentitySource');
    }
    /**
     * Grants permission to delete an identity source
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/mpa/latest/APIReference/API_DeleteIdentitySource.html
     */
    toDeleteIdentitySource() {
        return this.to('DeleteIdentitySource');
    }
    /**
     * Grants permission to delete an inactive approval team
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/mpa/latest/APIReference/API_DeleteInactiveApprovalTeamVersion.html
     */
    toDeleteInactiveApprovalTeamVersion() {
        return this.to('DeleteInactiveApprovalTeamVersion');
    }
    /**
     * Grants permission to delete a resource policy
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/mpa/latest/APIReference/API_DeleteResourcePolicy.html
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to retrieve details for an approval team
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/mpa/latest/APIReference/API_GetApprovalTeam.html
     */
    toGetApprovalTeam() {
        return this.to('GetApprovalTeam');
    }
    /**
     * Grants permission to retrieve details for an identity source
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/mpa/latest/APIReference/API_GetIdentitySource.html
     */
    toGetIdentitySource() {
        return this.to('GetIdentitySource');
    }
    /**
     * Grants permission to retrieve details for a policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mpa/latest/APIReference/API_GetPolicyVersion.html
     */
    toGetPolicyVersion() {
        return this.to('GetPolicyVersion');
    }
    /**
     * Grants permission to retrieve details for a specific resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mpa/latest/APIReference/API_GetResourcePolicy.html
     */
    toGetResourcePolicy() {
        return this.to('GetResourcePolicy');
    }
    /**
     * Grants permission to retrieve details for an approval session
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifRequestedOperation()
     * - .ifProtectedResourceAccount()
     *
     * https://docs.aws.amazon.com/mpa/latest/APIReference/API_GetSession.html
     */
    toGetSession() {
        return this.to('GetSession');
    }
    /**
     * Grants permission to list approval teams
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mpa/latest/APIReference/API_ListApprovalTeams.html
     */
    toListApprovalTeams() {
        return this.to('ListApprovalTeams');
    }
    /**
     * Grants permission to list identity sources
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mpa/latest/APIReference/API_ListIdentitySources.html
     */
    toListIdentitySources() {
        return this.to('ListIdentitySources');
    }
    /**
     * Grants permission to list policies
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mpa/latest/APIReference/API_ListPolicies.html
     */
    toListPolicies() {
        return this.to('ListPolicies');
    }
    /**
     * Grants permission to list the versions for policies
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mpa/latest/APIReference/API_ListPolicyVersions.html
     */
    toListPolicyVersions() {
        return this.to('ListPolicyVersions');
    }
    /**
     * Grants permission to list policies for a resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mpa/latest/APIReference/API_ListResourcePolicies.html
     */
    toListResourcePolicies() {
        return this.to('ListResourcePolicies');
    }
    /**
     * Grants permission to list approval sessions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mpa/latest/APIReference/API_ListSessions.html
     */
    toListSessions() {
        return this.to('ListSessions');
    }
    /**
     * Grants permission to list tags for a resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mpa/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to create or update policies for a resource
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/mpa/latest/APIReference/API_PutResourcePolicy.html
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to start the deletion process for an active approval team
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/mpa/latest/APIReference/API_StartActiveApprovalTeamDeletion.html
     */
    toStartActiveApprovalTeamDeletion() {
        return this.to('StartActiveApprovalTeamDeletion');
    }
    /**
     * Grants permission to start an approval session
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifRequestedOperation()
     * - .ifProtectedResourceAccount()
     *
     * https://docs.aws.amazon.com/mpa/latest/APIReference/API_StartSessionInternal.html
     */
    toStartSession() {
        return this.to('StartSession');
    }
    /**
     * Grants permission to tag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mpa/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mpa/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update approval team
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/mpa/latest/APIReference/API_UpdateApprovalTeam.html
     */
    toUpdateApprovalTeam() {
        return this.to('UpdateApprovalTeam');
    }
    /**
     * Adds a resource of type approval-team to the statement
     *
     * https://docs.aws.amazon.com/mpa/latest/userguide/mpa-concepts.html
     *
     * @param approvalTeamId - Identifier for the approvalTeamId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onApprovalTeam(approvalTeamId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mpa:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:approval-team/${approvalTeamId}`);
    }
    /**
     * Adds a resource of type identity-source to the statement
     *
     * https://docs.aws.amazon.com/mpa/latest/userguide/mpa-concepts.html
     *
     * @param identitySourceId - Identifier for the identitySourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIdentitySource(identitySourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mpa:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:identity-source/${identitySourceId}`);
    }
    /**
     * Adds a resource of type session to the statement
     *
     * https://docs.aws.amazon.com/mpa/latest/userguide/mpa-concepts.html
     *
     * @param sessionId - Identifier for the sessionId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSession(sessionId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mpa:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:session/${sessionId}`);
    }
    /**
     * Filters access by a tag key and value pair that is allowed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateApprovalTeam()
     * - .toCreateIdentitySource()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a tag key and value pair of a resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toCancelSession()
     * - .toCreateApprovalTeam()
     * - .toCreateIdentitySource()
     * - .toDeleteIdentitySource()
     * - .toDeleteInactiveApprovalTeamVersion()
     * - .toGetApprovalTeam()
     * - .toGetIdentitySource()
     * - .toGetSession()
     * - .toStartActiveApprovalTeamDeletion()
     * - .toStartSession()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateApprovalTeam()
     *
     * Applies to resource types:
     * - approval-team
     * - identity-source
     * - session
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a list of tag keys that are allowed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateApprovalTeam()
     * - .toCreateIdentitySource()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the account that owns the resource that is the target of the operation that requires approval
     *
     * https://docs.aws.amazon.com/mpa/latest/userguide/mpa-concepts.html
     *
     * Applies to actions:
     * - .toCancelSession()
     * - .toGetSession()
     * - .toStartSession()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifProtectedResourceAccount(value, operator) {
        return this.if(`ProtectedResourceAccount`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a requested operation that requires team approval before it can be executed
     *
     * https://docs.aws.amazon.com/mpa/latest/userguide/mpa-concepts.html
     *
     * Applies to actions:
     * - .toCancelSession()
     * - .toGetSession()
     * - .toStartSession()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestedOperation(value, operator) {
        return this.if(`RequestedOperation`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [mpa](https://docs.aws.amazon.com/service-authorization/latest/reference/list_multi-partyapproval.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'mpa';
        this.accessLevelList = {
            Write: [
                'CancelSession',
                'CreateApprovalTeam',
                'CreateIdentitySource',
                'DeleteIdentitySource',
                'DeleteInactiveApprovalTeamVersion',
                'StartActiveApprovalTeamDeletion',
                'StartSession',
                'UpdateApprovalTeam'
            ],
            'Permissions management': [
                'DeleteResourcePolicy',
                'PutResourcePolicy'
            ],
            Read: [
                'GetApprovalTeam',
                'GetIdentitySource',
                'GetPolicyVersion',
                'GetResourcePolicy',
                'GetSession'
            ],
            List: [
                'ListApprovalTeams',
                'ListIdentitySources',
                'ListPolicies',
                'ListPolicyVersions',
                'ListResourcePolicies',
                'ListSessions',
                'ListTagsForResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Mpa = Mpa;
//# sourceMappingURL=data:application/json;base64,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