import { AccessLevelList } from '../../shared/access-level';
import { PolicyStatement, Operator } from '../../shared';
import { aws_iam as iam } from "aws-cdk-lib";
/**
 * Statement provider for service [neptune-graph](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonneptuneanalytics.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class NeptuneGraph extends PolicyStatement {
    servicePrefix: string;
    /**
     * Grants permission to cancel an ongoing export task
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_CancelExportTask.html
     */
    toCancelExportTask(): this;
    /**
     * Grants permission to cancel an ongoing import task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_CancelImportTask.html
     */
    toCancelImportTask(): this;
    /**
     * Grants permission to cancel a query
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_CancelQuery.html
     */
    toCancelQuery(): this;
    /**
     * Grants permission to create a new graph
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifPublicConnectivity()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - kms:CreateGrant
     * - kms:Decrypt
     * - kms:DescribeKey
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_CreateGraph.html
     */
    toCreateGraph(): this;
    /**
     * Grants permission to create a new snapshot from an existing graph
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_CreateGraphSnapshot.html
     */
    toCreateGraphSnapshot(): this;
    /**
     * Grants permission to create a new graph while importing data into the new graph
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifPublicConnectivity()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - iam:PassRole
     * - kms:CreateGrant
     * - kms:Decrypt
     * - kms:DescribeKey
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_CreateGraphUsingImportTask.html
     */
    toCreateGraphUsingImportTask(): this;
    /**
     * Grants permission to create a new private graph endpoint to access the graph from within a vpc
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:CreateVpcEndpoint
     * - ec2:DescribeAvailabilityZones
     * - ec2:DescribeSecurityGroups
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcAttribute
     * - ec2:DescribeVpcEndpoints
     * - ec2:DescribeVpcs
     * - ec2:ModifyVpcEndpoint
     * - route53:AssociateVPCWithHostedZone
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_CreatePrivateGraphEndpoint.html
     */
    toCreatePrivateGraphEndpoint(): this;
    /**
     * Grants permission to delete data via query APIs on the graph
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_ExecuteQuery.html
     */
    toDeleteDataViaQuery(): this;
    /**
     * Grants permission to delete a graph
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_DeleteGraph.html
     */
    toDeleteGraph(): this;
    /**
     * Grants permission to delete a snapshot
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_DeleteGraphSnapshot.html
     */
    toDeleteGraphSnapshot(): this;
    /**
     * Grants permission to delete a private graph endpoint of a graph
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - ec2:DeleteVpcEndpoints
     * - ec2:DescribeAvailabilityZones
     * - ec2:DescribeSecurityGroups
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcAttribute
     * - ec2:DescribeVpcEndpoints
     * - ec2:DescribeVpcs
     * - ec2:ModifyVpcEndpoint
     * - route53:DisassociateVPCFromHostedZone
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_DeletePrivateGraphEndpoint.html
     */
    toDeletePrivateGraphEndpoint(): this;
    /**
     * Grants permission to get the engine status of the graph
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/#GetEngineStatus
     */
    toGetEngineStatus(): this;
    /**
     * Grants permission to get details about an export task
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_GetExportTask.html
     */
    toGetExportTask(): this;
    /**
     * Grants permission to get details about a graph
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_GetGraph.html
     */
    toGetGraph(): this;
    /**
     * Grants permission to get details about a snapshot
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_GetGraphSnapshot.html
     */
    toGetGraphSnapshot(): this;
    /**
     * Grants permission to get the summary for the data in the graph
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_GetGraphSummary.html
     */
    toGetGraphSummary(): this;
    /**
     * Grants permission to get details about an import task
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_GetImportTask.html
     */
    toGetImportTask(): this;
    /**
     * Grants permission to get details about a private graph endpoint of a graph
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_GetPrivateGraphEndpoint.html
     */
    toGetPrivateGraphEndpoint(): this;
    /**
     * Grants permission to check the status of a given query
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_GetQuery.html
     */
    toGetQueryStatus(): this;
    /**
     * Grants permission to get the statistics for the data in the graph
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/#GetStatisticsStatus
     */
    toGetStatisticsStatus(): this;
    /**
     * Grants permission to list the export tasks in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_ListExportTasks.html
     */
    toListExportTasks(): this;
    /**
     * Grants permission to list the snapshots in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_ListGraphSnapshots.html
     */
    toListGraphSnapshots(): this;
    /**
     * Grants permission to list the graphs in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_ListGraphs.html
     */
    toListGraphs(): this;
    /**
     * Grants permission to list the import tasks in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_ListImportTasks.html
     */
    toListImportTasks(): this;
    /**
     * Grants permission to list the private graph endpoints for a given graph
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_ListPrivateGraphEndpoints.html
     */
    toListPrivateGraphEndpoints(): this;
    /**
     * Grants permission to check the status of all active queries
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_ListQueries.html
     */
    toListQueries(): this;
    /**
     * Grants permission to lists tag for a Neptune Analytics resource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_ListTagsForResource.html
     */
    toListTagsForResource(): this;
    /**
     * Grants permission to read data via query APIs on the graph
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_ExecuteQuery.html
     */
    toReadDataViaQuery(): this;
    /**
     * Grants permission to reset a graph which deletes all data within the graph
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_ResetGraph.html
     */
    toResetGraph(): this;
    /**
     * Grants permission to create a new graph from an existing snapshot
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifPublicConnectivity()
     *
     * Dependent actions:
     * - kms:CreateGrant
     * - kms:Decrypt
     * - kms:DescribeKey
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_RestoreGraphFromSnapshot.html
     */
    toRestoreGraphFromSnapshot(): this;
    /**
     * Grants permission to export data from an existing graph
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_StartExportTask.html
     */
    toStartExportTask(): this;
    /**
     * Grants permission to start a graph
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - kms:Decrypt
     * - kms:DescribeKey
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_StartGraph.html
     */
    toStartGraph(): this;
    /**
     * Grants permission to import data into an existing graph
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_StartImportTask.html
     */
    toStartImportTask(): this;
    /**
     * Grants permission to stop a graph
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_StopGraph.html
     */
    toStopGraph(): this;
    /**
     * Grants permission to tag a Neptune Analytics resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_TagResource.html
     */
    toTagResource(): this;
    /**
     * Grants permission to untag a Neptune Analytics resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_UntagResource.html
     */
    toUntagResource(): this;
    /**
     * Grants permission to modify a graph
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifPublicConnectivity()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_UpdateGraph.html
     */
    toUpdateGraph(): this;
    /**
     * Grants permission to write data via query APIs on the graph
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/apiref/API_ExecuteQuery.html
     */
    toWriteDataViaQuery(): this;
    protected accessLevelList: AccessLevelList;
    /**
     * Adds a resource of type graph to the statement
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/userguide/iam-resources.html#graph
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onGraph(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type graph-snapshot to the statement
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/userguide/iam-resources.html#graph-snapshot
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onGraphSnapshot(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type import-task to the statement
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/userguide/iam-resources.html#import-task
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onImportTask(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type export-task to the statement
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/userguide/iam-resources.html#export-task
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onExportTask(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Filters access by a tag's key and value in a request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateGraph()
     * - .toCreateGraphSnapshot()
     * - .toCreateGraphUsingImportTask()
     * - .toCreatePrivateGraphEndpoint()
     * - .toRestoreGraphFromSnapshot()
     * - .toStartExportTask()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toCancelExportTask()
     * - .toCancelQuery()
     * - .toCreateGraphSnapshot()
     * - .toCreatePrivateGraphEndpoint()
     * - .toDeleteDataViaQuery()
     * - .toDeleteGraph()
     * - .toDeleteGraphSnapshot()
     * - .toDeletePrivateGraphEndpoint()
     * - .toGetEngineStatus()
     * - .toGetExportTask()
     * - .toGetGraph()
     * - .toGetGraphSnapshot()
     * - .toGetGraphSummary()
     * - .toGetPrivateGraphEndpoint()
     * - .toGetQueryStatus()
     * - .toGetStatisticsStatus()
     * - .toListPrivateGraphEndpoints()
     * - .toListQueries()
     * - .toListTagsForResource()
     * - .toReadDataViaQuery()
     * - .toResetGraph()
     * - .toRestoreGraphFromSnapshot()
     * - .toStartExportTask()
     * - .toStartGraph()
     * - .toStopGraph()
     * - .toUpdateGraph()
     * - .toWriteDataViaQuery()
     *
     * Applies to resource types:
     * - graph
     * - graph-snapshot
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the tag keys in a request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateGraph()
     * - .toCreateGraphSnapshot()
     * - .toCreateGraphUsingImportTask()
     * - .toCreatePrivateGraphEndpoint()
     * - .toRestoreGraphFromSnapshot()
     * - .toStartExportTask()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the value of the public connectivity parameter provided in the request or its default value, if unspecified. All access to graphs is IAM authenticated
     *
     * https://docs.aws.amazon.com/neptune-analytics/latest/userguide/iam-condition-keys.html#publicconnectivity
     *
     * Applies to actions:
     * - .toCreateGraph()
     * - .toCreateGraphUsingImportTask()
     * - .toRestoreGraphFromSnapshot()
     * - .toUpdateGraph()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifPublicConnectivity(value?: boolean): this;
    /**
     * Statement provider for service [neptune-graph](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonneptuneanalytics.html).
     *
     */
    constructor(props?: iam.PolicyStatementProps);
}
