"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NetworkFirewall = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [network-firewall](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsnetworkfirewall.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class NetworkFirewall extends shared_1.PolicyStatement {
    /**
     * Grants permission to accept pending Network Firewall attachments on a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_AcceptNetworkFirewallTransitGatewayAttachment.html
     */
    toAcceptNetworkFirewallTransitGatewayAttachment() {
        return this.to('AcceptNetworkFirewallTransitGatewayAttachment');
    }
    /**
     * Grants permission to associate availability zones to a firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_AssociateAvailabilityZones.html
     */
    toAssociateAvailabilityZones() {
        return this.to('AssociateAvailabilityZones');
    }
    /**
     * Grants permission to create an association between a firewall policy and a firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_AssociateFirewallPolicy.html
     */
    toAssociateFirewallPolicy() {
        return this.to('AssociateFirewallPolicy');
    }
    /**
     * Grants permission to associate VPC subnets to a firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_AssociateSubnets.html
     */
    toAssociateSubnets() {
        return this.to('AssociateSubnets');
    }
    /**
     * Grants permission to create an AWS Network Firewall firewall
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_CreateFirewall.html
     */
    toCreateFirewall() {
        return this.to('CreateFirewall');
    }
    /**
     * Grants permission to create an AWS Network Firewall firewall policy
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_CreateFirewallPolicy.html
     */
    toCreateFirewallPolicy() {
        return this.to('CreateFirewallPolicy');
    }
    /**
     * Grants permission to create an AWS Network Firewall rule group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_CreateRuleGroup.html
     */
    toCreateRuleGroup() {
        return this.to('CreateRuleGroup');
    }
    /**
     * Grants permission to create an AWS Network Firewall tls inspection configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_CreateTLSInspectionConfiguration.html
     */
    toCreateTLSInspectionConfiguration() {
        return this.to('CreateTLSInspectionConfiguration');
    }
    /**
     * Grants permission to create an AWS Network Firewall vpc endpoint association
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_CreateVpcEndpointAssociation.html
     */
    toCreateVpcEndpointAssociation() {
        return this.to('CreateVpcEndpointAssociation');
    }
    /**
     * Grants permission to delete a firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DeleteFirewall.html
     */
    toDeleteFirewall() {
        return this.to('DeleteFirewall');
    }
    /**
     * Grants permission to delete a firewall policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DeleteFirewallPolicy.html
     */
    toDeleteFirewallPolicy() {
        return this.to('DeleteFirewallPolicy');
    }
    /**
     * Grants permission to delete Network Firewall attachments on a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DeleteNetworkFirewallTransitGatewayAttachment.html
     */
    toDeleteNetworkFirewallTransitGatewayAttachment() {
        return this.to('DeleteNetworkFirewallTransitGatewayAttachment');
    }
    /**
     * Grants permission to delete a resource policy for a firewall policy or rule group or firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DeleteResourcePolicy.html
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to delete a rule group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DeleteRuleGroup.html
     */
    toDeleteRuleGroup() {
        return this.to('DeleteRuleGroup');
    }
    /**
     * Grants permission to delete a tls inspection configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DeleteTLSInspectionConfiguration.html
     */
    toDeleteTLSInspectionConfiguration() {
        return this.to('DeleteTLSInspectionConfiguration');
    }
    /**
     * Grants permission to delete a vpc endpoint association
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DeleteVpcEndpointAssociation.html
     */
    toDeleteVpcEndpointAssociation() {
        return this.to('DeleteVpcEndpointAssociation');
    }
    /**
     * Grants permission to retrieve the data objects that define a firewall
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DescribeFirewall.html
     */
    toDescribeFirewall() {
        return this.to('DescribeFirewall');
    }
    /**
     * Grants permission to retrieve the high-level information about a firewall
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DescribeFirewallMetadata.html
     */
    toDescribeFirewallMetadata() {
        return this.to('DescribeFirewallMetadata');
    }
    /**
     * Grants permission to retrieve the data objects that define a firewall policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DescribeFirewallPolicy.html
     */
    toDescribeFirewallPolicy() {
        return this.to('DescribeFirewallPolicy');
    }
    /**
     * Grants permission to describe a flow operation performed on a firewall
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DescribeFlowOperation.html
     */
    toDescribeFlowOperation() {
        return this.to('DescribeFlowOperation');
    }
    /**
     * Grants permission to describe the logging configuration of a firewall
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - logs:GetLogDelivery
     * - logs:ListLogDeliveries
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DescribeLoggingConfiguration.html
     */
    toDescribeLoggingConfiguration() {
        return this.to('DescribeLoggingConfiguration');
    }
    /**
     * Grants permission to describe a resource policy for a firewall policy or rule group or firewall
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DescribeResourcePolicy.html
     */
    toDescribeResourcePolicy() {
        return this.to('DescribeResourcePolicy');
    }
    /**
     * Grants permission to retrieve the data objects that define a rule group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DescribeRuleGroup.html
     */
    toDescribeRuleGroup() {
        return this.to('DescribeRuleGroup');
    }
    /**
     * Grants permission to retrieve the high-level information about a rule group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DescribeRuleGroupMetadata.html
     */
    toDescribeRuleGroupMetadata() {
        return this.to('DescribeRuleGroupMetadata');
    }
    /**
     * Grants permission to retrieve the summary information about a rule group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DescribeRuleGroupSummary.html
     */
    toDescribeRuleGroupSummary() {
        return this.to('DescribeRuleGroupSummary');
    }
    /**
     * Grants permission to retrieve the data objects that define a tls inspection configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DescribeTLSInspectionConfiguration.html
     */
    toDescribeTLSInspectionConfiguration() {
        return this.to('DescribeTLSInspectionConfiguration');
    }
    /**
     * Grants permission to retrieve the data objects that define a vpc endpoint association
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DescribeVpcEndpointAssociation.html
     */
    toDescribeVpcEndpointAssociation() {
        return this.to('DescribeVpcEndpointAssociation');
    }
    /**
     * Grants permission to disassociate availability zones to a firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DisassociateAvailabilityZones.html
     */
    toDisassociateAvailabilityZones() {
        return this.to('DisassociateAvailabilityZones');
    }
    /**
     * Grants permission to disassociate VPC subnets from a firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_DisassociateSubnets.html
     */
    toDisassociateSubnets() {
        return this.to('DisassociateSubnets');
    }
    /**
     * Grants permission to retrieve analysis report results of a firewall
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_GetAnalysisReportResults.html
     */
    toGetAnalysisReportResults() {
        return this.to('GetAnalysisReportResults');
    }
    /**
     * Grants permission to list firewall analysis reports
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_ListAnalysisReports.html
     */
    toListAnalysisReports() {
        return this.to('ListAnalysisReports');
    }
    /**
     * Grants permission to retrieve the metadata for firewall policies
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_ListFirewallPolicies.html
     */
    toListFirewallPolicies() {
        return this.to('ListFirewallPolicies');
    }
    /**
     * Grants permission to retrieve the metadata for firewalls
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_ListFirewalls.html
     */
    toListFirewalls() {
        return this.to('ListFirewalls');
    }
    /**
     * Grants permission to list results from a flow operation performed on a firewall
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_ListFlowOperationResults.html
     */
    toListFlowOperationResults() {
        return this.to('ListFlowOperationResults');
    }
    /**
     * Grants permission to list flow operations performed on a firewall
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_ListFlowOperations.html
     */
    toListFlowOperations() {
        return this.to('ListFlowOperations');
    }
    /**
     * Grants permission to retrieve the metadata for rule groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_ListRuleGroups.html
     */
    toListRuleGroups() {
        return this.to('ListRuleGroups');
    }
    /**
     * Grants permission to retrieve the metadata for tls inspection configurations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_ListTLSInspectionConfigurations.html
     */
    toListTLSInspectionConfigurations() {
        return this.to('ListTLSInspectionConfigurations');
    }
    /**
     * Grants permission to retrieve the tags for a resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to retrieve the metadata for vpc endpoint associations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_ListVpcEndpointAssociations.html
     */
    toListVpcEndpointAssociations() {
        return this.to('ListVpcEndpointAssociations');
    }
    /**
     * Grants permission to put a resource policy for a firewall policy or rule group or firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_PutResourcePolicy.html
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to reject pending Network Firewall attachments on a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_RejectNetworkFirewallTransitGatewayAttachment.html
     */
    toRejectNetworkFirewallTransitGatewayAttachment() {
        return this.to('RejectNetworkFirewallTransitGatewayAttachment');
    }
    /**
     * Grants permission to start an analysis report on a firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_GetAnalysisReportResults.html
     */
    toStartAnalysisReport() {
        return this.to('StartAnalysisReport');
    }
    /**
     * Grants permission to start capture operation on a firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_StartFlowCapture.html
     */
    toStartFlowCapture() {
        return this.to('StartFlowCapture');
    }
    /**
     * Grants permission to start flush operation on a firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_StartFlowFlush.html
     */
    toStartFlowFlush() {
        return this.to('StartFlowFlush');
    }
    /**
     * Grants permission to attach tags to a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to add or remove availability zone change protection for a firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_UpdateAvailabilityZoneChangeProtection.html
     */
    toUpdateAvailabilityZoneChangeProtection() {
        return this.to('UpdateAvailabilityZoneChangeProtection');
    }
    /**
     * Grants permission to modify firewall analysis settings of a firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_UpdateFirewallAnalysisSettings.html
     */
    toUpdateFirewallAnalysisSettings() {
        return this.to('UpdateFirewallAnalysisSettings');
    }
    /**
     * Grants permission to add or remove delete protection for a firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_UpdateFirewallDeleteProtection.html
     */
    toUpdateFirewallDeleteProtection() {
        return this.to('UpdateFirewallDeleteProtection');
    }
    /**
     * Grants permission to modify the description for a firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_UpdateFirewallDescription.html
     */
    toUpdateFirewallDescription() {
        return this.to('UpdateFirewallDescription');
    }
    /**
     * Grants permission to modify the encryption configuration of a firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_UpdateFirewallEncryptionConfiguration.html
     */
    toUpdateFirewallEncryptionConfiguration() {
        return this.to('UpdateFirewallEncryptionConfiguration');
    }
    /**
     * Grants permission to modify a firewall policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_UpdateFirewallPolicy.html
     */
    toUpdateFirewallPolicy() {
        return this.to('UpdateFirewallPolicy');
    }
    /**
     * Grants permission to add or remove firewall policy change protection for a firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_UpdateFirewallPolicyChangeProtection.html
     */
    toUpdateFirewallPolicyChangeProtection() {
        return this.to('UpdateFirewallPolicyChangeProtection');
    }
    /**
     * Grants permission to modify the logging configuration of a firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_UpdateLoggingConfiguration.html
     */
    toUpdateLoggingConfiguration() {
        return this.to('UpdateLoggingConfiguration');
    }
    /**
     * Grants permission to modify a rule group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_UpdateRuleGroup.html
     */
    toUpdateRuleGroup() {
        return this.to('UpdateRuleGroup');
    }
    /**
     * Grants permission to add or remove subnet change protection for a firewall
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_UpdateSubnetChangeProtection.html
     */
    toUpdateSubnetChangeProtection() {
        return this.to('UpdateSubnetChangeProtection');
    }
    /**
     * Grants permission to modify a tls inspection configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_UpdateTLSInspectionConfiguration.html
     */
    toUpdateTLSInspectionConfiguration() {
        return this.to('UpdateTLSInspectionConfiguration');
    }
    /**
     * Adds a resource of type Firewall to the statement
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_Firewall.html
     *
     * @param name - Identifier for the name.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFirewall(name, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:network-firewall:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:firewall/${name}`);
    }
    /**
     * Adds a resource of type FirewallPolicy to the statement
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_FirewallPolicyResponse.html
     *
     * @param name - Identifier for the name.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFirewallPolicy(name, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:network-firewall:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:firewall-policy/${name}`);
    }
    /**
     * Adds a resource of type StatefulRuleGroup to the statement
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_RuleGroupResponse.html
     *
     * @param name - Identifier for the name.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStatefulRuleGroup(name, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:network-firewall:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:stateful-rulegroup/${name}`);
    }
    /**
     * Adds a resource of type StatelessRuleGroup to the statement
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_RuleGroupResponse.html
     *
     * @param name - Identifier for the name.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStatelessRuleGroup(name, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:network-firewall:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:stateless-rulegroup/${name}`);
    }
    /**
     * Adds a resource of type TLSInspectionConfiguration to the statement
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_TLSInspectionConfigurationResponse.html
     *
     * @param name - Identifier for the name.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTLSInspectionConfiguration(name, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:network-firewall:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:tls-configuration/${name}`);
    }
    /**
     * Adds a resource of type VpcEndpointAssociation to the statement
     *
     * https://docs.aws.amazon.com/network-firewall/latest/APIReference/API_VpcEndpointAssociation.html
     *
     * @param name - Identifier for the name.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onVpcEndpointAssociation(name, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:network-firewall:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:vpc-endpoint-association/${name}`);
    }
    /**
     * Filters access by on the allowed set of values for each of the tags
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateFirewall()
     * - .toCreateFirewallPolicy()
     * - .toCreateRuleGroup()
     * - .toCreateTLSInspectionConfiguration()
     * - .toCreateVpcEndpointAssociation()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag value associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - Firewall
     * - FirewallPolicy
     * - StatefulRuleGroup
     * - StatelessRuleGroup
     * - TLSInspectionConfiguration
     * - VpcEndpointAssociation
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of mandatory tags in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateFirewall()
     * - .toCreateFirewallPolicy()
     * - .toCreateRuleGroup()
     * - .toCreateTLSInspectionConfiguration()
     * - .toCreateVpcEndpointAssociation()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [network-firewall](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsnetworkfirewall.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'network-firewall';
        this.accessLevelList = {
            Write: [
                'AcceptNetworkFirewallTransitGatewayAttachment',
                'AssociateAvailabilityZones',
                'AssociateFirewallPolicy',
                'AssociateSubnets',
                'CreateFirewall',
                'CreateFirewallPolicy',
                'CreateRuleGroup',
                'CreateTLSInspectionConfiguration',
                'CreateVpcEndpointAssociation',
                'DeleteFirewall',
                'DeleteFirewallPolicy',
                'DeleteNetworkFirewallTransitGatewayAttachment',
                'DeleteResourcePolicy',
                'DeleteRuleGroup',
                'DeleteTLSInspectionConfiguration',
                'DeleteVpcEndpointAssociation',
                'DisassociateAvailabilityZones',
                'DisassociateSubnets',
                'PutResourcePolicy',
                'RejectNetworkFirewallTransitGatewayAttachment',
                'StartAnalysisReport',
                'StartFlowCapture',
                'StartFlowFlush',
                'UpdateAvailabilityZoneChangeProtection',
                'UpdateFirewallAnalysisSettings',
                'UpdateFirewallDeleteProtection',
                'UpdateFirewallDescription',
                'UpdateFirewallEncryptionConfiguration',
                'UpdateFirewallPolicy',
                'UpdateFirewallPolicyChangeProtection',
                'UpdateLoggingConfiguration',
                'UpdateRuleGroup',
                'UpdateSubnetChangeProtection',
                'UpdateTLSInspectionConfiguration'
            ],
            Read: [
                'DescribeFirewall',
                'DescribeFirewallMetadata',
                'DescribeFirewallPolicy',
                'DescribeFlowOperation',
                'DescribeLoggingConfiguration',
                'DescribeResourcePolicy',
                'DescribeRuleGroup',
                'DescribeRuleGroupMetadata',
                'DescribeRuleGroupSummary',
                'DescribeTLSInspectionConfiguration',
                'DescribeVpcEndpointAssociation',
                'GetAnalysisReportResults',
                'ListFlowOperationResults'
            ],
            List: [
                'ListAnalysisReports',
                'ListFirewallPolicies',
                'ListFirewalls',
                'ListFlowOperations',
                'ListRuleGroups',
                'ListTLSInspectionConfigurations',
                'ListTagsForResource',
                'ListVpcEndpointAssociations'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.NetworkFirewall = NetworkFirewall;
//# sourceMappingURL=data:application/json;base64,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