"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Organizations = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [organizations](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsorganizations.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Organizations extends shared_1.PolicyStatement {
    /**
     * Grants permission to send a response to the originator of a handshake agreeing to the action proposed by the handshake request
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_AcceptHandshake.html
     */
    toAcceptHandshake() {
        return this.to('AcceptHandshake');
    }
    /**
     * Grants permission to attach a policy to a root, an organizational unit, or an individual account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_AttachPolicy.html
     */
    toAttachPolicy() {
        return this.to('AttachPolicy');
    }
    /**
     * Grants permission to cancel a handshake
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_CancelHandshake.html
     */
    toCancelHandshake() {
        return this.to('CancelHandshake');
    }
    /**
     * Grants permission to close an AWS account that is now a part of an Organizations, either created within the organization, or invited to join the organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_CloseAccount.html
     */
    toCloseAccount() {
        return this.to('CloseAccount');
    }
    /**
     * Grants permission to create an AWS account that is automatically a member of the organization with the credentials that made the request
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_CreateAccount.html
     */
    toCreateAccount() {
        return this.to('CreateAccount');
    }
    /**
     * Grants permission to create an AWS GovCloud (US) account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_CreateGovCloudAccount.html
     */
    toCreateGovCloudAccount() {
        return this.to('CreateGovCloudAccount');
    }
    /**
     * Grants permission to create an organization. The account with the credentials that calls the CreateOrganization operation automatically becomes the management account of the new organization
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_CreateOrganization.html
     */
    toCreateOrganization() {
        return this.to('CreateOrganization');
    }
    /**
     * Grants permission to create an organizational unit (OU) within a root or parent OU
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_CreateOrganizationalUnit.html
     */
    toCreateOrganizationalUnit() {
        return this.to('CreateOrganizationalUnit');
    }
    /**
     * Grants permission to create a policy that you can attach to a root, an organizational unit (OU), or an individual AWS account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPolicyType()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_CreatePolicy.html
     */
    toCreatePolicy() {
        return this.to('CreatePolicy');
    }
    /**
     * Grants permission to decline a handshake request. This sets the handshake state to DECLINED and effectively deactivates the request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DeclineHandshake.html
     */
    toDeclineHandshake() {
        return this.to('DeclineHandshake');
    }
    /**
     * Grants permission to delete the organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DeleteOrganization.html
     */
    toDeleteOrganization() {
        return this.to('DeleteOrganization');
    }
    /**
     * Grants permission to delete an organizational unit from a root or another OU
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DeleteOrganizationalUnit.html
     */
    toDeleteOrganizationalUnit() {
        return this.to('DeleteOrganizationalUnit');
    }
    /**
     * Grants permission to delete a policy from your organization
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DeletePolicy.html
     */
    toDeletePolicy() {
        return this.to('DeletePolicy');
    }
    /**
     * Grants permission to delete a resource policy from your organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DeleteResourcePolicy.html
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to deregister the specified member AWS account as a delegated administrator for the AWS service that is specified by ServicePrincipal
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifServicePrincipal()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DeregisterDelegatedAdministrator.html
     */
    toDeregisterDelegatedAdministrator() {
        return this.to('DeregisterDelegatedAdministrator');
    }
    /**
     * Grants permission to retrieve Organizations-related details about the specified account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeAccount.html
     */
    toDescribeAccount() {
        return this.to('DescribeAccount');
    }
    /**
     * Grants permission to retrieve the current status of an asynchronous request to create an account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeCreateAccountStatus.html
     */
    toDescribeCreateAccountStatus() {
        return this.to('DescribeCreateAccountStatus');
    }
    /**
     * Grants permission to retrieve the effective policy for an account
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeEffectivePolicy.html
     */
    toDescribeEffectivePolicy() {
        return this.to('DescribeEffectivePolicy');
    }
    /**
     * Grants permission to retrieve details about a previously requested handshake
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeHandshake.html
     */
    toDescribeHandshake() {
        return this.to('DescribeHandshake');
    }
    /**
     * Grants permission to retrieve details about the organization that the calling credentials belong to
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeOrganization.html
     */
    toDescribeOrganization() {
        return this.to('DescribeOrganization');
    }
    /**
     * Grants permission to retrieve details about an organizational unit (OU)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeOrganizationalUnit.html
     */
    toDescribeOrganizationalUnit() {
        return this.to('DescribeOrganizationalUnit');
    }
    /**
     * Grants permission to retrieve details about a policy
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribePolicy.html
     */
    toDescribePolicy() {
        return this.to('DescribePolicy');
    }
    /**
     * Grants permission to retrieve information about a resource policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DescribeResourcePolicy.html
     */
    toDescribeResourcePolicy() {
        return this.to('DescribeResourcePolicy');
    }
    /**
     * Grants permission to detach a policy from a target root, organizational unit, or account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DetachPolicy.html
     */
    toDetachPolicy() {
        return this.to('DetachPolicy');
    }
    /**
     * Grants permission to disable integration of an AWS service (the service that is specified by ServicePrincipal) with AWS Organizations
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifServicePrincipal()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DisableAWSServiceAccess.html
     */
    toDisableAWSServiceAccess() {
        return this.to('DisableAWSServiceAccess');
    }
    /**
     * Grants permission to disable an organization policy type in a root
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_DisablePolicyType.html
     */
    toDisablePolicyType() {
        return this.to('DisablePolicyType');
    }
    /**
     * Grants permission to enable integration of an AWS service (the service that is specified by ServicePrincipal) with AWS Organizations
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifServicePrincipal()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_EnableAWSServiceAccess.html
     */
    toEnableAWSServiceAccess() {
        return this.to('EnableAWSServiceAccess');
    }
    /**
     * Grants permission to start the process to enable all features in an organization, upgrading it from supporting only Consolidated Billing features
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_EnableAllFeatures.html
     */
    toEnableAllFeatures() {
        return this.to('EnableAllFeatures');
    }
    /**
     * Grants permission to enable a policy type in a root
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_EnablePolicyType.html
     */
    toEnablePolicyType() {
        return this.to('EnablePolicyType');
    }
    /**
     * Grants permission to send an invitation to another AWS account, asking it to join your organization as a member account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_InviteAccountToOrganization.html
     */
    toInviteAccountToOrganization() {
        return this.to('InviteAccountToOrganization');
    }
    /**
     * Grants permission to remove a member account from its parent organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_LeaveOrganization.html
     */
    toLeaveOrganization() {
        return this.to('LeaveOrganization');
    }
    /**
     * Grants permission to retrieve the list of the AWS services for which you enabled integration with your organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListAWSServiceAccessForOrganization.html
     */
    toListAWSServiceAccessForOrganization() {
        return this.to('ListAWSServiceAccessForOrganization');
    }
    /**
     * Grants permission to list all of the accounts in the organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListAccounts.html
     */
    toListAccounts() {
        return this.to('ListAccounts');
    }
    /**
     * Grants permission to list the accounts in an organization that are contained by a root or organizational unit (OU)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListAccountsForParent.html
     */
    toListAccountsForParent() {
        return this.to('ListAccountsForParent');
    }
    /**
     * Grants permission to list accounts that have invalid effective policies for a specified policy type
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListAccountsWithInvalidEffectivePolicy.html
     */
    toListAccountsWithInvalidEffectivePolicy() {
        return this.to('ListAccountsWithInvalidEffectivePolicy');
    }
    /**
     * Grants permission to list all of the OUs or accounts that are contained in a parent OU or root
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListChildren.html
     */
    toListChildren() {
        return this.to('ListChildren');
    }
    /**
     * Grants permission to list the asynchronous account creation requests that are currently being tracked for the organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListCreateAccountStatus.html
     */
    toListCreateAccountStatus() {
        return this.to('ListCreateAccountStatus');
    }
    /**
     * Grants permission to list the AWS accounts that are designated as delegated administrators in this organization
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifServicePrincipal()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListDelegatedAdministrators.html
     */
    toListDelegatedAdministrators() {
        return this.to('ListDelegatedAdministrators');
    }
    /**
     * Grants permission to list the AWS services for which the specified account is a delegated administrator in this organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListDelegatedServicesForAccount.html
     */
    toListDelegatedServicesForAccount() {
        return this.to('ListDelegatedServicesForAccount');
    }
    /**
     * Grants permission to list validation errors found in the effective policy for a specific account and policy type
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListEffectivePolicyValidationErrors.html
     */
    toListEffectivePolicyValidationErrors() {
        return this.to('ListEffectivePolicyValidationErrors');
    }
    /**
     * Grants permission to list all of the handshakes that are associated with an account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListHandshakesForAccount.html
     */
    toListHandshakesForAccount() {
        return this.to('ListHandshakesForAccount');
    }
    /**
     * Grants permission to list the handshakes that are associated with the organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListHandshakesForOrganization.html
     */
    toListHandshakesForOrganization() {
        return this.to('ListHandshakesForOrganization');
    }
    /**
     * Grants permission to list all of the organizational units (OUs) in a parent organizational unit or root
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListOrganizationalUnitsForParent.html
     */
    toListOrganizationalUnitsForParent() {
        return this.to('ListOrganizationalUnitsForParent');
    }
    /**
     * Grants permission to list the root or organizational units (OUs) that serve as the immediate parent of a child OU or account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListParents.html
     */
    toListParents() {
        return this.to('ListParents');
    }
    /**
     * Grants permission to list all of the policies in an organization
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListPolicies.html
     */
    toListPolicies() {
        return this.to('ListPolicies');
    }
    /**
     * Grants permission to list all of the policies that are directly attached to a root, organizational unit (OU), or account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListPoliciesForTarget.html
     */
    toListPoliciesForTarget() {
        return this.to('ListPoliciesForTarget');
    }
    /**
     * Grants permission to list all of the roots that are defined in the organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListRoots.html
     */
    toListRoots() {
        return this.to('ListRoots');
    }
    /**
     * Grants permission to list all tags for the specified resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list all the roots, OUs, and accounts to which a policy is attached
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_ListTargetsForPolicy.html
     */
    toListTargetsForPolicy() {
        return this.to('ListTargetsForPolicy');
    }
    /**
     * Grants permission to move an account from its current root or OU to another parent root or OU
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_MoveAccount.html
     */
    toMoveAccount() {
        return this.to('MoveAccount');
    }
    /**
     * Grants permission to create or update a resource policy
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_PutResourcePolicy.html
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to register the specified member account to administer the Organizations features of the AWS service that is specified by ServicePrincipal
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifServicePrincipal()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_RegisterDelegatedAdministrator.html
     */
    toRegisterDelegatedAdministrator() {
        return this.to('RegisterDelegatedAdministrator');
    }
    /**
     * Grants permission to remove the specified account from the organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_RemoveAccountFromOrganization.html
     */
    toRemoveAccountFromOrganization() {
        return this.to('RemoveAccountFromOrganization');
    }
    /**
     * Grants permission to add one or more tags to the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove one or more tags from the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to rename an organizational unit (OU)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_UpdateOrganizationalUnit.html
     */
    toUpdateOrganizationalUnit() {
        return this.to('UpdateOrganizationalUnit');
    }
    /**
     * Grants permission to update an existing policy with a new name, description, or content
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPolicyType()
     *
     * https://docs.aws.amazon.com/organizations/latest/APIReference/API_UpdatePolicy.html
     */
    toUpdatePolicy() {
        return this.to('UpdatePolicy');
    }
    /**
     * Adds a resource of type account to the statement
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_permissions_overview.html
     *
     * @param organizationId - Identifier for the organizationId.
     * @param accountId - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAccount(organizationId, accountId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:organizations::${account ?? this.defaultAccount}:account/o-${organizationId}/${accountId ?? this.defaultAccount}`);
    }
    /**
     * Adds a resource of type handshake to the statement
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_permissions_overview.html
     *
     * @param organizationId - Identifier for the organizationId.
     * @param handshakeType - Identifier for the handshakeType.
     * @param handshakeId - Identifier for the handshakeId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onHandshake(organizationId, handshakeType, handshakeId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:organizations::${account ?? this.defaultAccount}:handshake/o-${organizationId}/${handshakeType}/h-${handshakeId}`);
    }
    /**
     * Adds a resource of type organization to the statement
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_permissions_overview.html
     *
     * @param organizationId - Identifier for the organizationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onOrganization(organizationId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:organizations::${account ?? this.defaultAccount}:organization/o-${organizationId}`);
    }
    /**
     * Adds a resource of type organizationalunit to the statement
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_permissions_overview.html
     *
     * @param organizationId - Identifier for the organizationId.
     * @param organizationalUnitId - Identifier for the organizationalUnitId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onOrganizationalunit(organizationId, organizationalUnitId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:organizations::${account ?? this.defaultAccount}:ou/o-${organizationId}/ou-${organizationalUnitId}`);
    }
    /**
     * Adds a resource of type policy to the statement
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_permissions_overview.html
     *
     * @param organizationId - Identifier for the organizationId.
     * @param policyType - Identifier for the policyType.
     * @param policyId - Identifier for the policyId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPolicy(organizationId, policyType, policyId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:organizations::${account ?? this.defaultAccount}:policy/o-${organizationId}/${policyType}/p-${policyId}`);
    }
    /**
     * Adds a resource of type resourcepolicy to the statement
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_permissions_overview.html
     *
     * @param organizationId - Identifier for the organizationId.
     * @param resourcePolicyId - Identifier for the resourcePolicyId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onResourcepolicy(organizationId, resourcePolicyId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:organizations::${account ?? this.defaultAccount}:resourcepolicy/o-${organizationId}/rp-${resourcePolicyId}`);
    }
    /**
     * Adds a resource of type awspolicy to the statement
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_permissions_overview.html
     *
     * @param policyType - Identifier for the policyType.
     * @param policyId - Identifier for the policyId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAwspolicy(policyType, policyId, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:organizations::aws:policy/${policyType}/p-${policyId}`);
    }
    /**
     * Adds a resource of type root to the statement
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_permissions_overview.html
     *
     * @param organizationId - Identifier for the organizationId.
     * @param rootId - Identifier for the rootId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRoot(organizationId, rootId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:organizations::${account ?? this.defaultAccount}:root/o-${organizationId}/r-${rootId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateAccount()
     * - .toCreateGovCloudAccount()
     * - .toCreateOrganizationalUnit()
     * - .toCreatePolicy()
     * - .toInviteAccountToOrganization()
     * - .toPutResourcePolicy()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toCreateAccount()
     * - .toCreateGovCloudAccount()
     * - .toCreatePolicy()
     *
     * Applies to resource types:
     * - account
     * - organizationalunit
     * - policy
     * - resourcepolicy
     * - root
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateAccount()
     * - .toCreateGovCloudAccount()
     * - .toCreateOrganizationalUnit()
     * - .toCreatePolicy()
     * - .toInviteAccountToOrganization()
     * - .toPutResourcePolicy()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the specified policy type names
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_permissions_overview.html#orgs_permissions_conditionkeys
     *
     * Applies to actions:
     * - .toAttachPolicy()
     * - .toCreatePolicy()
     * - .toDeletePolicy()
     * - .toDescribeEffectivePolicy()
     * - .toDescribePolicy()
     * - .toDetachPolicy()
     * - .toDisablePolicyType()
     * - .toEnablePolicyType()
     * - .toListAccountsWithInvalidEffectivePolicy()
     * - .toListEffectivePolicyValidationErrors()
     * - .toListPolicies()
     * - .toListPoliciesForTarget()
     * - .toListTargetsForPolicy()
     * - .toTagResource()
     * - .toUpdatePolicy()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPolicyType(value, operator) {
        return this.if(`PolicyType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the specified service principal names
     *
     * https://docs.aws.amazon.com/organizations/latest/userguide/orgs_permissions_overview.html#orgs_permissions_conditionkeys
     *
     * Applies to actions:
     * - .toDeregisterDelegatedAdministrator()
     * - .toDisableAWSServiceAccess()
     * - .toEnableAWSServiceAccess()
     * - .toListDelegatedAdministrators()
     * - .toRegisterDelegatedAdministrator()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifServicePrincipal(value, operator) {
        return this.if(`ServicePrincipal`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [organizations](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsorganizations.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'organizations';
        this.accessLevelList = {
            Write: [
                'AcceptHandshake',
                'AttachPolicy',
                'CancelHandshake',
                'CloseAccount',
                'CreateAccount',
                'CreateGovCloudAccount',
                'CreateOrganization',
                'CreateOrganizationalUnit',
                'CreatePolicy',
                'DeclineHandshake',
                'DeleteOrganization',
                'DeleteOrganizationalUnit',
                'DeletePolicy',
                'DeleteResourcePolicy',
                'DeregisterDelegatedAdministrator',
                'DetachPolicy',
                'DisableAWSServiceAccess',
                'DisablePolicyType',
                'EnableAWSServiceAccess',
                'EnableAllFeatures',
                'EnablePolicyType',
                'InviteAccountToOrganization',
                'LeaveOrganization',
                'MoveAccount',
                'PutResourcePolicy',
                'RegisterDelegatedAdministrator',
                'RemoveAccountFromOrganization',
                'UpdateOrganizationalUnit',
                'UpdatePolicy'
            ],
            Read: [
                'DescribeAccount',
                'DescribeCreateAccountStatus',
                'DescribeEffectivePolicy',
                'DescribeHandshake',
                'DescribeOrganization',
                'DescribeOrganizationalUnit',
                'DescribePolicy',
                'DescribeResourcePolicy'
            ],
            List: [
                'ListAWSServiceAccessForOrganization',
                'ListAccounts',
                'ListAccountsForParent',
                'ListAccountsWithInvalidEffectivePolicy',
                'ListChildren',
                'ListCreateAccountStatus',
                'ListDelegatedAdministrators',
                'ListDelegatedServicesForAccount',
                'ListEffectivePolicyValidationErrors',
                'ListHandshakesForAccount',
                'ListHandshakesForOrganization',
                'ListOrganizationalUnitsForParent',
                'ListParents',
                'ListPolicies',
                'ListPoliciesForTarget',
                'ListRoots',
                'ListTagsForResource',
                'ListTargetsForPolicy'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Organizations = Organizations;
//# sourceMappingURL=data:application/json;base64,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