"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Outposts = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [outposts](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsoutposts.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Outposts extends shared_1.PolicyStatement {
    /**
     * Grants permission to cancel a capacity task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_CancelCapacityTask.html
     */
    toCancelCapacityTask() {
        return this.to('CancelCapacityTask');
    }
    /**
     * Grants permission to cancel an order
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_CancelOrder.html
     */
    toCancelOrder() {
        return this.to('CancelOrder');
    }
    /**
     * Grants permission to create an order
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_CreateOrder.html
     */
    toCreateOrder() {
        return this.to('CreateOrder');
    }
    /**
     * Grants permission to create an Outpost
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_CreateOutpost.html
     */
    toCreateOutpost() {
        return this.to('CreateOutpost');
    }
    /**
     * Grants permission to create a private connectivity configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/outposts/latest/userguide/how-outposts-works.html#private-connectivity
     */
    toCreatePrivateConnectivityConfig() {
        return this.to('CreatePrivateConnectivityConfig');
    }
    /**
     * Grants permission to create a site
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_CreateSite.html
     */
    toCreateSite() {
        return this.to('CreateSite');
    }
    /**
     * Grants permission to delete an Outpost
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_DeleteOutpost.html
     */
    toDeleteOutpost() {
        return this.to('DeleteOutpost');
    }
    /**
     * Grants permission to delete a site
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_DeleteSite.html
     */
    toDeleteSite() {
        return this.to('DeleteSite');
    }
    /**
     * Grants permission to get information about the specified capacity task
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_GetCapacityTask.html
     */
    toGetCapacityTask() {
        return this.to('GetCapacityTask');
    }
    /**
     * Grants permission to get a catalog item
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_GetCatalogItem.html
     */
    toGetCatalogItem() {
        return this.to('GetCatalogItem');
    }
    /**
     * Grants permission to get information about the connection for your Outpost server
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_GetConnection.html
     */
    toGetConnection() {
        return this.to('GetConnection');
    }
    /**
     * Grants permission to get information about an order
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_GetOrder.html
     */
    toGetOrder() {
        return this.to('GetOrder');
    }
    /**
     * Grants permission to get information about the specified Outpost
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_GetOutpost.html
     */
    toGetOutpost() {
        return this.to('GetOutpost');
    }
    /**
     * Grants permission to get Outpost billing information for the specified Outpost
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_GetOutpostBillingInformation.html
     */
    toGetOutpostBillingInformation() {
        return this.to('GetOutpostBillingInformation');
    }
    /**
     * Grants permission to get the instance types for the specified Outpost
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_GetOutpostInstanceTypes.html
     */
    toGetOutpostInstanceTypes() {
        return this.to('GetOutpostInstanceTypes');
    }
    /**
     * Grants permission to get the supported instance types for the specified Outpost
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_GetOutpostSupportedInstanceTypes.html
     */
    toGetOutpostSupportedInstanceTypes() {
        return this.to('GetOutpostSupportedInstanceTypes');
    }
    /**
     * Grants permission to get a private connectivity configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/outposts/latest/userguide/how-outposts-works.html#private-connectivity
     */
    toGetPrivateConnectivityConfig() {
        return this.to('GetPrivateConnectivityConfig');
    }
    /**
     * Grants permission to get a site
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_GetSite.html
     */
    toGetSite() {
        return this.to('GetSite');
    }
    /**
     * Grants permission to get a site address
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_GetSiteAddress.html
     */
    toGetSiteAddress() {
        return this.to('GetSiteAddress');
    }
    /**
     * Grants permission to list all running instances for the specified Outpost
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_ListAssetInstances.html
     */
    toListAssetInstances() {
        return this.to('ListAssetInstances');
    }
    /**
     * Grants permission to list the assets for your Outpost
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_ListAssets.html
     */
    toListAssets() {
        return this.to('ListAssets');
    }
    /**
     * Grants permission to list all running instances that are blocking the capacity task from running for the specified Outpost
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_ListBlockingInstancesForCapacityTask.html
     */
    toListBlockingInstancesForCapacityTask() {
        return this.to('ListBlockingInstancesForCapacityTask');
    }
    /**
     * Grants permission to list the capacity tasks for your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_ListCapacityTasks.html
     */
    toListCapacityTasks() {
        return this.to('ListCapacityTasks');
    }
    /**
     * Grants permission to list all catalog items
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_ListCatalogItems.html
     */
    toListCatalogItems() {
        return this.to('ListCatalogItems');
    }
    /**
     * Grants permission to list the orders for your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_ListOrders.html
     */
    toListOrders() {
        return this.to('ListOrders');
    }
    /**
     * Grants permission to list the Outposts for your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_ListOutposts.html
     */
    toListOutposts() {
        return this.to('ListOutposts');
    }
    /**
     * Grants permission to list the sites for your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_ListSites.html
     */
    toListSites() {
        return this.to('ListSites');
    }
    /**
     * Grants permission to list tags for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to create a capacity task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_StartCapacityTask.html
     */
    toStartCapacityTask() {
        return this.to('StartCapacityTask');
    }
    /**
     * Grants permission to start a connection for your Outpost server
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_StartConnection.html
     */
    toStartConnection() {
        return this.to('StartConnection');
    }
    /**
     * Grants permission to tag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update an Outpost
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_UpdateOutpost.html
     */
    toUpdateOutpost() {
        return this.to('UpdateOutpost');
    }
    /**
     * Grants permission to update a site
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_UpdateSite.html
     */
    toUpdateSite() {
        return this.to('UpdateSite');
    }
    /**
     * Grants permission to update the site address
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_UpdateSiteAddress.html
     */
    toUpdateSiteAddress() {
        return this.to('UpdateSiteAddress');
    }
    /**
     * Grants permission to update the physical properties of a rack at a site
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/outposts/latest/APIReference/API_UpdateSiteRackPhysicalProperties.html
     */
    toUpdateSiteRackPhysicalProperties() {
        return this.to('UpdateSiteRackPhysicalProperties');
    }
    /**
     * Adds a resource of type outpost to the statement
     *
     * https://docs.aws.amazon.com/outposts/latest/userguide/what-is-outposts.html
     *
     * @param outpostId - Identifier for the outpostId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onOutpost(outpostId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:outposts:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:outpost/${outpostId}`);
    }
    /**
     * Adds a resource of type site to the statement
     *
     * https://docs.aws.amazon.com/outposts/latest/userguide/what-is-outposts.html
     *
     * @param siteId - Identifier for the siteId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSite(siteId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:outposts:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:site/${siteId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/outposts/latest/userguide/identity-access-management.html
     *
     * Applies to actions:
     * - .toCreateOutpost()
     * - .toCreateSite()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/outposts/latest/userguide/identity-access-management.html
     *
     * Applies to actions:
     * - .toCreateOutpost()
     * - .toCreateSite()
     *
     * Applies to resource types:
     * - outpost
     * - site
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/outposts/latest/userguide/identity-access-management.html
     *
     * Applies to actions:
     * - .toCreateOutpost()
     * - .toCreateSite()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [outposts](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsoutposts.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'outposts';
        this.accessLevelList = {
            Write: [
                'CancelCapacityTask',
                'CancelOrder',
                'CreateOrder',
                'CreateOutpost',
                'CreatePrivateConnectivityConfig',
                'CreateSite',
                'DeleteOutpost',
                'DeleteSite',
                'StartCapacityTask',
                'StartConnection',
                'UpdateOutpost',
                'UpdateSite',
                'UpdateSiteAddress',
                'UpdateSiteRackPhysicalProperties'
            ],
            Read: [
                'GetCapacityTask',
                'GetCatalogItem',
                'GetConnection',
                'GetOrder',
                'GetOutpost',
                'GetOutpostBillingInformation',
                'GetOutpostInstanceTypes',
                'GetOutpostSupportedInstanceTypes',
                'GetPrivateConnectivityConfig',
                'GetSite',
                'GetSiteAddress',
                'ListTagsForResource'
            ],
            List: [
                'ListAssetInstances',
                'ListAssets',
                'ListBlockingInstancesForCapacityTask',
                'ListCapacityTasks',
                'ListCatalogItems',
                'ListOrders',
                'ListOutposts',
                'ListSites'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Outposts = Outposts;
//# sourceMappingURL=data:application/json;base64,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