"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Panorama = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [panorama](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awspanorama.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Panorama extends shared_1.PolicyStatement {
    /**
     * Grants permission to create an AWS Panorama Application Instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/panorama/latest/api/API_CreateApplicationInstance.html
     */
    toCreateApplicationInstance() {
        return this.to('CreateApplicationInstance');
    }
    /**
     * Grants permission to create a job for an AWS Panorama Appliance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/panorama/latest/api/API_CreateJobForDevices.html
     */
    toCreateJobForDevices() {
        return this.to('CreateJobForDevices');
    }
    /**
     * Grants permission to create an AWS Panorama Node
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/panorama/latest/api/API_CreateNodeFromTemplateJob.html
     */
    toCreateNodeFromTemplateJob() {
        return this.to('CreateNodeFromTemplateJob');
    }
    /**
     * Grants permission to create an AWS Panorama Package
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/panorama/latest/api/API_CreatePackage.html
     */
    toCreatePackage() {
        return this.to('CreatePackage');
    }
    /**
     * Grants permission to create an AWS Panorama Package
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/panorama/latest/api/API_CreatePackageImportJob.html
     */
    toCreatePackageImportJob() {
        return this.to('CreatePackageImportJob');
    }
    /**
     * Grants permission to deregister an AWS Panorama Appliance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/panorama/latest/api/API_DeleteDevice.html
     */
    toDeleteDevice() {
        return this.to('DeleteDevice');
    }
    /**
     * Grants permission to delete an AWS Panorama Package
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/panorama/latest/api/API_DeletePackage.html
     */
    toDeletePackage() {
        return this.to('DeletePackage');
    }
    /**
     * Grants permission to deregister an AWS Panorama package version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/panorama/latest/api/API_DeregisterPackageVersion.html
     */
    toDeregisterPackageVersion() {
        return this.to('DeregisterPackageVersion');
    }
    /**
     * Grants permission to view details about an AWS Panorama application instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/panorama/latest/api/API_DescribeApplicationInstance.html
     */
    toDescribeApplicationInstance() {
        return this.to('DescribeApplicationInstance');
    }
    /**
     * Grants permission to view details about an AWS Panorama application instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/panorama/latest/api/API_DescribeApplicationInstanceDetails.html
     */
    toDescribeApplicationInstanceDetails() {
        return this.to('DescribeApplicationInstanceDetails');
    }
    /**
     * Grants permission to view details about an AWS Panorama Appliance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/panorama/latest/api/API_DescribeDevice.html
     */
    toDescribeDevice() {
        return this.to('DescribeDevice');
    }
    /**
     * Grants permission to view job details for an AWS Panorama Appliance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/panorama/latest/api/API_DescribeDeviceJob.html
     */
    toDescribeDeviceJob() {
        return this.to('DescribeDeviceJob');
    }
    /**
     * Grants permission to view details about an AWS Panorama application node
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/panorama/latest/api/API_DescribeNode.html
     */
    toDescribeNode() {
        return this.to('DescribeNode');
    }
    /**
     * Grants permission to view details about AWS Panorama application node
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/panorama/latest/api/API_DescribeNodeFromTemplateJob.html
     */
    toDescribeNodeFromTemplateJob() {
        return this.to('DescribeNodeFromTemplateJob');
    }
    /**
     * Grants permission to view details about an AWS Panorama package
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/panorama/latest/api/API_DescribePackage.html
     */
    toDescribePackage() {
        return this.to('DescribePackage');
    }
    /**
     * Grants permission to view details about an AWS Panorama package
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/panorama/latest/api/API_DescribePackageImportJob.html
     */
    toDescribePackageImportJob() {
        return this.to('DescribePackageImportJob');
    }
    /**
     * Grants permission to view details about an AWS Panorama package version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/panorama/latest/api/API_DescribePackageVersion.html
     */
    toDescribePackageVersion() {
        return this.to('DescribePackageVersion');
    }
    /**
     * Grants permission to view details about a software version for the AWS Panorama Appliance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awspanorama.html
     */
    toDescribeSoftware() {
        return this.to('DescribeSoftware');
    }
    /**
     * Grants permission to generate a WebSocket endpoint for communication with AWS Panorama
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awspanorama.html
     */
    toGetWebSocketURL() {
        return this.to('GetWebSocketURL');
    }
    /**
     * Grants permission to retrieve a list of application instance dependencies in AWS Panorama
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/panorama/latest/api/API_ListApplicationInstanceDependencies.html
     */
    toListApplicationInstanceDependencies() {
        return this.to('ListApplicationInstanceDependencies');
    }
    /**
     * Grants permission to retrieve a list of node instances of application instances in AWS Panorama
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/panorama/latest/api/API_ListApplicationInstanceNodeInstances.html
     */
    toListApplicationInstanceNodeInstances() {
        return this.to('ListApplicationInstanceNodeInstances');
    }
    /**
     * Grants permission to retrieve a list of application instances in AWS Panorama
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/panorama/latest/api/API_ListApplicationInstances.html
     */
    toListApplicationInstances() {
        return this.to('ListApplicationInstances');
    }
    /**
     * Grants permission to retrieve a list of appliances in AWS Panorama
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/panorama/latest/api/API_ListDevices.html
     */
    toListDevices() {
        return this.to('ListDevices');
    }
    /**
     * Grants permission to retrieve a list of jobs for an AWS Panorama Appliance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/panorama/latest/api/API_ListDevicesJobs.html
     */
    toListDevicesJobs() {
        return this.to('ListDevicesJobs');
    }
    /**
     * Grants permission to retrieve a list of Nodes for an AWS Panorama Appliance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/panorama/latest/api/API_ListNodeFromTemplateJobs.html
     */
    toListNodeFromTemplateJobs() {
        return this.to('ListNodeFromTemplateJobs');
    }
    /**
     * Grants permission to retrieve a list of nodes in AWS Panorama
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/panorama/latest/api/API_ListNodes.html
     */
    toListNodes() {
        return this.to('ListNodes');
    }
    /**
     * Grants permission to retrieve a list of packages in AWS Panorama
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/panorama/latest/api/API_ListPackageImportJobs.html
     */
    toListPackageImportJobs() {
        return this.to('ListPackageImportJobs');
    }
    /**
     * Grants permission to retrieve a list of packages in AWS Panorama
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/panorama/latest/api/API_ListPackages.html
     */
    toListPackages() {
        return this.to('ListPackages');
    }
    /**
     * Grants permission to retrieve a list of tags for a resource in AWS Panorama
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/panorama/latest/api/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to register an AWS Panorama Appliance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/panorama/latest/api/API_ProvisionDevice.html
     */
    toProvisionDevice() {
        return this.to('ProvisionDevice');
    }
    /**
     * Grants permission to register an AWS Panorama package version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/panorama/latest/api/API_RegisterPackageVersion.html
     */
    toRegisterPackageVersion() {
        return this.to('RegisterPackageVersion');
    }
    /**
     * Grants permission to remove an AWS Panorama application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/panorama/latest/api/API_RemoveApplicationInstance.html
     */
    toRemoveApplicationInstance() {
        return this.to('RemoveApplicationInstance');
    }
    /**
     * Grants permission to signal camera nodes in an application instance to pause or resume
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/panorama/latest/api/API_SignalApplicationInstanceNodeInstances.html
     */
    toSignalApplicationInstanceNodeInstances() {
        return this.to('SignalApplicationInstanceNodeInstances');
    }
    /**
     * Grants permission to add tags to a resource in AWS Panorama
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/panorama/latest/api/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from a resource in AWS Panorama
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/panorama/latest/api/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to modify basic settings for an AWS Panorama Appliance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/panorama/latest/api/API_UpdateDeviceMetadata.html
     */
    toUpdateDeviceMetadata() {
        return this.to('UpdateDeviceMetadata');
    }
    /**
     * Adds a resource of type device to the statement
     *
     * https://docs.aws.amazon.com/panorama/latest/dev/gettingstarted-concepts.html#gettingstarted-concepts-appliance
     *
     * @param deviceId - Identifier for the deviceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDevice(deviceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:panorama:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:device/${deviceId}`);
    }
    /**
     * Adds a resource of type package to the statement
     *
     * https://docs.aws.amazon.com/panorama/latest/dev/gettingstarted-concepts.html#gettingstarted-concepts-node
     *
     * @param packageId - Identifier for the packageId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPackage(packageId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:panorama:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:package/${packageId}`);
    }
    /**
     * Adds a resource of type applicationInstance to the statement
     *
     * https://docs.aws.amazon.com/panorama/latest/dev/gettingstarted-concepts.html#gettingstarted-concepts-application
     *
     * @param applicationInstanceId - Identifier for the applicationInstanceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onApplicationInstance(applicationInstanceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:panorama:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:applicationInstance/${applicationInstanceId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateApplicationInstance()
     * - .toCreatePackage()
     * - .toProvisionDevice()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - device
     * - package
     * - applicationInstance
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateApplicationInstance()
     * - .toCreatePackage()
     * - .toProvisionDevice()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [panorama](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awspanorama.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'panorama';
        this.accessLevelList = {
            Write: [
                'CreateApplicationInstance',
                'CreateJobForDevices',
                'CreateNodeFromTemplateJob',
                'CreatePackage',
                'CreatePackageImportJob',
                'DeleteDevice',
                'DeletePackage',
                'DeregisterPackageVersion',
                'ProvisionDevice',
                'RegisterPackageVersion',
                'RemoveApplicationInstance',
                'SignalApplicationInstanceNodeInstances',
                'UpdateDeviceMetadata'
            ],
            Read: [
                'DescribeApplicationInstance',
                'DescribeApplicationInstanceDetails',
                'DescribeDevice',
                'DescribeDeviceJob',
                'DescribeNode',
                'DescribeNodeFromTemplateJob',
                'DescribePackage',
                'DescribePackageImportJob',
                'DescribePackageVersion',
                'DescribeSoftware',
                'GetWebSocketURL',
                'ListTagsForResource'
            ],
            List: [
                'ListApplicationInstanceDependencies',
                'ListApplicationInstanceNodeInstances',
                'ListApplicationInstances',
                'ListDevices',
                'ListDevicesJobs',
                'ListNodeFromTemplateJobs',
                'ListNodes',
                'ListPackageImportJobs',
                'ListPackages'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Panorama = Panorama;
//# sourceMappingURL=data:application/json;base64,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