import { AccessLevelList } from '../../shared/access-level';
import { PolicyStatement, Operator } from '../../shared';
import { aws_iam as iam } from "aws-cdk-lib";
/**
 * Statement provider for service [pcs](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsparallelcomputingservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class Pcs extends PolicyStatement {
    servicePrefix: string;
    /**
     * Grants permission to configure vended log delivery for AWS PCS cluster logs
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pcs/latest/userguide/monitoring_scheduler-logs.html
     */
    toAllowVendedLogDeliveryForResource(): this;
    /**
     * Grants permission to create clusters
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:CreateNetworkInterfacePermission
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSecurityGroups
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     * - ec2:GetSecurityGroupsForVpc
     * - iam:CreateServiceLinkedRole
     * - secretsmanager:CreateSecret
     * - secretsmanager:TagResource
     *
     * https://docs.aws.amazon.com/pcs/latest/APIReference/API_CreateCluster.html
     */
    toCreateCluster(): this;
    /**
     * Grants permission to create compute node groups
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:CreateFleet
     * - ec2:CreateLaunchTemplate
     * - ec2:CreateLaunchTemplateVersion
     * - ec2:CreateTags
     * - ec2:DescribeImages
     * - ec2:DescribeInstanceStatus
     * - ec2:DescribeInstanceTypes
     * - ec2:DescribeInstances
     * - ec2:DescribeLaunchTemplateVersions
     * - ec2:DescribeLaunchTemplates
     * - ec2:DescribeSecurityGroups
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     * - ec2:RunInstances
     * - iam:GetInstanceProfile
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/pcs/latest/APIReference/API_CreateComputeNodeGroup.html
     */
    toCreateComputeNodeGroup(): this;
    /**
     * Grants permission to create queues
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pcs/latest/APIReference/API_CreateQueue.html
     */
    toCreateQueue(): this;
    /**
     * Grants permission to delete clusters
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:DeleteNetworkInterface
     * - secretsmanager:DeleteSecret
     *
     * https://docs.aws.amazon.com/pcs/latest/APIReference/API_DeleteCluster.html
     */
    toDeleteCluster(): this;
    /**
     * Grants permission to delete compute node groups
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:DeleteLaunchTemplate
     * - ec2:TerminateInstances
     *
     * https://docs.aws.amazon.com/pcs/latest/APIReference/API_DeleteComputeNodeGroup.html
     */
    toDeleteComputeNodeGroup(): this;
    /**
     * Grants permission to delete queues
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pcs/latest/APIReference/API_DeleteQueue.html
     */
    toDeleteQueue(): this;
    /**
     * Grants permission to get cluster properties
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pcs/latest/APIReference/API_GetCluster.html
     */
    toGetCluster(): this;
    /**
     * Grants permission to get compute node group properties
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pcs/latest/APIReference/API_GetComputeNodeGroup.html
     */
    toGetComputeNodeGroup(): this;
    /**
     * Grants permission to get queue properties
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pcs/latest/APIReference/API_GetQueue.html
     */
    toGetQueue(): this;
    /**
     * Grants permission to list clusters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pcs/latest/APIReference/API_ListClusters.html
     */
    toListClusters(): this;
    /**
     * Grants permission to list compute node groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pcs/latest/APIReference/API_ListComputeNodeGroups.html
     */
    toListComputeNodeGroups(): this;
    /**
     * Grants permission to list queues
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/pcs/latest/APIReference/API_ListQueues.html
     */
    toListQueues(): this;
    /**
     * Grants permission to list the tags for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pcs/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource(): this;
    /**
     * Grants permission to register a compute instance in a compute node group
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - secretsmanager:GetSecretValue
     *
     * https://docs.aws.amazon.com/pcs/latest/APIReference/API_RegisterComputeNodeGroupInstance.html
     */
    toRegisterComputeNodeGroupInstance(): this;
    /**
     * Grants permission to tag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pcs/latest/APIReference/API_TagResource.html
     */
    toTagResource(): this;
    /**
     * Grants permission to untag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pcs/latest/APIReference/API_UntagResource.html
     */
    toUntagResource(): this;
    /**
     * Grants permission to update compute node group properties
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateFleet
     * - ec2:CreateLaunchTemplate
     * - ec2:CreateLaunchTemplateVersion
     * - ec2:CreateTags
     * - ec2:DescribeImages
     * - ec2:DescribeInstanceStatus
     * - ec2:DescribeInstanceTypes
     * - ec2:DescribeInstances
     * - ec2:DescribeLaunchTemplateVersions
     * - ec2:DescribeLaunchTemplates
     * - ec2:DescribeSecurityGroups
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     * - ec2:RunInstances
     * - iam:GetInstanceProfile
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/pcs/latest/APIReference/API_UpdateComputeNodeGroup.html
     */
    toUpdateComputeNodeGroup(): this;
    /**
     * Grants permission to update queue properties
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pcs/latest/APIReference/API_UpdateQueue.html
     */
    toUpdateQueue(): this;
    protected accessLevelList: AccessLevelList;
    /**
     * Adds a resource of type cluster to the statement
     *
     * https://docs.aws.amazon.com/pcs/latest/APIReference/API_Cluster.html
     *
     * @param clusterIdentifier - Identifier for the clusterIdentifier.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCluster(clusterIdentifier: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type computenodegroup to the statement
     *
     * https://docs.aws.amazon.com/pcs/latest/APIReference/API_ComputeNodeGroup.html
     *
     * @param clusterIdentifier - Identifier for the clusterIdentifier.
     * @param computeNodeGroupIdentifier - Identifier for the computeNodeGroupIdentifier.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onComputenodegroup(clusterIdentifier: string, computeNodeGroupIdentifier: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type queue to the statement
     *
     * https://docs.aws.amazon.com/pcs/latest/APIReference/API_Queue.html
     *
     * @param clusterIdentifier - Identifier for the clusterIdentifier.
     * @param queueIdentifier - Identifier for the queueIdentifier.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onQueue(clusterIdentifier: string, queueIdentifier: string, account?: string, region?: string, partition?: string): this;
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateCluster()
     * - .toCreateComputeNodeGroup()
     * - .toCreateQueue()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toCreateCluster()
     * - .toCreateComputeNodeGroup()
     * - .toCreateQueue()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * Applies to resource types:
     * - cluster
     * - computenodegroup
     * - queue
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateCluster()
     * - .toCreateComputeNodeGroup()
     * - .toCreateQueue()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value: string | string[], operator?: Operator | string): this;
    /**
     * Statement provider for service [pcs](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsparallelcomputingservice.html).
     *
     */
    constructor(props?: iam.PolicyStatementProps);
}
