"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Partnercentral = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [partnercentral](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awspartnercentralselling.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Partnercentral extends shared_1.PolicyStatement {
    /**
     * Grants permission to accept Engagement Invitations on AWS Partner Central
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_AcceptEngagementInvitation.html
     */
    toAcceptEngagementInvitation() {
        return this.to('AcceptEngagementInvitation');
    }
    /**
     * Grants permission to assign Opportunities on AWS Partner Central
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_AssignOpportunity.html
     */
    toAssignOpportunity() {
        return this.to('AssignOpportunity');
    }
    /**
     * Grants permission to associate Opportunities on AWS Partner Central with other entities
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCatalog()
     * - .ifRelatedEntityType()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_AssociateOpportunity.html
     */
    toAssociateOpportunity() {
        return this.to('AssociateOpportunity');
    }
    /**
     * Grants permission to creating engagements in AWS Partner Central
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_CreateEngagement.html
     */
    toCreateEngagement() {
        return this.to('CreateEngagement');
    }
    /**
     * Grants permission to creating engagement invitations in AWS Partner Central
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_CreateEngagementInvitation.html
     */
    toCreateEngagementInvitation() {
        return this.to('CreateEngagementInvitation');
    }
    /**
     * Grants permission to create new Opportunities on AWS Partner Central
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_CreateOpportunity.html
     */
    toCreateOpportunity() {
        return this.to('CreateOpportunity');
    }
    /**
     * Grants permission to creating resource snapshots in AWS Partner Central
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_CreateResourceSnapshot.html
     */
    toCreateResourceSnapshot() {
        return this.to('CreateResourceSnapshot');
    }
    /**
     * Grants permission to creating resource snapshot jobs in AWS Partner Central
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_CreateResourceSnapshotJob.html
     */
    toCreateResourceSnapshotJob() {
        return this.to('CreateResourceSnapshotJob');
    }
    /**
     * Grants permission to deleting resource snapshot jobs on AWS Partner Central
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_DeleteResourceSnapshotJob.html
     */
    toDeleteResourceSnapshotJob() {
        return this.to('DeleteResourceSnapshotJob');
    }
    /**
     * Grants permission to disassociate Opportunities on AWS Partner Central from other entities
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCatalog()
     * - .ifRelatedEntityType()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_DisassociateOpportunity.html
     */
    toDisassociateOpportunity() {
        return this.to('DisassociateOpportunity');
    }
    /**
     * Grants permission to retrieve AWS Opportunity Summaries for Opportunities on AWS Partner Central
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_GetAwsOpportunitySummary.html
     */
    toGetAwsOpportunitySummary() {
        return this.to('GetAwsOpportunitySummary');
    }
    /**
     * Grants permission to retrieval of engagement details in AWS Partner Central
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_GetEngagement.html
     */
    toGetEngagement() {
        return this.to('GetEngagement');
    }
    /**
     * Grants permission to retrieve details of Engagement Invitations on AWS Partner Central
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_GetEngagementInvitation.html
     */
    toGetEngagementInvitation() {
        return this.to('GetEngagementInvitation');
    }
    /**
     * Grants permission to retrieve details of Opportunities on AWS Partner Central
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_GetOpportunity.html
     */
    toGetOpportunity() {
        return this.to('GetOpportunity');
    }
    /**
     * Grants permission to retrieving resource snapshot details in AWS Partner Central
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_GetResourceSnapshot.html
     */
    toGetResourceSnapshot() {
        return this.to('GetResourceSnapshot');
    }
    /**
     * Grants permission to retrieving resource snapshot job details in AWS Partner Central
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_GetResourceSnapshotJob.html
     */
    toGetResourceSnapshotJob() {
        return this.to('GetResourceSnapshotJob');
    }
    /**
     * Grants permission to retrieving system settings settings in AWS Partner Central
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_GetSellingSystemSettings.html
     */
    toGetSellingSystemSettings() {
        return this.to('GetSellingSystemSettings');
    }
    /**
     * Grants permission to listing engagements by accepting invitation tasks in AWS Partner Central
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_ListEngagementByAcceptingInvitationTasks.html
     */
    toListEngagementByAcceptingInvitationTasks() {
        return this.to('ListEngagementByAcceptingInvitationTasks');
    }
    /**
     * Grants permission to listing engagements from opportunity tasks in AWS Partner Central
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_ListEngagementFromOpportunityTasks.html
     */
    toListEngagementFromOpportunityTasks() {
        return this.to('ListEngagementFromOpportunityTasks');
    }
    /**
     * Grants permission to list Engagement Invitations on AWS Partner Central
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_ListEngagementInvitations.html
     */
    toListEngagementInvitations() {
        return this.to('ListEngagementInvitations');
    }
    /**
     * Grants permission to listing engagement members in AWS Partner Central
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_ListEngagementMembers.html
     */
    toListEngagementMembers() {
        return this.to('ListEngagementMembers');
    }
    /**
     * Grants permission to listing engagement resource associations in AWS Partner Central
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_ListEngagementResourceAssociations.html
     */
    toListEngagementResourceAssociations() {
        return this.to('ListEngagementResourceAssociations');
    }
    /**
     * Grants permission to listing engagements in AWS Partner Central
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_ListEngagements.html
     */
    toListEngagements() {
        return this.to('ListEngagements');
    }
    /**
     * Grants permission to list Opportunities on AWS Partner Central
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_ListOpportunities.html
     */
    toListOpportunities() {
        return this.to('ListOpportunities');
    }
    /**
     * Grants permission to listing resource snapshot jobs in AWS Partner Central
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_ListResourceSnapshotJobs.html
     */
    toListResourceSnapshotJobs() {
        return this.to('ListResourceSnapshotJobs');
    }
    /**
     * Grants permission to listing resource snapshots in AWS Partner Central
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_ListResourceSnapshots.html
     */
    toListResourceSnapshots() {
        return this.to('ListResourceSnapshots');
    }
    /**
     * Grants permission to list Solutions on AWS Partner Central
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_ListSolutions.html
     */
    toListSolutions() {
        return this.to('ListSolutions');
    }
    /**
     * Grants permission to add lists tags to a resource. Supported resource: ResourceSnapshotJob
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to put system settings settings in AWS Partner Central
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_PutSellingSystemSettings.html
     */
    toPutSellingSystemSettings() {
        return this.to('PutSellingSystemSettings');
    }
    /**
     * Grants permission to reject Engagement Invitations on AWS Partner Central
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_RejectEngagementInvitation.html
     */
    toRejectEngagementInvitation() {
        return this.to('RejectEngagementInvitation');
    }
    /**
     * Grants permission to initiate tasks that start Engagements on AWS Partner Central by accepting an Engagement Invitation
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifCatalog()
     *
     * Dependent actions:
     * - partnercentral:AcceptEngagementInvitation
     * - partnercentral:CreateOpportunity
     * - partnercentral:CreateResourceSnapshotJob
     * - partnercentral:GetEngagementInvitation
     * - partnercentral:StartResourceSnapshotJob
     * - partnercentral:SubmitOpportunity
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_StartEngagementByAcceptingInvitationTask.html
     */
    toStartEngagementByAcceptingInvitationTask() {
        return this.to('StartEngagementByAcceptingInvitationTask');
    }
    /**
     * Grants permission to initiate tasks that start Engagements from Opportunities on AWS Partner Central
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifCatalog()
     *
     * Dependent actions:
     * - partnercentral:CreateEngagement
     * - partnercentral:CreateEngagementInvitation
     * - partnercentral:CreateResourceSnapshotJob
     * - partnercentral:GetOpportunity
     * - partnercentral:StartResourceSnapshotJob
     * - partnercentral:SubmitOpportunity
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_StartEngagementFromOpportunityTask.html
     */
    toStartEngagementFromOpportunityTask() {
        return this.to('StartEngagementFromOpportunityTask');
    }
    /**
     * Grants permission to starting resource snapshot jobs in AWS Partner Central
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_StartResourceSnapshotJob.html
     */
    toStartResourceSnapshotJob() {
        return this.to('StartResourceSnapshotJob');
    }
    /**
     * Grants permission to stopping resource snapshot jobs in AWS Partner Central
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_StopResourceSnapshotJob.html
     */
    toStopResourceSnapshotJob() {
        return this.to('StopResourceSnapshotJob');
    }
    /**
     * Grants permission to submit Opportunities on AWS Partner Central
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_SubmitOpportunity.html
     */
    toSubmitOpportunity() {
        return this.to('SubmitOpportunity');
    }
    /**
     * Grants permission to add new tags to a resource. Supported resource: ResourceSnapshotJob
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from a resource. Supported resource: ResourceSnapshotJob
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update Opportunities on AWS Partner Central
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCatalog()
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_UpdateOpportunity.html
     */
    toUpdateOpportunity() {
        return this.to('UpdateOpportunity');
    }
    /**
     * Adds a resource of type Engagement to the statement
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/working-with-multi-partner-opportunities.html
     *
     * @param catalog - Identifier for the catalog.
     * @param identifier - Identifier for the identifier.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onEngagement(catalog, identifier, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:partnercentral:${region ?? this.defaultRegion}::catalog/${catalog}/engagement/${identifier}`);
    }
    /**
     * Adds a resource of type engagement-by-accepting-invitation-task to the statement
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/working-with-multi-partner-opportunities.html
     *
     * @param catalog - Identifier for the catalog.
     * @param taskId - Identifier for the taskId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onEngagementByAcceptingInvitationTask(catalog, taskId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:partnercentral:${region ?? this.defaultRegion}::catalog/${catalog}/engagement-by-accepting-invitation-task/${taskId}`);
    }
    /**
     * Adds a resource of type engagement-from-opportunity-task to the statement
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/working-with-multi-partner-opportunities.html
     *
     * @param catalog - Identifier for the catalog.
     * @param taskId - Identifier for the taskId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onEngagementFromOpportunityTask(catalog, taskId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:partnercentral:${region ?? this.defaultRegion}::catalog/${catalog}/engagement-from-opportunity-task/${taskId}`);
    }
    /**
     * Adds a resource of type engagement-invitation to the statement
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/working-with-multi-partner-opportunities.html
     *
     * @param catalog - Identifier for the catalog.
     * @param identifier - Identifier for the identifier.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onEngagementInvitation(catalog, identifier, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:partnercentral:${region ?? this.defaultRegion}::catalog/${catalog}/engagement-invitation/${identifier}`);
    }
    /**
     * Adds a resource of type Opportunity to the statement
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/working-with-your-opportunities.html
     *
     * @param catalog - Identifier for the catalog.
     * @param identifier - Identifier for the identifier.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onOpportunity(catalog, identifier, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:partnercentral:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:catalog/${catalog}/opportunity/${identifier}`);
    }
    /**
     * Adds a resource of type resource-snapshot-job to the statement
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/working-with-multi-partner-opportunities.html
     *
     * @param catalog - Identifier for the catalog.
     * @param identifier - Identifier for the identifier.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onResourceSnapshotJob(catalog, identifier, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:partnercentral:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:catalog/${catalog}/resource-snapshot-job/${identifier}`);
    }
    /**
     * Adds a resource of type ResourceSnapshot to the statement
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/working-with-multi-partner-opportunities.html
     *
     * @param catalog - Identifier for the catalog.
     * @param engagementIdentifier - Identifier for the engagementIdentifier.
     * @param resourceType - Identifier for the resourceType.
     * @param resourceIdentifier - Identifier for the resourceIdentifier.
     * @param templateIdentifier - Identifier for the templateIdentifier.
     * @param snapshotRevision - Identifier for the snapshotRevision.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onResourceSnapshot(catalog, engagementIdentifier, resourceType, resourceIdentifier, templateIdentifier, snapshotRevision, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:partnercentral:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:catalog/${catalog}/engagement/${engagementIdentifier}/resource/${resourceType}/${resourceIdentifier}/template/${templateIdentifier}/resource-snapshot/${snapshotRevision}`);
    }
    /**
     * Adds a resource of type Solution to the statement
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/API_ListSolutions.html
     *
     * @param catalog - Identifier for the catalog.
     * @param identifier - Identifier for the identifier.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onSolution(catalog, identifier, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:partnercentral:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:catalog/${catalog}/solution/${identifier}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateOpportunity()
     * - .toCreateResourceSnapshotJob()
     * - .toStartEngagementByAcceptingInvitationTask()
     * - .toStartEngagementFromOpportunityTask()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toListTagsForResource()
     * - .toStartEngagementByAcceptingInvitationTask()
     * - .toStartEngagementFromOpportunityTask()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * Applies to resource types:
     * - Opportunity
     * - resource-snapshot-job
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateOpportunity()
     * - .toCreateResourceSnapshotJob()
     * - .toListTagsForResource()
     * - .toStartEngagementByAcceptingInvitationTask()
     * - .toStartEngagementFromOpportunityTask()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a specific Catalog. Accepted values: [AWS, Sandbox]
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/access-control.html#managing-permissions-with-condition-keyspage.html#Catalog
     *
     * Applies to actions:
     * - .toAcceptEngagementInvitation()
     * - .toAssignOpportunity()
     * - .toAssociateOpportunity()
     * - .toCreateEngagement()
     * - .toCreateEngagementInvitation()
     * - .toCreateOpportunity()
     * - .toCreateResourceSnapshot()
     * - .toCreateResourceSnapshotJob()
     * - .toDeleteResourceSnapshotJob()
     * - .toDisassociateOpportunity()
     * - .toGetAwsOpportunitySummary()
     * - .toGetEngagement()
     * - .toGetEngagementInvitation()
     * - .toGetOpportunity()
     * - .toGetResourceSnapshot()
     * - .toGetResourceSnapshotJob()
     * - .toGetSellingSystemSettings()
     * - .toListEngagementByAcceptingInvitationTasks()
     * - .toListEngagementFromOpportunityTasks()
     * - .toListEngagementInvitations()
     * - .toListEngagementMembers()
     * - .toListEngagementResourceAssociations()
     * - .toListEngagements()
     * - .toListOpportunities()
     * - .toListResourceSnapshotJobs()
     * - .toListResourceSnapshots()
     * - .toListSolutions()
     * - .toListTagsForResource()
     * - .toRejectEngagementInvitation()
     * - .toStartEngagementByAcceptingInvitationTask()
     * - .toStartEngagementFromOpportunityTask()
     * - .toStartResourceSnapshotJob()
     * - .toStopResourceSnapshotJob()
     * - .toSubmitOpportunity()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateOpportunity()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCatalog(value, operator) {
        return this.if(`Catalog`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by entity types for Opportunity association. Accepted values: [Solutions, AwsProducts, AwsMarketplaceOffers]
     *
     * https://docs.aws.amazon.com/partner-central/latest/APIReference/access-control.html#managing-permissions-with-condition-keyspage.html#RelatedEntityType
     *
     * Applies to actions:
     * - .toAssociateOpportunity()
     * - .toDisassociateOpportunity()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRelatedEntityType(value, operator) {
        return this.if(`RelatedEntityType`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [partnercentral](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awspartnercentralselling.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'partnercentral';
        this.accessLevelList = {
            Write: [
                'AcceptEngagementInvitation',
                'AssignOpportunity',
                'AssociateOpportunity',
                'CreateEngagement',
                'CreateEngagementInvitation',
                'CreateOpportunity',
                'CreateResourceSnapshot',
                'CreateResourceSnapshotJob',
                'DeleteResourceSnapshotJob',
                'DisassociateOpportunity',
                'PutSellingSystemSettings',
                'RejectEngagementInvitation',
                'StartEngagementByAcceptingInvitationTask',
                'StartEngagementFromOpportunityTask',
                'StartResourceSnapshotJob',
                'StopResourceSnapshotJob',
                'SubmitOpportunity',
                'UpdateOpportunity'
            ],
            Read: [
                'GetAwsOpportunitySummary',
                'GetEngagement',
                'GetEngagementInvitation',
                'GetOpportunity',
                'GetResourceSnapshot',
                'GetResourceSnapshotJob',
                'GetSellingSystemSettings',
                'ListEngagementMembers',
                'ListEngagementResourceAssociations',
                'ListResourceSnapshots',
                'ListTagsForResource'
            ],
            List: [
                'ListEngagementByAcceptingInvitationTasks',
                'ListEngagementFromOpportunityTasks',
                'ListEngagementInvitations',
                'ListEngagements',
                'ListOpportunities',
                'ListResourceSnapshotJobs',
                'ListSolutions'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Partnercentral = Partnercentral;
//# sourceMappingURL=data:application/json;base64,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