"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AcmPca = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [acm-pca](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsprivatecertificateauthority.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class AcmPca extends shared_1.PolicyStatement {
    /**
     * Grants permission to create an AWS Private CA and its associated private key and configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/privateca/latest/APIReference/API_CreateCertificateAuthority.html
     */
    toCreateCertificateAuthority() {
        return this.to('CreateCertificateAuthority');
    }
    /**
     * Grants permission to create an audit report for an AWS Private CA
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/privateca/latest/APIReference/API_CreateCertificateAuthorityAuditReport.html
     */
    toCreateCertificateAuthorityAuditReport() {
        return this.to('CreateCertificateAuthorityAuditReport');
    }
    /**
     * Grants permission to create a permission for an AWS Private CA
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/privateca/latest/APIReference/API_CreatePermission.html
     */
    toCreatePermission() {
        return this.to('CreatePermission');
    }
    /**
     * Grants permission to delete an AWS Private CA and its associated private key and configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/privateca/latest/APIReference/API_DeleteCertificateAuthority.html
     */
    toDeleteCertificateAuthority() {
        return this.to('DeleteCertificateAuthority');
    }
    /**
     * Grants permission to delete a permission for an AWS Private CA
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/privateca/latest/APIReference/API_DeletePermission.html
     */
    toDeletePermission() {
        return this.to('DeletePermission');
    }
    /**
     * Grants permission to delete the policy for an AWS Private CA
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/privateca/latest/APIReference/API_DeletePolicy.html
     */
    toDeletePolicy() {
        return this.to('DeletePolicy');
    }
    /**
     * Grants permission to return a list of the configuration and status fields contained in the specified AWS Private CA
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/privateca/latest/APIReference/API_DescribeCertificateAuthority.html
     */
    toDescribeCertificateAuthority() {
        return this.to('DescribeCertificateAuthority');
    }
    /**
     * Grants permission to return the status and information about an AWS Private CA audit report
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/privateca/latest/APIReference/API_DescribeCertificateAuthorityAuditReport.html
     */
    toDescribeCertificateAuthorityAuditReport() {
        return this.to('DescribeCertificateAuthorityAuditReport');
    }
    /**
     * Grants permission to retrieve an AWS Private CA certificate and certificate chain for the certificate authority specified by an ARN
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/privateca/latest/APIReference/API_GetCertificate.html
     */
    toGetCertificate() {
        return this.to('GetCertificate');
    }
    /**
     * Grants permission to retrieve an AWS Private CA certificate and certificate chain for the certificate authority specified by an ARN
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/privateca/latest/APIReference/API_GetCertificateAuthorityCertificate.html
     */
    toGetCertificateAuthorityCertificate() {
        return this.to('GetCertificateAuthorityCertificate');
    }
    /**
     * Grants permission to retrieve an AWS Private CA certificate signing request (CSR) for the certificate-authority specified by an ARN
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/privateca/latest/APIReference/API_GetCertificateAuthorityCsr.html
     */
    toGetCertificateAuthorityCsr() {
        return this.to('GetCertificateAuthorityCsr');
    }
    /**
     * Grants permission to retrieve the policy on an AWS Private CA
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/privateca/latest/APIReference/API_GetPolicy.html
     */
    toGetPolicy() {
        return this.to('GetPolicy');
    }
    /**
     * Grants permission to import an SSL/TLS certificate into AWS Private CA for use as the CA certificate of an AWS Private CA
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/privateca/latest/APIReference/API_ImportCertificateAuthorityCertificate.html
     */
    toImportCertificateAuthorityCertificate() {
        return this.to('ImportCertificateAuthorityCertificate');
    }
    /**
     * Grants permission to issue an AWS Private CA certificate
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTemplateArn()
     *
     * https://docs.aws.amazon.com/privateca/latest/APIReference/API_IssueCertificate.html
     */
    toIssueCertificate() {
        return this.to('IssueCertificate');
    }
    /**
     * Grants permission to retrieve a list of the AWS Private CA certificate authority ARNs, and a summary of the status of each CA in the calling account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/privateca/latest/APIReference/API_ListCertificateAuthorities.html
     */
    toListCertificateAuthorities() {
        return this.to('ListCertificateAuthorities');
    }
    /**
     * Grants permission to list the permissions that have been applied to the AWS Private CA certificate authority
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/privateca/latest/APIReference/API_ListPermissions.html
     */
    toListPermissions() {
        return this.to('ListPermissions');
    }
    /**
     * Grants permission to list the tags that have been applied to the AWS Private CA certificate authority
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/privateca/latest/APIReference/API_ListTags.html
     */
    toListTags() {
        return this.to('ListTags');
    }
    /**
     * Grants permission to put a policy on an AWS Private CA
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/privateca/latest/APIReference/API_PutPolicy.html
     */
    toPutPolicy() {
        return this.to('PutPolicy');
    }
    /**
     * Grants permission to restore an AWS Private CA from the deleted state to the state it was in when deleted
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/privateca/latest/APIReference/API_RestoreCertificateAuthority.html
     */
    toRestoreCertificateAuthority() {
        return this.to('RestoreCertificateAuthority');
    }
    /**
     * Grants permission to revoke a certificate issued by an AWS Private CA
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/privateca/latest/APIReference/API_RevokeCertificate.html
     */
    toRevokeCertificate() {
        return this.to('RevokeCertificate');
    }
    /**
     * Grants permission to add one or more tags to an AWS Private CA
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/privateca/latest/APIReference/API_TagCertificateAuthority.html
     */
    toTagCertificateAuthority() {
        return this.to('TagCertificateAuthority');
    }
    /**
     * Grants permission to remove one or more tags from an AWS Private CA
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/privateca/latest/APIReference/API_UntagCertificateAuthority.html
     */
    toUntagCertificateAuthority() {
        return this.to('UntagCertificateAuthority');
    }
    /**
     * Grants permission to update the configuration of an AWS Private CA
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/privateca/latest/APIReference/API_UpdateCertificateAuthority.html
     */
    toUpdateCertificateAuthority() {
        return this.to('UpdateCertificateAuthority');
    }
    /**
     * Adds a resource of type certificate-authority to the statement
     *
     * https://docs.aws.amazon.com/privateca/latest/userguide/api-permissions.html
     *
     * @param certificateAuthorityId - Identifier for the certificateAuthorityId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCertificateAuthority(certificateAuthorityId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:acm-pca:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:certificate-authority/${certificateAuthorityId}`);
    }
    /**
     * Filters access by the arn of the certificate template used in Issue Certificate request
     *
     * https://docs.aws.amazon.com/privateca/latest/userguide/UsingTemplates.html#template-varieties
     *
     * Applies to actions:
     * - .toIssueCertificate()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifTemplateArn(value, operator) {
        return this.if(`TemplateArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateCertificateAuthority()
     * - .toTagCertificateAuthority()
     * - .toUntagCertificateAuthority()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toCreateCertificateAuthority()
     *
     * Applies to resource types:
     * - certificate-authority
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateCertificateAuthority()
     * - .toTagCertificateAuthority()
     * - .toUntagCertificateAuthority()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [acm-pca](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsprivatecertificateauthority.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'acm-pca';
        this.accessLevelList = {
            Write: [
                'CreateCertificateAuthority',
                'CreateCertificateAuthorityAuditReport',
                'DeleteCertificateAuthority',
                'ImportCertificateAuthorityCertificate',
                'IssueCertificate',
                'RestoreCertificateAuthority',
                'RevokeCertificate',
                'UpdateCertificateAuthority'
            ],
            'Permissions management': [
                'CreatePermission',
                'DeletePermission',
                'DeletePolicy',
                'PutPolicy'
            ],
            Read: [
                'DescribeCertificateAuthority',
                'DescribeCertificateAuthorityAuditReport',
                'GetCertificate',
                'GetCertificateAuthorityCertificate',
                'GetCertificateAuthorityCsr',
                'GetPolicy',
                'ListPermissions',
                'ListTags'
            ],
            List: [
                'ListCertificateAuthorities'
            ],
            Tagging: [
                'TagCertificateAuthority',
                'UntagCertificateAuthority'
            ]
        };
    }
}
exports.AcmPca = AcmPca;
//# sourceMappingURL=data:application/json;base64,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