"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Qbusiness = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [qbusiness](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonqbusiness.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Qbusiness extends shared_1.PolicyStatement {
    /**
     * Grants permission to configure vended log delivery for Amazon Q Business application resource
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    toAllowVendedLogDeliveryForResource() {
        return this.to('AllowVendedLogDeliveryForResource');
    }
    /**
     * Grants permission to associate resource based policy statement to the application
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - qbusiness:PutResourcePolicy
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_AssociatePermission.html
     */
    toAssociatePermission() {
        return this.to('AssociatePermission');
    }
    /**
     * Grants permission to batch delete document
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_BatchDeleteDocument.html
     */
    toBatchDeleteDocument() {
        return this.to('BatchDeleteDocument');
    }
    /**
     * Grants permission to batch put document
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_BatchPutDocument.html
     */
    toBatchPutDocument() {
        return this.to('BatchPutDocument');
    }
    /**
     * Grants permission to cancel a subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_CancelSubscription.html
     */
    toCancelSubscription() {
        return this.to('CancelSubscription');
    }
    /**
     * Grants permission to chat using an application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_Chat.html
     */
    toChat() {
        return this.to('Chat');
    }
    /**
     * Grants permission to chat synchronously using an application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ChatSync.html
     */
    toChatSync() {
        return this.to('ChatSync');
    }
    /**
     * Grants permission to check if a user has access to a document
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_CheckDocumentAccess.html
     */
    toCheckDocumentAccess() {
        return this.to('CheckDocumentAccess');
    }
    /**
     * Grants permission to create a unique URL for anonymous Amazon Q Business web experience
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_CreateAnonymousWebExperienceUrl.html
     */
    toCreateAnonymousWebExperienceUrl() {
        return this.to('CreateAnonymousWebExperienceUrl');
    }
    /**
     * Grants permission to create an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_CreateApplication.html
     */
    toCreateApplication() {
        return this.to('CreateApplication');
    }
    /**
     * Grants permission to create a chat response configuration to the application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_CreateChatResponseConfiguration.html
     */
    toCreateChatResponseConfiguration() {
        return this.to('CreateChatResponseConfiguration');
    }
    /**
     * Grants permission to create DataAccessor to the application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - qbusiness:CreateDataAccessorWithTti
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_CreateDataAccessor.html
     */
    toCreateDataAccessor() {
        return this.to('CreateDataAccessor');
    }
    /**
     * Grants permission to create AWS IAM Identity center Trusted Token Issuer based DataAccessor to the application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_CreateDataAccessor.html
     */
    toCreateDataAccessorWithTti() {
        return this.to('CreateDataAccessorWithTti');
    }
    /**
     * Grants permission to create a data source for a given application and index
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_CreateDataSource.html
     */
    toCreateDataSource() {
        return this.to('CreateDataSource');
    }
    /**
     * Grants permission to create an index for a given application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_CreateIndex.html
     */
    toCreateIndex() {
        return this.to('CreateIndex');
    }
    /**
     * Grants permission to create a new integration for a Q Business application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_CreateIntegration.html
     */
    toCreateIntegration() {
        return this.to('CreateIntegration');
    }
    /**
     * Grants permission to create a plugin for a given application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_CreatePlugin.html
     */
    toCreatePlugin() {
        return this.to('CreatePlugin');
    }
    /**
     * Grants permission to create a retriever for a given application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_CreateRetriever.html
     */
    toCreateRetriever() {
        return this.to('CreateRetriever');
    }
    /**
     * Grants permission to create a subscription
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifIdentitystoreUserId()
     * - .ifIdentitystoreGroupId()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_CreateSubscription.html
     */
    toCreateSubscription() {
        return this.to('CreateSubscription');
    }
    /**
     * Grants permission to create a user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_CreateUser.html
     */
    toCreateUser() {
        return this.to('CreateUser');
    }
    /**
     * Grants permission to create a web experience for a given application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_CreateWebExperience.html
     */
    toCreateWebExperience() {
        return this.to('CreateWebExperience');
    }
    /**
     * Grants permission to delete an application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_DeleteApplication.html
     */
    toDeleteApplication() {
        return this.to('DeleteApplication');
    }
    /**
     * Grants permission to delete an attachment in the current chat context
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_DeleteAttachment.html
     */
    toDeleteAttachment() {
        return this.to('DeleteAttachment');
    }
    /**
     * Grants permission to delete chat controls configuration for an application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_DeleteChatControlsConfiguration.html
     */
    toDeleteChatControlsConfiguration() {
        return this.to('DeleteChatControlsConfiguration');
    }
    /**
     * Grants permission to delete a chat response configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_DeleteChatResponseConfiguration.html
     */
    toDeleteChatResponseConfiguration() {
        return this.to('DeleteChatResponseConfiguration');
    }
    /**
     * Grants permission to delete a conversation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_DeleteConversation.html
     */
    toDeleteConversation() {
        return this.to('DeleteConversation');
    }
    /**
     * Grants permission to delete DataAccessor
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_DeleteDataAccessor.html
     */
    toDeleteDataAccessor() {
        return this.to('DeleteDataAccessor');
    }
    /**
     * Grants permission to delete a DataSource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_DeleteDataSource.html
     */
    toDeleteDataSource() {
        return this.to('DeleteDataSource');
    }
    /**
     * Grants permission to delete a group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_DeleteGroup.html
     */
    toDeleteGroup() {
        return this.to('DeleteGroup');
    }
    /**
     * Grants permission to delete an index
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_DeleteIndex.html
     */
    toDeleteIndex() {
        return this.to('DeleteIndex');
    }
    /**
     * Grants permission to delete an integration for a Q Business application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_DeleteIntegration.html
     */
    toDeleteIntegration() {
        return this.to('DeleteIntegration');
    }
    /**
     * Grants permission to delete a plugin
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_DeletePlugin.html
     */
    toDeletePlugin() {
        return this.to('DeletePlugin');
    }
    /**
     * Grants permission to delete a retriever
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_DeleteRetriever.html
     */
    toDeleteRetriever() {
        return this.to('DeleteRetriever');
    }
    /**
     * Grants permission to delete a user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_DeleteUser.html
     */
    toDeleteUser() {
        return this.to('DeleteUser');
    }
    /**
     * Grants permission to delete a web-experience
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_DeleteWebExperience.html
     */
    toDeleteWebExperience() {
        return this.to('DeleteWebExperience');
    }
    /**
     * Grants permission to disable the ACL crawl while creating the Amazon Q Business data source resource
     *
     * Access Level: Write
     */
    toDisableAclOnDataSource() {
        return this.to('DisableAclOnDataSource');
    }
    /**
     * Grants permission to disassociate resource based policy statement to the application
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - qbusiness:PutResourcePolicy
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_DisassociatePermission.html
     */
    toDisassociatePermission() {
        return this.to('DisassociatePermission');
    }
    /**
     * Grants permission to get an application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_GetApplication.html
     */
    toGetApplication() {
        return this.to('GetApplication');
    }
    /**
     * Grants permission to get chat controls configuration for an application
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_GetChatControlsConfiguration.html
     */
    toGetChatControlsConfiguration() {
        return this.to('GetChatControlsConfiguration');
    }
    /**
     * Grants permission to get a chat response configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_GetChatResponseConfiguration.html
     */
    toGetChatResponseConfiguration() {
        return this.to('GetChatResponseConfiguration');
    }
    /**
     * Grants permission to get DataAccessor
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_GetDataAccessor.html
     */
    toGetDataAccessor() {
        return this.to('GetDataAccessor');
    }
    /**
     * Grants permission to get a data source
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_GetDataSource.html
     */
    toGetDataSource() {
        return this.to('GetDataSource');
    }
    /**
     * Grants permission to get a document content
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_GetDocumentContent.html
     */
    toGetDocumentContent() {
        return this.to('GetDocumentContent');
    }
    /**
     * Grants permission to get a group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_GetGroup.html
     */
    toGetGroup() {
        return this.to('GetGroup');
    }
    /**
     * Grants permission to get an index
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_GetIndex.html
     */
    toGetIndex() {
        return this.to('GetIndex');
    }
    /**
     * Grants permission to get an integration for a Q Business application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_GetIntegration.html
     */
    toGetIntegration() {
        return this.to('GetIntegration');
    }
    /**
     * Grants permission to get the media associated to a system message
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_GetMedia.html
     */
    toGetMedia() {
        return this.to('GetMedia');
    }
    /**
     * Grants permission to get a plugin
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_GetPlugin.html
     */
    toGetPlugin() {
        return this.to('GetPlugin');
    }
    /**
     * Grants permission to get resource based policy of the application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_GetPolicy.html
     */
    toGetPolicy() {
        return this.to('GetPolicy');
    }
    /**
     * Grants permission to get a retriever
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_GetRetriever.html
     */
    toGetRetriever() {
        return this.to('GetRetriever');
    }
    /**
     * Grants permission to get a user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_GetUser.html
     */
    toGetUser() {
        return this.to('GetUser');
    }
    /**
     * Grants permission to get a web-experience
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_GetWebExperience.html
     */
    toGetWebExperience() {
        return this.to('GetWebExperience');
    }
    /**
     * Grants permission to list the applications
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListApplications.html
     */
    toListApplications() {
        return this.to('ListApplications');
    }
    /**
     * Grants permission to list attachments in the current chat context
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListAttachments.html
     */
    toListAttachments() {
        return this.to('ListAttachments');
    }
    /**
     * Grants permission to list chat response configurations for an application
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListChatResponseConfigurations.html
     */
    toListChatResponseConfigurations() {
        return this.to('ListChatResponseConfigurations');
    }
    /**
     * Grants permission to list all conversations for an application
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListConversations.html
     */
    toListConversations() {
        return this.to('ListConversations');
    }
    /**
     * Grants permission to list DataAccessors for the application
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListDataAccessors.html
     */
    toListDataAccessors() {
        return this.to('ListDataAccessors');
    }
    /**
     * Grants permission to get Data Source sync job history
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListDataSourceSyncJobs.html
     */
    toListDataSourceSyncJobs() {
        return this.to('ListDataSourceSyncJobs');
    }
    /**
     * Grants permission to list the data sources of an application and an index
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListDataSources.html
     */
    toListDataSources() {
        return this.to('ListDataSources');
    }
    /**
     * Grants permission to list all documents
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListDocuments.html
     */
    toListDocuments() {
        return this.to('ListDocuments');
    }
    /**
     * Grants permission to list groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListGroups.html
     */
    toListGroups() {
        return this.to('ListGroups');
    }
    /**
     * Grants permission to list the indices of an application
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListIndices.html
     */
    toListIndices() {
        return this.to('ListIndices');
    }
    /**
     * Grants permission to list all integrations for a Q Business application
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListIntegrations.html
     */
    toListIntegrations() {
        return this.to('ListIntegrations');
    }
    /**
     * Grants permission to list all messages
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListMessages.html
     */
    toListMessages() {
        return this.to('ListMessages');
    }
    /**
     * Grants permission to list the plugins actions of a plugin within application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListPluginActions.html
     */
    toListPluginActions() {
        return this.to('ListPluginActions');
    }
    /**
     * Grants permission to list all the actions for a plugin type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListPluginTypeActions.html
     */
    toListPluginTypeActions() {
        return this.to('ListPluginTypeActions');
    }
    /**
     * Grants permission to list all the plugin type metadata
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListPluginTypeMetadata.html
     */
    toListPluginTypeMetadata() {
        return this.to('ListPluginTypeMetadata');
    }
    /**
     * Grants permission to list the plugins of an application
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListPlugins.html
     */
    toListPlugins() {
        return this.to('ListPlugins');
    }
    /**
     * Grants permission to list the retrievers of an application
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListRetrievers.html
     */
    toListRetrievers() {
        return this.to('ListRetrievers');
    }
    /**
     * Grants permission to list subscriptions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListSubscriptions.html
     */
    toListSubscriptions() {
        return this.to('ListSubscriptions');
    }
    /**
     * Grants permission to list tags for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list the web experiences of an application
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListWebExperiences.html
     */
    toListWebExperiences() {
        return this.to('ListWebExperiences');
    }
    /**
     * Grants permission to put feedback about a conversation message
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_PutFeedback.html
     */
    toPutFeedback() {
        return this.to('PutFeedback');
    }
    /**
     * Grants permission to put a group of users
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_PutGroup.html
     */
    toPutGroup() {
        return this.to('PutGroup');
    }
    /**
     * Grants permission to put resource based policy statement to the application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_AssociatePermission.html
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to search relevant content from the Amazon Q Business Application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_SearchRelevantContent.html
     */
    toSearchRelevantContent() {
        return this.to('SearchRelevantContent');
    }
    /**
     * Grants permission to start Data Source sync job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_StartDataSourceSyncJob.html
     */
    toStartDataSourceSyncJob() {
        return this.to('StartDataSourceSyncJob');
    }
    /**
     * Grants permission to start deployment for an integration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_StartDeployment.html
     */
    toStartDeployment() {
        return this.to('StartDeployment');
    }
    /**
     * Grants permission to stop Data Source sync job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_StopDataSourceSyncJob.html
     */
    toStopDataSourceSyncJob() {
        return this.to('StopDataSourceSyncJob');
    }
    /**
     * Grants permission to tag a resource with given key value pairs
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove the tag with the given key from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update an Application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_UpdateApplication.html
     */
    toUpdateApplication() {
        return this.to('UpdateApplication');
    }
    /**
     * Grants permission to update chat controls configuration for an application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_UpdateChatControlsConfiguration.html
     */
    toUpdateChatControlsConfiguration() {
        return this.to('UpdateChatControlsConfiguration');
    }
    /**
     * Grants permission to update a chat response configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_UpdateChatResponseConfiguration.html
     */
    toUpdateChatResponseConfiguration() {
        return this.to('UpdateChatResponseConfiguration');
    }
    /**
     * Grants permission to update DataAccessor
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_UpdateDataAccessor.html
     */
    toUpdateDataAccessor() {
        return this.to('UpdateDataAccessor');
    }
    /**
     * Grants permission to update a DataSource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_UpdateDataSource.html
     */
    toUpdateDataSource() {
        return this.to('UpdateDataSource');
    }
    /**
     * Grants permission to update an index
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_UpdateIndex.html
     */
    toUpdateIndex() {
        return this.to('UpdateIndex');
    }
    /**
     * Grants permission to update an integration for a Q Business application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_UpdateIntegration.html
     */
    toUpdateIntegration() {
        return this.to('UpdateIntegration');
    }
    /**
     * Grants permission to update a plugin
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_UpdatePlugin.html
     */
    toUpdatePlugin() {
        return this.to('UpdatePlugin');
    }
    /**
     * Grants permission to update a Retriever
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_UpdateRetriever.html
     */
    toUpdateRetriever() {
        return this.to('UpdateRetriever');
    }
    /**
     * Grants permission to update a subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_UpdateSubscription.html
     */
    toUpdateSubscription() {
        return this.to('UpdateSubscription');
    }
    /**
     * Grants permission to update a user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_UpdateUser.html
     */
    toUpdateUser() {
        return this.to('UpdateUser');
    }
    /**
     * Grants permission to update a WebExperience
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_UpdateWebExperience.html
     */
    toUpdateWebExperience() {
        return this.to('UpdateWebExperience');
    }
    /**
     * Adds a resource of type application to the statement
     *
     * https://docs.aws.amazon.com/amazonq/latest/business-use-dg/create-application.html
     *
     * @param applicationId - Identifier for the applicationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onApplication(applicationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:qbusiness:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:application/${applicationId}`);
    }
    /**
     * Adds a resource of type integration to the statement
     *
     * https://docs.aws.amazon.com/amazonq/latest/business-use-dg/create-integration.html
     *
     * @param applicationId - Identifier for the applicationId.
     * @param integrationId - Identifier for the integrationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIntegration(applicationId, integrationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:qbusiness:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:application/${applicationId}/integration/${integrationId}`);
    }
    /**
     * Adds a resource of type retriever to the statement
     *
     * https://docs.aws.amazon.com/amazonq/latest/business-use-dg/select-retriever.html
     *
     * @param applicationId - Identifier for the applicationId.
     * @param retrieverId - Identifier for the retrieverId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRetriever(applicationId, retrieverId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:qbusiness:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:application/${applicationId}/retriever/${retrieverId}`);
    }
    /**
     * Adds a resource of type index to the statement
     *
     * https://docs.aws.amazon.com/amazonq/latest/business-use-dg/select-retriever.html
     *
     * @param applicationId - Identifier for the applicationId.
     * @param indexId - Identifier for the indexId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIndex(applicationId, indexId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:qbusiness:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:application/${applicationId}/index/${indexId}`);
    }
    /**
     * Adds a resource of type data-source to the statement
     *
     * https://docs.aws.amazon.com/amazonq/latest/business-use-dg/connect-data.html
     *
     * @param applicationId - Identifier for the applicationId.
     * @param indexId - Identifier for the indexId.
     * @param dataSourceId - Identifier for the dataSourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDataSource(applicationId, indexId, dataSourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:qbusiness:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:application/${applicationId}/index/${indexId}/data-source/${dataSourceId}`);
    }
    /**
     * Adds a resource of type plugin to the statement
     *
     * https://docs.aws.amazon.com/amazonq/latest/business-use-dg/plugins.html
     *
     * @param applicationId - Identifier for the applicationId.
     * @param pluginId - Identifier for the pluginId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPlugin(applicationId, pluginId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:qbusiness:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:application/${applicationId}/plugin/${pluginId}`);
    }
    /**
     * Adds a resource of type web-experience to the statement
     *
     * https://docs.aws.amazon.com/amazonq/latest/business-use-dg/using-web-experience.html
     *
     * @param applicationId - Identifier for the applicationId.
     * @param webExperienceId - Identifier for the webExperienceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWebExperience(applicationId, webExperienceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:qbusiness:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:application/${applicationId}/web-experience/${webExperienceId}`);
    }
    /**
     * Adds a resource of type subscription to the statement
     *
     * https://docs.aws.amazon.com/amazonq/latest/business-use-dg/subscriptions.html
     *
     * @param applicationId - Identifier for the applicationId.
     * @param subscriptionId - Identifier for the subscriptionId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onSubscription(applicationId, subscriptionId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:qbusiness:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:application/${applicationId}/subscription/${subscriptionId}`);
    }
    /**
     * Adds a resource of type data-accessor to the statement
     *
     * https://docs.aws.amazon.com/amazonq/latest/business-use-dg/data-accessors.html
     *
     * @param applicationId - Identifier for the applicationId.
     * @param dataAccessorId - Identifier for the dataAccessorId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDataAccessor(applicationId, dataAccessorId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:qbusiness:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:application/${applicationId}/data-accessor/${dataAccessorId}`);
    }
    /**
     * Adds a resource of type chat-response-configuration to the statement
     *
     * https://docs.aws.amazon.com/amazonq/latest/business-use-dg/response-customization.html
     *
     * @param applicationId - Identifier for the applicationId.
     * @param chatResponseConfigurationId - Identifier for the chatResponseConfigurationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onChatResponseConfiguration(applicationId, chatResponseConfigurationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:qbusiness:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:application/${applicationId}/chat-response-configuration/${chatResponseConfigurationId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateApplication()
     * - .toCreateChatResponseConfiguration()
     * - .toCreateDataAccessor()
     * - .toCreateDataSource()
     * - .toCreateIndex()
     * - .toCreateIntegration()
     * - .toCreatePlugin()
     * - .toCreateRetriever()
     * - .toCreateWebExperience()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toAllowVendedLogDeliveryForResource()
     *
     * Applies to resource types:
     * - application
     * - integration
     * - retriever
     * - index
     * - data-source
     * - plugin
     * - web-experience
     * - data-accessor
     * - chat-response-configuration
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateApplication()
     * - .toCreateChatResponseConfiguration()
     * - .toCreateDataAccessor()
     * - .toCreateDataSource()
     * - .toCreateIndex()
     * - .toCreateIntegration()
     * - .toCreatePlugin()
     * - .toCreateRetriever()
     * - .toCreateWebExperience()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by IAM Identity Center Group ID
     *
     * https://docs.aws.amazon.com/amazonq/latest/qbusiness-ug/security_iam_service-with-iam.html
     *
     * Applies to actions:
     * - .toCreateSubscription()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifIdentitystoreGroupId(value, operator) {
        return this.if(`identitystore:GroupId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by IAM Identity Center User ID
     *
     * https://docs.aws.amazon.com/amazonq/latest/qbusiness-ug/security_iam_service-with-iam.html
     *
     * Applies to actions:
     * - .toCreateSubscription()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifIdentitystoreUserId(value, operator) {
        return this.if(`identitystore:UserId`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [qbusiness](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonqbusiness.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'qbusiness';
        this.accessLevelList = {
            'Permissions management': [
                'AllowVendedLogDeliveryForResource'
            ],
            Write: [
                'AssociatePermission',
                'BatchDeleteDocument',
                'BatchPutDocument',
                'CancelSubscription',
                'CreateAnonymousWebExperienceUrl',
                'CreateApplication',
                'CreateChatResponseConfiguration',
                'CreateDataAccessor',
                'CreateDataAccessorWithTti',
                'CreateDataSource',
                'CreateIndex',
                'CreateIntegration',
                'CreatePlugin',
                'CreateRetriever',
                'CreateSubscription',
                'CreateUser',
                'CreateWebExperience',
                'DeleteApplication',
                'DeleteAttachment',
                'DeleteChatControlsConfiguration',
                'DeleteChatResponseConfiguration',
                'DeleteConversation',
                'DeleteDataAccessor',
                'DeleteDataSource',
                'DeleteGroup',
                'DeleteIndex',
                'DeleteIntegration',
                'DeletePlugin',
                'DeleteRetriever',
                'DeleteUser',
                'DeleteWebExperience',
                'DisableAclOnDataSource',
                'DisassociatePermission',
                'PutFeedback',
                'PutGroup',
                'PutResourcePolicy',
                'StartDataSourceSyncJob',
                'StartDeployment',
                'StopDataSourceSyncJob',
                'UpdateApplication',
                'UpdateChatControlsConfiguration',
                'UpdateChatResponseConfiguration',
                'UpdateDataAccessor',
                'UpdateDataSource',
                'UpdateIndex',
                'UpdateIntegration',
                'UpdatePlugin',
                'UpdateRetriever',
                'UpdateSubscription',
                'UpdateUser',
                'UpdateWebExperience'
            ],
            Read: [
                'Chat',
                'ChatSync',
                'CheckDocumentAccess',
                'GetApplication',
                'GetChatResponseConfiguration',
                'GetDataAccessor',
                'GetDataSource',
                'GetDocumentContent',
                'GetGroup',
                'GetIndex',
                'GetIntegration',
                'GetMedia',
                'GetPlugin',
                'GetPolicy',
                'GetRetriever',
                'GetUser',
                'GetWebExperience',
                'ListPluginActions',
                'ListPluginTypeActions',
                'ListPluginTypeMetadata',
                'ListTagsForResource',
                'SearchRelevantContent'
            ],
            List: [
                'GetChatControlsConfiguration',
                'ListApplications',
                'ListAttachments',
                'ListChatResponseConfigurations',
                'ListConversations',
                'ListDataAccessors',
                'ListDataSourceSyncJobs',
                'ListDataSources',
                'ListDocuments',
                'ListGroups',
                'ListIndices',
                'ListIntegrations',
                'ListMessages',
                'ListPlugins',
                'ListRetrievers',
                'ListSubscriptions',
                'ListWebExperiences'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Qbusiness = Qbusiness;
//# sourceMappingURL=data:application/json;base64,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