"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Qapps = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [qapps](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonqbusinessqapps.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Qapps extends shared_1.PolicyStatement {
    /**
     * Grants permission to associate a library item review in the Q Business application environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUserIsAppOwner()
     * - .ifAppIsPublished()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_qapps_AssociateLibraryItemReview.html
     */
    toAssociateLibraryItemReview() {
        return this.to('AssociateLibraryItemReview');
    }
    /**
     * Grants permission to associate Q App with a user in the Q Business application environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUserIsAppOwner()
     * - .ifAppIsPublished()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_qapps_AssociateQAppWithUser.html
     */
    toAssociateQAppWithUser() {
        return this.to('AssociateQAppWithUser');
    }
    /**
     * Grants permission to create the categories of a library in the Q Business application environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_qapps_BatchCreateCategory.html
     */
    toBatchCreateCategory() {
        return this.to('BatchCreateCategory');
    }
    /**
     * Grants permission to delete the categories of a library in the Q Business application environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_qapps_BatchDeleteCategory.html
     */
    toBatchDeleteCategory() {
        return this.to('BatchDeleteCategory');
    }
    /**
     * Grants permission to update the categories of a library in the Q Business application environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_qapps_BatchUpdateCategory.html
     */
    toBatchUpdateCategory() {
        return this.to('BatchUpdateCategory');
    }
    /**
     * Grants permission to copy Q App in the Q Business application environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUserIsAppOwner()
     * - .ifAppIsPublished()
     *
     * https://docs.aws.amazon.com/amazonq/latest/qbusiness-ug/purpose-built-qapps.html
     */
    toCopyQApp() {
        return this.to('CopyQApp');
    }
    /**
     * Grants permission to create a library item in the Q Business application environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUserIsAppOwner()
     * - .ifAppIsPublished()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_qapps_CreateLibraryItem.html
     */
    toCreateLibraryItem() {
        return this.to('CreateLibraryItem');
    }
    /**
     * Grants permission to create a library item review in the Q Business application environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUserIsAppOwner()
     * - .ifAppIsPublished()
     *
     * https://docs.aws.amazon.com/amazonq/latest/qbusiness-ug/purpose-built-qapps.html
     */
    toCreateLibraryItemReview() {
        return this.to('CreateLibraryItemReview');
    }
    /**
     * Grants permission to create Q App in the Q Business application environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_qapps_CreateQApp.html
     */
    toCreateQApp() {
        return this.to('CreateQApp');
    }
    /**
     * Grants permission to subscribe to a Q App event bus topic in the Q Business application environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/qbusiness-ug/purpose-built-qapps.html
     */
    toCreateSubscriptionToken() {
        return this.to('CreateSubscriptionToken');
    }
    /**
     * Grants permission to delete a library item in the Q Business application environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUserIsAppOwner()
     * - .ifAppIsPublished()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_qapps_DeleteLibraryItem.html
     */
    toDeleteLibraryItem() {
        return this.to('DeleteLibraryItem');
    }
    /**
     * Grants permission to delete Q App in the Q Business application environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUserIsAppOwner()
     * - .ifAppIsPublished()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_qapps_DeleteQApp.html
     */
    toDeleteQApp() {
        return this.to('DeleteQApp');
    }
    /**
     * Grants permission to get Q App sharing permissions in the Q Business application environment
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifUserIsAppOwner()
     * - .ifAppIsPublished()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_qapps_DescribeQAppPermissions.html
     */
    toDescribeQAppPermissions() {
        return this.to('DescribeQAppPermissions');
    }
    /**
     * Grants permission to disassociate a library item review in the Q Business application environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUserIsAppOwner()
     * - .ifAppIsPublished()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_qapps_DisassociateLibraryItemReview.html
     */
    toDisassociateLibraryItemReview() {
        return this.to('DisassociateLibraryItemReview');
    }
    /**
     * Grants permission to disassociate Q App with a user in the Q Business application environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUserIsAppOwner()
     * - .ifAppIsPublished()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_qapps_DisassociateQAppFromUser.html
     */
    toDisassociateQAppFromUser() {
        return this.to('DisassociateQAppFromUser');
    }
    /**
     * Grants permission to export Q App session data in the Q Business application environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/qbusiness-ug/purpose-built-qapps.html
     */
    toExportQAppSessionData() {
        return this.to('ExportQAppSessionData');
    }
    /**
     * Grants permission to get a library item in the Q Business application environment
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifUserIsAppOwner()
     * - .ifAppIsPublished()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_qapps_GetLibraryItem.html
     */
    toGetLibraryItem() {
        return this.to('GetLibraryItem');
    }
    /**
     * Grants permission to get Q App in the Q Business application environment
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifUserIsAppOwner()
     * - .ifAppIsPublished()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_qapps_GetQApp.html
     */
    toGetQApp() {
        return this.to('GetQApp');
    }
    /**
     * Grants permission to get Q App session in the Q Business application environment
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifUserIsAppOwner()
     * - .ifAppIsPublished()
     * - .ifUserIsSessionModerator()
     * - .ifSessionIsShared()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_qapps_GetQAppSession.html
     */
    toGetQAppSession() {
        return this.to('GetQAppSession');
    }
    /**
     * Grants permission to get Q App session metadata in the Q Business application environment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/qbusiness-ug/purpose-built-qapps.html
     */
    toGetQAppSessionMetadata() {
        return this.to('GetQAppSessionMetadata');
    }
    /**
     * Grants permission to import a document to Q App or Q App Session in the Q Business application environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUserIsAppOwner()
     * - .ifAppIsPublished()
     * - .ifUserIsSessionModerator()
     * - .ifSessionIsShared()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_qapps_ImportDocument.html
     */
    toImportDocument() {
        return this.to('ImportDocument');
    }
    /**
     * Grants permission to list categories in the Q Business application environment
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_qapps_ListCategories.html
     */
    toListCategories() {
        return this.to('ListCategories');
    }
    /**
     * Grants permission to list library items in the Q Business application environment
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_qapps_ListLibraryItems.html
     */
    toListLibraryItems() {
        return this.to('ListLibraryItems');
    }
    /**
     * Grants permission to get Q App session data in the Q Business application environment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/qbusiness-ug/purpose-built-qapps.html
     */
    toListQAppSessionData() {
        return this.to('ListQAppSessionData');
    }
    /**
     * Grants permission to list Q Apps in the Q Business application environment
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_qapps_ListQApps.html
     */
    toListQApps() {
        return this.to('ListQApps');
    }
    /**
     * Grants permission to list tags for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to predict problem statement from conversation log in the Q Business application environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/qbusiness-ug/purpose-built-qapps.html
     */
    toPredictProblemStatementFromConversation() {
        return this.to('PredictProblemStatementFromConversation');
    }
    /**
     * Grants permission to predict Q App from conversation log or problem statement in the Q Business application environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_qapps_PredictQApp.html
     */
    toPredictQApp() {
        return this.to('PredictQApp');
    }
    /**
     * Grants permission to predict Q App metadata from problem statement in the Q Business application environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/qbusiness-ug/purpose-built-qapps.html
     */
    toPredictQAppFromProblemStatement() {
        return this.to('PredictQAppFromProblemStatement');
    }
    /**
     * Grants permission to start Q App session in the Q Business application environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUserIsAppOwner()
     * - .ifAppIsPublished()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_qapps_StartQAppSession.html
     */
    toStartQAppSession() {
        return this.to('StartQAppSession');
    }
    /**
     * Grants permission to stop Q App session in the Q Business application environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUserIsAppOwner()
     * - .ifAppIsPublished()
     * - .ifUserIsSessionModerator()
     * - .ifSessionIsShared()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_qapps_StopQAppSession.html
     */
    toStopQAppSession() {
        return this.to('StopQAppSession');
    }
    /**
     * Grants permission to tag a resource with given key value pairs
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove the tag with the given key from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a library item in the Q Business application environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUserIsAppOwner()
     * - .ifAppIsPublished()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_qapps_UpdateLibraryItem.html
     */
    toUpdateLibraryItem() {
        return this.to('UpdateLibraryItem');
    }
    /**
     * Grants permission to update the metadata of a library item in the Q Business application environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAppIsPublished()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_qapps_UpdateLibraryItemMetadata.html
     */
    toUpdateLibraryItemMetadata() {
        return this.to('UpdateLibraryItemMetadata');
    }
    /**
     * Grants permission to update Q App in the Q Business application environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUserIsAppOwner()
     * - .ifAppIsPublished()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_qapps_UpdateQApp.html
     */
    toUpdateQApp() {
        return this.to('UpdateQApp');
    }
    /**
     * Grants permission to update Q App sharing permissions in the Q Business application environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUserIsAppOwner()
     * - .ifAppIsPublished()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_qapps_UpdateQAppPermissions.html
     */
    toUpdateQAppPermissions() {
        return this.to('UpdateQAppPermissions');
    }
    /**
     * Grants permission to update Q App session in the Q Business application environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUserIsAppOwner()
     * - .ifAppIsPublished()
     * - .ifUserIsSessionModerator()
     * - .ifSessionIsShared()
     *
     * https://docs.aws.amazon.com/amazonq/latest/api-reference/API_qapps_UpdateQAppSession.html
     */
    toUpdateQAppSession() {
        return this.to('UpdateQAppSession');
    }
    /**
     * Grants permission to update Q App session metadata in the Q Business application environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonq/latest/qbusiness-ug/purpose-built-qapps.html
     */
    toUpdateQAppSessionMetadata() {
        return this.to('UpdateQAppSessionMetadata');
    }
    /**
     * Adds a resource of type application to the statement
     *
     * https://docs.aws.amazon.com/amazonq/latest/qbusiness-ug/create-app.html
     *
     * @param applicationId - Identifier for the applicationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onApplication(applicationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:qbusiness:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:application/${applicationId}`);
    }
    /**
     * Adds a resource of type qapp to the statement
     *
     * https://docs.aws.amazon.com/amazonq/latest/qbusiness-ug/purpose-built-qapps.html
     *
     * @param applicationId - Identifier for the applicationId.
     * @param appId - Identifier for the appId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onQapp(applicationId, appId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:qapps:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:application/${applicationId}/qapp/${appId}`);
    }
    /**
     * Adds a resource of type qapp-session to the statement
     *
     * https://docs.aws.amazon.com/amazonq/latest/qbusiness-ug/purpose-built-qapps.html
     *
     * @param applicationId - Identifier for the applicationId.
     * @param appId - Identifier for the appId.
     * @param sessionId - Identifier for the sessionId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onQappSession(applicationId, appId, sessionId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:qapps:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:application/${applicationId}/qapp/${appId}/session/${sessionId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateQApp()
     * - .toStartQAppSession()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - qapp
     * - qapp-session
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateQApp()
     * - .toStartQAppSession()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by whether Q App is published
     *
     * https://docs.aws.amazon.com/amazonq/latest/qbusiness-ug/security-iam.html
     *
     * Applies to actions:
     * - .toAssociateLibraryItemReview()
     * - .toAssociateQAppWithUser()
     * - .toCopyQApp()
     * - .toCreateLibraryItem()
     * - .toCreateLibraryItemReview()
     * - .toDeleteLibraryItem()
     * - .toDeleteQApp()
     * - .toDescribeQAppPermissions()
     * - .toDisassociateLibraryItemReview()
     * - .toDisassociateQAppFromUser()
     * - .toGetLibraryItem()
     * - .toGetQApp()
     * - .toGetQAppSession()
     * - .toImportDocument()
     * - .toStartQAppSession()
     * - .toStopQAppSession()
     * - .toUpdateLibraryItem()
     * - .toUpdateLibraryItemMetadata()
     * - .toUpdateQApp()
     * - .toUpdateQAppPermissions()
     * - .toUpdateQAppSession()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAppIsPublished(value, operator) {
        return this.if(`AppIsPublished`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by whether Q App Session is shared
     *
     * https://docs.aws.amazon.com/amazonq/latest/qbusiness-ug/security-iam.html
     *
     * Applies to actions:
     * - .toGetQAppSession()
     * - .toImportDocument()
     * - .toStopQAppSession()
     * - .toUpdateQAppSession()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSessionIsShared(value, operator) {
        return this.if(`SessionIsShared`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by whether requester is Q App owner
     *
     * https://docs.aws.amazon.com/amazonq/latest/qbusiness-ug/security-iam.html
     *
     * Applies to actions:
     * - .toAssociateLibraryItemReview()
     * - .toAssociateQAppWithUser()
     * - .toCopyQApp()
     * - .toCreateLibraryItem()
     * - .toCreateLibraryItemReview()
     * - .toDeleteLibraryItem()
     * - .toDeleteQApp()
     * - .toDescribeQAppPermissions()
     * - .toDisassociateLibraryItemReview()
     * - .toDisassociateQAppFromUser()
     * - .toGetLibraryItem()
     * - .toGetQApp()
     * - .toGetQAppSession()
     * - .toImportDocument()
     * - .toStartQAppSession()
     * - .toStopQAppSession()
     * - .toUpdateLibraryItem()
     * - .toUpdateQApp()
     * - .toUpdateQAppPermissions()
     * - .toUpdateQAppSession()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifUserIsAppOwner(value, operator) {
        return this.if(`UserIsAppOwner`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by whether requester is Q App Session moderator
     *
     * https://docs.aws.amazon.com/amazonq/latest/qbusiness-ug/security-iam.html
     *
     * Applies to actions:
     * - .toGetQAppSession()
     * - .toImportDocument()
     * - .toStopQAppSession()
     * - .toUpdateQAppSession()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifUserIsSessionModerator(value, operator) {
        return this.if(`UserIsSessionModerator`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [qapps](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonqbusinessqapps.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'qapps';
        this.accessLevelList = {
            Write: [
                'AssociateLibraryItemReview',
                'AssociateQAppWithUser',
                'BatchCreateCategory',
                'BatchDeleteCategory',
                'BatchUpdateCategory',
                'CopyQApp',
                'CreateLibraryItem',
                'CreateLibraryItemReview',
                'CreateQApp',
                'CreateSubscriptionToken',
                'DeleteLibraryItem',
                'DeleteQApp',
                'DisassociateLibraryItemReview',
                'DisassociateQAppFromUser',
                'ExportQAppSessionData',
                'ImportDocument',
                'PredictProblemStatementFromConversation',
                'PredictQApp',
                'PredictQAppFromProblemStatement',
                'StartQAppSession',
                'StopQAppSession',
                'UpdateLibraryItem',
                'UpdateLibraryItemMetadata',
                'UpdateQApp',
                'UpdateQAppPermissions',
                'UpdateQAppSession',
                'UpdateQAppSessionMetadata'
            ],
            Read: [
                'DescribeQAppPermissions',
                'GetLibraryItem',
                'GetQApp',
                'GetQAppSession',
                'GetQAppSessionMetadata',
                'ListQAppSessionData',
                'ListTagsForResource'
            ],
            List: [
                'ListCategories',
                'ListLibraryItems',
                'ListQApps'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Qapps = Qapps;
//# sourceMappingURL=data:application/json;base64,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