"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Wisdom = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [wisdom](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonqinconnect.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Wisdom extends shared_1.PolicyStatement {
    /**
     * Grants permission to activate a message template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_ActivateMessageTemplate.html
     */
    toActivateMessageTemplate() {
        return this.to('ActivateMessageTemplate');
    }
    /**
     * Grants permission to configure vended log delivery for an assistant
     *
     * Access Level: Permissions management
     */
    toAllowVendedLogDeliveryForResource() {
        return this.to('AllowVendedLogDeliveryForResource');
    }
    /**
     * Grants permission to create an ai agent
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_CreateAIAgent.html
     */
    toCreateAIAgent() {
        return this.to('CreateAIAgent');
    }
    /**
     * Grants permission to create an ai agent version
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_CreateAIAgentVersion.html
     */
    toCreateAIAgentVersion() {
        return this.to('CreateAIAgentVersion');
    }
    /**
     * Grants permission to create an ai guardrail
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_CreateAIGuardrail.html
     */
    toCreateAIGuardrail() {
        return this.to('CreateAIGuardrail');
    }
    /**
     * Grants permission to create an ai guardrail version
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_CreateAIGuardrailVersion.html
     */
    toCreateAIGuardrailVersion() {
        return this.to('CreateAIGuardrailVersion');
    }
    /**
     * Grants permission to create an ai prompt
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_CreateAIPrompt.html
     */
    toCreateAIPrompt() {
        return this.to('CreateAIPrompt');
    }
    /**
     * Grants permission to create an ai prompt version
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_CreateAIPromptVersion.html
     */
    toCreateAIPromptVersion() {
        return this.to('CreateAIPromptVersion');
    }
    /**
     * Grants permission to create an assistant
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_CreateAssistant.html
     */
    toCreateAssistant() {
        return this.to('CreateAssistant');
    }
    /**
     * Grants permission to create an association between an assistant and another resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_CreateAssistantAssociation.html
     */
    toCreateAssistantAssociation() {
        return this.to('CreateAssistantAssociation');
    }
    /**
     * Grants permission to create content
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_CreateContent.html
     */
    toCreateContent() {
        return this.to('CreateContent');
    }
    /**
     * Grants permission to create a content association
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_CreateContentAssociation.html
     */
    toCreateContentAssociation() {
        return this.to('CreateContentAssociation');
    }
    /**
     * Grants permission to create a knowledge base
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_CreateKnowledgeBase.html
     */
    toCreateKnowledgeBase() {
        return this.to('CreateKnowledgeBase');
    }
    /**
     * Grants permission to create a message template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_CreateMessageTemplate.html
     */
    toCreateMessageTemplate() {
        return this.to('CreateMessageTemplate');
    }
    /**
     * Grants permission to create an attachment to a message template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_CreateMessageTemplateAttachment.html
     */
    toCreateMessageTemplateAttachment() {
        return this.to('CreateMessageTemplateAttachment');
    }
    /**
     * Grants permission to create a version of a message template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_CreateMessageTemplateVersion.html
     */
    toCreateMessageTemplateVersion() {
        return this.to('CreateMessageTemplateVersion');
    }
    /**
     * Grants permission to create quick response
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_CreateQuickResponse.html
     */
    toCreateQuickResponse() {
        return this.to('CreateQuickResponse');
    }
    /**
     * Grants permission to create a session
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_CreateSession.html
     */
    toCreateSession() {
        return this.to('CreateSession');
    }
    /**
     * Grants permission to deactivate a message template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_DeactivateMessageTemplate.html
     */
    toDeactivateMessageTemplate() {
        return this.to('DeactivateMessageTemplate');
    }
    /**
     * Grants permission to delete an ai agent
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_DeleteAIAgent.html
     */
    toDeleteAIAgent() {
        return this.to('DeleteAIAgent');
    }
    /**
     * Grants permission to delete an ai agent version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_DeleteAIAgentVersion.html
     */
    toDeleteAIAgentVersion() {
        return this.to('DeleteAIAgentVersion');
    }
    /**
     * Grants permission to delete an ai guardrail
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_DeleteAIGuardrail.html
     */
    toDeleteAIGuardrail() {
        return this.to('DeleteAIGuardrail');
    }
    /**
     * Grants permission to delete an ai guardrail version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_DeleteAIGuardrailVersion.html
     */
    toDeleteAIGuardrailVersion() {
        return this.to('DeleteAIGuardrailVersion');
    }
    /**
     * Grants permission to delete an ai prompt
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_DeleteAIPrompt.html
     */
    toDeleteAIPrompt() {
        return this.to('DeleteAIPrompt');
    }
    /**
     * Grants permission to delete an ai prompt version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_DeleteAIPromptVersion.html
     */
    toDeleteAIPromptVersion() {
        return this.to('DeleteAIPromptVersion');
    }
    /**
     * Grants permission to delete an assistant
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_DeleteAssistant.html
     */
    toDeleteAssistant() {
        return this.to('DeleteAssistant');
    }
    /**
     * Grants permission to delete an assistant association
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_DeleteAssistantAssociation.html
     */
    toDeleteAssistantAssociation() {
        return this.to('DeleteAssistantAssociation');
    }
    /**
     * Grants permission to delete content
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_DeleteContent.html
     */
    toDeleteContent() {
        return this.to('DeleteContent');
    }
    /**
     * Grants permission to delete a content association
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_DeleteContentAssociation.html
     */
    toDeleteContentAssociation() {
        return this.to('DeleteContentAssociation');
    }
    /**
     * Grants permission to delete a import job of a knowledge base
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_DeleteImportJob.html
     */
    toDeleteImportJob() {
        return this.to('DeleteImportJob');
    }
    /**
     * Grants permission to delete a knowledge base
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_DeleteKnowledgeBase.html
     */
    toDeleteKnowledgeBase() {
        return this.to('DeleteKnowledgeBase');
    }
    /**
     * Grants permission to delete a message template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_DeleteMessageTemplate.html
     */
    toDeleteMessageTemplate() {
        return this.to('DeleteMessageTemplate');
    }
    /**
     * Grants permission to delete an attachment from a message template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_DeleteMessageTemplateAttachment.html
     */
    toDeleteMessageTemplateAttachment() {
        return this.to('DeleteMessageTemplateAttachment');
    }
    /**
     * Grants permission to delete quick response
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_DeleteQuickResponse.html
     */
    toDeleteQuickResponse() {
        return this.to('DeleteQuickResponse');
    }
    /**
     * Grants permission to retrieve information about an ai agent
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_GetAIAgent.html
     */
    toGetAIAgent() {
        return this.to('GetAIAgent');
    }
    /**
     * Grants permission to retrieve information about an ai guardrail
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_GetAIGuardrail.html
     */
    toGetAIGuardrail() {
        return this.to('GetAIGuardrail');
    }
    /**
     * Grants permission to retrieve information about an ai prompt
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_GetAIPrompt.html
     */
    toGetAIPrompt() {
        return this.to('GetAIPrompt');
    }
    /**
     * Grants permission to retrieve information about an assistant
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_GetAssistant.html
     */
    toGetAssistant() {
        return this.to('GetAssistant');
    }
    /**
     * Grants permission to retrieve information about an assistant association
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_GetAssistantAssociation.html
     */
    toGetAssistantAssociation() {
        return this.to('GetAssistantAssociation');
    }
    /**
     * Grants permission to retrieve content, including a pre-signed URL to download the content
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_GetContent.html
     */
    toGetContent() {
        return this.to('GetContent');
    }
    /**
     * Grants permission to retrieve information about a content association
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_GetContentAssociation.html
     */
    toGetContentAssociation() {
        return this.to('GetContentAssociation');
    }
    /**
     * Grants permission to retrieve summary information about the content
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_GetContentSummary.html
     */
    toGetContentSummary() {
        return this.to('GetContentSummary');
    }
    /**
     * Grants permission to retrieve information about the import job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_GetImportJob.html
     */
    toGetImportJob() {
        return this.to('GetImportJob');
    }
    /**
     * Grants permission to retrieve information about the knowledge base
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_GetKnowledgeBase.html
     */
    toGetKnowledgeBase() {
        return this.to('GetKnowledgeBase');
    }
    /**
     * Grants permission to retrieve a message template
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifMessageTemplateRoutingProfileArn()
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_GetMessageTemplate.html
     */
    toGetMessageTemplate() {
        return this.to('GetMessageTemplate');
    }
    /**
     * Grants permission to retrieve for next message in a session
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_GetNextMessage.html
     */
    toGetNextMessage() {
        return this.to('GetNextMessage');
    }
    /**
     * Grants permission to retrieve content
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_GetQuickResponse.html
     */
    toGetQuickResponse() {
        return this.to('GetQuickResponse');
    }
    /**
     * Grants permission to retrieve recommendations for the specified session
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_GetRecommendations.html
     */
    toGetRecommendations() {
        return this.to('GetRecommendations');
    }
    /**
     * Grants permission to retrieve information for a specified session
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_GetSession.html
     */
    toGetSession() {
        return this.to('GetSession');
    }
    /**
     * Grants permission to list information about ai agent versions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_ListAIAgentVersions.html
     */
    toListAIAgentVersions() {
        return this.to('ListAIAgentVersions');
    }
    /**
     * Grants permission to list information about ai agents
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_ListAIAgents.html
     */
    toListAIAgents() {
        return this.to('ListAIAgents');
    }
    /**
     * Grants permission to list information about ai guardrail versions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_ListAIGuardrailVersions.html
     */
    toListAIGuardrailVersions() {
        return this.to('ListAIGuardrailVersions');
    }
    /**
     * Grants permission to list information about ai guardrails
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_ListAIGuardrails.html
     */
    toListAIGuardrails() {
        return this.to('ListAIGuardrails');
    }
    /**
     * Grants permission to list information about ai prompt versions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_ListAIPromptVersions.html
     */
    toListAIPromptVersions() {
        return this.to('ListAIPromptVersions');
    }
    /**
     * Grants permission to list information about ai prompts
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_ListAIPrompts.html
     */
    toListAIPrompts() {
        return this.to('ListAIPrompts');
    }
    /**
     * Grants permission to list information about assistant associations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_ListAssistantAssociations.html
     */
    toListAssistantAssociations() {
        return this.to('ListAssistantAssociations');
    }
    /**
     * Grants permission to list information about assistants
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_ListAssistants.html
     */
    toListAssistants() {
        return this.to('ListAssistants');
    }
    /**
     * Grants permission to list information about content associations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_ListContentAssociations.html
     */
    toListContentAssociations() {
        return this.to('ListContentAssociations');
    }
    /**
     * Grants permission to list the content with a knowledge base
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_ListContents.html
     */
    toListContents() {
        return this.to('ListContents');
    }
    /**
     * Grants permission to list information about knowledge bases
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_ListImportJobs.html
     */
    toListImportJobs() {
        return this.to('ListImportJobs');
    }
    /**
     * Grants permission to list information about knowledge bases
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_ListKnowledgeBases.html
     */
    toListKnowledgeBases() {
        return this.to('ListKnowledgeBases');
    }
    /**
     * Grants permission to list message template versions for the specified message template
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_ListMessageTemplateVersions.html
     */
    toListMessageTemplateVersions() {
        return this.to('ListMessageTemplateVersions');
    }
    /**
     * Grants permission to list the message templates for a knowledge base
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_ListMessageTemplates.html
     */
    toListMessageTemplates() {
        return this.to('ListMessageTemplates');
    }
    /**
     * Grants permission to list messages in a session
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_ListMessages.html
     */
    toListMessages() {
        return this.to('ListMessages');
    }
    /**
     * Grants permission to list the quick response with a knowledge base
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_ListQuickResponses.html
     */
    toListQuickResponses() {
        return this.to('ListQuickResponses');
    }
    /**
     * Grants permission to list the tags for the specified resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to remove the specified recommendations from the specified assistant's queue of newly available recommendations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_NotifyRecommendationsReceived.html
     */
    toNotifyRecommendationsReceived() {
        return this.to('NotifyRecommendationsReceived');
    }
    /**
     * Grants permission to submit feedback
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_PutFeedback.html
     */
    toPutFeedback() {
        return this.to('PutFeedback');
    }
    /**
     * Grants permission to perform a manual search against the specified assistant
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_QueryAssistant.html
     */
    toQueryAssistant() {
        return this.to('QueryAssistant');
    }
    /**
     * Grants permission to remove an ai agent from an assistant
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_RemoveAssistantAIAgent.html
     */
    toRemoveAssistantAIAgent() {
        return this.to('RemoveAssistantAIAgent');
    }
    /**
     * Grants permission to remove a URI template from a knowledge base
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_RemoveKnowledgeBaseTemplateUri.html
     */
    toRemoveKnowledgeBaseTemplateUri() {
        return this.to('RemoveKnowledgeBaseTemplateUri');
    }
    /**
     * Grants permission to render a message template
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifMessageTemplateRoutingProfileArn()
     *
     * Dependent actions:
     * - wisdom:GetMessageTemplate
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_RenderMessageTemplate.html
     */
    toRenderMessageTemplate() {
        return this.to('RenderMessageTemplate');
    }
    /**
     * Grants permission to retrieve knowledge content from specified assistant associations
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_Retrieve.html
     */
    toRetrieve() {
        return this.to('Retrieve');
    }
    /**
     * Grants permission to search for content referencing a specified knowledge base. Can be used to get a specific content resource by its name
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_SearchContent.html
     */
    toSearchContent() {
        return this.to('SearchContent');
    }
    /**
     * Grants permission to search for message templates referencing a specified knowledge base
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifSearchFilterRoutingProfileArn()
     * - .ifSearchFilterQualifier()
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_SearchMessageTemplates.html
     */
    toSearchMessageTemplates() {
        return this.to('SearchMessageTemplates');
    }
    /**
     * Grants permission to search for quick response referencing a specified knowledge base
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifSearchFilterRoutingProfileArn()
     *
     * Dependent actions:
     * - wisdom:GetQuickResponse
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_SearchQuickResponses.html
     */
    toSearchQuickResponses() {
        return this.to('SearchQuickResponses');
    }
    /**
     * Grants permission to search for sessions referencing a specified assistant. Can be used to et a specific session resource by its name
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_SearchSessions.html
     */
    toSearchSessions() {
        return this.to('SearchSessions');
    }
    /**
     * Grants permission to send a message
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_SendMessage.html
     */
    toSendMessage() {
        return this.to('SendMessage');
    }
    /**
     * Grants permission to get a URL to upload content to a knowledge base
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_StartContentUpload.html
     */
    toStartContentUpload() {
        return this.to('StartContentUpload');
    }
    /**
     * Grants permission to create multiple quick responses
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_StartImportJob.html
     */
    toStartImportJob() {
        return this.to('StartImportJob');
    }
    /**
     * Grants permission to add the specified tags to the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove the specified tags from the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update information about an ai agent
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_UpdateAIAgent.html
     */
    toUpdateAIAgent() {
        return this.to('UpdateAIAgent');
    }
    /**
     * Grants permission to update information about an ai guardrail
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_UpdateAIGuardrail.html
     */
    toUpdateAIGuardrail() {
        return this.to('UpdateAIGuardrail');
    }
    /**
     * Grants permission to update information about an ai prompt
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_UpdateAIPrompt.html
     */
    toUpdateAIPrompt() {
        return this.to('UpdateAIPrompt');
    }
    /**
     * Grants permission to update assistant information about an ai agent
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_UpdateAssistantAIAgent.html
     */
    toUpdateAssistantAIAgent() {
        return this.to('UpdateAssistantAIAgent');
    }
    /**
     * Grants permission to update information about the content
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_UpdateContent.html
     */
    toUpdateContent() {
        return this.to('UpdateContent');
    }
    /**
     * Grants permission to update the template URI of a knowledge base
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_UpdateKnowledgeBaseTemplateUri.html
     */
    toUpdateKnowledgeBaseTemplateUri() {
        return this.to('UpdateKnowledgeBaseTemplateUri');
    }
    /**
     * Grants permission to update content of the message template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_UpdateMessageTemplate.html
     */
    toUpdateMessageTemplate() {
        return this.to('UpdateMessageTemplate');
    }
    /**
     * Grants permission to update metadata of the message template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_UpdateMessageTemplateMetadata.html
     */
    toUpdateMessageTemplateMetadata() {
        return this.to('UpdateMessageTemplateMetadata');
    }
    /**
     * Grants permission to update information or content of the quick response
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_UpdateQuickResponse.html
     */
    toUpdateQuickResponse() {
        return this.to('UpdateQuickResponse');
    }
    /**
     * Grants permission to update a session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_UpdateSession.html
     */
    toUpdateSession() {
        return this.to('UpdateSession');
    }
    /**
     * Grants permission to update data stored in a session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_UpdateSessionData.html
     */
    toUpdateSessionData() {
        return this.to('UpdateSessionData');
    }
    /**
     * Adds a resource of type AIAgent to the statement
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_AIAgentData.html
     *
     * @param assistantId - Identifier for the assistantId.
     * @param aIAgentId - Identifier for the aIAgentId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAIAgent(assistantId, aIAgentId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:wisdom:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:ai-agent/${assistantId}/${aIAgentId}`);
    }
    /**
     * Adds a resource of type AIPrompt to the statement
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_AIPromptData.html
     *
     * @param assistantId - Identifier for the assistantId.
     * @param aIPromptId - Identifier for the aIPromptId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAIPrompt(assistantId, aIPromptId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:wisdom:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:ai-prompt/${assistantId}/${aIPromptId}`);
    }
    /**
     * Adds a resource of type AIGuardrail to the statement
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_AIGuardrailData.html
     *
     * @param assistantId - Identifier for the assistantId.
     * @param aIGuardrailId - Identifier for the aIGuardrailId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAIGuardrail(assistantId, aIGuardrailId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:wisdom:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:ai-guardrail/${assistantId}/${aIGuardrailId}`);
    }
    /**
     * Adds a resource of type Assistant to the statement
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_AssistantData.html
     *
     * @param assistantId - Identifier for the assistantId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAssistant(assistantId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:wisdom:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:assistant/${assistantId}`);
    }
    /**
     * Adds a resource of type AssistantAssociation to the statement
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_AssistantAssociationData.html
     *
     * @param assistantId - Identifier for the assistantId.
     * @param assistantAssociationId - Identifier for the assistantAssociationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAssistantAssociation(assistantId, assistantAssociationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:wisdom:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:association/${assistantId}/${assistantAssociationId}`);
    }
    /**
     * Adds a resource of type Content to the statement
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_ContentData.html
     *
     * @param knowledgeBaseId - Identifier for the knowledgeBaseId.
     * @param contentId - Identifier for the contentId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onContent(knowledgeBaseId, contentId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:wisdom:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:content/${knowledgeBaseId}/${contentId}`);
    }
    /**
     * Adds a resource of type ContentAssociation to the statement
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_ContentAssociationData.html
     *
     * @param knowledgeBaseId - Identifier for the knowledgeBaseId.
     * @param contentId - Identifier for the contentId.
     * @param contentAssociationId - Identifier for the contentAssociationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onContentAssociation(knowledgeBaseId, contentId, contentAssociationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:wisdom:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:content-association/${knowledgeBaseId}/${contentId}/${contentAssociationId}`);
    }
    /**
     * Adds a resource of type KnowledgeBase to the statement
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_KnowledgeBaseData.html
     *
     * @param knowledgeBaseId - Identifier for the knowledgeBaseId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onKnowledgeBase(knowledgeBaseId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:wisdom:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:knowledge-base/${knowledgeBaseId}`);
    }
    /**
     * Adds a resource of type MessageTemplate to the statement
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_MessageTemplateData.html
     *
     * @param knowledgeBaseId - Identifier for the knowledgeBaseId.
     * @param messageTemplateId - Identifier for the messageTemplateId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifMessageTemplateRoutingProfileArn()
     */
    onMessageTemplate(knowledgeBaseId, messageTemplateId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:wisdom:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:message-template/${knowledgeBaseId}/${messageTemplateId}`);
    }
    /**
     * Adds a resource of type Session to the statement
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_SessionData.html
     *
     * @param assistantId - Identifier for the assistantId.
     * @param sessionId - Identifier for the sessionId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSession(assistantId, sessionId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:wisdom:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:session/${assistantId}/${sessionId}`);
    }
    /**
     * Adds a resource of type QuickResponse to the statement
     *
     * https://docs.aws.amazon.com/wisdom/latest/APIReference/API_QuickResponseData.html
     *
     * @param knowledgeBaseId - Identifier for the knowledgeBaseId.
     * @param quickResponseId - Identifier for the quickResponseId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onQuickResponse(knowledgeBaseId, quickResponseId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:wisdom:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:quick-response/${knowledgeBaseId}/${quickResponseId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateAIAgent()
     * - .toCreateAIAgentVersion()
     * - .toCreateAIGuardrail()
     * - .toCreateAIGuardrailVersion()
     * - .toCreateAIPrompt()
     * - .toCreateAIPromptVersion()
     * - .toCreateAssistant()
     * - .toCreateAssistantAssociation()
     * - .toCreateContent()
     * - .toCreateContentAssociation()
     * - .toCreateKnowledgeBase()
     * - .toCreateMessageTemplate()
     * - .toCreateQuickResponse()
     * - .toCreateSession()
     * - .toStartImportJob()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toTagResource()
     * - .toUntagResource()
     *
     * Applies to resource types:
     * - AIAgent
     * - AIPrompt
     * - AIGuardrail
     * - Assistant
     * - AssistantAssociation
     * - Content
     * - ContentAssociation
     * - KnowledgeBase
     * - MessageTemplate
     * - Session
     * - QuickResponse
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateAIAgent()
     * - .toCreateAIAgentVersion()
     * - .toCreateAIGuardrail()
     * - .toCreateAIGuardrailVersion()
     * - .toCreateAIPrompt()
     * - .toCreateAIPromptVersion()
     * - .toCreateAssistant()
     * - .toCreateAssistantAssociation()
     * - .toCreateContent()
     * - .toCreateContentAssociation()
     * - .toCreateKnowledgeBase()
     * - .toCreateMessageTemplate()
     * - .toCreateQuickResponse()
     * - .toCreateSession()
     * - .toStartImportJob()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the connect routing profile arns associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonconnectwisdom.html#amazonconnectwisdom-policy-keys
     *
     * Applies to actions:
     * - .toGetMessageTemplate()
     * - .toRenderMessageTemplate()
     *
     * Applies to resource types:
     * - MessageTemplate
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifMessageTemplateRoutingProfileArn(value, operator) {
        return this.if(`MessageTemplate/RoutingProfileArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the qualifiers that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonconnectwisdom.html#amazonconnectwisdom-policy-keys
     *
     * Applies to actions:
     * - .toSearchMessageTemplates()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSearchFilterQualifier(value, operator) {
        return this.if(`SearchFilter/Qualifier`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the connect routing profile arn that is passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonconnectwisdom.html#amazonconnectwisdom-policy-keys
     *
     * Applies to actions:
     * - .toSearchMessageTemplates()
     * - .toSearchQuickResponses()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifSearchFilterRoutingProfileArn(value, operator) {
        return this.if(`SearchFilter/RoutingProfileArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Statement provider for service [wisdom](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonqinconnect.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'wisdom';
        this.accessLevelList = {
            Write: [
                'ActivateMessageTemplate',
                'CreateAIAgent',
                'CreateAIAgentVersion',
                'CreateAIGuardrail',
                'CreateAIGuardrailVersion',
                'CreateAIPrompt',
                'CreateAIPromptVersion',
                'CreateAssistant',
                'CreateAssistantAssociation',
                'CreateContent',
                'CreateContentAssociation',
                'CreateKnowledgeBase',
                'CreateMessageTemplate',
                'CreateMessageTemplateAttachment',
                'CreateMessageTemplateVersion',
                'CreateQuickResponse',
                'CreateSession',
                'DeactivateMessageTemplate',
                'DeleteAIAgent',
                'DeleteAIAgentVersion',
                'DeleteAIGuardrail',
                'DeleteAIGuardrailVersion',
                'DeleteAIPrompt',
                'DeleteAIPromptVersion',
                'DeleteAssistant',
                'DeleteAssistantAssociation',
                'DeleteContent',
                'DeleteContentAssociation',
                'DeleteImportJob',
                'DeleteKnowledgeBase',
                'DeleteMessageTemplate',
                'DeleteMessageTemplateAttachment',
                'DeleteQuickResponse',
                'NotifyRecommendationsReceived',
                'PutFeedback',
                'RemoveAssistantAIAgent',
                'RemoveKnowledgeBaseTemplateUri',
                'SendMessage',
                'StartContentUpload',
                'StartImportJob',
                'UpdateAIAgent',
                'UpdateAIGuardrail',
                'UpdateAIPrompt',
                'UpdateAssistantAIAgent',
                'UpdateContent',
                'UpdateKnowledgeBaseTemplateUri',
                'UpdateMessageTemplate',
                'UpdateMessageTemplateMetadata',
                'UpdateQuickResponse',
                'UpdateSession',
                'UpdateSessionData'
            ],
            'Permissions management': [
                'AllowVendedLogDeliveryForResource'
            ],
            Read: [
                'GetAIAgent',
                'GetAIGuardrail',
                'GetAIPrompt',
                'GetAssistant',
                'GetAssistantAssociation',
                'GetContent',
                'GetContentAssociation',
                'GetContentSummary',
                'GetImportJob',
                'GetKnowledgeBase',
                'GetMessageTemplate',
                'GetNextMessage',
                'GetQuickResponse',
                'GetRecommendations',
                'GetSession',
                'ListTagsForResource',
                'QueryAssistant',
                'RenderMessageTemplate',
                'Retrieve',
                'SearchContent',
                'SearchMessageTemplates',
                'SearchQuickResponses',
                'SearchSessions'
            ],
            List: [
                'ListAIAgentVersions',
                'ListAIAgents',
                'ListAIGuardrailVersions',
                'ListAIGuardrails',
                'ListAIPromptVersions',
                'ListAIPrompts',
                'ListAssistantAssociations',
                'ListAssistants',
                'ListContentAssociations',
                'ListContents',
                'ListImportJobs',
                'ListKnowledgeBases',
                'ListMessageTemplateVersions',
                'ListMessageTemplates',
                'ListMessages',
                'ListQuickResponses'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Wisdom = Wisdom;
//# sourceMappingURL=data:application/json;base64,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