import { AccessLevelList } from '../../shared/access-level';
import { PolicyStatement, Operator } from '../../shared';
import { aws_iam as iam } from "aws-cdk-lib";
/**
 * Statement provider for service [quicksight](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonquicksight.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class Quicksight extends PolicyStatement {
    servicePrefix: string;
    /**
     * Grants permission to enable setting default access to AWS resources
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/accessing-data-sources.html
     */
    toAccountConfigurations(): this;
    /**
     * Grants permission to create reviewed answers for a topic
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_BatchCreateTopicReviewedAnswer.html
     */
    toBatchCreateTopicReviewedAnswer(): this;
    /**
     * Grants permission to delete reviewed answers for a topic
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_BatchDeleteTopicReviewedAnswer.html
     */
    toBatchDeleteTopicReviewedAnswer(): this;
    /**
     * Grants permission to cancel a SPICE ingestions on a dataset
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CancelIngestion.html
     */
    toCancelIngestion(): this;
    /**
     * Grants permission to create an account customization for QuickSight account or namespace
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateAccountCustomization.html
     */
    toCreateAccountCustomization(): this;
    /**
     * Grants permission to subscribe to QuickSight
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifEdition()
     * - .ifDirectoryType()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateAccountSubscription.html
     */
    toCreateAccountSubscription(): this;
    /**
     * Grants permission to provision Amazon QuickSight administrators, authors, and readers
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    toCreateAdmin(): this;
    /**
     * Grants permission to create an analysis from a template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateAnalysis.html
     */
    toCreateAnalysis(): this;
    /**
     * Grants permission to create an Amazon QuickSight brand
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateBrand.html
     */
    toCreateBrand(): this;
    /**
     * Grants permission to create a QuickSight custom permissions resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateCustomPermissions.html
     */
    toCreateCustomPermissions(): this;
    /**
     * Grants permission to create a QuickSight Dashboard
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateDashboard.html
     */
    toCreateDashboard(): this;
    /**
     * Grants permission to create a dataset
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - quicksight:PassDataSource
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateDataSet.html
     */
    toCreateDataSet(): this;
    /**
     * Grants permission to create a data source
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateDataSource.html
     */
    toCreateDataSource(): this;
    /**
     * Grants permission to create a QuickSight email customization template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/customizing-quicksight-email-templates.html
     */
    toCreateEmailCustomizationTemplate(): this;
    /**
     * Grants permission to create a QuickSight folder
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateFolder.html
     */
    toCreateFolder(): this;
    /**
     * Grants permission to add a QuickSight Dashboard, Analysis or Dataset to a QuickSight Folder
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateFolderMembership.html
     */
    toCreateFolderMembership(): this;
    /**
     * Grants permission to create a QuickSight group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateGroup.html
     */
    toCreateGroup(): this;
    /**
     * Grants permission to add a QuickSight user to a QuickSight group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateGroupMembership.html
     */
    toCreateGroupMembership(): this;
    /**
     * Grants permission to create an assignment with one specified IAM Policy ARN that will be assigned to specified groups or users of QuickSight
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateIAMPolicyAssignment.html
     */
    toCreateIAMPolicyAssignment(): this;
    /**
     * Grants permission to start a SPICE ingestion on a dataset
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateIngestion.html
     */
    toCreateIngestion(): this;
    /**
     * Grants permission to create an QuickSight namespace
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ds:CreateIdentityPoolDirectory
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateNamespace.html
     */
    toCreateNamespace(): this;
    /**
     * Grants permission to provision Amazon QuickSight readers
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    toCreateReader(): this;
    /**
     * Grants permission to create a refresh schedule for a dataset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateRefreshSchedule.html
     */
    toCreateRefreshSchedule(): this;
    /**
     * Grants permission to add a group member to a role
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifGroup()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateRoleMembership.html
     */
    toCreateRoleMembership(): this;
    /**
     * Grants permission to create a template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateTemplate.html
     */
    toCreateTemplate(): this;
    /**
     * Grants permission to create a template alias
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateTemplateAlias.html
     */
    toCreateTemplateAlias(): this;
    /**
     * Grants permission to create a theme
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateTheme.html
     */
    toCreateTheme(): this;
    /**
     * Grants permission to create an alias for a theme version
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateThemeAlias.html
     */
    toCreateThemeAlias(): this;
    /**
     * Grants permission to create a topic
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - quicksight:PassDataSet
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateTopic.html
     */
    toCreateTopic(): this;
    /**
     * Grants permission to create a refresh schedule for a topic
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateTopicRefreshSchedule.html
     */
    toCreateTopicRefreshSchedule(): this;
    /**
     * Grants permission to provision Amazon QuickSight authors and readers
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    toCreateUser(): this;
    /**
     * Grants permission to create a vpc connection
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CreateVPCConnection.html
     */
    toCreateVPCConnection(): this;
    /**
     * Grants permission to remove the custom permission associated with an account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteAccountCustomPermission.html
     */
    toDeleteAccountCustomPermission(): this;
    /**
     * Grants permission to delete an account customization for QuickSight account or namespace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteAccountCustomization.html
     */
    toDeleteAccountCustomization(): this;
    /**
     * Grants permission to delete a QuickSight account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteAccountSubscription.html
     */
    toDeleteAccountSubscription(): this;
    /**
     * Grants permission to delete an analysis
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteAnalysis.html
     */
    toDeleteAnalysis(): this;
    /**
     * Grants permission to delete an Amazon QuickSight brand
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteBrand.html
     */
    toDeleteBrand(): this;
    /**
     * Grants permission to delete a brand assignment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteBrandAssignment.html
     */
    toDeleteBrandAssignment(): this;
    /**
     * Grants permission to delete a QuickSight custom permissions resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteCustomPermissions.html
     */
    toDeleteCustomPermissions(): this;
    /**
     * Grants permission to delete a QuickSight Dashboard
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteDashboard.html
     */
    toDeleteDashboard(): this;
    /**
     * Grants permission to delete a dataset
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteDataSet.html
     */
    toDeleteDataSet(): this;
    /**
     * Grants permission to delete dataset refresh properties for a dataset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteDataSetRefreshProperties.html
     */
    toDeleteDataSetRefreshProperties(): this;
    /**
     * Grants permission to delete a data source
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteDataSource.html
     */
    toDeleteDataSource(): this;
    /**
     * Grants permission to delete linked QBusiness application for QuickSight account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteDefaultQBusinessApplication.html
     */
    toDeleteDefaultQBusinessApplication(): this;
    /**
     * Grants permission to delete a QuickSight email customization template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/customizing-quicksight-email-templates.html
     */
    toDeleteEmailCustomizationTemplate(): this;
    /**
     * Grants permission to delete a QuickSight Folder
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteFolder.html
     */
    toDeleteFolder(): this;
    /**
     * Grants permission to remove a QuickSight Dashboard, Analysis or Dataset from a QuickSight Folder
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteFolderMembership.html
     */
    toDeleteFolderMembership(): this;
    /**
     * Grants permission to remove a user group from QuickSight
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteGroup.html
     */
    toDeleteGroup(): this;
    /**
     * Grants permission to remove a user from a group so that he/she is no longer a member of the group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteGroupMembership.html
     */
    toDeleteGroupMembership(): this;
    /**
     * Grants permission to update an existing assignment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteIAMPolicyAssignment.html
     */
    toDeleteIAMPolicyAssignment(): this;
    /**
     * Grants permission to remove AWS services for trusted identity propagation in QuickSight
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteIdentityPropagationConfig.html
     */
    toDeleteIdentityPropagationConfig(): this;
    /**
     * Grants permission to delete a QuickSight namespace
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ds:DeleteDirectory
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteNamespace.html
     */
    toDeleteNamespace(): this;
    /**
     * Grants permission to delete a refresh schedule for a dataset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteRefreshSchedule.html
     */
    toDeleteRefreshSchedule(): this;
    /**
     * Grants permission to remove the custom permission associated with a role
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteRoleCustomPermission.html
     */
    toDeleteRoleCustomPermission(): this;
    /**
     * Grants permission to remove a group member from a role
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifGroup()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteRoleMembership.html
     */
    toDeleteRoleMembership(): this;
    /**
     * Grants permission to delete a template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteTemplate.html
     */
    toDeleteTemplate(): this;
    /**
     * Grants permission to delete a template alias
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteTemplateAlias.html
     */
    toDeleteTemplateAlias(): this;
    /**
     * Grants permission to delete a theme
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteTheme.html
     */
    toDeleteTheme(): this;
    /**
     * Grants permission to delete the alias of a theme
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteThemeAlias.html
     */
    toDeleteThemeAlias(): this;
    /**
     * Grants permission to delete a topic
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteTopic.html
     */
    toDeleteTopic(): this;
    /**
     * Grants permission to delete a refresh schedule for a topic
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteTopicRefreshSchedule.html
     */
    toDeleteTopicRefreshSchedule(): this;
    /**
     * Grants permission to delete a QuickSight user, given the user name
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteUser.html
     */
    toDeleteUser(): this;
    /**
     * Grants permission to delete a user identified by its principal ID
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteUserByPrincipalId.html
     */
    toDeleteUserByPrincipalId(): this;
    /**
     * Grants permission to remove the custom permission associated with a user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteUserCustomPermission.html
     */
    toDeleteUserCustomPermission(): this;
    /**
     * Grants permission to delete a vpc connection
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DeleteVPCConnection.html
     */
    toDeleteVPCConnection(): this;
    /**
     * Grants permission to describe the custom permission associated with an account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeAccountCustomPermission.html
     */
    toDescribeAccountCustomPermission(): this;
    /**
     * Grants permission to describe an account customization for QuickSight account or namespace
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeAccountCustomization.html
     */
    toDescribeAccountCustomization(): this;
    /**
     * Grants permission to describe the administrative account settings for QuickSight account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeAccountSettings.html
     */
    toDescribeAccountSettings(): this;
    /**
     * Grants permission to describe a QuickSight account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeAccountSubscription.html
     */
    toDescribeAccountSubscription(): this;
    /**
     * Grants permission to describe an analysis
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeAnalysis.html
     */
    toDescribeAnalysis(): this;
    /**
     * Grants permission to describe permissions for an analysis
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeAnalysisPermissions.html
     */
    toDescribeAnalysisPermissions(): this;
    /**
     * Grants permission to describe an asset bundle export job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeAssetBundleExportJob.html
     */
    toDescribeAssetBundleExportJob(): this;
    /**
     * Grants permission to describe an asset bundle import job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeAssetBundleImportJob.html
     */
    toDescribeAssetBundleImportJob(): this;
    /**
     * Grants permission to describe a brand
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeBrand.html
     */
    toDescribeBrand(): this;
    /**
     * Grants permission to describe a brand assignment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeBrandAssignment.html
     */
    toDescribeBrandAssignment(): this;
    /**
     * Grants permission to describes the published version of the brand
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeBrandPublishedVersion.html
     */
    toDescribeBrandPublishedVersion(): this;
    /**
     * Grants permission to describe a custom permissions resource in a QuickSight account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeCustomPermissions.html
     */
    toDescribeCustomPermissions(): this;
    /**
     * Grants permission to describe a QuickSight Dashboard
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeDashboard.html
     */
    toDescribeDashboard(): this;
    /**
     * Grants permission to describe permissions for a QuickSight Dashboard
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeDashboardPermissions.html
     */
    toDescribeDashboardPermissions(): this;
    /**
     * Grants permission to describe a dashboard snapshot job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeDashboardSnapshotJob.html
     */
    toDescribeDashboardSnapshotJob(): this;
    /**
     * Grants permission to describe result of a dashboard snapshot job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeDashboardSnapshotJobResult.html
     */
    toDescribeDashboardSnapshotJobResult(): this;
    /**
     * Grants permission to describe dashboards qa configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeDashboardsQAConfiguration.html
     */
    toDescribeDashboardsQAConfiguration(): this;
    /**
     * Grants permission to describe a dataset
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeDataSet.html
     */
    toDescribeDataSet(): this;
    /**
     * Grants permission to describe the resource policy of a dataset
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeDataSetPermissions.html
     */
    toDescribeDataSetPermissions(): this;
    /**
     * Grants permission to describe refresh properties for a dataset
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeDataSetRefreshProperties.html
     */
    toDescribeDataSetRefreshProperties(): this;
    /**
     * Grants permission to describe a data source
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeDataSource.html
     */
    toDescribeDataSource(): this;
    /**
     * Grants permission to describe the resource policy of a data source
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeDataSourcePermissions.html
     */
    toDescribeDataSourcePermissions(): this;
    /**
     * Grants permission to describe linked QBusiness application Id for QuickSight account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeDefaultQBusinessApplication.html
     */
    toDescribeDefaultQBusinessApplication(): this;
    /**
     * Grants permission to describe a QuickSight email customization template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/customizing-quicksight-email-templates.html
     */
    toDescribeEmailCustomizationTemplate(): this;
    /**
     * Grants permission to describe a QuickSight Folder
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeFolder.html
     */
    toDescribeFolder(): this;
    /**
     * Grants permission to describe permissions for a QuickSight Folder
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeFolderPermissions.html
     */
    toDescribeFolderPermissions(): this;
    /**
     * Grants permission to describe resolved permissions for a QuickSight Folder
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeFolderResolvedPermissions.html
     */
    toDescribeFolderResolvedPermissions(): this;
    /**
     * Grants permission to describe a QuickSight group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeGroup.html
     */
    toDescribeGroup(): this;
    /**
     * Grants permission to describe a QuickSight group member
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeGroupMembership.html
     */
    toDescribeGroupMembership(): this;
    /**
     * Grants permission to describe an existing assignment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeIAMPolicyAssignment.html
     */
    toDescribeIAMPolicyAssignment(): this;
    /**
     * Grants permission to describe a SPICE ingestion on a dataset
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeIngestion.html
     */
    toDescribeIngestion(): this;
    /**
     * Grants permission to describe the IP restrictions for QuickSight account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeIpRestriction.html
     */
    toDescribeIpRestriction(): this;
    /**
     * Grants permission to describe QuickSight key registration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeKeyRegistration.html
     */
    toDescribeKeyRegistration(): this;
    /**
     * Grants permission to describe a QuickSight namespace
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeNamespace.html
     */
    toDescribeNamespace(): this;
    /**
     * Grants permission to describe a personalization configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeQPersonalizationConfiguration.html
     */
    toDescribeQPersonalizationConfiguration(): this;
    /**
     * Grants permission to describe QuickSight Q Search configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeQuickSightQSearchConfiguration.html
     */
    toDescribeQuickSightQSearchConfiguration(): this;
    /**
     * Grants permission to describe a refresh schedule for a dataset
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeRefreshSchedule.html
     */
    toDescribeRefreshSchedule(): this;
    /**
     * Grants permission to describe the custom permission associated with a role
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeRoleCustomPermission.html
     */
    toDescribeRoleCustomPermission(): this;
    /**
     * Grants permission to describe a template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeTemplate.html
     */
    toDescribeTemplate(): this;
    /**
     * Grants permission to describe a template alias
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeTemplateAlias.html
     */
    toDescribeTemplateAlias(): this;
    /**
     * Grants permission to describe permissions for a template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeTemplatePermissions.html
     */
    toDescribeTemplatePermissions(): this;
    /**
     * Grants permission to describe a theme
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeTheme.html
     */
    toDescribeTheme(): this;
    /**
     * Grants permission to describe a theme alias
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeThemeAlias.html
     */
    toDescribeThemeAlias(): this;
    /**
     * Grants permission to describe permissions for a theme
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeThemePermissions.html
     */
    toDescribeThemePermissions(): this;
    /**
     * Grants permission to describe a topic
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeTopic.html
     */
    toDescribeTopic(): this;
    /**
     * Grants permission to describe the resource policy of a topic
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeTopicPermissions.html
     */
    toDescribeTopicPermissions(): this;
    /**
     * Grants permission to describe the refresh status of a topic
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeTopicRefresh.html
     */
    toDescribeTopicRefresh(): this;
    /**
     * Grants permission to describe a refresh schedule for a topic
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeTopicRefreshSchedule.html
     */
    toDescribeTopicRefreshSchedule(): this;
    /**
     * Grants permission to describe a QuickSight user given the user name
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeUser.html
     */
    toDescribeUser(): this;
    /**
     * Grants permission to describe a vpc connection
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DescribeVPCConnection.html
     */
    toDescribeVPCConnection(): this;
    /**
     * Grants permission to generate a URL used to embed a QuickSight Dashboard or Q Topic for a user not registered with QuickSight
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAllowedEmbeddingDomains()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_GenerateEmbedUrlForAnonymousUser.html
     */
    toGenerateEmbedUrlForAnonymousUser(): this;
    /**
     * Grants permission to generate a URL used to embed a QuickSight Dashboard for a user registered with QuickSight
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAllowedEmbeddingDomains()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_GenerateEmbedUrlForRegisteredUser.html
     */
    toGenerateEmbedUrlForRegisteredUser(): this;
    /**
     * Grants permission to generate a URL used to embed a QuickSight Experience for a user registered with QuickSight using Identity-enhanced role session
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAllowedEmbeddingDomains()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_GenerateEmbedUrlForRegisteredUserWithIdentity.html
     */
    toGenerateEmbedUrlForRegisteredUserWithIdentity(): this;
    /**
     * Grants permission to get a URL used to embed a QuickSight Dashboard for a user not registered with QuickSight
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    toGetAnonymousUserEmbedUrl(): this;
    /**
     * Grants permission to get an auth code representing a QuickSight user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    toGetAuthCode(): this;
    /**
     * Grants permission to get a URL used to embed a QuickSight Dashboard
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_GetDashboardEmbedUrl.html
     */
    toGetDashboardEmbedUrl(): this;
    /**
     * Grants permission to use Amazon QuickSight, in Enterprise edition, to identify and display the Microsoft Active Directory (Microsoft Active Directory) directory groups that are mapped to roles in Amazon QuickSight
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    toGetGroupMapping(): this;
    /**
     * Grants permission to get a URL to embed QuickSight console experience
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_GetSessionEmbedUrl.html
     */
    toGetSessionEmbedUrl(): this;
    /**
     * Grants permission to list all analyses in an account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListAnalyses.html
     */
    toListAnalyses(): this;
    /**
     * Grants permission to list all asset bundle export jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListAssetBundleExportJobs.html
     */
    toListAssetBundleExportJobs(): this;
    /**
     * Grants permission to list all asset bundle import jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListAssetBundleImportJobs.html
     */
    toListAssetBundleImportJobs(): this;
    /**
     * Grants permission to lists all brands in an Amazon QuickSight account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListBrands.html
     */
    toListBrands(): this;
    /**
     * Grants permission to list custom permissions resources in QuickSight account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListCustomPermissions.html
     */
    toListCustomPermissions(): this;
    /**
     * Grants permission to list all registered customer managed keys
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/key-management.html
     */
    toListCustomerManagedKeys(): this;
    /**
     * Grants permission to list all versions of a QuickSight Dashboard
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListDashboardVersions.html
     */
    toListDashboardVersions(): this;
    /**
     * Grants permission to list all Dashboards in a QuickSight Account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListDashboards.html
     */
    toListDashboards(): this;
    /**
     * Grants permission to list all datasets
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListDataSets.html
     */
    toListDataSets(): this;
    /**
     * Grants permission to list all data sources
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListDataSources.html
     */
    toListDataSources(): this;
    /**
     * Grants permission to list all members in a folder
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListFolderMembers.html
     */
    toListFolderMembers(): this;
    /**
     * Grants permission to list all Folders in a QuickSight Account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListFolders.html
     */
    toListFolders(): this;
    /**
     * Grants permission to list all Folders in which a QuickSight resource is a member
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListFoldersForResource.html
     */
    toListFoldersForResource(): this;
    /**
     * Grants permission to list member users in a group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListGroupMemberships.html
     */
    toListGroupMemberships(): this;
    /**
     * Grants permission to list all user groups in QuickSight
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListGroups.html
     */
    toListGroups(): this;
    /**
     * Grants permission to list all assignments in the current Amazon QuickSight account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListIAMPolicyAssignments.html
     */
    toListIAMPolicyAssignments(): this;
    /**
     * Grants permission to list all assignments assigned to a user and the groups it belongs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListIAMPolicyAssignmentsForUser.html
     */
    toListIAMPolicyAssignmentsForUser(): this;
    /**
     * Grants permission to list AWS services enabled for trusted identity propagation in QuickSight
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListIdentityPropagationConfigs.html
     */
    toListIdentityPropagationConfigs(): this;
    /**
     * Grants permission to list all SPICE ingestions on a dataset
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListIngestions.html
     */
    toListIngestions(): this;
    /**
     * Grants permission to list a user's KMS keys
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/key-management.html
     */
    toListKMSKeysForUser(): this;
    /**
     * Grants permission to lists all namespaces in a QuickSight account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListNamespaces.html
     */
    toListNamespaces(): this;
    /**
     * Grants permission to list all refresh schedules on a dataset
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListRefreshSchedules.html
     */
    toListRefreshSchedules(): this;
    /**
     * Grants permission to list the members of a role
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListRoleMemberships.html
     */
    toListRoleMemberships(): this;
    /**
     * Grants permission to list tags of a QuickSight resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource(): this;
    /**
     * Grants permission to list all aliases for a template
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListTemplateAliases.html
     */
    toListTemplateAliases(): this;
    /**
     * Grants permission to list all versions of a template
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListTemplateVersions.html
     */
    toListTemplateVersions(): this;
    /**
     * Grants permission to list all templates in a QuickSight account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListTemplates.html
     */
    toListTemplates(): this;
    /**
     * Grants permission to list all aliases of a theme
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListThemeAliases.html
     */
    toListThemeAliases(): this;
    /**
     * Grants permission to list all versions of a theme
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListThemeVersions.html
     */
    toListThemeVersions(): this;
    /**
     * Grants permission to list all themes in an account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListThemes.html
     */
    toListThemes(): this;
    /**
     * Grants permission to list all refresh schedules on a topic
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListTopicRefreshSchedules.html
     */
    toListTopicRefreshSchedules(): this;
    /**
     * Grants permission to list all reviewed answers for topic
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListTopicReviewedAnswers.html
     */
    toListTopicReviewedAnswers(): this;
    /**
     * Grants permission to list all topics
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListTopics.html
     */
    toListTopics(): this;
    /**
     * Grants permission to list groups that a given user is a member of
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListUserGroups.html
     */
    toListUserGroups(): this;
    /**
     * Grants permission to list all of the QuickSight users belonging to this account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListUsers.html
     */
    toListUsers(): this;
    /**
     * Grants permission to list all vpc connections
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_ListVPCConnections.html
     */
    toListVPCConnections(): this;
    /**
     * Grants permission to use a dataset for a template
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/qs-api-overview.html
     */
    toPassDataSet(): this;
    /**
     * Grants permission to use a data source for a data set
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/qs-api-overview.html
     */
    toPassDataSource(): this;
    /**
     * Grants permission to predict QA results
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_PredictQAResults.html
     */
    toPredictQAResults(): this;
    /**
     * Grants permission to put dataset refresh properties for a dataset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_PutDataSetRefreshProperties.html
     */
    toPutDataSetRefreshProperties(): this;
    /**
     * Grants permission to register a customer managed key
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/key-management.html
     */
    toRegisterCustomerManagedKey(): this;
    /**
     * Grants permission to create a QuickSight user, whose identity is associated with the IAM identity/role specified in the request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_RegisterUser.html
     */
    toRegisterUser(): this;
    /**
     * Grants permission to remove a customer managed key
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/key-management.html
     */
    toRemoveCustomerManagedKey(): this;
    /**
     * Grants permission to restore a deleted analysis
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_RestoreAnalysis.html
     */
    toRestoreAnalysis(): this;
    /**
     * Grants permission to manage scoping policies for permissions to AWS resources
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/accessing-data-sources.html
     */
    toScopeDownPolicy(): this;
    /**
     * Grants permission to search for a sub-set of analyses
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_SearchAnalyses.html
     */
    toSearchAnalyses(): this;
    /**
     * Grants permission to search for a sub-set of QuickSight Dashboards
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_SearchDashboards.html
     */
    toSearchDashboards(): this;
    /**
     * Grants permission to search for a sub-set of QuickSight DatSets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_SearchDataSets.html
     */
    toSearchDataSets(): this;
    /**
     * Grants permission to search for a sub-set of QuickSight Data Sources
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_SearchDataSources.html
     */
    toSearchDataSources(): this;
    /**
     * Grants permission to use Amazon QuickSight, in Enterprise edition, to display your Microsoft Active Directory directory groups so that you can choose which ones to map to roles in Amazon QuickSight
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    toSearchDirectoryGroups(): this;
    /**
     * Grants permission to search for a sub-set of QuickSight Folders
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_SearchFolders.html
     */
    toSearchFolders(): this;
    /**
     * Grants permission to search for a sub-set of QuickSight groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_SearchGroups.html
     */
    toSearchGroups(): this;
    /**
     * Grants permission to search for a sub-set of topics
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_SearchTopics.html
     */
    toSearchTopics(): this;
    /**
     * Grants permission to search the QuickSight users belonging to this account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    toSearchUsers(): this;
    /**
     * Grants permission to use Amazon QuickSight, in Enterprise edition, to display your Microsoft Active Directory directory groups so that you can choose which ones to map to roles in Amazon QuickSight
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    toSetGroupMapping(): this;
    /**
     * Grants permission to start an asset bundle export job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_StartAssetBundleExportJob.html
     */
    toStartAssetBundleExportJob(): this;
    /**
     * Grants permission to start an asset bundle import job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_StartAssetBundleImportJob.html
     */
    toStartAssetBundleImportJob(): this;
    /**
     * Grants permission to start a dashboard snapshot job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_StartDashboardSnapshotJob.html
     */
    toStartDashboardSnapshotJob(): this;
    /**
     * Grants permission to start a dashboard snapshot job schedule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_StartDashboardSnapshotJobSchedule.html
     */
    toStartDashboardSnapshotJobSchedule(): this;
    /**
     * Grants permission to subscribe to Amazon QuickSight, and also to allow the user to upgrade the subscription to Enterprise edition
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifEdition()
     * - .ifDirectoryType()
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    toSubscribe(): this;
    /**
     * Grants permission to add tags to a QuickSight resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_TagResource.html
     */
    toTagResource(): this;
    /**
     * Grants permission to unsubscribe from Amazon QuickSight, which permanently deletes all users and their resources from Amazon QuickSight
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     */
    toUnsubscribe(): this;
    /**
     * Grants permission to remove tags from a QuickSight resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UntagResource.html
     */
    toUntagResource(): this;
    /**
     * Grants permission to update the custom permission associated with an account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateAccountCustomPermission.html
     */
    toUpdateAccountCustomPermission(): this;
    /**
     * Grants permission to update an account customization for QuickSight account or namespace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateAccountCustomization.html
     */
    toUpdateAccountCustomization(): this;
    /**
     * Grants permission to update the administrative account settings for QuickSight account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateAccountSettings.html
     */
    toUpdateAccountSettings(): this;
    /**
     * Grants permission to update an analysis
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateAnalysis.html
     */
    toUpdateAnalysis(): this;
    /**
     * Grants permission to update permissions for an analysis
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateAnalysisPermissions.html
     */
    toUpdateAnalysisPermissions(): this;
    /**
     * Grants permission to update QuickSight IAM Identity Center application with Token Exchange grant
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateApplicationWithTokenExchangeGrant.html
     */
    toUpdateApplicationWithTokenExchangeGrant(): this;
    /**
     * Grants permission to update a brand
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateBrand.html
     */
    toUpdateBrand(): this;
    /**
     * Grants permission to update a brand assignment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateBrandAssignment.html
     */
    toUpdateBrandAssignment(): this;
    /**
     * Grants permission to update the published version of a brand
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateBrandPublishedVersion.html
     */
    toUpdateBrandPublishedVersion(): this;
    /**
     * Grants permission to update a QuickSight custom permissions resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateCustomPermissions.html
     */
    toUpdateCustomPermissions(): this;
    /**
     * Grants permission to update a QuickSight Dashboard
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateDashboard.html
     */
    toUpdateDashboard(): this;
    /**
     * Grants permission to update a QuickSight Dashboard's links
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateDashboardLinks.html
     */
    toUpdateDashboardLinks(): this;
    /**
     * Grants permission to update permissions for a QuickSight Dashboard
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateDashboardPermissions.html
     */
    toUpdateDashboardPermissions(): this;
    /**
     * Grants permission to update a QuickSight Dashboard's Published Version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateDashboardPublishedVersion.html
     */
    toUpdateDashboardPublishedVersion(): this;
    /**
     * Grants permission to update dashboards qa configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateDashboardsQAConfiguration.html
     */
    toUpdateDashboardsQAConfiguration(): this;
    /**
     * Grants permission to update a dataset
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - quicksight:PassDataSource
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateDataSet.html
     */
    toUpdateDataSet(): this;
    /**
     * Grants permission to update the resource policy of a dataset
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateDataSetPermissions.html
     */
    toUpdateDataSetPermissions(): this;
    /**
     * Grants permission to update a data source
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateDataSource.html
     */
    toUpdateDataSource(): this;
    /**
     * Grants permission to update the resource policy of a data source
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateDataSourcePermissions.html
     */
    toUpdateDataSourcePermissions(): this;
    /**
     * Grants permission to update linked QBusiness application Id for QuickSight account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateDefaultQBusinessApplication.html
     */
    toUpdateDefaultQBusinessApplication(): this;
    /**
     * Grants permission to update a QuickSight email customization template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/customizing-quicksight-email-templates.html
     */
    toUpdateEmailCustomizationTemplate(): this;
    /**
     * Grants permission to update a QuickSight Folder
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateFolder.html
     */
    toUpdateFolder(): this;
    /**
     * Grants permission to update permissions for a QuickSight Folder
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateFolderPermissions.html
     */
    toUpdateFolderPermissions(): this;
    /**
     * Grants permission to change group description
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateGroup.html
     */
    toUpdateGroup(): this;
    /**
     * Grants permission to update an existing assignment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateIAMPolicyAssignment.html
     */
    toUpdateIAMPolicyAssignment(): this;
    /**
     * Grants permission to add and update AWS services for trusted identity propagation in QuickSight
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateIdentityPropagationConfig.html
     */
    toUpdateIdentityPropagationConfig(): this;
    /**
     * Grants permission to update the IP restrictions for QuickSight account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateIpRestriction.html
     */
    toUpdateIpRestriction(): this;
    /**
     * Grants permission to update QuickSight key registration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateKeyRegistration.html
     */
    toUpdateKeyRegistration(): this;
    /**
     * Grants permission to enable or disable public sharing on an account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdatePublicSharingSettings.html
     */
    toUpdatePublicSharingSettings(): this;
    /**
     * Grants permission to update a personalization configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateQPersonalizationConfiguration.html
     */
    toUpdateQPersonalizationConfiguration(): this;
    /**
     * Grants permission to update QuickSight Q Search configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateQuickSightQSearchConfiguration.html
     */
    toUpdateQuickSightQSearchConfiguration(): this;
    /**
     * Grants permission to update a refresh schedule for a dataset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateRefreshSchedule.html
     */
    toUpdateRefreshSchedule(): this;
    /**
     * Grants permission to update resource-level permissions in QuickSight
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/security_iam_service-with-iam.html
     */
    toUpdateResourcePermissions(): this;
    /**
     * Grants permission to update the custom permission associated with a role
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateRoleCustomPermission.html
     */
    toUpdateRoleCustomPermission(): this;
    /**
     * Grants permission to update QuickSight SPICE capacity configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateSPICECapacityConfiguration.html
     */
    toUpdateSPICECapacityConfiguration(): this;
    /**
     * Grants permission to update a template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateTemplate.html
     */
    toUpdateTemplate(): this;
    /**
     * Grants permission to update a template alias
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateTemplateAlias.html
     */
    toUpdateTemplateAlias(): this;
    /**
     * Grants permission to update permissions for a template
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateTemplatePermissions.html
     */
    toUpdateTemplatePermissions(): this;
    /**
     * Grants permission to update a theme
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateTheme.html
     */
    toUpdateTheme(): this;
    /**
     * Grants permission to update the alias of a theme
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateThemeAlias.html
     */
    toUpdateThemeAlias(): this;
    /**
     * Grants permission to update permissions for a theme
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateThemePermissions.html
     */
    toUpdateThemePermissions(): this;
    /**
     * Grants permission to update a topic
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - quicksight:PassDataSet
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateTopic.html
     */
    toUpdateTopic(): this;
    /**
     * Grants permission to update the resource policy of a topic
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateTopicPermissions.html
     */
    toUpdateTopicPermissions(): this;
    /**
     * Grants permission to update a refresh schedule for a topic
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateTopicRefreshSchedule.html
     */
    toUpdateTopicRefreshSchedule(): this;
    /**
     * Grants permission to update an Amazon QuickSight user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateUser.html
     */
    toUpdateUser(): this;
    /**
     * Grants permission to update the custom permission associated with a user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateUserCustomPermission.html
     */
    toUpdateUserCustomPermission(): this;
    /**
     * Grants permission to update a vpc connection
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_UpdateVPCConnection.html
     */
    toUpdateVPCConnection(): this;
    protected accessLevelList: AccessLevelList;
    /**
     * Adds a resource of type account to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_AccountInfo.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAccount(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type user to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_User.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onUser(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type group to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_Group.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onGroup(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type analysis to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_Analysis.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAnalysis(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type dashboard to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_Dashboard.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDashboard(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type template to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_Template.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTemplate(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type vpcconnection to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_VPCConnection.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onVpcconnection(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type assetBundleExportJob to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_StartAssetBundleExportJob.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAssetBundleExportJob(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type assetBundleImportJob to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_StartAssetBundleImportJob.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAssetBundleImportJob(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type datasource to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DataSource.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDatasource(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type dataset to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DataSet.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDataset(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type ingestion to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_Ingestion.html
     *
     * @param datasetId - Identifier for the datasetId.
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIngestion(datasetId: string, resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type refreshschedule to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_RefreshSchedule.html
     *
     * @param datasetId - Identifier for the datasetId.
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onRefreshschedule(datasetId: string, resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type theme to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_Theme.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTheme(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type assignment to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_IAMPolicyAssignment.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAssignment(resourceId: string, account?: string, partition?: string): this;
    /**
     * Adds a resource of type customization to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_AccountCustomization.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCustomization(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type namespace to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_NamespaceInfoV2.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onNamespace(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type folder to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_Folder.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFolder(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type emailCustomizationTemplate to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/customizing-quicksight-email-templates.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onEmailCustomizationTemplate(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type topic to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_TopicDetails.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTopic(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type dashboardSnapshotJob to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_DashboardSnapshotJob.html
     *
     * @param dashboardId - Identifier for the dashboardId.
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDashboardSnapshotJob(dashboardId: string, resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type brand to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_BrandDetail.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBrand(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type custompermissions to the statement
     *
     * https://docs.aws.amazon.com/quicksight/latest/APIReference/API_CustomPermissions.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCustompermissions(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Filters access by tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toBatchCreateTopicReviewedAnswer()
     * - .toBatchDeleteTopicReviewedAnswer()
     * - .toCancelIngestion()
     * - .toCreateAccountCustomization()
     * - .toCreateAnalysis()
     * - .toCreateBrand()
     * - .toCreateCustomPermissions()
     * - .toCreateDashboard()
     * - .toCreateDataSet()
     * - .toCreateDataSource()
     * - .toCreateFolder()
     * - .toCreateGroupMembership()
     * - .toCreateIngestion()
     * - .toCreateTemplate()
     * - .toCreateTemplateAlias()
     * - .toCreateTheme()
     * - .toCreateThemeAlias()
     * - .toCreateTopic()
     * - .toCreateVPCConnection()
     * - .toDeleteDataSet()
     * - .toDeleteDataSource()
     * - .toDeleteTopic()
     * - .toDeleteVPCConnection()
     * - .toDescribeDataSet()
     * - .toDescribeDataSetPermissions()
     * - .toDescribeDataSource()
     * - .toDescribeDataSourcePermissions()
     * - .toDescribeIngestion()
     * - .toDescribeTopic()
     * - .toDescribeTopicPermissions()
     * - .toDescribeTopicRefresh()
     * - .toDescribeVPCConnection()
     * - .toListDataSets()
     * - .toListDataSources()
     * - .toListIngestions()
     * - .toListTopicReviewedAnswers()
     * - .toListTopics()
     * - .toListVPCConnections()
     * - .toPassDataSet()
     * - .toPassDataSource()
     * - .toTagResource()
     * - .toUpdateDataSet()
     * - .toUpdateDataSetPermissions()
     * - .toUpdateDataSource()
     * - .toUpdateDataSourcePermissions()
     * - .toUpdateTopic()
     * - .toUpdateTopicPermissions()
     * - .toUpdateVPCConnection()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - analysis
     * - dashboard
     * - template
     * - vpcconnection
     * - datasource
     * - dataset
     * - ingestion
     * - theme
     * - customization
     * - folder
     * - topic
     * - dashboardSnapshotJob
     * - brand
     * - custompermissions
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by tag keys
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toBatchCreateTopicReviewedAnswer()
     * - .toBatchDeleteTopicReviewedAnswer()
     * - .toCancelIngestion()
     * - .toCreateAccountCustomization()
     * - .toCreateAnalysis()
     * - .toCreateBrand()
     * - .toCreateCustomPermissions()
     * - .toCreateDashboard()
     * - .toCreateDataSet()
     * - .toCreateDataSource()
     * - .toCreateFolder()
     * - .toCreateGroupMembership()
     * - .toCreateIngestion()
     * - .toCreateTemplate()
     * - .toCreateTemplateAlias()
     * - .toCreateTheme()
     * - .toCreateThemeAlias()
     * - .toCreateTopic()
     * - .toCreateVPCConnection()
     * - .toDeleteDataSet()
     * - .toDeleteDataSource()
     * - .toDeleteTopic()
     * - .toDeleteVPCConnection()
     * - .toDescribeDataSet()
     * - .toDescribeDataSetPermissions()
     * - .toDescribeDataSource()
     * - .toDescribeDataSourcePermissions()
     * - .toDescribeIngestion()
     * - .toDescribeTopic()
     * - .toDescribeTopicPermissions()
     * - .toDescribeTopicRefresh()
     * - .toDescribeVPCConnection()
     * - .toListDataSets()
     * - .toListDataSources()
     * - .toListIngestions()
     * - .toListTopicReviewedAnswers()
     * - .toListTopics()
     * - .toListVPCConnections()
     * - .toPassDataSet()
     * - .toPassDataSource()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateDataSet()
     * - .toUpdateDataSetPermissions()
     * - .toUpdateDataSource()
     * - .toUpdateDataSourcePermissions()
     * - .toUpdateTopic()
     * - .toUpdateTopicPermissions()
     * - .toUpdateVPCConnection()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the allowed embedding domains
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/embedded-dashboards-for-authenticated-users-step-1.html
     *
     * Applies to actions:
     * - .toGenerateEmbedUrlForAnonymousUser()
     * - .toGenerateEmbedUrlForRegisteredUser()
     * - .toGenerateEmbedUrlForRegisteredUserWithIdentity()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAllowedEmbeddingDomains(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the user management options
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/security-scp.html
     *
     * Applies to actions:
     * - .toCreateAccountSubscription()
     * - .toSubscribe()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDirectoryType(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the edition of QuickSight
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/security-scp.html
     *
     * Applies to actions:
     * - .toCreateAccountSubscription()
     * - .toSubscribe()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEdition(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by QuickSight group ARN
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     *
     * Applies to actions:
     * - .toCreateRoleMembership()
     * - .toDeleteRoleMembership()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifGroup(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by IAM user or role ARN
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     *
     * Applies to actions:
     * - .toRegisterUser()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifIamArn(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by KMS key ARNs
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/key-management.html
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifKmsKeyArns(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by session name
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     *
     * Applies to actions:
     * - .toRegisterUser()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSessionName(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by user name
     *
     * https://docs.aws.amazon.com/quicksight/latest/user/iam-actions.html
     *
     * Applies to actions:
     * - .toCreateGroupMembership()
     * - .toDeleteGroupMembership()
     * - .toDescribeGroupMembership()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifUserName(value: string | string[], operator?: Operator | string): this;
    /**
     * Statement provider for service [quicksight](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonquicksight.html).
     *
     */
    constructor(props?: iam.PolicyStatementProps);
}
