"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RedshiftServerless = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [redshift-serverless](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonredshiftserverless.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class RedshiftServerless extends shared_1.PolicyStatement {
    /**
     * Grants permission to convert a recovery point to a snapshot
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_ConvertRecoveryPointToSnapshot.html
     */
    toConvertRecoveryPointToSnapshot() {
        return this.to('ConvertRecoveryPointToSnapshot');
    }
    /**
     * Grants permission to create a custom domain association in Amazon Redshift Serverless
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - acm:DescribeCertificate
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_CreateCustomDomainAssociation.html
     */
    toCreateCustomDomainAssociation() {
        return this.to('CreateCustomDomainAssociation');
    }
    /**
     * Grants permission to create an Amazon Redshift Serverless managed VPC endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_CreateEndpointAccess.html
     */
    toCreateEndpointAccess() {
        return this.to('CreateEndpointAccess');
    }
    /**
     * Grants permission to create an Amazon Redshift Serverless namespace
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - kms:CreateGrant
     * - kms:Decrypt
     * - kms:DescribeKey
     * - kms:GenerateDataKey
     * - kms:RetireGrant
     * - secretsmanager:CreateSecret
     * - secretsmanager:DeleteSecret
     * - secretsmanager:DescribeSecret
     * - secretsmanager:GetRandomPassword
     * - secretsmanager:RotateSecret
     * - secretsmanager:TagResource
     * - secretsmanager:UpdateSecret
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_CreateNamespace.html
     */
    toCreateNamespace() {
        return this.to('CreateNamespace');
    }
    /**
     * Grants permission to purchase a capacity reservation according to a specific reservation offering, for a specified number of RPUs
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_CreateReservation.html
     */
    toCreateReservation() {
        return this.to('CreateReservation');
    }
    /**
     * Grants permission to create a scheduled action for a specified Amazon Redshift Serverless namespace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_CreateScheduledAction.html
     */
    toCreateScheduledAction() {
        return this.to('CreateScheduledAction');
    }
    /**
     * Grants permission to create a snapshot of all databases in a namespace
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_CreateSnapshot.html
     */
    toCreateSnapshot() {
        return this.to('CreateSnapshot');
    }
    /**
     * Grants permission to create a snapshot copy configuration for a specified Amazon Redshift Serverless namespace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_CreateSnapshotCopyConfiguration.html
     */
    toCreateSnapshotCopyConfiguration() {
        return this.to('CreateSnapshotCopyConfiguration');
    }
    /**
     * Grants permission to create a usage limit for a specified Amazon Redshift Serverless usage type
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_CreateUsageLimit.html
     */
    toCreateUsageLimit() {
        return this.to('CreateUsageLimit');
    }
    /**
     * Grants permission to create a workgroup in Amazon Redshift Serverless
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_CreateWorkgroup.html
     */
    toCreateWorkgroup() {
        return this.to('CreateWorkgroup');
    }
    /**
     * Grants permission to delete a custom domain association
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_DeleteCustomDomainAssociation.html
     */
    toDeleteCustomDomainAssociation() {
        return this.to('DeleteCustomDomainAssociation');
    }
    /**
     * Grants permission to delete an Amazon Redshift Serverless managed VPC endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_DeleteEndpointAccess.html
     */
    toDeleteEndpointAccess() {
        return this.to('DeleteEndpointAccess');
    }
    /**
     * Grants permission to delete a namespace from Amazon Redshift Serverless
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:DescribeKey
     * - kms:RetireGrant
     * - secretsmanager:DeleteSecret
     * - secretsmanager:DescribeSecret
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_DeleteNamespace.html
     */
    toDeleteNamespace() {
        return this.to('DeleteNamespace');
    }
    /**
     * Grants permission to delete the specified resource policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_DeleteResourcePolicy.html
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to delete a scheduled action from Amazon Redshift Serverless
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_DeleteScheduledAction.html
     */
    toDeleteScheduledAction() {
        return this.to('DeleteScheduledAction');
    }
    /**
     * Grants permission to delete a snapshot from Amazon Redshift Serverless
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_DeleteSnapshot.html
     */
    toDeleteSnapshot() {
        return this.to('DeleteSnapshot');
    }
    /**
     * Grants permission to delete a snapshot copy configuration for a Amazon Redshift Serverless namespace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_DeleteSnapshotCopyConfiguration.html
     */
    toDeleteSnapshotCopyConfiguration() {
        return this.to('DeleteSnapshotCopyConfiguration');
    }
    /**
     * Grants permission to delete a usage limit from Amazon Redshift Serverless
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_DeleteUsageLimit.html
     */
    toDeleteUsageLimit() {
        return this.to('DeleteUsageLimit');
    }
    /**
     * Grants permission to delete a workgroup
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_DeleteWorkgroup.html
     */
    toDeleteWorkgroup() {
        return this.to('DeleteWorkgroup');
    }
    /**
     * Grants permission to see on the Amazon Redshift Serverless console the remaining number of free trial credits and their expiration date
     *
     * Access Level: Read
     *
     * https://aws.amazon.com/redshift/free-trial/
     */
    toDescribeOneTimeCredit() {
        return this.to('DescribeOneTimeCredit');
    }
    /**
     * Grants permission to get a database user name and temporary password with temporary authorization to log on to Amazon Redshift Serverless
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_GetCredentials.html
     */
    toGetCredentials() {
        return this.to('GetCredentials');
    }
    /**
     * Grants permission to get information about a specific custom domain association
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_GetCustomDomainAssociation.html
     */
    toGetCustomDomainAssociation() {
        return this.to('GetCustomDomainAssociation');
    }
    /**
     * Grants permission to create an Amazon Redshift Serverless managed VPC endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_GetEndpointAccess.html
     */
    toGetEndpointAccess() {
        return this.to('GetEndpointAccess');
    }
    /**
     * Grants permission to create a Amazon Redshift Managed Serverless workgroup with the specified configuration settings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_GetManagedWorkgroup.html
     */
    toGetManagedWorkgroup() {
        return this.to('GetManagedWorkgroup');
    }
    /**
     * Grants permission to get information about a namespace in Amazon Redshift Serverless
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_GetNamespace.html
     */
    toGetNamespace() {
        return this.to('GetNamespace');
    }
    /**
     * Grants permission to get information about a recovery point
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_GetRecoveryPoint.html
     */
    toGetRecoveryPoint() {
        return this.to('GetRecoveryPoint');
    }
    /**
     * Grants permission to get a particular reservation object
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_GetReservation.html
     */
    toGetReservation() {
        return this.to('GetReservation');
    }
    /**
     * Grants permission to get a particular reservation offering
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_GetReservationOffering.html
     */
    toGetReservationOffering() {
        return this.to('GetReservationOffering');
    }
    /**
     * Grants permission to get a resource policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_GetResourcePolicy.html
     */
    toGetResourcePolicy() {
        return this.to('GetResourcePolicy');
    }
    /**
     * Grants permission to get information about a specific scheduled action
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_GetScheduledAction.html
     */
    toGetScheduledAction() {
        return this.to('GetScheduledAction');
    }
    /**
     * Grants permission to get information about a specific snapshot
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_GetSnapshot.html
     */
    toGetSnapshot() {
        return this.to('GetSnapshot');
    }
    /**
     * Grants permission to get table restore status about a specific snapshot
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_GetTableRestoreStatus.html
     */
    toGetTableRestoreStatus() {
        return this.to('GetTableRestoreStatus');
    }
    /**
     * Grants permission to get information about a track in Amazon Redshift Serverless
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_GetTrack.html
     */
    toGetTrack() {
        return this.to('GetTrack');
    }
    /**
     * Grants permission to get information about a usage limit in Amazon Redshift Serverless
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_GetUsageLimit.html
     */
    toGetUsageLimit() {
        return this.to('GetUsageLimit');
    }
    /**
     * Grants permission to get information about a specific workgroup
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_GetWorkgroup.html
     */
    toGetWorkgroup() {
        return this.to('GetWorkgroup');
    }
    /**
     * Grants permission to list custom domain associations in Amazon Redshift Serverless
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_ListCustomDomainAssociations.html
     */
    toListCustomDomainAssociations() {
        return this.to('ListCustomDomainAssociations');
    }
    /**
     * Grants permission to list EndpointAccess objects and relevant information
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_ListEndpointAccess.html
     */
    toListEndpointAccess() {
        return this.to('ListEndpointAccess');
    }
    /**
     * Grants permission to list managed workgroups in Amazon Redshift Serverless
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_ListManagedWorkgroups.html
     */
    toListManagedWorkgroups() {
        return this.to('ListManagedWorkgroups');
    }
    /**
     * Grants permission to list namespaces in Amazon Redshift Serverless
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_ListNamespaces.html
     */
    toListNamespaces() {
        return this.to('ListNamespaces');
    }
    /**
     * Grants permission to list an array of recovery points
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_ListRecoveryPoints.html
     */
    toListRecoveryPoints() {
        return this.to('ListRecoveryPoints');
    }
    /**
     * Grants permission to list all available capacity reservation offerings
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_ListReservationOfferings.html
     */
    toListReservationOfferings() {
        return this.to('ListReservationOfferings');
    }
    /**
     * Grants permission to list all reservations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_ListReservations.html
     */
    toListReservations() {
        return this.to('ListReservations');
    }
    /**
     * Grants permission to list scheduled actions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_ListScheduledActions.html
     */
    toListScheduledActions() {
        return this.to('ListScheduledActions');
    }
    /**
     * Grants permission to list SnapshotCopyConfiguration objects and relevant information
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_ListSnapshotCopyConfigurations.html
     */
    toListSnapshotCopyConfigurations() {
        return this.to('ListSnapshotCopyConfigurations');
    }
    /**
     * Grants permission to list snapshots
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_ListSnapshots.html
     */
    toListSnapshots() {
        return this.to('ListSnapshots');
    }
    /**
     * Grants permission to list table restore status
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_ListTableRestoreStatus.html
     */
    toListTableRestoreStatus() {
        return this.to('ListTableRestoreStatus');
    }
    /**
     * Grants permission to list the tags assigned to a resource
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list tracks available in Amazon Redshift Serverless
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_ListTracks.html
     */
    toListTracks() {
        return this.to('ListTracks');
    }
    /**
     * Grants permission to list all usage limits within Amazon Redshift Serverless
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_ListUsageLimits.html
     */
    toListUsageLimits() {
        return this.to('ListUsageLimits');
    }
    /**
     * Grants permission to list workgroups in Amazon Redshift Serverless
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_ListWorkgroups.html
     */
    toListWorkgroups() {
        return this.to('ListWorkgroups');
    }
    /**
     * Grants permission to create or update a resource policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_PutResourcePolicy.html
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to restore the data from a recovery point
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_RestoreFromRecoveryPoint.html
     */
    toRestoreFromRecoveryPoint() {
        return this.to('RestoreFromRecoveryPoint');
    }
    /**
     * Grants permission to restore a namespace from a snapshot
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:CreateGrant
     * - kms:Decrypt
     * - kms:DescribeKey
     * - kms:GenerateDataKey
     * - kms:RetireGrant
     * - secretsmanager:CreateSecret
     * - secretsmanager:DeleteSecret
     * - secretsmanager:DescribeSecret
     * - secretsmanager:GetRandomPassword
     * - secretsmanager:RotateSecret
     * - secretsmanager:TagResource
     * - secretsmanager:UpdateSecret
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_RestoreFromSnapshot.html
     */
    toRestoreFromSnapshot() {
        return this.to('RestoreFromSnapshot');
    }
    /**
     * Grants permission to restore a table from a recovery point
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_RestoreTableFromRecoveryPoint.html
     */
    toRestoreTableFromRecoveryPoint() {
        return this.to('RestoreTableFromRecoveryPoint');
    }
    /**
     * Grants permission to restore a table from a snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_RestoreTableFromSnapshot.html
     */
    toRestoreTableFromSnapshot() {
        return this.to('RestoreTableFromSnapshot');
    }
    /**
     * Grants permission to assign one or more tags to a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove a tag or set of tags from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a certificate associated with a custom domain
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - acm:DescribeCertificate
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_UpdateCustomDomainAssociation.html
     */
    toUpdateCustomDomainAssociation() {
        return this.to('UpdateCustomDomainAssociation');
    }
    /**
     * Grants permission to update an Amazon Redshift Serverless managed VPC endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_UpdateEndpointAccess.html
     */
    toUpdateEndpointAccess() {
        return this.to('UpdateEndpointAccess');
    }
    /**
     * Grants permission to update a namespace with the specified configuration settings
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:CreateGrant
     * - kms:Decrypt
     * - kms:DescribeKey
     * - kms:GenerateDataKey
     * - kms:RetireGrant
     * - secretsmanager:CreateSecret
     * - secretsmanager:DeleteSecret
     * - secretsmanager:DescribeSecret
     * - secretsmanager:GetRandomPassword
     * - secretsmanager:RotateSecret
     * - secretsmanager:TagResource
     * - secretsmanager:UpdateSecret
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_UpdateNamespace.html
     */
    toUpdateNamespace() {
        return this.to('UpdateNamespace');
    }
    /**
     * Grants permission to update a scheduled action
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_UpdateScheduledAction.html
     */
    toUpdateScheduledAction() {
        return this.to('UpdateScheduledAction');
    }
    /**
     * Grants permission to update a snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_UpdateSnapshot.html
     */
    toUpdateSnapshot() {
        return this.to('UpdateSnapshot');
    }
    /**
     * Grants permission to update a snapshot copy configuration for a Amazon Redshift Serverless namespace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_UpdateSnapshotCopyConfiguration.html
     */
    toUpdateSnapshotCopyConfiguration() {
        return this.to('UpdateSnapshotCopyConfiguration');
    }
    /**
     * Grants permission to update a usage limit in Amazon Redshift Serverless
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_UpdateUsageLimit.html
     */
    toUpdateUsageLimit() {
        return this.to('UpdateUsageLimit');
    }
    /**
     * Grants permission to update an Amazon Redshift Serverless workgroup with the specified configuration settings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift-serverless/latest/APIReference/API_UpdateWorkgroup.html
     */
    toUpdateWorkgroup() {
        return this.to('UpdateWorkgroup');
    }
    /**
     * Adds a resource of type namespace to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/serverless-workgroup-namespace.html
     *
     * @param namespaceId - Identifier for the namespaceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onNamespace(namespaceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:redshift-serverless:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:namespace/${namespaceId}`);
    }
    /**
     * Adds a resource of type snapshot to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/serverless-snapshots-recovery.html
     *
     * @param snapshotId - Identifier for the snapshotId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSnapshot(snapshotId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:redshift-serverless:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:snapshot/${snapshotId}`);
    }
    /**
     * Adds a resource of type workgroup to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/serverless-workgroup-namespace.html
     *
     * @param workgroupId - Identifier for the workgroupId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWorkgroup(workgroupId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:redshift-serverless:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:workgroup/${workgroupId}`);
    }
    /**
     * Adds a resource of type managed-workgroup to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/serverless-managed-workgroup-namespace.html
     *
     * @param managedWorkgroupName - Identifier for the managedWorkgroupName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onManagedWorkgroup(managedWorkgroupName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:redshift-serverless:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:managed-workgroup/${managedWorkgroupName}`);
    }
    /**
     * Adds a resource of type recoveryPoint to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/serverless-snapshots-recovery.html
     *
     * @param recoveryPointId - Identifier for the recoveryPointId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRecoveryPoint(recoveryPointId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:redshift-serverless:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:recoverypoint/${recoveryPointId}`);
    }
    /**
     * Adds a resource of type endpointAccess to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/serverless-connecting.html
     *
     * @param endpointAccessId - Identifier for the endpointAccessId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onEndpointAccess(endpointAccessId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:redshift-serverless:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:managedvpcendpoint/${endpointAccessId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toConvertRecoveryPointToSnapshot()
     * - .toCreateNamespace()
     * - .toCreateSnapshot()
     * - .toCreateWorkgroup()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toListTagsForResource()
     * - .toTagResource()
     *
     * Applies to resource types:
     * - namespace
     * - snapshot
     * - workgroup
     * - recoveryPoint
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toConvertRecoveryPointToSnapshot()
     * - .toCreateNamespace()
     * - .toCreateSnapshot()
     * - .toCreateWorkgroup()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the endpoint access identifier
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEndpointAccessId(value, operator) {
        return this.if(`endpointAccessId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the managed workgroup identifier
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifManagedWorkgroupName(value, operator) {
        return this.if(`managedWorkgroupName`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the namespace identifier
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifNamespaceId(value, operator) {
        return this.if(`namespaceId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the recovery point identifier
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRecoveryPointId(value, operator) {
        return this.if(`recoveryPointId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the snapshot identifier
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSnapshotId(value, operator) {
        return this.if(`snapshotId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the table restore request identifier
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTableRestoreRequestId(value, operator) {
        return this.if(`tableRestoreRequestId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the workgroup identifier
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifWorkgroupId(value, operator) {
        return this.if(`workgroupId`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [redshift-serverless](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonredshiftserverless.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'redshift-serverless';
        this.accessLevelList = {
            Write: [
                'ConvertRecoveryPointToSnapshot',
                'CreateCustomDomainAssociation',
                'CreateEndpointAccess',
                'CreateNamespace',
                'CreateReservation',
                'CreateScheduledAction',
                'CreateSnapshot',
                'CreateSnapshotCopyConfiguration',
                'CreateUsageLimit',
                'CreateWorkgroup',
                'DeleteCustomDomainAssociation',
                'DeleteEndpointAccess',
                'DeleteNamespace',
                'DeleteResourcePolicy',
                'DeleteScheduledAction',
                'DeleteSnapshot',
                'DeleteSnapshotCopyConfiguration',
                'DeleteUsageLimit',
                'DeleteWorkgroup',
                'GetCredentials',
                'PutResourcePolicy',
                'RestoreFromRecoveryPoint',
                'RestoreFromSnapshot',
                'RestoreTableFromRecoveryPoint',
                'RestoreTableFromSnapshot',
                'UpdateCustomDomainAssociation',
                'UpdateEndpointAccess',
                'UpdateNamespace',
                'UpdateScheduledAction',
                'UpdateSnapshot',
                'UpdateSnapshotCopyConfiguration',
                'UpdateUsageLimit',
                'UpdateWorkgroup'
            ],
            Read: [
                'DescribeOneTimeCredit',
                'GetCustomDomainAssociation',
                'GetEndpointAccess',
                'GetManagedWorkgroup',
                'GetNamespace',
                'GetRecoveryPoint',
                'GetReservation',
                'GetReservationOffering',
                'GetResourcePolicy',
                'GetScheduledAction',
                'GetSnapshot',
                'GetTableRestoreStatus',
                'GetTrack',
                'GetUsageLimit',
                'GetWorkgroup'
            ],
            List: [
                'ListCustomDomainAssociations',
                'ListEndpointAccess',
                'ListManagedWorkgroups',
                'ListNamespaces',
                'ListRecoveryPoints',
                'ListReservationOfferings',
                'ListReservations',
                'ListScheduledActions',
                'ListSnapshotCopyConfigurations',
                'ListSnapshots',
                'ListTableRestoreStatus',
                'ListTagsForResource',
                'ListTracks',
                'ListUsageLimits',
                'ListWorkgroups'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.RedshiftServerless = RedshiftServerless;
//# sourceMappingURL=data:application/json;base64,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