"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Repostspace = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [repostspace](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsrepostprivate.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Repostspace extends shared_1.PolicyStatement {
    /**
     * Grants permission to add a role to users and groups in a private re:Post channel in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/repostprivate/latest/APIReference/API_BatchAddChannelRoleToAccessors.html
     */
    toBatchAddChannelRoleToAccessors() {
        return this.to('BatchAddChannelRoleToAccessors');
    }
    /**
     * Grants permission to add a role to users and groups in a private re:Post in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/repostprivate/latest/APIReference/API_BatchAddRole.html
     */
    toBatchAddRole() {
        return this.to('BatchAddRole');
    }
    /**
     * Grants permission to remove a role from users and groups in a private re:Post channel in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/repostprivate/latest/APIReference/API_BatchRemoveChannelRoleFromAccessors.html
     */
    toBatchRemoveChannelRoleFromAccessors() {
        return this.to('BatchRemoveChannelRoleFromAccessors');
    }
    /**
     * Grants permission to remove a role from users and groups in a private re:Post in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/repostprivate/latest/APIReference/API_BatchRemoveRole.html
     */
    toBatchRemoveRole() {
        return this.to('BatchRemoveRole');
    }
    /**
     * Grants permission to create a new channel in private re:Post in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/repostprivate/latest/APIReference/API_CreateChannel.html
     */
    toCreateChannel() {
        return this.to('CreateChannel');
    }
    /**
     * Grants permission to create a new private re:Post in your account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/repostprivate/latest/APIReference/API_CreateSpace.html
     */
    toCreateSpace() {
        return this.to('CreateSpace');
    }
    /**
     * Grants permission to delete a private re:Post from your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/repostprivate/latest/APIReference/API_DeleteSpace.html
     */
    toDeleteSpace() {
        return this.to('DeleteSpace');
    }
    /**
     * Grants permission to remove an administrator to a private re:Post in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/repostprivate/latest/APIReference/API_DeregisterAdmin.html
     */
    toDeregisterAdmin() {
        return this.to('DeregisterAdmin');
    }
    /**
     * Grants permission to get the description for a channel in private re:Post in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/repostprivate/latest/APIReference/API_GetChannel.html
     */
    toGetChannel() {
        return this.to('GetChannel');
    }
    /**
     * Grants permission to get the description for a private re:Post in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/repostprivate/latest/APIReference/API_GetSpace.html
     */
    toGetSpace() {
        return this.to('GetSpace');
    }
    /**
     * Grants permission to list all channels in a private re:Post in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/repostprivate/latest/APIReference/API_ListChannels.html
     */
    toListChannels() {
        return this.to('ListChannels');
    }
    /**
     * Grants permission to list all private re:Posts in your account
     *
     * Access Level: Read
     */
    toListSpaces() {
        return this.to('ListSpaces');
    }
    /**
     * Grants permission to list the tags associated with a resource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/repostprivate/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to add an administrator to a private re:Post in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/repostprivate/latest/APIReference/API_RegisterAdmin.html
     */
    toRegisterAdmin() {
        return this.to('RegisterAdmin');
    }
    /**
     * Grants permission to send invites to users of a private re:Post in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/repostprivate/latest/APIReference/API_SendInvites.html
     */
    toSendInvites() {
        return this.to('SendInvites');
    }
    /**
     * Grants permission to tag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/repostprivate/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/repostprivate/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a channel in private re:Post in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/repostprivate/latest/APIReference/API_UpdateChannel.html
     */
    toUpdateChannel() {
        return this.to('UpdateChannel');
    }
    /**
     * Grants permission to update a private re:Post in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/repostprivate/latest/APIReference/API_UpdateSpace.html
     */
    toUpdateSpace() {
        return this.to('UpdateSpace');
    }
    /**
     * Adds a resource of type space to the statement
     *
     * https://docs.aws.amazon.com/repostprivate/latest/userguide/
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSpace(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:repostspace:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:space/${resourceId}`);
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateSpace()
     * - .toListTagsForResource()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - space
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateSpace()
     * - .toListTagsForResource()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [repostspace](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsrepostprivate.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'repostspace';
        this.accessLevelList = {
            Write: [
                'BatchAddChannelRoleToAccessors',
                'BatchAddRole',
                'BatchRemoveChannelRoleFromAccessors',
                'BatchRemoveRole',
                'CreateChannel',
                'CreateSpace',
                'DeleteSpace',
                'DeregisterAdmin',
                'RegisterAdmin',
                'SendInvites',
                'UpdateChannel',
                'UpdateSpace'
            ],
            Read: [
                'GetChannel',
                'GetSpace',
                'ListChannels',
                'ListSpaces',
                'ListTagsForResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Repostspace = Repostspace;
//# sourceMappingURL=data:application/json;base64,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