"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ResourceExplorer2 = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [resource-explorer-2](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsresourceexplorer.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class ResourceExplorer2 extends shared_1.PolicyStatement {
    /**
     * Grants permission to set the specified view as the default for this AWS Region in this AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_AssociateDefaultView.html
     */
    toAssociateDefaultView() {
        return this.to('AssociateDefaultView');
    }
    /**
     * Grants permission to retrieve details about views that you specify by a list of ARNs
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - resource-explorer-2:GetView
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_BatchGetView.html
     */
    toBatchGetView() {
        return this.to('BatchGetView');
    }
    /**
     * Grants permission to turn on Resource Explorer in the AWS Region in which you called this operation by creating an index
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_CreateIndex.html
     */
    toCreateIndex() {
        return this.to('CreateIndex');
    }
    /**
     * Grants permission to create managed view
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/userguide/API_ManagedView.html
     */
    toCreateManagedView() {
        return this.to('CreateManagedView');
    }
    /**
     * Grants permission to create a view that users can query
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_CreateView.html
     */
    toCreateView() {
        return this.to('CreateView');
    }
    /**
     * Grants permission to turn off Resource Explorer in the specified AWS Region by deleting the index
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_DeleteIndex.html
     */
    toDeleteIndex() {
        return this.to('DeleteIndex');
    }
    /**
     * Grants permission to delete the specified view's resource policy
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/userguide/manage-views-share.html
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to delete a view
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_DeleteView.html
     */
    toDeleteView() {
        return this.to('DeleteView');
    }
    /**
     * Grants permission to remove the default view for the AWS Region in which you call this operation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_DisassociateDefaultView.html
     */
    toDisassociateDefaultView() {
        return this.to('DisassociateDefaultView');
    }
    /**
     * Grants permission to Resource Explorer to access account level data within your AWS Organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_GetAccountLevelServiceConfiguration.html
     */
    toGetAccountLevelServiceConfiguration() {
        return this.to('GetAccountLevelServiceConfiguration');
    }
    /**
     * Grants permission to retrieve the Amazon resource name (ARN) of the view that is the default for the AWS Region in which you call this operation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_GetDefaultView.html
     */
    toGetDefaultView() {
        return this.to('GetDefaultView');
    }
    /**
     * Grants permission to retrieve information about the index in the AWS Region in which you call this operation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_GetIndex.html
     */
    toGetIndex() {
        return this.to('GetIndex');
    }
    /**
     * Grants permission to get managed view
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_GetManagedView.html
     */
    toGetManagedView() {
        return this.to('GetManagedView');
    }
    /**
     * Grants permission to retrieve information about the specified view's resource policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/userguide/manage-views-share.html
     */
    toGetResourcePolicy() {
        return this.to('GetResourcePolicy');
    }
    /**
     * Grants permission to retrieve information about the specified view
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_GetView.html
     */
    toGetView() {
        return this.to('GetView');
    }
    /**
     * Grants permission to list the indexes in all AWS Regions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_ListIndexes.html
     */
    toListIndexes() {
        return this.to('ListIndexes');
    }
    /**
     * Grants permission to list the organization member account's indexes in all AWS Regions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_ListIndexesForMembers.html
     */
    toListIndexesForMembers() {
        return this.to('ListIndexesForMembers');
    }
    /**
     * Grants permission to list managed views
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_ListManagedViews.html
     */
    toListManagedViews() {
        return this.to('ListManagedViews');
    }
    /**
     * Grants permission to retrieve a list of all resource types currently supported by Resource Explorer
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_ListSupportedResourceTypes.html
     */
    toListSupportedResourceTypes() {
        return this.to('ListSupportedResourceTypes');
    }
    /**
     * Grants permission to list the tags that are attached to the specified resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list the Amazon resource names (ARNs) of all of the views available in the AWS Region in which you call this operation
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_ListViews.html
     */
    toListViews() {
        return this.to('ListViews');
    }
    /**
     * Grants permission to update the specified view's resource policy
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/userguide/manage-views-share.html
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to search for resources and display details about all resources that match the specified criteria
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifOperation()
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_Search.html
     */
    toSearch() {
        return this.to('Search');
    }
    /**
     * Grants permission to add one or more tag key and value pairs to the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove one or more tag key and value pairs from the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to change the type of the index from LOCAL to AGGREGATOR or back
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_UpdateIndexType.html
     */
    toUpdateIndexType() {
        return this.to('UpdateIndexType');
    }
    /**
     * Grants permission to modify some of the details of a view
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_UpdateView.html
     */
    toUpdateView() {
        return this.to('UpdateView');
    }
    /**
     * Adds a resource of type view to the statement
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_View.html
     *
     * @param viewName - Identifier for the viewName.
     * @param viewUuid - Identifier for the viewUuid.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onView(viewName, viewUuid, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:resource-explorer-2:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:view/${viewName}/${viewUuid}`);
    }
    /**
     * Adds a resource of type index to the statement
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/apireference/API_Index.html
     *
     * @param indexUuid - Identifier for the indexUuid.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIndex(indexUuid, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:resource-explorer-2:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:index/${indexUuid}`);
    }
    /**
     * Adds a resource of type managed-view to the statement
     *
     * https://docs.aws.amazon.com/resource-explorer/latest/userguide/API_ManagedView.html
     *
     * @param managedViewName - Identifier for the managedViewName.
     * @param managedViewUuid - Identifier for the managedViewUuid.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onManagedView(managedViewName, managedViewUuid, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:resource-explorer-2:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:managed-view/${managedViewName}/${managedViewUuid}`);
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateIndex()
     * - .toCreateView()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keyss attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - view
     * - index
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateIndex()
     * - .toCreateView()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the actual operation that is being invoked, available values: Search, ListResources
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsresourceexplorer.html
     *
     * Applies to actions:
     * - .toSearch()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifOperation(value, operator) {
        return this.if(`Operation`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [resource-explorer-2](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsresourceexplorer.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'resource-explorer-2';
        this.accessLevelList = {
            Write: [
                'AssociateDefaultView',
                'CreateIndex',
                'CreateManagedView',
                'CreateView',
                'DeleteIndex',
                'DeleteView',
                'DisassociateDefaultView',
                'UpdateIndexType',
                'UpdateView'
            ],
            Read: [
                'BatchGetView',
                'GetAccountLevelServiceConfiguration',
                'GetDefaultView',
                'GetIndex',
                'GetManagedView',
                'GetResourcePolicy',
                'GetView',
                'ListTagsForResource',
                'Search'
            ],
            'Permissions management': [
                'DeleteResourcePolicy',
                'PutResourcePolicy'
            ],
            List: [
                'ListIndexes',
                'ListIndexesForMembers',
                'ListManagedViews',
                'ListSupportedResourceTypes',
                'ListViews'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.ResourceExplorer2 = ResourceExplorer2;
//# sourceMappingURL=data:application/json;base64,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