"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Route53RecoveryControlConfig = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [route53-recovery-control-config](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonroute53recoverycontrols.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Route53RecoveryControlConfig extends shared_1.PolicyStatement {
    /**
     * Grants permission to create a cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/recovery-cluster/latest/api/cluster.html
     */
    toCreateCluster() {
        return this.to('CreateCluster');
    }
    /**
     * Grants permission to create a control panel
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/recovery-cluster/latest/api/controlpanel.html
     */
    toCreateControlPanel() {
        return this.to('CreateControlPanel');
    }
    /**
     * Grants permission to create a routing control
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/recovery-cluster/latest/api/routingcontrol.html
     */
    toCreateRoutingControl() {
        return this.to('CreateRoutingControl');
    }
    /**
     * Grants permission to create a safety rule
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/recovery-cluster/latest/api/safetyrule.html
     */
    toCreateSafetyRule() {
        return this.to('CreateSafetyRule');
    }
    /**
     * Grants permission to delete a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/recovery-cluster/latest/api/cluster-clusterarn.html
     */
    toDeleteCluster() {
        return this.to('DeleteCluster');
    }
    /**
     * Grants permission to delete a control panel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/recovery-cluster/latest/api/controlpanel-controlpanelarn.html
     */
    toDeleteControlPanel() {
        return this.to('DeleteControlPanel');
    }
    /**
     * Grants permission to delete the RAM access control policy for a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/r53recovery/latest/dg/routing-control.failover-different-accounts.html
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to delete a routing control
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/recovery-cluster/latest/api/routingcontrol-routingcontrolarn.html
     */
    toDeleteRoutingControl() {
        return this.to('DeleteRoutingControl');
    }
    /**
     * Grants permission to delete a safety rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/recovery-cluster/latest/api/safetyrule-safetyrulearn.html
     */
    toDeleteSafetyRule() {
        return this.to('DeleteSafetyRule');
    }
    /**
     * Grants permission to describe a cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/recovery-cluster/latest/api/cluster-clusterarn.html
     */
    toDescribeCluster() {
        return this.to('DescribeCluster');
    }
    /**
     * Grants permission to describe a control panel
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/recovery-cluster/latest/api/controlpanel-controlpanelarn.html
     */
    toDescribeControlPanel() {
        return this.to('DescribeControlPanel');
    }
    /**
     * Grants permission to describe a routing control
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/recovery-cluster/latest/api/routingcontrol-routingcontrolarn.html
     */
    toDescribeRoutingControl() {
        return this.to('DescribeRoutingControl');
    }
    /**
     * Grants permission to describe a routing control
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/recovery-cluster/latest/api/routingcontrol-routingcontrolarn.html
     */
    toDescribeRoutingControlByName() {
        return this.to('DescribeRoutingControlByName');
    }
    /**
     * Grants permission to describe a safety rule
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/recovery-cluster/latest/api/safetyrule-safetyrulearn.html
     */
    toDescribeSafetyRule() {
        return this.to('DescribeSafetyRule');
    }
    /**
     * Grants permission to get the resource policy of a cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/recovery-cluster/latest/api/cluster/resourcepolicy-resourcearn.html
     */
    toGetResourcePolicy() {
        return this.to('GetResourcePolicy');
    }
    /**
     * Grants permission to list associated Route 53 health checks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/recovery-cluster/latest/api/routingcontrol-routingcontrolarn-associatedroute53healthchecks.html
     */
    toListAssociatedRoute53HealthChecks() {
        return this.to('ListAssociatedRoute53HealthChecks');
    }
    /**
     * Grants permission to list clusters
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/recovery-cluster/latest/api/cluster.html
     */
    toListClusters() {
        return this.to('ListClusters');
    }
    /**
     * Grants permission to list control panels
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/recovery-cluster/latest/api/controlpanels.html
     */
    toListControlPanels() {
        return this.to('ListControlPanels');
    }
    /**
     * Grants permission to list routing controls
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/recovery-cluster/latest/api/controlpanel-controlpanelarn-routingcontrols.html
     */
    toListRoutingControls() {
        return this.to('ListRoutingControls');
    }
    /**
     * Grants permission to list safety rules
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/recovery-cluster/latest/api/controlpanel-controlpanelarn-safetyrules.html
     */
    toListSafetyRules() {
        return this.to('ListSafetyRules');
    }
    /**
     * Grants permission to list tags for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/recovery-cluster/latest/api/tags-resource-arn.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to define the RAM access control policy for a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/r53recovery/latest/dg/routing-control.failover-different-accounts.html
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to tag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/recovery-cluster/latest/api/tags-resource-arn.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/recovery-cluster/latest/api/tags-resource-arn.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/recovery-cluster/latest/api/cluster.html
     */
    toUpdateCluster() {
        return this.to('UpdateCluster');
    }
    /**
     * Grants permission to update a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/recovery-cluster/latest/api/controlpanel.html
     */
    toUpdateControlPanel() {
        return this.to('UpdateControlPanel');
    }
    /**
     * Grants permission to update a routing control
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/recovery-cluster/latest/api/routingcontrol.html
     */
    toUpdateRoutingControl() {
        return this.to('UpdateRoutingControl');
    }
    /**
     * Grants permission to update a safety rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/recovery-cluster/latest/api/safetyrule.html
     */
    toUpdateSafetyRule() {
        return this.to('UpdateSafetyRule');
    }
    /**
     * Adds a resource of type cluster to the statement
     *
     * https://docs.aws.amazon.com/recovery-cluster/latest/api/cluster.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCluster(resourceId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:route53-recovery-control::${account ?? this.defaultAccount}:cluster/${resourceId}`);
    }
    /**
     * Adds a resource of type controlpanel to the statement
     *
     * https://docs.aws.amazon.com/recovery-cluster/latest/api/controlpanel.html
     *
     * @param controlPanelId - Identifier for the controlPanelId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onControlpanel(controlPanelId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:route53-recovery-control::${account ?? this.defaultAccount}:controlpanel/${controlPanelId}`);
    }
    /**
     * Adds a resource of type routingcontrol to the statement
     *
     * https://docs.aws.amazon.com/recovery-cluster/latest/api/routingcontrol.html
     *
     * @param controlPanelId - Identifier for the controlPanelId.
     * @param routingControlId - Identifier for the routingControlId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onRoutingcontrol(controlPanelId, routingControlId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:route53-recovery-control::${account ?? this.defaultAccount}:controlpanel/${controlPanelId}/routingcontrol/${routingControlId}`);
    }
    /**
     * Adds a resource of type safetyrule to the statement
     *
     * https://docs.aws.amazon.com/recovery-cluster/latest/api/safetyrule.html
     *
     * @param controlPanelId - Identifier for the controlPanelId.
     * @param safetyRuleId - Identifier for the safetyRuleId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSafetyrule(controlPanelId, safetyRuleId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:route53-recovery-control::${account ?? this.defaultAccount}:controlpanel/${controlPanelId}/safetyrule/${safetyRuleId}`);
    }
    /**
     * Filters access by a tag's key and value in a request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateCluster()
     * - .toCreateControlPanel()
     * - .toCreateSafetyRule()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - cluster
     * - controlpanel
     * - safetyrule
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateCluster()
     * - .toCreateControlPanel()
     * - .toCreateSafetyRule()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [route53-recovery-control-config](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonroute53recoverycontrols.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'route53-recovery-control-config';
        this.accessLevelList = {
            Write: [
                'CreateCluster',
                'CreateControlPanel',
                'CreateRoutingControl',
                'CreateSafetyRule',
                'DeleteCluster',
                'DeleteControlPanel',
                'DeleteResourcePolicy',
                'DeleteRoutingControl',
                'DeleteSafetyRule',
                'PutResourcePolicy',
                'UpdateCluster',
                'UpdateControlPanel',
                'UpdateRoutingControl',
                'UpdateSafetyRule'
            ],
            Read: [
                'DescribeCluster',
                'DescribeControlPanel',
                'DescribeRoutingControl',
                'DescribeRoutingControlByName',
                'DescribeSafetyRule',
                'GetResourcePolicy',
                'ListClusters',
                'ListControlPanels',
                'ListRoutingControls',
                'ListSafetyRules',
                'ListTagsForResource'
            ],
            List: [
                'ListAssociatedRoute53HealthChecks'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Route53RecoveryControlConfig = Route53RecoveryControlConfig;
//# sourceMappingURL=data:application/json;base64,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