"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Route53RecoveryReadiness = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [route53-recovery-readiness](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonroute53recoveryreadiness.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Route53RecoveryReadiness extends shared_1.PolicyStatement {
    /**
     * Grants permission to create a new cell
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/recovery-readiness/latest/api/cells.html
     */
    toCreateCell() {
        return this.to('CreateCell');
    }
    /**
     * Grants permission to create a cross account authorization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/recovery-readiness/latest/api/crossaccountauthorizations.html
     */
    toCreateCrossAccountAuthorization() {
        return this.to('CreateCrossAccountAuthorization');
    }
    /**
     * Grants permission to create a readiness check
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/recovery-readiness/latest/api/readinesschecks.html
     */
    toCreateReadinessCheck() {
        return this.to('CreateReadinessCheck');
    }
    /**
     * Grants permission to create a recovery group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/recovery-readiness/latest/api/recoverygroups.html
     */
    toCreateRecoveryGroup() {
        return this.to('CreateRecoveryGroup');
    }
    /**
     * Grants permission to create a resource set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/recovery-readiness/latest/api/resourcesets.html
     */
    toCreateResourceSet() {
        return this.to('CreateResourceSet');
    }
    /**
     * Grants permission to delete a cell
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/recovery-readiness/latest/api/cells-cellname.html
     */
    toDeleteCell() {
        return this.to('DeleteCell');
    }
    /**
     * Grants permission to delete a cross account authorization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/recovery-readiness/latest/api/crossaccountauthorizations-crossaccountauthorization.html
     */
    toDeleteCrossAccountAuthorization() {
        return this.to('DeleteCrossAccountAuthorization');
    }
    /**
     * Grants permission to delete a readiness check
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/recovery-readiness/latest/api/readinesschecks-readinesscheckname.html
     */
    toDeleteReadinessCheck() {
        return this.to('DeleteReadinessCheck');
    }
    /**
     * Grants permission to delete a recovery group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/recovery-readiness/latest/api/recoverygroups-recoverygroupname.html
     */
    toDeleteRecoveryGroup() {
        return this.to('DeleteRecoveryGroup');
    }
    /**
     * Grants permission to delete a resource set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/recovery-readiness/latest/api/resourcesets-resourcesetname.html
     */
    toDeleteResourceSet() {
        return this.to('DeleteResourceSet');
    }
    /**
     * Grants permission to get architecture recommendations for a recovery group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/recovery-readiness/latest/api/recoverygroups-recoverygroupname-architecturerecommendations.html
     */
    toGetArchitectureRecommendations() {
        return this.to('GetArchitectureRecommendations');
    }
    /**
     * Grants permission to get information about a cell
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/recovery-readiness/latest/api/cells-cellname.html
     */
    toGetCell() {
        return this.to('GetCell');
    }
    /**
     * Grants permission to get a readiness summary for a cell
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/recovery-readiness/latest/api/cellreadiness-cellname.html
     */
    toGetCellReadinessSummary() {
        return this.to('GetCellReadinessSummary');
    }
    /**
     * Grants permission to get information about a readiness check
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/recovery-readiness/latest/api/readinesschecks-readinesscheckname.html
     */
    toGetReadinessCheck() {
        return this.to('GetReadinessCheck');
    }
    /**
     * Grants permission to get the readiness status for an individual resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/recovery-readiness/latest/api/readinesschecks-readinesscheckname-resource-resourceidentifier-status.html
     */
    toGetReadinessCheckResourceStatus() {
        return this.to('GetReadinessCheckResourceStatus');
    }
    /**
     * Grants permission to get the status of a readiness check (for a resource set)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/recovery-readiness/latest/api/readinesschecks-readinesscheckname-status.html
     */
    toGetReadinessCheckStatus() {
        return this.to('GetReadinessCheckStatus');
    }
    /**
     * Grants permission to get information about a recovery group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/recovery-readiness/latest/api/recoverygroups-recoverygroupname.html
     */
    toGetRecoveryGroup() {
        return this.to('GetRecoveryGroup');
    }
    /**
     * Grants permission to get a readiness summary for a recovery group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/recovery-readiness/latest/api/recoverygroupreadiness-recoverygroupname.html
     */
    toGetRecoveryGroupReadinessSummary() {
        return this.to('GetRecoveryGroupReadinessSummary');
    }
    /**
     * Grants permission to get information about a resource set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/recovery-readiness/latest/api/resourcesets-resourcesetname.html
     */
    toGetResourceSet() {
        return this.to('GetResourceSet');
    }
    /**
     * Grants permission to list cells
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/recovery-readiness/latest/api/cells.html
     */
    toListCells() {
        return this.to('ListCells');
    }
    /**
     * Grants permission to list cross account authorizations
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/recovery-readiness/latest/api/crossaccountauthorizations.html
     */
    toListCrossAccountAuthorizations() {
        return this.to('ListCrossAccountAuthorizations');
    }
    /**
     * Grants permission to list readiness checks
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/recovery-readiness/latest/api/readinesschecks.html
     */
    toListReadinessChecks() {
        return this.to('ListReadinessChecks');
    }
    /**
     * Grants permission to list recovery groups
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/recovery-readiness/latest/api/recoverygroups.html
     */
    toListRecoveryGroups() {
        return this.to('ListRecoveryGroups');
    }
    /**
     * Grants permission to list resource sets
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/recovery-readiness/latest/api/resourcesets.html
     */
    toListResourceSets() {
        return this.to('ListResourceSets');
    }
    /**
     * Grants permission to list readiness rules
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/recovery-readiness/latest/api/rules.html
     */
    toListRules() {
        return this.to('ListRules');
    }
    /**
     * Grants permission to list tags for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/recovery-readiness/latest/api/tags-resource-arn.html
     */
    toListTagsForResources() {
        return this.to('ListTagsForResources');
    }
    /**
     * Grants permission to add a tag to a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/recovery-readiness/latest/api/tags-resource-arn.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove a tag from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/recovery-readiness/latest/api/tags-resource-arn.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a cell
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/recovery-readiness/latest/api/cells-cellname.html
     */
    toUpdateCell() {
        return this.to('UpdateCell');
    }
    /**
     * Grants permission to update a readiness check
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/recovery-readiness/latest/api/readinesschecks-readinesscheckname.html
     */
    toUpdateReadinessCheck() {
        return this.to('UpdateReadinessCheck');
    }
    /**
     * Grants permission to update a recovery group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/recovery-readiness/latest/api/recoverygroups-recoverygroupname.html
     */
    toUpdateRecoveryGroup() {
        return this.to('UpdateRecoveryGroup');
    }
    /**
     * Grants permission to update a resource set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/recovery-readiness/latest/api/resourcesets-resourcesetname.html
     */
    toUpdateResourceSet() {
        return this.to('UpdateResourceSet');
    }
    /**
     * Adds a resource of type readinesscheck to the statement
     *
     * https://docs.aws.amazon.com/r53recovery/latest/dg/recovery-readiness.readiness-checks.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onReadinesscheck(resourceId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:route53-recovery-readiness::${account ?? this.defaultAccount}:readiness-check/${resourceId}`);
    }
    /**
     * Adds a resource of type resourceset to the statement
     *
     * https://docs.aws.amazon.com/r53recovery/latest/dg/recovery-readiness.readiness-checks.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onResourceset(resourceId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:route53-recovery-readiness::${account ?? this.defaultAccount}:resource-set/${resourceId}`);
    }
    /**
     * Adds a resource of type cell to the statement
     *
     * https://docs.aws.amazon.com/r53recovery/latest/dg/recovery-readiness.recovery-groups.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCell(resourceId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:route53-recovery-readiness::${account ?? this.defaultAccount}:cell/${resourceId}`);
    }
    /**
     * Adds a resource of type recoverygroup to the statement
     *
     * https://docs.aws.amazon.com/r53recovery/latest/dg/recovery-readiness.recovery-groups.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRecoverygroup(resourceId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:route53-recovery-readiness::${account ?? this.defaultAccount}:recovery-group/${resourceId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateCell()
     * - .toCreateReadinessCheck()
     * - .toCreateRecoveryGroup()
     * - .toCreateResourceSet()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - readinesscheck
     * - resourceset
     * - cell
     * - recoverygroup
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateCell()
     * - .toCreateReadinessCheck()
     * - .toCreateRecoveryGroup()
     * - .toCreateResourceSet()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateCell()
     * - .toUpdateReadinessCheck()
     * - .toUpdateRecoveryGroup()
     * - .toUpdateResourceSet()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [route53-recovery-readiness](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonroute53recoveryreadiness.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'route53-recovery-readiness';
        this.accessLevelList = {
            Write: [
                'CreateCell',
                'CreateCrossAccountAuthorization',
                'CreateReadinessCheck',
                'CreateRecoveryGroup',
                'CreateResourceSet',
                'DeleteCell',
                'DeleteCrossAccountAuthorization',
                'DeleteReadinessCheck',
                'DeleteRecoveryGroup',
                'DeleteResourceSet',
                'UpdateCell',
                'UpdateReadinessCheck',
                'UpdateRecoveryGroup',
                'UpdateResourceSet'
            ],
            Read: [
                'GetArchitectureRecommendations',
                'GetCell',
                'GetCellReadinessSummary',
                'GetReadinessCheck',
                'GetReadinessCheckResourceStatus',
                'GetReadinessCheckStatus',
                'GetRecoveryGroup',
                'GetRecoveryGroupReadinessSummary',
                'GetResourceSet',
                'ListCells',
                'ListCrossAccountAuthorizations',
                'ListReadinessChecks',
                'ListRecoveryGroups',
                'ListResourceSets',
                'ListRules',
                'ListTagsForResources'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Route53RecoveryReadiness = Route53RecoveryReadiness;
//# sourceMappingURL=data:application/json;base64,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