"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3express = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [s3express](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazons3express.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class S3express extends shared_1.PolicyStatement {
    /**
     * Grants permission to create a new access point
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifLocationName()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateAccessPoint.html
     */
    toCreateAccessPoint() {
        return this.to('CreateAccessPoint');
    }
    /**
     * Grants permission to create a new bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifLocationName()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_CreateBucket.html
     */
    toCreateBucket() {
        return this.to('CreateBucket');
    }
    /**
     * Grants permission to Create Session token which is used for object APIs such as PutObject, GetObject, etc
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSessionMode()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifXAmzServerSideEncryption()
     * - .ifXAmzServerSideEncryptionAwsKmsKeyId()
     * - .ifAllAccessRestrictedToLocalZoneGroup()
     * - .ifPermissions()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_CreateSession.html
     */
    toCreateSession() {
        return this.to('CreateSession');
    }
    /**
     * Grants permission to delete the access point named in the URI
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPoint.html
     */
    toDeleteAccessPoint() {
        return this.to('DeleteAccessPoint');
    }
    /**
     * Grants permission to delete the policy on a specified access point
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPointPolicy.html
     */
    toDeleteAccessPointPolicy() {
        return this.to('DeleteAccessPointPolicy');
    }
    /**
     * Grants permission to delete the scope configuration on a specified access point
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPointScope.html
     */
    toDeleteAccessPointScope() {
        return this.to('DeleteAccessPointScope');
    }
    /**
     * Grants permission to delete the bucket named in the URI
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucket.html
     */
    toDeleteBucket() {
        return this.to('DeleteBucket');
    }
    /**
     * Grants permission to delete the policy on a specified bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucketPolicy.html
     */
    toDeleteBucketPolicy() {
        return this.to('DeleteBucketPolicy');
    }
    /**
     * Grants permission to return configuration information about the specified access point
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPoint.html
     */
    toGetAccessPoint() {
        return this.to('GetAccessPoint');
    }
    /**
     * Grants permission to return the access point policy associated with the specified access point
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointPolicy.html
     */
    toGetAccessPointPolicy() {
        return this.to('GetAccessPointPolicy');
    }
    /**
     * Grants permission to return the scope configuration associated with the specified access point
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointScope.html
     */
    toGetAccessPointScope() {
        return this.to('GetAccessPointScope');
    }
    /**
     * Grants permission to return the policy of the specified bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketPolicy.html
     */
    toGetBucketPolicy() {
        return this.to('GetBucketPolicy');
    }
    /**
     * Grants permission to return the default encryption configuration for a directory bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketEncryption.html
     */
    toGetEncryptionConfiguration() {
        return this.to('GetEncryptionConfiguration');
    }
    /**
     * Grants permission to return the lifecycle configuration information set on a directory bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketLifecycleConfiguration.html
     */
    toGetLifecycleConfiguration() {
        return this.to('GetLifecycleConfiguration');
    }
    /**
     * Grants permission to list access points
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListAccessPointsForDirectoryBuckets.html
     */
    toListAccessPointsForDirectoryBuckets() {
        return this.to('ListAccessPointsForDirectoryBuckets');
    }
    /**
     * Grants permission to list all directory buckets owned by the authenticated sender of the request
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListDirectoryBuckets.html
     */
    toListAllMyDirectoryBuckets() {
        return this.to('ListAllMyDirectoryBuckets');
    }
    /**
     * Grants permission to lists all of the tags for a specified resource
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to associate an access policy with a specified access point
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutAccessPointPolicy.html
     */
    toPutAccessPointPolicy() {
        return this.to('PutAccessPointPolicy');
    }
    /**
     * Grants permission to associate an access point with a specified access point scope configuration
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutAccessPointScope.html
     */
    toPutAccessPointScope() {
        return this.to('PutAccessPointScope');
    }
    /**
     * Grants permission to add or replace a bucket policy on a bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketPolicy.html
     */
    toPutBucketPolicy() {
        return this.to('PutBucketPolicy');
    }
    /**
     * Grants permission to set the encryption configuration for a directory bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketEncryption.html
     */
    toPutEncryptionConfiguration() {
        return this.to('PutEncryptionConfiguration');
    }
    /**
     * Grants permission to create a new lifecycle configuration for the directory bucket or replace an existing lifecycle configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketLifecycleConfiguration.html
     */
    toPutLifecycleConfiguration() {
        return this.to('PutLifecycleConfiguration');
    }
    /**
     * Grants permission to create a new user-defined tag or update an existing tag
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove the specified user-defined tags from an S3 resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifResourceAccount()
     * - .ifSignatureversion()
     * - .ifTlsVersion()
     * - .ifXAmzContentSha256()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Adds a resource of type bucket to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-express-security-iam.html
     *
     * @param bucketName - Identifier for the bucketName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifBucketTag()
     */
    onBucket(bucketName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:s3express:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:bucket/${bucketName}`);
    }
    /**
     * Adds a resource of type accesspoint to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/access-points.html
     *
     * @param accessPointName - Identifier for the accessPointName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAccessPointTag()
     */
    onAccesspoint(accessPointName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:s3express:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:accesspoint/${accessPointName}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/directory-buckets-tagging.html#example-user-policy-request-tag
     *
     * Applies to actions:
     * - .toCreateAccessPoint()
     * - .toCreateBucket()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/directory-buckets-tagging.html#example-user-policy-resource-tag
     *
     * Applies to resource types:
     * - bucket
     * - accesspoint
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/directory-buckets-tagging.html#example-user-policy-tag-keys
     *
     * Applies to actions:
     * - .toCreateAccessPoint()
     * - .toCreateBucket()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the network origin (Internet or VPC)
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/creating-access-points.html#access-points-policies
     *
     * Applies to actions:
     * - .toCreateAccessPoint()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointScope()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointScope()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointScope()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAccessPointNetworkOrigin(value, operator) {
        return this.if(`AccessPointNetworkOrigin`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag key-value pairs attached to the access point
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/access-points-db-tagging.html#example-access-points-db-policy-bucket-tag
     *
     * Applies to resource types:
     * - accesspoint
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAccessPointTag(tagKey, value, operator) {
        return this.if(`AccessPointTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by AWS Local Zone network border group(s) provided in this condition key
     *
     * Applies to actions:
     * - .toCreateSession()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAllAccessRestrictedToLocalZoneGroup(value, operator) {
        return this.if(`AllAccessRestrictedToLocalZoneGroup`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag key-value pairs attached to the bucket
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/directory-buckets-tagging.html#example-policy-bucket-tag
     *
     * Applies to resource types:
     * - bucket
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifBucketTag(tagKey, value, operator) {
        return this.if(`BucketTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the AWS Account ID that owns the access point
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/creating-access-points.html#access-points-policies
     *
     * Applies to actions:
     * - .toCreateAccessPoint()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointScope()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointScope()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointScope()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDataAccessPointAccount(value, operator) {
        return this.if(`DataAccessPointAccount`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by an access point Amazon Resource Name (ARN)
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/creating-access-points.html#access-points-policies
     *
     * Applies to actions:
     * - .toCreateAccessPoint()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointScope()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointScope()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointScope()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifDataAccessPointArn(value, operator) {
        return this.if(`DataAccessPointArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by a specific Availability Zone or Local Zone ID
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/amazon-s3-express-zonal-policy-keys.html#example-location-name
     *
     * Applies to actions:
     * - .toCreateAccessPoint()
     * - .toCreateBucket()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLocationName(value, operator) {
        return this.if(`LocationName`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the permission requested by Access Point Scope configuration, such as GetObject, PutObject
     *
     * Applies to actions:
     * - .toCreateSession()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPermissions(value, operator) {
        return this.if(`Permissions`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the resource owner AWS account ID
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/amazon-s3-express-zonal-policy-keys.html#example-object-resource-account
     *
     * Applies to actions:
     * - .toCreateAccessPoint()
     * - .toCreateBucket()
     * - .toCreateSession()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointScope()
     * - .toDeleteBucket()
     * - .toDeleteBucketPolicy()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointScope()
     * - .toGetBucketPolicy()
     * - .toGetEncryptionConfiguration()
     * - .toGetLifecycleConfiguration()
     * - .toListAccessPointsForDirectoryBuckets()
     * - .toListAllMyDirectoryBuckets()
     * - .toListTagsForResource()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointScope()
     * - .toPutBucketPolicy()
     * - .toPutEncryptionConfiguration()
     * - .toPutLifecycleConfiguration()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceAccount(value, operator) {
        return this.if(`ResourceAccount`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the permission requested by CreateSession API, such as ReadOnly and ReadWrite
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/amazon-s3-express-zonal-policy-keys.html#example-session-mode
     *
     * Applies to actions:
     * - .toCreateSession()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSessionMode(value, operator) {
        return this.if(`SessionMode`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the TLS version used by the client
     *
     * Applies to actions:
     * - .toCreateAccessPoint()
     * - .toCreateBucket()
     * - .toCreateSession()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointScope()
     * - .toDeleteBucket()
     * - .toDeleteBucketPolicy()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointScope()
     * - .toGetBucketPolicy()
     * - .toGetEncryptionConfiguration()
     * - .toGetLifecycleConfiguration()
     * - .toListAccessPointsForDirectoryBuckets()
     * - .toListAllMyDirectoryBuckets()
     * - .toListTagsForResource()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointScope()
     * - .toPutBucketPolicy()
     * - .toPutEncryptionConfiguration()
     * - .toPutLifecycleConfiguration()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifTlsVersion(value, operator) {
        return this.if(`TlsVersion`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access by authentication method
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .toCreateAccessPoint()
     * - .toCreateBucket()
     * - .toCreateSession()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointScope()
     * - .toDeleteBucket()
     * - .toDeleteBucketPolicy()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointScope()
     * - .toGetBucketPolicy()
     * - .toGetEncryptionConfiguration()
     * - .toGetLifecycleConfiguration()
     * - .toListAccessPointsForDirectoryBuckets()
     * - .toListAllMyDirectoryBuckets()
     * - .toListTagsForResource()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointScope()
     * - .toPutBucketPolicy()
     * - .toPutEncryptionConfiguration()
     * - .toPutLifecycleConfiguration()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAuthType(value, operator) {
        return this.if(`authType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the age in milliseconds of the request signature
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .toCreateSession()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifSignatureAge(value, operator) {
        return this.if(`signatureAge`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access by the AWS Signature Version used on the request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .toCreateAccessPoint()
     * - .toCreateBucket()
     * - .toCreateSession()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointScope()
     * - .toDeleteBucket()
     * - .toDeleteBucketPolicy()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointScope()
     * - .toGetBucketPolicy()
     * - .toGetEncryptionConfiguration()
     * - .toGetLifecycleConfiguration()
     * - .toListAccessPointsForDirectoryBuckets()
     * - .toListAllMyDirectoryBuckets()
     * - .toListTagsForResource()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointScope()
     * - .toPutBucketPolicy()
     * - .toPutEncryptionConfiguration()
     * - .toPutLifecycleConfiguration()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSignatureversion(value, operator) {
        return this.if(`signatureversion`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by unsigned content in your bucket
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .toCreateAccessPoint()
     * - .toCreateBucket()
     * - .toCreateSession()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteAccessPointScope()
     * - .toDeleteBucket()
     * - .toDeleteBucketPolicy()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointScope()
     * - .toGetBucketPolicy()
     * - .toGetEncryptionConfiguration()
     * - .toGetLifecycleConfiguration()
     * - .toListAccessPointsForDirectoryBuckets()
     * - .toListAllMyDirectoryBuckets()
     * - .toListTagsForResource()
     * - .toPutAccessPointPolicy()
     * - .toPutAccessPointScope()
     * - .toPutBucketPolicy()
     * - .toPutEncryptionConfiguration()
     * - .toPutLifecycleConfiguration()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzContentSha256(value, operator) {
        return this.if(`x-amz-content-sha256`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by server-side encryption
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-express-data-protection.html
     *
     * Applies to actions:
     * - .toCreateSession()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzServerSideEncryption(value, operator) {
        return this.if(`x-amz-server-side-encryption`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by AWS KMS customer managed key for server-side encryption
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-express-UsingKMSEncryption.html#s3-express-require-sse-kms
     *
     * Applies to actions:
     * - .toCreateSession()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifXAmzServerSideEncryptionAwsKmsKeyId(value, operator) {
        return this.if(`x-amz-server-side-encryption-aws-kms-key-id`, value, operator ?? 'ArnLike');
    }
    /**
     * Statement provider for service [s3express](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazons3express.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 's3express';
        this.accessLevelList = {
            Write: [
                'CreateAccessPoint',
                'CreateBucket',
                'CreateSession',
                'DeleteAccessPoint',
                'DeleteBucket',
                'PutEncryptionConfiguration',
                'PutLifecycleConfiguration'
            ],
            'Permissions management': [
                'DeleteAccessPointPolicy',
                'DeleteAccessPointScope',
                'DeleteBucketPolicy',
                'PutAccessPointPolicy',
                'PutAccessPointScope',
                'PutBucketPolicy'
            ],
            Read: [
                'GetAccessPoint',
                'GetAccessPointPolicy',
                'GetAccessPointScope',
                'GetBucketPolicy',
                'GetEncryptionConfiguration',
                'GetLifecycleConfiguration'
            ],
            List: [
                'ListAccessPointsForDirectoryBuckets',
                'ListAllMyDirectoryBuckets',
                'ListTagsForResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.S3express = S3express;
//# sourceMappingURL=data:application/json;base64,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