"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3tables = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [s3tables](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazons3tables.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class S3tables extends shared_1.PolicyStatement {
    /**
     * Grants permission to create a namespace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3TableBuckets_CreateNamespace.html
     */
    toCreateNamespace() {
        return this.to('CreateNamespace');
    }
    /**
     * Grants permission to create a table
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifNamespace()
     * - .ifSSEAlgorithm()
     * - .ifKMSKeyArn()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3TableBuckets_CreateTable.html
     */
    toCreateTable() {
        return this.to('CreateTable');
    }
    /**
     * Grants permission to create a table bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSSEAlgorithm()
     * - .ifKMSKeyArn()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3TableBuckets_CreateTableBucket.html
     */
    toCreateTableBucket() {
        return this.to('CreateTableBucket');
    }
    /**
     * Grants permission to delete a namespace
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifNamespace()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3TableBuckets_DeleteNamespace.html
     */
    toDeleteNamespace() {
        return this.to('DeleteNamespace');
    }
    /**
     * Grants permission to delete a table
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifNamespace()
     * - .ifTableName()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3TableBuckets_DeleteTable.html
     */
    toDeleteTable() {
        return this.to('DeleteTable');
    }
    /**
     * Grants permission to delete a table bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3TableBuckets_DeleteTableBucket.html
     */
    toDeleteTableBucket() {
        return this.to('DeleteTableBucket');
    }
    /**
     * Grants permission to delete encryption configuration on a table bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3TableBuckets_DeleteTableBucketEncryption.html
     */
    toDeleteTableBucketEncryption() {
        return this.to('DeleteTableBucketEncryption');
    }
    /**
     * Grants permission to delete a policy on a table bucket
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3TableBuckets_DeleteTableBucketPolicy.html
     */
    toDeleteTableBucketPolicy() {
        return this.to('DeleteTableBucketPolicy');
    }
    /**
     * Grants permission to delete a policy on a table
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifNamespace()
     * - .ifTableName()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3TableBuckets_DeleteTablePolicy.html
     */
    toDeleteTablePolicy() {
        return this.to('DeleteTablePolicy');
    }
    /**
     * Grants permission to get a namespace
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifNamespace()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3TableBuckets_GetNamespace.html
     */
    toGetNamespace() {
        return this.to('GetNamespace');
    }
    /**
     * Grants permission to retrieve a table
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifNamespace()
     * - .ifTableName()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3TableBuckets_GetTable.html
     */
    toGetTable() {
        return this.to('GetTable');
    }
    /**
     * Grants permission to retrieve a table bucket
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3TableBuckets_GetTableBucket.html
     */
    toGetTableBucket() {
        return this.to('GetTableBucket');
    }
    /**
     * Grants permission to retrieve encryption configuration on a table bucket
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3TableBuckets_GetTableBucketEncryption.html
     */
    toGetTableBucketEncryption() {
        return this.to('GetTableBucketEncryption');
    }
    /**
     * Grants permission to retrieve a maintenance configuration on a table bucket
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3TableBuckets_GetTableBucketMaintenanceConfiguration.html
     */
    toGetTableBucketMaintenanceConfiguration() {
        return this.to('GetTableBucketMaintenanceConfiguration');
    }
    /**
     * Grants permission to retrieve a policy on a table bucket
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3TableBuckets_GetTableBucketPolicy.html
     */
    toGetTableBucketPolicy() {
        return this.to('GetTableBucketPolicy');
    }
    /**
     * Grants permission to read metadata and data objects from a table storage endpoint using S3 APIs
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifNamespace()
     * - .ifTableName()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-setting-up.html#s3-tables-actions
     */
    toGetTableData() {
        return this.to('GetTableData');
    }
    /**
     * Grants permission to retrieve encryption configuration on a table
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifNamespace()
     * - .ifTableName()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3TableBuckets_GetTableEncryption.html
     */
    toGetTableEncryption() {
        return this.to('GetTableEncryption');
    }
    /**
     * Grants permission to retrieve a maintenance configuration on a table
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifNamespace()
     * - .ifTableName()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3TableBuckets_GetTableMaintenanceConfiguration.html
     */
    toGetTableMaintenanceConfiguration() {
        return this.to('GetTableMaintenanceConfiguration');
    }
    /**
     * Grants permission to retrieve the status of maintenance jobs on a table
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifNamespace()
     * - .ifTableName()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3TableBuckets_GetTableMaintenanceJobStatus.html
     */
    toGetTableMaintenanceJobStatus() {
        return this.to('GetTableMaintenanceJobStatus');
    }
    /**
     * Grants permission to retrieve the metadata location of a table
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifNamespace()
     * - .ifTableName()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3TableBuckets_GetTableMetadataLocation.html
     */
    toGetTableMetadataLocation() {
        return this.to('GetTableMetadataLocation');
    }
    /**
     * Grants permission to retrieve a policy on a table
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifNamespace()
     * - .ifTableName()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3TableBuckets_GetTablePolicy.html
     */
    toGetTablePolicy() {
        return this.to('GetTablePolicy');
    }
    /**
     * Grants permission to list namespaces
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3TableBuckets_ListNamespaces.html
     */
    toListNamespaces() {
        return this.to('ListNamespaces');
    }
    /**
     * Grants permission to list table buckets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3TableBuckets_ListTableBuckets.html
     */
    toListTableBuckets() {
        return this.to('ListTableBuckets');
    }
    /**
     * Grants permission to list tables
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifNamespace()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3TableBuckets_ListTables.html
     */
    toListTables() {
        return this.to('ListTables');
    }
    /**
     * Grants permission to put or overwrite encryption configuration on a table bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifKMSKeyArn()
     * - .ifSSEAlgorithm()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3TableBuckets_PutTableBucketEncryption.html
     */
    toPutTableBucketEncryption() {
        return this.to('PutTableBucketEncryption');
    }
    /**
     * Grants permission to put a maintenance configuration on a table bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3TableBuckets_PutTableBucketMaintenanceConfiguration.html
     */
    toPutTableBucketMaintenanceConfiguration() {
        return this.to('PutTableBucketMaintenanceConfiguration');
    }
    /**
     * Grants permission to create or overwrite a policy on a table bucket
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3TableBuckets_PutTableBucketPolicy.html
     */
    toPutTableBucketPolicy() {
        return this.to('PutTableBucketPolicy');
    }
    /**
     * Grants permission to write metadata and data objects to a table storage endpoint using S3 APIs
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifNamespace()
     * - .ifTableName()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-setting-up.html#s3-tables-actions
     */
    toPutTableData() {
        return this.to('PutTableData');
    }
    /**
     * Grants permission to put encryption configuration on a table
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifNamespace()
     * - .ifSSEAlgorithm()
     * - .ifKMSKeyArn()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-setting-up.html#s3-tables-actions
     */
    toPutTableEncryption() {
        return this.to('PutTableEncryption');
    }
    /**
     * Grants permission to put a maintenance configuration on a table
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifNamespace()
     * - .ifTableName()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3TableBuckets_PutTableMaintenanceConfiguration.html
     */
    toPutTableMaintenanceConfiguration() {
        return this.to('PutTableMaintenanceConfiguration');
    }
    /**
     * Grants permission to create or overwrite a policy on a table
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifNamespace()
     * - .ifTableName()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3TableBuckets_PutTablePolicy.html
     */
    toPutTablePolicy() {
        return this.to('PutTablePolicy');
    }
    /**
     * Grants permission to rename a table or move a table across namespaces
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifNamespace()
     * - .ifTableName()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3TableBuckets_RenameTable.html
     */
    toRenameTable() {
        return this.to('RenameTable');
    }
    /**
     * Grants permission to update the metadata location of a table
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifNamespace()
     * - .ifTableName()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_s3TableBuckets_UpdateTableMetadataLocation.html
     */
    toUpdateTableMetadataLocation() {
        return this.to('UpdateTableMetadataLocation');
    }
    /**
     * Adds a resource of type TableBucket to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-buckets.html
     *
     * @param tableBucketName - Identifier for the tableBucketName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onTableBucket(tableBucketName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:s3tables:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:bucket/${tableBucketName}`);
    }
    /**
     * Adds a resource of type Table to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-tables.html
     *
     * @param tableBucketName - Identifier for the tableBucketName.
     * @param tableID - Identifier for the tableID.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifNamespace()
     * - .ifTableName()
     */
    onTable(tableBucketName, tableID, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:s3tables:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:bucket/${tableBucketName}/table/${tableID}`);
    }
    /**
     * Filters access by the AWS KMS key ARN for the key used to encrypt a table
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-setting-up.htmls3-tables-setting-up.html
     *
     * Applies to actions:
     * - .toCreateTable()
     * - .toCreateTableBucket()
     * - .toPutTableBucketEncryption()
     * - .toPutTableEncryption()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifKMSKeyArn(value, operator) {
        return this.if(`KMSKeyArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the server-side encryption algorithm used to encrypt a table
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-setting-up.htmls3-tables-setting-up.html
     *
     * Applies to actions:
     * - .toCreateTable()
     * - .toCreateTableBucket()
     * - .toPutTableBucketEncryption()
     * - .toPutTableEncryption()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSSEAlgorithm(value, operator) {
        return this.if(`SSEAlgorithm`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the namespaces created in the table bucket
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-setting-up.htmls3-tables-setting-up.html
     *
     * Applies to actions:
     * - .toCreateTable()
     * - .toDeleteNamespace()
     * - .toDeleteTable()
     * - .toDeleteTablePolicy()
     * - .toGetNamespace()
     * - .toGetTable()
     * - .toGetTableData()
     * - .toGetTableEncryption()
     * - .toGetTableMaintenanceConfiguration()
     * - .toGetTableMaintenanceJobStatus()
     * - .toGetTableMetadataLocation()
     * - .toGetTablePolicy()
     * - .toListTables()
     * - .toPutTableData()
     * - .toPutTableEncryption()
     * - .toPutTableMaintenanceConfiguration()
     * - .toPutTablePolicy()
     * - .toRenameTable()
     * - .toUpdateTableMetadataLocation()
     *
     * Applies to resource types:
     * - Table
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifNamespace(value, operator) {
        return this.if(`namespace`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the name of the tables in the table bucket
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-tables-setting-up.htmls3-tables-setting-up.html
     *
     * Applies to actions:
     * - .toDeleteTable()
     * - .toDeleteTablePolicy()
     * - .toGetTable()
     * - .toGetTableData()
     * - .toGetTableEncryption()
     * - .toGetTableMaintenanceConfiguration()
     * - .toGetTableMaintenanceJobStatus()
     * - .toGetTableMetadataLocation()
     * - .toGetTablePolicy()
     * - .toPutTableData()
     * - .toPutTableMaintenanceConfiguration()
     * - .toPutTablePolicy()
     * - .toRenameTable()
     * - .toUpdateTableMetadataLocation()
     *
     * Applies to resource types:
     * - Table
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTableName(value, operator) {
        return this.if(`tableName`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [s3tables](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazons3tables.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 's3tables';
        this.accessLevelList = {
            Write: [
                'CreateNamespace',
                'CreateTable',
                'CreateTableBucket',
                'DeleteNamespace',
                'DeleteTable',
                'DeleteTableBucket',
                'DeleteTableBucketEncryption',
                'PutTableBucketEncryption',
                'PutTableBucketMaintenanceConfiguration',
                'PutTableData',
                'PutTableEncryption',
                'PutTableMaintenanceConfiguration',
                'RenameTable',
                'UpdateTableMetadataLocation'
            ],
            'Permissions management': [
                'DeleteTableBucketPolicy',
                'DeleteTablePolicy',
                'PutTableBucketPolicy',
                'PutTablePolicy'
            ],
            Read: [
                'GetNamespace',
                'GetTable',
                'GetTableBucket',
                'GetTableBucketEncryption',
                'GetTableBucketMaintenanceConfiguration',
                'GetTableBucketPolicy',
                'GetTableData',
                'GetTableEncryption',
                'GetTableMaintenanceConfiguration',
                'GetTableMaintenanceJobStatus',
                'GetTableMetadataLocation',
                'GetTablePolicy'
            ],
            List: [
                'ListNamespaces',
                'ListTableBuckets',
                'ListTables'
            ]
        };
    }
}
exports.S3tables = S3tables;
//# sourceMappingURL=data:application/json;base64,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