import { AccessLevelList } from '../../shared/access-level';
import { PolicyStatement, Operator } from '../../shared';
import { aws_iam as iam } from "aws-cdk-lib";
/**
 * Statement provider for service [s3vectors](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazons3vectors.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class S3vectors extends PolicyStatement {
    servicePrefix: string;
    /**
     * Grants permission to create a new vector index within a specified vector bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_S3VectorBuckets_CreateIndex.html
     */
    toCreateIndex(): this;
    /**
     * Grants permission to create a new vector bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSseType()
     * - .ifKmsKeyArn()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_S3VectorBuckets_CreateVectorBucket.html
     */
    toCreateVectorBucket(): this;
    /**
     * Grants permission to delete a specified vector index
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_S3VectorBuckets_DeleteIndex.html
     */
    toDeleteIndex(): this;
    /**
     * Grants permission to delete a specified vector bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_S3VectorBuckets_DeleteVectorBucket.html
     */
    toDeleteVectorBucket(): this;
    /**
     * Grants permission to delete the IAM resource policy from a specified vector bucket
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_S3VectorBuckets_DeleteVectorBucketPolicy.html
     */
    toDeleteVectorBucketPolicy(): this;
    /**
     * Grants permission to delete a batch of vectors from a specified vector index
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_S3VectorBuckets_DeleteVectors.html
     */
    toDeleteVectors(): this;
    /**
     * Grants permission to get the attributes of a specified vector index
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_S3VectorBuckets_GetIndex.html
     */
    toGetIndex(): this;
    /**
     * Grants permission to get the attributes of a specified vector bucket
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_S3VectorBuckets_GetVectorBucket.html
     */
    toGetVectorBucket(): this;
    /**
     * Grants permission to get the IAM resource policy for a specific vector bucket
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_S3VectorBuckets_GetVectorBucketPolicy.html
     */
    toGetVectorBucketPolicy(): this;
    /**
     * Grants permission to get a batch of vectors by their vector keys
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_S3VectorBuckets_GetVectors.html
     */
    toGetVectors(): this;
    /**
     * Grants permission to get a paginated list of all indexes in a specified vector bucket
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_S3VectorBuckets_ListIndexes.html
     */
    toListIndexes(): this;
    /**
     * Grants permission to get a paginated list of all vector buckets in the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_S3VectorBuckets_ListVectorBuckets.html
     */
    toListVectorBuckets(): this;
    /**
     * Grants permission to get a paginated list of all vectors in a specified vector index
     *
     * Access Level: List
     *
     * Dependent actions:
     * - s3vectors:GetVectors
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_S3VectorBuckets_ListVectors.html
     */
    toListVectors(): this;
    /**
     * Grants permission to add an IAM resource policy to a specified vector bucket
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_S3VectorBuckets_PutVectorBucketPolicy.html
     */
    toPutVectorBucketPolicy(): this;
    /**
     * Grants permission to add a batch of vectors to a specified vector index
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_S3VectorBuckets_PutVectors.html
     */
    toPutVectors(): this;
    /**
     * Grants permission to find approximate nearest neighbors within a specified search vector index for a given query vector
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - s3vectors:GetVectors
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_S3VectorBuckets_QueryVectors.html
     */
    toQueryVectors(): this;
    protected accessLevelList: AccessLevelList;
    /**
     * Adds a resource of type Index to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-vectors-access-management.html
     *
     * @param bucketName - Identifier for the bucketName.
     * @param indexName - Identifier for the indexName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onIndex(bucketName: string, indexName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type VectorBucket to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-vectors-access-management.html
     *
     * @param bucketName - Identifier for the bucketName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onVectorBucket(bucketName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Filters access by the AWS KMS key ARN for the key used to encrypt a vector bucket
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-vectors-access-management.html#s3-vectors-condition-keyss3-vectors-access-management.html
     *
     * Applies to actions:
     * - .toCreateVectorBucket()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifKmsKeyArn(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by server-side encryption type
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-vectors-access-management.html#s3-vectors-condition-keyss3-vectors-access-management.html
     *
     * Applies to actions:
     * - .toCreateVectorBucket()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSseType(value: string | string[], operator?: Operator | string): this;
    /**
     * Statement provider for service [s3vectors](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazons3vectors.html).
     *
     */
    constructor(props?: iam.PolicyStatementProps);
}
