"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3vectors = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [s3vectors](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazons3vectors.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class S3vectors extends shared_1.PolicyStatement {
    /**
     * Grants permission to create a new vector index within a specified vector bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_S3VectorBuckets_CreateIndex.html
     */
    toCreateIndex() {
        return this.to('CreateIndex');
    }
    /**
     * Grants permission to create a new vector bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSseType()
     * - .ifKmsKeyArn()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_S3VectorBuckets_CreateVectorBucket.html
     */
    toCreateVectorBucket() {
        return this.to('CreateVectorBucket');
    }
    /**
     * Grants permission to delete a specified vector index
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_S3VectorBuckets_DeleteIndex.html
     */
    toDeleteIndex() {
        return this.to('DeleteIndex');
    }
    /**
     * Grants permission to delete a specified vector bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_S3VectorBuckets_DeleteVectorBucket.html
     */
    toDeleteVectorBucket() {
        return this.to('DeleteVectorBucket');
    }
    /**
     * Grants permission to delete the IAM resource policy from a specified vector bucket
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_S3VectorBuckets_DeleteVectorBucketPolicy.html
     */
    toDeleteVectorBucketPolicy() {
        return this.to('DeleteVectorBucketPolicy');
    }
    /**
     * Grants permission to delete a batch of vectors from a specified vector index
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_S3VectorBuckets_DeleteVectors.html
     */
    toDeleteVectors() {
        return this.to('DeleteVectors');
    }
    /**
     * Grants permission to get the attributes of a specified vector index
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_S3VectorBuckets_GetIndex.html
     */
    toGetIndex() {
        return this.to('GetIndex');
    }
    /**
     * Grants permission to get the attributes of a specified vector bucket
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_S3VectorBuckets_GetVectorBucket.html
     */
    toGetVectorBucket() {
        return this.to('GetVectorBucket');
    }
    /**
     * Grants permission to get the IAM resource policy for a specific vector bucket
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_S3VectorBuckets_GetVectorBucketPolicy.html
     */
    toGetVectorBucketPolicy() {
        return this.to('GetVectorBucketPolicy');
    }
    /**
     * Grants permission to get a batch of vectors by their vector keys
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_S3VectorBuckets_GetVectors.html
     */
    toGetVectors() {
        return this.to('GetVectors');
    }
    /**
     * Grants permission to get a paginated list of all indexes in a specified vector bucket
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_S3VectorBuckets_ListIndexes.html
     */
    toListIndexes() {
        return this.to('ListIndexes');
    }
    /**
     * Grants permission to get a paginated list of all vector buckets in the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_S3VectorBuckets_ListVectorBuckets.html
     */
    toListVectorBuckets() {
        return this.to('ListVectorBuckets');
    }
    /**
     * Grants permission to get a paginated list of all vectors in a specified vector index
     *
     * Access Level: List
     *
     * Dependent actions:
     * - s3vectors:GetVectors
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_S3VectorBuckets_ListVectors.html
     */
    toListVectors() {
        return this.to('ListVectors');
    }
    /**
     * Grants permission to add an IAM resource policy to a specified vector bucket
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_S3VectorBuckets_PutVectorBucketPolicy.html
     */
    toPutVectorBucketPolicy() {
        return this.to('PutVectorBucketPolicy');
    }
    /**
     * Grants permission to add a batch of vectors to a specified vector index
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_S3VectorBuckets_PutVectors.html
     */
    toPutVectors() {
        return this.to('PutVectors');
    }
    /**
     * Grants permission to find approximate nearest neighbors within a specified search vector index for a given query vector
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - s3vectors:GetVectors
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_S3VectorBuckets_QueryVectors.html
     */
    toQueryVectors() {
        return this.to('QueryVectors');
    }
    /**
     * Adds a resource of type Index to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-vectors-access-management.html
     *
     * @param bucketName - Identifier for the bucketName.
     * @param indexName - Identifier for the indexName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onIndex(bucketName, indexName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:s3vectors:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:bucket/${bucketName}/index/${indexName}`);
    }
    /**
     * Adds a resource of type VectorBucket to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-vectors-access-management.html
     *
     * @param bucketName - Identifier for the bucketName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onVectorBucket(bucketName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:s3vectors:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:bucket/${bucketName}`);
    }
    /**
     * Filters access by the AWS KMS key ARN for the key used to encrypt a vector bucket
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-vectors-access-management.html#s3-vectors-condition-keyss3-vectors-access-management.html
     *
     * Applies to actions:
     * - .toCreateVectorBucket()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifKmsKeyArn(value, operator) {
        return this.if(`kmsKeyArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by server-side encryption type
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/userguide/s3-vectors-access-management.html#s3-vectors-condition-keyss3-vectors-access-management.html
     *
     * Applies to actions:
     * - .toCreateVectorBucket()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSseType(value, operator) {
        return this.if(`sseType`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [s3vectors](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazons3vectors.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 's3vectors';
        this.accessLevelList = {
            Write: [
                'CreateIndex',
                'CreateVectorBucket',
                'DeleteIndex',
                'DeleteVectorBucket',
                'DeleteVectors',
                'PutVectors'
            ],
            'Permissions management': [
                'DeleteVectorBucketPolicy',
                'PutVectorBucketPolicy'
            ],
            Read: [
                'GetIndex',
                'GetVectorBucket',
                'GetVectorBucketPolicy',
                'GetVectors',
                'QueryVectors'
            ],
            List: [
                'ListIndexes',
                'ListVectorBuckets',
                'ListVectors'
            ]
        };
    }
}
exports.S3vectors = S3vectors;
//# sourceMappingURL=data:application/json;base64,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