"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Sagemaker = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [sagemaker](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonsagemaker.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Sagemaker extends shared_1.PolicyStatement {
    /**
     * Grants permission to associate a lineage entity (artifact, context, action, experiment, experiment-trial-component) to each other
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_AddAssociation.html
     */
    toAddAssociation() {
        return this.to('AddAssociation');
    }
    /**
     * Grants permission to add or overwrite one or more tags for the specified Amazon SageMaker resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifTaggingAction()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_AddTags.html
     */
    toAddTags() {
        return this.to('AddTags');
    }
    /**
     * Grants permission to associate a trial component with a trial
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_AssociateTrialComponent.html
     */
    toAssociateTrialComponent() {
        return this.to('AssociateTrialComponent');
    }
    /**
     * Grants permission to attach an Amazon EBS volume to a SageMaker HyperPod cluster node
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:AttachVolume
     * - ec2:DescribeVolumes
     * - eks:DescribeCluster
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_AttachClusterNodeVolume.html
     */
    toAttachClusterNodeVolume() {
        return this.to('AttachClusterNodeVolume');
    }
    /**
     * Grants permission to add multiple nodes at a time to a SageMaker HyperPod cluster
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - eks:DescribeCluster
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_BatchAddClusterNodes.html
     */
    toBatchAddClusterNodes() {
        return this.to('BatchAddClusterNodes');
    }
    /**
     * Grants permission to batch delete SageMaker HyperPod cluster nodes
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - eks:DescribeCluster
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_BatchDeleteClusterNodes.html
     */
    toBatchDeleteClusterNodes() {
        return this.to('BatchDeleteClusterNodes');
    }
    /**
     * Grants permission to describe one or more ModelPackages
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_BatchDescribeModelPackage.html
     */
    toBatchDescribeModelPackage() {
        return this.to('BatchDescribeModelPackage');
    }
    /**
     * Grants permission to retrieve metrics associated with SageMaker Resources such as Training Jobs or Trial Components
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/
     */
    toBatchGetMetrics() {
        return this.to('BatchGetMetrics');
    }
    /**
     * Grants permission to get a batch of records from one or more feature groups
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_feature_store_BatchGetRecord.html
     */
    toBatchGetRecord() {
        return this.to('BatchGetRecord');
    }
    /**
     * Grants permission to publish metrics associated with a SageMaker Resource such as a Training Job or Trial Component
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/
     */
    toBatchPutMetrics() {
        return this.to('BatchPutMetrics');
    }
    /**
     * Grants permission for Partner App SDK to access the Partner App for reading or writing data use cases
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/partner-apps-onboard.html
     */
    toCallPartnerAppApi() {
        return this.to('CallPartnerAppApi');
    }
    /**
     * Grants permission to create an action
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateAction.html
     */
    toCreateAction() {
        return this.to('CreateAction');
    }
    /**
     * Grants permission to create an algorithm
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateAlgorithm.html
     */
    toCreateAlgorithm() {
        return this.to('CreateAlgorithm');
    }
    /**
     * Grants permission to create an App for a SageMaker UserProfile or Space
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceTypes()
     * - .ifImageArns()
     * - .ifImageVersionArns()
     * - .ifOwnerUserProfileArn()
     * - .ifSpaceSharingType()
     *
     * Dependent actions:
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateApp.html
     */
    toCreateApp() {
        return this.to('CreateApp');
    }
    /**
     * Grants permission to create an AppImageConfig
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateAppImageConfig.html
     */
    toCreateAppImageConfig() {
        return this.to('CreateAppImageConfig');
    }
    /**
     * Grants permission to create an artifact
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateArtifact.html
     */
    toCreateArtifact() {
        return this.to('CreateArtifact');
    }
    /**
     * Grants permission to create an AutoML job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInterContainerTrafficEncryption()
     * - .ifOutputKmsKey()
     * - .ifVolumeKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * Dependent actions:
     * - iam:PassRole
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateAutoMLJob.html
     */
    toCreateAutoMLJob() {
        return this.to('CreateAutoMLJob');
    }
    /**
     * Grants permission to create a V2 AutoML job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInterContainerTrafficEncryption()
     * - .ifOutputKmsKey()
     * - .ifVolumeKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * Dependent actions:
     * - iam:PassRole
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateAutoMLJobV2.html
     */
    toCreateAutoMLJobV2() {
        return this.to('CreateAutoMLJobV2');
    }
    /**
     * Grants permission to create a SageMaker HyperPod cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceTypes()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * Dependent actions:
     * - ec2:DescribeImages
     * - ec2:DescribeSnapshots
     * - ec2:ModifyImageAttribute
     * - ec2:ModifySnapshotAttribute
     * - eks:AssociateAccessPolicy
     * - eks:CreateAccessEntry
     * - eks:DeleteAccessEntry
     * - eks:DescribeAccessEntry
     * - eks:DescribeCluster
     * - iam:CreateServiceLinkedRole
     * - iam:PassRole
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateCluster.html
     */
    toCreateCluster() {
        return this.to('CreateCluster');
    }
    /**
     * Grants permission to create a cluster scheduler config
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - eks:AssociateAccessPolicy
     * - eks:DescribeCluster
     * - eks:ListAssociatedAccessPolicies
     * - sagemaker:AddTags
     * - sagemaker:DescribeCluster
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateClusterSchedulerConfig.html
     */
    toCreateClusterSchedulerConfig() {
        return this.to('CreateClusterSchedulerConfig');
    }
    /**
     * Grants permission to create a CodeRepository
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateCodeRepository.html
     */
    toCreateCodeRepository() {
        return this.to('CreateCodeRepository');
    }
    /**
     * Grants permission to create a compilation job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateCompilationJob.html
     */
    toCreateCompilationJob() {
        return this.to('CreateCompilationJob');
    }
    /**
     * Grants permission to create a compute quota
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - eks:AssociateAccessPolicy
     * - eks:DescribeCluster
     * - eks:ListAssociatedAccessPolicies
     * - sagemaker:AddTags
     * - sagemaker:DescribeCluster
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateComputeQuota.html
     */
    toCreateComputeQuota() {
        return this.to('CreateComputeQuota');
    }
    /**
     * Grants permission to create a context
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateContext.html
     */
    toCreateContext() {
        return this.to('CreateContext');
    }
    /**
     * Grants permission to create a data quality job definition
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceTypes()
     * - .ifInterContainerTrafficEncryption()
     * - .ifMaxRuntimeInSeconds()
     * - .ifNetworkIsolation()
     * - .ifOutputKmsKey()
     * - .ifVolumeKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * Dependent actions:
     * - iam:PassRole
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateDataQualityJobDefinition.html
     */
    toCreateDataQualityJobDefinition() {
        return this.to('CreateDataQualityJobDefinition');
    }
    /**
     * Grants permission to create a device fleet
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateDeviceFleet.html
     */
    toCreateDeviceFleet() {
        return this.to('CreateDeviceFleet');
    }
    /**
     * Grants permission to create a Domain for SageMaker Studio
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAppNetworkAccessType()
     * - .ifInstanceTypes()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     * - .ifDomainSharingOutputKmsKey()
     * - .ifVolumeKmsKey()
     * - .ifImageArns()
     * - .ifImageVersionArns()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - iam:PassRole
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateDomain.html
     */
    toCreateDomain() {
        return this.to('CreateDomain');
    }
    /**
     * Grants permission to create an edge deployment plan
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateEdgeDeploymentPlan.html
     */
    toCreateEdgeDeploymentPlan() {
        return this.to('CreateEdgeDeploymentPlan');
    }
    /**
     * Grants permission to create an edge deployment stage
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateEdgeDeploymentStage.html
     */
    toCreateEdgeDeploymentStage() {
        return this.to('CreateEdgeDeploymentStage');
    }
    /**
     * Grants permission to create an edge packaging job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateEdgePackagingJob.html
     */
    toCreateEdgePackagingJob() {
        return this.to('CreateEdgePackagingJob');
    }
    /**
     * Grants permission to create an endpoint using the endpoint configuration specified in the request
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateEndpoint.html
     */
    toCreateEndpoint() {
        return this.to('CreateEndpoint');
    }
    /**
     * Grants permission to create an endpoint configuration that can be deployed using Amazon SageMaker hosting services
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAcceleratorTypes()
     * - .ifInstanceTypes()
     * - .ifModelArn()
     * - .ifVolumeKmsKey()
     * - .ifServerlessMaxConcurrency()
     * - .ifServerlessMemorySize()
     * - .ifNetworkIsolation()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * Dependent actions:
     * - iam:PassRole
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateEndpointConfig.html
     */
    toCreateEndpointConfig() {
        return this.to('CreateEndpointConfig');
    }
    /**
     * Grants permission to create an experiment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateExperiment.html
     */
    toCreateExperiment() {
        return this.to('CreateExperiment');
    }
    /**
     * Grants permission to create a feature group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifFeatureGroupOnlineStoreKmsKey()
     * - .ifFeatureGroupOfflineStoreKmsKey()
     * - .ifFeatureGroupOfflineStoreS3Uri()
     * - .ifFeatureGroupEnableOnlineStore()
     * - .ifFeatureGroupOfflineStoreConfig()
     * - .ifFeatureGroupDisableGlueTableCreation()
     *
     * Dependent actions:
     * - iam:PassRole
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateFeatureGroup.html
     */
    toCreateFeatureGroup() {
        return this.to('CreateFeatureGroup');
    }
    /**
     * Grants permission to create a flow definition, which defines settings for a human workflow
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifWorkteamArn()
     * - .ifWorkteamType()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateFlowDefinition.html
     */
    toCreateFlowDefinition() {
        return this.to('CreateFlowDefinition');
    }
    /**
     * Grants permission to create a hub
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateHub.html
     */
    toCreateHub() {
        return this.to('CreateHub');
    }
    /**
     * Grants permission to generate S3 presigned URLs with GetObject permission for accessing model artifacts
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateHubContentPresignedUrls.html
     */
    toCreateHubContentPresignedUrls() {
        return this.to('CreateHubContentPresignedUrls');
    }
    /**
     * Grants permission to create hub content reference
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateHubContentReference.html
     */
    toCreateHubContentReference() {
        return this.to('CreateHubContentReference');
    }
    /**
     * Grants permission to define the settings you will use for the human review workflow user interface
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateHumanTaskUi.html
     */
    toCreateHumanTaskUi() {
        return this.to('CreateHumanTaskUi');
    }
    /**
     * Grants permission to create a hyper parameter tuning job that can be deployed using Amazon SageMaker
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifFileSystemAccessMode()
     * - .ifFileSystemDirectoryPath()
     * - .ifFileSystemId()
     * - .ifFileSystemType()
     * - .ifInstanceTypes()
     * - .ifInterContainerTrafficEncryption()
     * - .ifMaxRuntimeInSeconds()
     * - .ifNetworkIsolation()
     * - .ifOutputKmsKey()
     * - .ifVolumeKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * Dependent actions:
     * - iam:PassRole
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateHyperParameterTuningJob.html
     */
    toCreateHyperParameterTuningJob() {
        return this.to('CreateHyperParameterTuningJob');
    }
    /**
     * Grants permission to create a SageMaker Image
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateImage.html
     */
    toCreateImage() {
        return this.to('CreateImage');
    }
    /**
     * Grants permission to create a SageMaker ImageVersion
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateImageVersion.html
     */
    toCreateImageVersion() {
        return this.to('CreateImageVersion');
    }
    /**
     * Grants permission to create an inference component on an endpoint
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifModelArn()
     *
     * Dependent actions:
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateInferenceComponent.html
     */
    toCreateInferenceComponent() {
        return this.to('CreateInferenceComponent');
    }
    /**
     * Grants permission to create an inference experiment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateInferenceExperiment.html
     */
    toCreateInferenceExperiment() {
        return this.to('CreateInferenceExperiment');
    }
    /**
     * Grants permission to create an inference recommendations job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateInferenceRecommendationsJob.html
     */
    toCreateInferenceRecommendationsJob() {
        return this.to('CreateInferenceRecommendationsJob');
    }
    /**
     * Grants permission to start a labeling job. A labeling job takes unlabeled data in and produces labeled data as output, which can be used for training SageMaker models
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifWorkteamArn()
     * - .ifWorkteamType()
     * - .ifVolumeKmsKey()
     * - .ifOutputKmsKey()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateLabelingJob.html
     */
    toCreateLabelingJob() {
        return this.to('CreateLabelingJob');
    }
    /**
     * Grants permission to create a lineage group policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/Welcome.html
     */
    toCreateLineageGroupPolicy() {
        return this.to('CreateLineageGroupPolicy');
    }
    /**
     * Grants permission to create an MLflow tracking server
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateMlflowTrackingServer.html
     */
    toCreateMlflowTrackingServer() {
        return this.to('CreateMlflowTrackingServer');
    }
    /**
     * Grants permission to create a model in Amazon SageMaker. In the request, you specify a name for the model and describe one or more containers
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifNetworkIsolation()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     * - .ifDirectGatedModelAccess()
     *
     * Dependent actions:
     * - iam:PassRole
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateModel.html
     */
    toCreateModel() {
        return this.to('CreateModel');
    }
    /**
     * Grants permission to create a model bias job definition
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceTypes()
     * - .ifInterContainerTrafficEncryption()
     * - .ifMaxRuntimeInSeconds()
     * - .ifNetworkIsolation()
     * - .ifOutputKmsKey()
     * - .ifVolumeKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * Dependent actions:
     * - iam:PassRole
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateModelBiasJobDefinition.html
     */
    toCreateModelBiasJobDefinition() {
        return this.to('CreateModelBiasJobDefinition');
    }
    /**
     * Grants permission to create a model card
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateModelCard.html
     */
    toCreateModelCard() {
        return this.to('CreateModelCard');
    }
    /**
     * Grants permission to create an export job for a model card
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateModelCardExportJob.html
     */
    toCreateModelCardExportJob() {
        return this.to('CreateModelCardExportJob');
    }
    /**
     * Grants permission to create a model explainability job definition
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceTypes()
     * - .ifInterContainerTrafficEncryption()
     * - .ifMaxRuntimeInSeconds()
     * - .ifNetworkIsolation()
     * - .ifOutputKmsKey()
     * - .ifVolumeKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * Dependent actions:
     * - iam:PassRole
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateModelExplainabilityJobDefinition.html
     */
    toCreateModelExplainabilityJobDefinition() {
        return this.to('CreateModelExplainabilityJobDefinition');
    }
    /**
     * Grants permission to create a ModelPackage
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifModelApprovalStatus()
     * - .ifCustomerMetadataProperties()
     * - .ifModelLifeCycle()
     * - .ifModelLifeCycle()
     *
     * Dependent actions:
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateModelPackage.html
     */
    toCreateModelPackage() {
        return this.to('CreateModelPackage');
    }
    /**
     * Grants permission to create a ModelPackageGroup
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateModelPackageGroup.html
     */
    toCreateModelPackageGroup() {
        return this.to('CreateModelPackageGroup');
    }
    /**
     * Grants permission to create a model quality job definition
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceTypes()
     * - .ifInterContainerTrafficEncryption()
     * - .ifMaxRuntimeInSeconds()
     * - .ifNetworkIsolation()
     * - .ifOutputKmsKey()
     * - .ifVolumeKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * Dependent actions:
     * - iam:PassRole
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateModelQualityJobDefinition.html
     */
    toCreateModelQualityJobDefinition() {
        return this.to('CreateModelQualityJobDefinition');
    }
    /**
     * Grants permission to create a monitoring schedule
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceTypes()
     * - .ifInterContainerTrafficEncryption()
     * - .ifMaxRuntimeInSeconds()
     * - .ifNetworkIsolation()
     * - .ifOutputKmsKey()
     * - .ifVolumeKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * Dependent actions:
     * - iam:PassRole
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateMonitoringSchedule.html
     */
    toCreateMonitoringSchedule() {
        return this.to('CreateMonitoringSchedule');
    }
    /**
     * Grants permission to create an Amazon SageMaker notebook instance. A notebook instance is an Amazon EC2 instance running on a Jupyter Notebook
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAcceleratorTypes()
     * - .ifDirectInternetAccess()
     * - .ifInstanceTypes()
     * - .ifMinimumInstanceMetadataServiceVersion()
     * - .ifRootAccess()
     * - .ifVolumeKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * Dependent actions:
     * - iam:PassRole
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateNotebookInstance.html
     */
    toCreateNotebookInstance() {
        return this.to('CreateNotebookInstance');
    }
    /**
     * Grants permission to create a notebook instance lifecycle configuration that can be deployed using Amazon SageMaker
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateNotebookInstanceLifecycleConfig.html
     */
    toCreateNotebookInstanceLifecycleConfig() {
        return this.to('CreateNotebookInstanceLifecycleConfig');
    }
    /**
     * Grants permission to create an optimization job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateOptimizationJob.html
     */
    toCreateOptimizationJob() {
        return this.to('CreateOptimizationJob');
    }
    /**
     * Grants permission to create an Amazon SageMaker Partner AI App
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreatePartnerApp.html
     */
    toCreatePartnerApp() {
        return this.to('CreatePartnerApp');
    }
    /**
     * Grants permission to return a URL that you can use from your browser to connect to the Amazon SageMaker Partner AI App
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreatePartnerAppPresignedUrl.html
     */
    toCreatePartnerAppPresignedUrl() {
        return this.to('CreatePartnerAppPresignedUrl');
    }
    /**
     * Grants permission to create a pipeline
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreatePipeline.html
     */
    toCreatePipeline() {
        return this.to('CreatePipeline');
    }
    /**
     * Grants permission to return a URL that you can use from your browser to connect to the Domain as a specified UserProfile when AuthMode is 'IAM'
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreatePresignedDomainUrl.html
     */
    toCreatePresignedDomainUrl() {
        return this.to('CreatePresignedDomainUrl');
    }
    /**
     * Grants permission to return a URL that you can use from your browser to connect to the MLflow tracking server
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreatePresignedMlflowTrackingServerUrl.html
     */
    toCreatePresignedMlflowTrackingServerUrl() {
        return this.to('CreatePresignedMlflowTrackingServerUrl');
    }
    /**
     * Grants permission to create a URL that you can use from your browser to connect to the Notebook Instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreatePresignedNotebookInstanceUrl.html
     */
    toCreatePresignedNotebookInstanceUrl() {
        return this.to('CreatePresignedNotebookInstanceUrl');
    }
    /**
     * Grants permission to start a processing job. After processing completes, Amazon SageMaker saves the resulting artifacts and other optional output to an Amazon S3 location that you specify
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceTypes()
     * - .ifMaxRuntimeInSeconds()
     * - .ifNetworkIsolation()
     * - .ifOutputKmsKey()
     * - .ifVolumeKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     * - .ifInterContainerTrafficEncryption()
     *
     * Dependent actions:
     * - iam:PassRole
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateProcessingJob.html
     */
    toCreateProcessingJob() {
        return this.to('CreateProcessingJob');
    }
    /**
     * Grants permission to create a Project
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateProject.html
     */
    toCreateProject() {
        return this.to('CreateProject');
    }
    /**
     * Grants permission to create a reserved capacity
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateReservedCapacity.html
     */
    toCreateReservedCapacity() {
        return this.to('CreateReservedCapacity');
    }
    /**
     * Grants permission to create a shared model in a SageMaker Studio application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/canvas-collaborate-permissions.html
     */
    toCreateSharedModel() {
        return this.to('CreateSharedModel');
    }
    /**
     * Grants permission to create a Space for a SageMaker Domain
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceTypes()
     * - .ifImageArns()
     * - .ifImageVersionArns()
     * - .ifOwnerUserProfileArn()
     * - .ifRemoteAccess()
     * - .ifSpaceSharingType()
     *
     * Dependent actions:
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateSpace.html
     */
    toCreateSpace() {
        return this.to('CreateSpace');
    }
    /**
     * Grants permission to create a Studio Lifecycle Configuration that can be deployed using Amazon SageMaker
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateStudioLifecycleConfig.html
     */
    toCreateStudioLifecycleConfig() {
        return this.to('CreateStudioLifecycleConfig');
    }
    /**
     * Grants permission to start a model training job. After training completes, Amazon SageMaker saves the resulting model artifacts and other optional output to an Amazon S3 location that you specify
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifFileSystemAccessMode()
     * - .ifFileSystemDirectoryPath()
     * - .ifFileSystemId()
     * - .ifFileSystemType()
     * - .ifInstanceTypes()
     * - .ifInterContainerTrafficEncryption()
     * - .ifMaxRuntimeInSeconds()
     * - .ifNetworkIsolation()
     * - .ifOutputKmsKey()
     * - .ifVolumeKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     * - .ifKeepAlivePeriod()
     * - .ifEnableRemoteDebug()
     * - .ifDirectGatedModelAccess()
     *
     * Dependent actions:
     * - iam:PassRole
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateTrainingJob.html
     */
    toCreateTrainingJob() {
        return this.to('CreateTrainingJob');
    }
    /**
     * Grants permission to create a training plan that allocates resources for scheduling workloads within a specified time range
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - sagemaker:AddTags
     * - sagemaker:CreateReservedCapacity
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateTrainingPlan.html
     */
    toCreateTrainingPlan() {
        return this.to('CreateTrainingPlan');
    }
    /**
     * Grants permission to start a transform job. After the results are obtained, Amazon SageMaker saves them to an Amazon S3 location that you specify
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceTypes()
     * - .ifModelArn()
     * - .ifOutputKmsKey()
     * - .ifVolumeKmsKey()
     *
     * Dependent actions:
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateTransformJob.html
     */
    toCreateTransformJob() {
        return this.to('CreateTransformJob');
    }
    /**
     * Grants permission to create a trial
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateTrial.html
     */
    toCreateTrial() {
        return this.to('CreateTrial');
    }
    /**
     * Grants permission to create a trial component
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateTrialComponent.html
     */
    toCreateTrialComponent() {
        return this.to('CreateTrialComponent');
    }
    /**
     * Grants permission to create a UserProfile for a SageMaker Domain
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifVpcSecurityGroupIds()
     * - .ifInstanceTypes()
     * - .ifDomainSharingOutputKmsKey()
     * - .ifImageArns()
     * - .ifImageVersionArns()
     *
     * Dependent actions:
     * - iam:PassRole
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateUserProfile.html
     */
    toCreateUserProfile() {
        return this.to('CreateUserProfile');
    }
    /**
     * Grants permission to create a workforce
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateWorkforce.html
     */
    toCreateWorkforce() {
        return this.to('CreateWorkforce');
    }
    /**
     * Grants permission to create a workteam
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateWorkteam.html
     */
    toCreateWorkteam() {
        return this.to('CreateWorkteam');
    }
    /**
     * Grants permission to delete an action
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteAction.html
     */
    toDeleteAction() {
        return this.to('DeleteAction');
    }
    /**
     * Grants permission to delete an algorithm
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteAlgorithm.html
     */
    toDeleteAlgorithm() {
        return this.to('DeleteAlgorithm');
    }
    /**
     * Grants permission to delete an App
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifOwnerUserProfileArn()
     * - .ifSpaceSharingType()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteApp.html
     */
    toDeleteApp() {
        return this.to('DeleteApp');
    }
    /**
     * Grants permission to delete an AppImageConfig
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteAppImageConfig.html
     */
    toDeleteAppImageConfig() {
        return this.to('DeleteAppImageConfig');
    }
    /**
     * Grants permission to delete an artifact
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteArtifact.html
     */
    toDeleteArtifact() {
        return this.to('DeleteArtifact');
    }
    /**
     * Grants permission to delete the association from a lineage entity (artifact, context, action, experiment, experiment-trial-component) to another
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteAssociation.html
     */
    toDeleteAssociation() {
        return this.to('DeleteAssociation');
    }
    /**
     * Grants permission to delete a SageMaker HyperPod cluster
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - eks:DeleteAccessEntry
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteCluster.html
     */
    toDeleteCluster() {
        return this.to('DeleteCluster');
    }
    /**
     * Grants permission to delete a cluster scheduler config
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteClusterSchedulerConfig.html
     */
    toDeleteClusterSchedulerConfig() {
        return this.to('DeleteClusterSchedulerConfig');
    }
    /**
     * Grants permission to delete a CodeRepository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteCodeRepository.html
     */
    toDeleteCodeRepository() {
        return this.to('DeleteCodeRepository');
    }
    /**
     * Grants permission to delete a compilation job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteCompilationJob.html
     */
    toDeleteCompilationJob() {
        return this.to('DeleteCompilationJob');
    }
    /**
     * Grants permission to delete a compute quota
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteComputeQuota.html
     */
    toDeleteComputeQuota() {
        return this.to('DeleteComputeQuota');
    }
    /**
     * Grants permission to delete a context
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteContext.html
     */
    toDeleteContext() {
        return this.to('DeleteContext');
    }
    /**
     * Grants permission to delete the data quality job definition created using the CreateDataQualityJobDefinition API
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteDataQualityJobDefinition.html
     */
    toDeleteDataQualityJobDefinition() {
        return this.to('DeleteDataQualityJobDefinition');
    }
    /**
     * Grants permission to delete a device fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteDeviceFleet.html
     */
    toDeleteDeviceFleet() {
        return this.to('DeleteDeviceFleet');
    }
    /**
     * Grants permission to delete a Domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteDomain.html
     */
    toDeleteDomain() {
        return this.to('DeleteDomain');
    }
    /**
     * Grants permission to delete an edge deployment plan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteEdgeDeploymentPlan.html
     */
    toDeleteEdgeDeploymentPlan() {
        return this.to('DeleteEdgeDeploymentPlan');
    }
    /**
     * Grants permission to delete an edge deployment stage
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteEdgeDeploymentStage.html
     */
    toDeleteEdgeDeploymentStage() {
        return this.to('DeleteEdgeDeploymentStage');
    }
    /**
     * Grants permission to delete an endpoint. Amazon SageMaker frees up all the resources that were deployed when the endpoint was created
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteEndpoint.html
     */
    toDeleteEndpoint() {
        return this.to('DeleteEndpoint');
    }
    /**
     * Grants permission to delete the endpoint configuration created using the CreateEndpointConfig API. The DeleteEndpointConfig API deletes only the specified configuration. It does not delete any endpoints created using the configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteEndpointConfig.html
     */
    toDeleteEndpointConfig() {
        return this.to('DeleteEndpointConfig');
    }
    /**
     * Grants permission to delete an experiment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteExperiment.html
     */
    toDeleteExperiment() {
        return this.to('DeleteExperiment');
    }
    /**
     * Grants permission to delete a feature group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteFeatureGroup.html
     */
    toDeleteFeatureGroup() {
        return this.to('DeleteFeatureGroup');
    }
    /**
     * Grants permission to delete the specified flow definition
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteFlowDefinition.html
     */
    toDeleteFlowDefinition() {
        return this.to('DeleteFlowDefinition');
    }
    /**
     * Grants permission to delete hubs
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteHub.html
     */
    toDeleteHub() {
        return this.to('DeleteHub');
    }
    /**
     * Grants permission to delete hub content
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteHubContent.html
     */
    toDeleteHubContent() {
        return this.to('DeleteHubContent');
    }
    /**
     * Grants permission to delete hub content reference
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteHubContentReference.html
     */
    toDeleteHubContentReference() {
        return this.to('DeleteHubContentReference');
    }
    /**
     * Grants permission to delete a specified human loop
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteHumanLoop.html
     */
    toDeleteHumanLoop() {
        return this.to('DeleteHumanLoop');
    }
    /**
     * Grants permission to delete the specified human task user interface (worker task template)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteHumanTaskUi.html
     */
    toDeleteHumanTaskUi() {
        return this.to('DeleteHumanTaskUi');
    }
    /**
     * Grants permission to delete a hyper parameter tuning job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteHyperParameterTuningJob.html
     */
    toDeleteHyperParameterTuningJob() {
        return this.to('DeleteHyperParameterTuningJob');
    }
    /**
     * Grants permission to delete a SageMaker Image
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteImage.html
     */
    toDeleteImage() {
        return this.to('DeleteImage');
    }
    /**
     * Grants permission to delete a SageMaker ImageVersion
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteImageVersion.html
     */
    toDeleteImageVersion() {
        return this.to('DeleteImageVersion');
    }
    /**
     * Grants permission to delete an inference component. Amazon SageMaker frees up the resources that were reserved when the inference component was created
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteInferenceComponent.html
     */
    toDeleteInferenceComponent() {
        return this.to('DeleteInferenceComponent');
    }
    /**
     * Grants permission to delete an inference experiment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteInferenceExperiment.html
     */
    toDeleteInferenceExperiment() {
        return this.to('DeleteInferenceExperiment');
    }
    /**
     * Grants permission to delete a lineage group policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/Welcome.html
     */
    toDeleteLineageGroupPolicy() {
        return this.to('DeleteLineageGroupPolicy');
    }
    /**
     * Grants permission to delete an MLflow tracking server
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteMlflowTrackingServer.html
     */
    toDeleteMlflowTrackingServer() {
        return this.to('DeleteMlflowTrackingServer');
    }
    /**
     * Grants permission to delete a model created using the CreateModel API. The DeleteModel API deletes only the model entry in Amazon SageMaker that you created by calling the CreateModel API. It does not delete model artifacts, inference code, or the IAM role that you specified when creating the model
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteModel.html
     */
    toDeleteModel() {
        return this.to('DeleteModel');
    }
    /**
     * Grants permission to delete the model bias job definition created using the CreateModelBiasJobDefinition API
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteModelBiasJobDefinition.html
     */
    toDeleteModelBiasJobDefinition() {
        return this.to('DeleteModelBiasJobDefinition');
    }
    /**
     * Grants permission to delete a model card
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteModelCard.html
     */
    toDeleteModelCard() {
        return this.to('DeleteModelCard');
    }
    /**
     * Grants permission to delete the model explainability job definition created using the CreateModelExplainabilityJobDefinition API
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteModelExplainabilityJobDefinition.html
     */
    toDeleteModelExplainabilityJobDefinition() {
        return this.to('DeleteModelExplainabilityJobDefinition');
    }
    /**
     * Grants permission to delete a ModelPackage
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteModelPackage.html
     */
    toDeleteModelPackage() {
        return this.to('DeleteModelPackage');
    }
    /**
     * Grants permission to delete a ModelPackageGroup
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteModelPackageGroup.html
     */
    toDeleteModelPackageGroup() {
        return this.to('DeleteModelPackageGroup');
    }
    /**
     * Grants permission to delete a ModelPackageGroup policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteModelPackageGroupPolicy.html
     */
    toDeleteModelPackageGroupPolicy() {
        return this.to('DeleteModelPackageGroupPolicy');
    }
    /**
     * Grants permission to delete the model quality job definition created using the CreateModelQualityJobDefinition API
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteModelQualityJobDefinition.html
     */
    toDeleteModelQualityJobDefinition() {
        return this.to('DeleteModelQualityJobDefinition');
    }
    /**
     * Grants permission to delete a monitoring schedule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteMonitoringSchedule.html
     */
    toDeleteMonitoringSchedule() {
        return this.to('DeleteMonitoringSchedule');
    }
    /**
     * Grants permission to delete a Amazon SageMaker notebook instance. Before you can delete a notebook instance, you must call the StopNotebookInstance API
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteNotebookInstance.html
     */
    toDeleteNotebookInstance() {
        return this.to('DeleteNotebookInstance');
    }
    /**
     * Grants permission to delete a notebook instance lifecycle configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteNotebookInstanceLifecycleConfig.html
     */
    toDeleteNotebookInstanceLifecycleConfig() {
        return this.to('DeleteNotebookInstanceLifecycleConfig');
    }
    /**
     * Grants permission to delete an optimization job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteOptimizationJob.html
     */
    toDeleteOptimizationJob() {
        return this.to('DeleteOptimizationJob');
    }
    /**
     * Grants permission to delete an Amazon SageMaker Partner AI App
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeletePartnerApp.html
     */
    toDeletePartnerApp() {
        return this.to('DeletePartnerApp');
    }
    /**
     * Grants permission to delete a pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeletePipeline.html
     */
    toDeletePipeline() {
        return this.to('DeletePipeline');
    }
    /**
     * Grants permission to delete a project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteProject.html
     */
    toDeleteProject() {
        return this.to('DeleteProject');
    }
    /**
     * Grants permission to delete a record from a feature group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_feature_store_DeleteRecord.html
     */
    toDeleteRecord() {
        return this.to('DeleteRecord');
    }
    /**
     * Grants AWS Resource Access Manager permission to delete a resource policy on a SageMaker resource that supports cross-account sharing
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteResourcePolicy.html
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to delete a Space
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifOwnerUserProfileArn()
     * - .ifSpaceSharingType()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteSpace.html
     */
    toDeleteSpace() {
        return this.to('DeleteSpace');
    }
    /**
     * Grants permission to delete a Studio Lifecycle Configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteStudioLifecycleConfig.html
     */
    toDeleteStudioLifecycleConfig() {
        return this.to('DeleteStudioLifecycleConfig');
    }
    /**
     * Grants permission to delete the specified set of tags from an Amazon SageMaker resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteTags.html
     */
    toDeleteTags() {
        return this.to('DeleteTags');
    }
    /**
     * Grants permission to delete a trial
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteTrial.html
     */
    toDeleteTrial() {
        return this.to('DeleteTrial');
    }
    /**
     * Grants permission to delete a trial component
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteTrialComponent.html
     */
    toDeleteTrialComponent() {
        return this.to('DeleteTrialComponent');
    }
    /**
     * Grants permission to delete a UserProfile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteUserProfile.html
     */
    toDeleteUserProfile() {
        return this.to('DeleteUserProfile');
    }
    /**
     * Grants permission to delete a workforce
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteWorkforce.html
     */
    toDeleteWorkforce() {
        return this.to('DeleteWorkforce');
    }
    /**
     * Grants permission to delete a workteam
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeleteWorkteam.html
     */
    toDeleteWorkteam() {
        return this.to('DeleteWorkteam');
    }
    /**
     * Grants permission to deploy a model in hub to an endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/jumpstart-curated-hubs-admin-guide.html
     */
    toDeployHubModel() {
        return this.to('DeployHubModel');
    }
    /**
     * Grants permission to deregister a set of devices
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DeregisterDevices.html
     */
    toDeregisterDevices() {
        return this.to('DeregisterDevices');
    }
    /**
     * Grants permission to get information about an action
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeAction.html
     */
    toDescribeAction() {
        return this.to('DescribeAction');
    }
    /**
     * Grants permission to describe an algorithm
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeAlgorithm.html
     */
    toDescribeAlgorithm() {
        return this.to('DescribeAlgorithm');
    }
    /**
     * Grants permission to describe an App
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeApp.html
     */
    toDescribeApp() {
        return this.to('DescribeApp');
    }
    /**
     * Grants permission to describe an AppImageConfig
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeAppImageConfig.html
     */
    toDescribeAppImageConfig() {
        return this.to('DescribeAppImageConfig');
    }
    /**
     * Grants permission to get information about an artifact
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeArtifact.html
     */
    toDescribeArtifact() {
        return this.to('DescribeArtifact');
    }
    /**
     * Grants permission to describe an AutoML job that was created via the CreateAutoMLJob API
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeAutoMLJob.html
     */
    toDescribeAutoMLJob() {
        return this.to('DescribeAutoMLJob');
    }
    /**
     * Grants permission to describe an AutoML job that was created via the CreateAutoMLJobV2 API
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeAutoMLJobV2.html
     */
    toDescribeAutoMLJobV2() {
        return this.to('DescribeAutoMLJobV2');
    }
    /**
     * Grants permission to return information about a SageMaker HyperPod cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeCluster.html
     */
    toDescribeCluster() {
        return this.to('DescribeCluster');
    }
    /**
     * Grants permission to return information about an Event within a SageMaker HyperPod cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeClusterEvent.html
     */
    toDescribeClusterEvent() {
        return this.to('DescribeClusterEvent');
    }
    /**
     * Grants permission to get information about the inference operator for a SageMaker HyperPod cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/sagemaker-hyperpod-inference.html
     */
    toDescribeClusterInference() {
        return this.to('DescribeClusterInference');
    }
    /**
     * Grants permission to return information about a SageMaker HyperPod cluster node
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeClusterNode.html
     */
    toDescribeClusterNode() {
        return this.to('DescribeClusterNode');
    }
    /**
     * Grants permission to get information about a cluster scheduler config
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeClusterSchedulerConfig.html
     */
    toDescribeClusterSchedulerConfig() {
        return this.to('DescribeClusterSchedulerConfig');
    }
    /**
     * Grants permission to describe a CodeRepository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeCodeRepository.html
     */
    toDescribeCodeRepository() {
        return this.to('DescribeCodeRepository');
    }
    /**
     * Grants permission to return information about a compilation job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeCompilationJob.html
     */
    toDescribeCompilationJob() {
        return this.to('DescribeCompilationJob');
    }
    /**
     * Grants permission to get information about a compute quota
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeComputeQuota.html
     */
    toDescribeComputeQuota() {
        return this.to('DescribeComputeQuota');
    }
    /**
     * Grants permission to get information about a context
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeContext.html
     */
    toDescribeContext() {
        return this.to('DescribeContext');
    }
    /**
     * Grants permission to return information about a data quality job definition
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeDataQualityJobDefinition.html
     */
    toDescribeDataQualityJobDefinition() {
        return this.to('DescribeDataQualityJobDefinition');
    }
    /**
     * Grants permission to access information about a device
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeDevice.html
     */
    toDescribeDevice() {
        return this.to('DescribeDevice');
    }
    /**
     * Grants permission to access information about a device fleet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeDeviceFleet.html
     */
    toDescribeDeviceFleet() {
        return this.to('DescribeDeviceFleet');
    }
    /**
     * Grants permission to describe a Domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeDomain.html
     */
    toDescribeDomain() {
        return this.to('DescribeDomain');
    }
    /**
     * Grants permission to access information about an edge deployment plan
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeEdgeDeploymentPlan.html
     */
    toDescribeEdgeDeploymentPlan() {
        return this.to('DescribeEdgeDeploymentPlan');
    }
    /**
     * Grants permission to access information about an edge packaging job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeEdgePackagingJob.html
     */
    toDescribeEdgePackagingJob() {
        return this.to('DescribeEdgePackagingJob');
    }
    /**
     * Grants permission to return the description of an endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeEndpoint.html
     */
    toDescribeEndpoint() {
        return this.to('DescribeEndpoint');
    }
    /**
     * Grants permission to return the description of an endpoint configuration, which was created using the CreateEndpointConfig API
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeEndpointConfig.html
     */
    toDescribeEndpointConfig() {
        return this.to('DescribeEndpointConfig');
    }
    /**
     * Grants permission to return information about an experiment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeExperiment.html
     */
    toDescribeExperiment() {
        return this.to('DescribeExperiment');
    }
    /**
     * Grants permission to return information about a feature group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeFeatureGroup.html
     */
    toDescribeFeatureGroup() {
        return this.to('DescribeFeatureGroup');
    }
    /**
     * Grants permission to return information about a feature metadata
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeFeatureMetadata.html
     */
    toDescribeFeatureMetadata() {
        return this.to('DescribeFeatureMetadata');
    }
    /**
     * Grants permission to return information about the specified flow definition
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeFlowDefinition.html
     */
    toDescribeFlowDefinition() {
        return this.to('DescribeFlowDefinition');
    }
    /**
     * Grants permission to describe hubs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeHub.html
     */
    toDescribeHub() {
        return this.to('DescribeHub');
    }
    /**
     * Grants permission to describe hub content
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeHubContent.html
     */
    toDescribeHubContent() {
        return this.to('DescribeHubContent');
    }
    /**
     * Grants permission to return information about the specified human loop
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeHumanLoop.html
     */
    toDescribeHumanLoop() {
        return this.to('DescribeHumanLoop');
    }
    /**
     * Grants permission to return detailed information about the specified human review workflow user interface
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeHumanTaskUi.html
     */
    toDescribeHumanTaskUi() {
        return this.to('DescribeHumanTaskUi');
    }
    /**
     * Grants permission to describe a hyper parameter tuning job that was created via the CreateHyperParameterTuningJob API
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeHyperParameterTuningJob.html
     */
    toDescribeHyperParameterTuningJob() {
        return this.to('DescribeHyperParameterTuningJob');
    }
    /**
     * Grants permission to return information about a SageMaker Image
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeImage.html
     */
    toDescribeImage() {
        return this.to('DescribeImage');
    }
    /**
     * Grants permission to return information about a SageMaker ImageVersion
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeImageVersion.html
     */
    toDescribeImageVersion() {
        return this.to('DescribeImageVersion');
    }
    /**
     * Grants permission to return the description of an inference component
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeInferenceComponent.html
     */
    toDescribeInferenceComponent() {
        return this.to('DescribeInferenceComponent');
    }
    /**
     * Grants permission to get information about an inference experiment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeInferenceExperiment.html
     */
    toDescribeInferenceExperiment() {
        return this.to('DescribeInferenceExperiment');
    }
    /**
     * Grants permission to get information about an inference recommendations job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeInferenceRecommendationsJob.html
     */
    toDescribeInferenceRecommendationsJob() {
        return this.to('DescribeInferenceRecommendationsJob');
    }
    /**
     * Grants permission to return information about a labeling job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeLabelingJob.html
     */
    toDescribeLabelingJob() {
        return this.to('DescribeLabelingJob');
    }
    /**
     * Grants permission to describe a lineage group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeLineageGroup.html
     */
    toDescribeLineageGroup() {
        return this.to('DescribeLineageGroup');
    }
    /**
     * Grants permission to get information about an MLflow tracking server
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeMlflowTrackingServer.html
     */
    toDescribeMlflowTrackingServer() {
        return this.to('DescribeMlflowTrackingServer');
    }
    /**
     * Grants permission to describe a model that you created using the CreateModel API
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeModel.html
     */
    toDescribeModel() {
        return this.to('DescribeModel');
    }
    /**
     * Grants permission to return information about a model bias job definition
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeModelBiasJobDefinition.html
     */
    toDescribeModelBiasJobDefinition() {
        return this.to('DescribeModelBiasJobDefinition');
    }
    /**
     * Grants permission to get information about a model card
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeModelCard.html
     */
    toDescribeModelCard() {
        return this.to('DescribeModelCard');
    }
    /**
     * Grants permission to get information about a model card export job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeModelCardExportJob.html
     */
    toDescribeModelCardExportJob() {
        return this.to('DescribeModelCardExportJob');
    }
    /**
     * Grants permission to return information about a model explainability job definition
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeModelExplainabilityJobDefinition.html
     */
    toDescribeModelExplainabilityJobDefinition() {
        return this.to('DescribeModelExplainabilityJobDefinition');
    }
    /**
     * Grants permission to describe a ModelPackage
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeModelPackage.html
     */
    toDescribeModelPackage() {
        return this.to('DescribeModelPackage');
    }
    /**
     * Grants permission to describe a ModelPackageGroup
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeModelPackageGroup.html
     */
    toDescribeModelPackageGroup() {
        return this.to('DescribeModelPackageGroup');
    }
    /**
     * Grants permission to return information about a model quality job definition
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeModelQualityJobDefinition.html
     */
    toDescribeModelQualityJobDefinition() {
        return this.to('DescribeModelQualityJobDefinition');
    }
    /**
     * Grants permission to return information about a monitoring schedule
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeMonitoringSchedule.html
     */
    toDescribeMonitoringSchedule() {
        return this.to('DescribeMonitoringSchedule');
    }
    /**
     * Grants permission to return information about a notebook instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeNotebookInstance.html
     */
    toDescribeNotebookInstance() {
        return this.to('DescribeNotebookInstance');
    }
    /**
     * Grants permission to describe a notebook instance lifecycle configuration that was created via the CreateNotebookInstanceLifecycleConfig API
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeNotebookInstanceLifecycleConfig.html
     */
    toDescribeNotebookInstanceLifecycleConfig() {
        return this.to('DescribeNotebookInstanceLifecycleConfig');
    }
    /**
     * Grants permission to return information about an optimization job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeOptimizationJob.html
     */
    toDescribeOptimizationJob() {
        return this.to('DescribeOptimizationJob');
    }
    /**
     * Grants permission to describe an Amazon SageMaker Partner AI App
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribePartnerApp.html
     */
    toDescribePartnerApp() {
        return this.to('DescribePartnerApp');
    }
    /**
     * Grants permission to get information about a pipeline
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifPipelineVersionId()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribePipeline.html
     */
    toDescribePipeline() {
        return this.to('DescribePipeline');
    }
    /**
     * Grants permission to get the pipeline definition for a pipeline execution
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribePipelineDefinitionForExecution.html
     */
    toDescribePipelineDefinitionForExecution() {
        return this.to('DescribePipelineDefinitionForExecution');
    }
    /**
     * Grants permission to get information about a pipeline execution
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribePipelineExecution.html
     */
    toDescribePipelineExecution() {
        return this.to('DescribePipelineExecution');
    }
    /**
     * Grants permission to return information about a processing job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeProcessingJob.html
     */
    toDescribeProcessingJob() {
        return this.to('DescribeProcessingJob');
    }
    /**
     * Grants permission to describe a project
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeProject.html
     */
    toDescribeProject() {
        return this.to('DescribeProject');
    }
    /**
     * Grants permission to return information about a specified Reserved Capacity
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeReservedCapacity.html
     */
    toDescribeReservedCapacity() {
        return this.to('DescribeReservedCapacity');
    }
    /**
     * Grants permission to describe a shared model in a SageMaker Studio application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/canvas-collaborate-permissions.html
     */
    toDescribeSharedModel() {
        return this.to('DescribeSharedModel');
    }
    /**
     * Grants permission to describe a Space
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeSpace.html
     */
    toDescribeSpace() {
        return this.to('DescribeSpace');
    }
    /**
     * Grants permission to describe a Studio Lifecycle Configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeStudioLifecycleConfig.html
     */
    toDescribeStudioLifecycleConfig() {
        return this.to('DescribeStudioLifecycleConfig');
    }
    /**
     * Grants permission to return information about a subscribed workteam
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeSubscribedWorkteam.html
     */
    toDescribeSubscribedWorkteam() {
        return this.to('DescribeSubscribedWorkteam');
    }
    /**
     * Grants permission to return information about a training job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeTrainingJob.html
     */
    toDescribeTrainingJob() {
        return this.to('DescribeTrainingJob');
    }
    /**
     * Grants permission to return information about a specified training plan
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeTrainingPlan.html
     */
    toDescribeTrainingPlan() {
        return this.to('DescribeTrainingPlan');
    }
    /**
     * Grants permission to return information about a transform job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeTransformJob.html
     */
    toDescribeTransformJob() {
        return this.to('DescribeTransformJob');
    }
    /**
     * Grants permission to return information about a trial
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeTrial.html
     */
    toDescribeTrial() {
        return this.to('DescribeTrial');
    }
    /**
     * Grants permission to return information about a trial component
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeTrialComponent.html
     */
    toDescribeTrialComponent() {
        return this.to('DescribeTrialComponent');
    }
    /**
     * Grants permission to describe a UserProfile
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeUserProfile.html
     */
    toDescribeUserProfile() {
        return this.to('DescribeUserProfile');
    }
    /**
     * Grants permission to return information about a workforce
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeWorkforce.html
     */
    toDescribeWorkforce() {
        return this.to('DescribeWorkforce');
    }
    /**
     * Grants permission to return information about a workteam
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DescribeWorkteam.html
     */
    toDescribeWorkteam() {
        return this.to('DescribeWorkteam');
    }
    /**
     * Grants permission to detach an Amazon EBS volume from a SageMaker HyperPod cluster node
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:DescribeVolumes
     * - ec2:DetachVolume
     * - eks:DescribeCluster
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DetachClusterNodeVolume.html
     */
    toDetachClusterNodeVolume() {
        return this.to('DetachClusterNodeVolume');
    }
    /**
     * Grants permission to disable a SageMaker Service Catalog Portfolio
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DisableSagemakerServicecatalogPortfolio.html
     */
    toDisableSagemakerServicecatalogPortfolio() {
        return this.to('DisableSagemakerServicecatalogPortfolio');
    }
    /**
     * Grants permission to disassociate a trial component from a trial
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_DisassociateTrialComponent.html
     */
    toDisassociateTrialComponent() {
        return this.to('DisassociateTrialComponent');
    }
    /**
     * Grants permission to enable a SageMaker Service Catalog Portfolio
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_EnableSagemakerServicecatalogPortfolio.html
     */
    toEnableSagemakerServicecatalogPortfolio() {
        return this.to('EnableSagemakerServicecatalogPortfolio');
    }
    /**
     * Grants permission to get deployment plan for device
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_edge_GetDeployments.html
     */
    toGetDeployments() {
        return this.to('GetDeployments');
    }
    /**
     * Grants permission to access a summary of the devices in a device fleet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_GetDeviceFleetReport.html
     */
    toGetDeviceFleetReport() {
        return this.to('GetDeviceFleetReport');
    }
    /**
     * Grants permission to get device registration. After you deploy a model onto edge devices this api is used to get current device registration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_edge_GetDeviceRegistration.html
     */
    toGetDeviceRegistration() {
        return this.to('GetDeviceRegistration');
    }
    /**
     * Grants permission to retreive a lineage group policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_GetLineageGroupPolicy.html
     */
    toGetLineageGroupPolicy() {
        return this.to('GetLineageGroupPolicy');
    }
    /**
     * Grants permission to get a ModelPackageGroup policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_GetModelPackageGroupPolicy.html
     */
    toGetModelPackageGroupPolicy() {
        return this.to('GetModelPackageGroupPolicy');
    }
    /**
     * Grants permission to get a record from a feature group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_feature_store_GetRecord.html
     */
    toGetRecord() {
        return this.to('GetRecord');
    }
    /**
     * Grants AWS Resource Access Manager permission to retrieve a resource policy on a SageMaker resource that supports cross-account sharing
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_GetResourcePolicy.html
     */
    toGetResourcePolicy() {
        return this.to('GetResourcePolicy');
    }
    /**
     * Grants permission to get a SageMaker Service Catalog Portfolio
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_GetSagemakerServicecatalogPortfolioStatus.html
     */
    toGetSagemakerServicecatalogPortfolioStatus() {
        return this.to('GetSagemakerServicecatalogPortfolioStatus');
    }
    /**
     * Grants permission to get a scaling policy configuration recommendation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_GetScalingConfigurationRecommendation.html
     */
    toGetScalingConfigurationRecommendation() {
        return this.to('GetScalingConfigurationRecommendation');
    }
    /**
     * Grants permission to get search suggestions when provided with a keyword
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_GetSearchSuggestions.html
     */
    toGetSearchSuggestions() {
        return this.to('GetSearchSuggestions');
    }
    /**
     * Grants permission to import hub content
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - sagemaker:AddTags
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ImportHubContent.html
     */
    toImportHubContent() {
        return this.to('ImportHubContent');
    }
    /**
     * Grants permission to invoke an endpoint. After you deploy a model into production using Amazon SageMaker hosting services, your client applications use this API to get inferences from the model hosted at the specified endpoint
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifTargetModel()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_runtime_InvokeEndpoint.html
     */
    toInvokeEndpoint() {
        return this.to('InvokeEndpoint');
    }
    /**
     * Grants permission to get inferences from the hosted model at the specified endpoint in an asynchronous manner
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_runtime_InvokeEndpointAsync.html
     */
    toInvokeEndpointAsync() {
        return this.to('InvokeEndpointAsync');
    }
    /**
     * Grants permission to get the inference response as a stream from the specified endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_runtime_InvokeEndpointWithResponseStream.html
     */
    toInvokeEndpointWithResponseStream() {
        return this.to('InvokeEndpointWithResponseStream');
    }
    /**
     * Grants permission to list actions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListActions.html
     */
    toListActions() {
        return this.to('ListActions');
    }
    /**
     * Grants permission to list Algorithms
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListAlgorithms.html
     */
    toListAlgorithms() {
        return this.to('ListAlgorithms');
    }
    /**
     * Grants permission to list Aliases that belong to a SageMaker Image or Sagemaker ImageVersion
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListAliases.html
     */
    toListAliases() {
        return this.to('ListAliases');
    }
    /**
     * Grants permission to list the AppImageConfigs in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListAppImageConfigs.html
     */
    toListAppImageConfigs() {
        return this.to('ListAppImageConfigs');
    }
    /**
     * Grants permission to list the Apps in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListApps.html
     */
    toListApps() {
        return this.to('ListApps');
    }
    /**
     * Grants permission to list artifacts
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListArtifacts.html
     */
    toListArtifacts() {
        return this.to('ListArtifacts');
    }
    /**
     * Grants permission to list associations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListAssociations.html
     */
    toListAssociations() {
        return this.to('ListAssociations');
    }
    /**
     * Grants permission to list AutoML jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListAutoMLJobs.html
     */
    toListAutoMLJobs() {
        return this.to('ListAutoMLJobs');
    }
    /**
     * Grants permission to lists candidates for an AutoML job
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListCandidatesForAutoMLJob.html
     */
    toListCandidatesForAutoMLJob() {
        return this.to('ListCandidatesForAutoMLJob');
    }
    /**
     * Grants permission to list events within a SageMaker HyperPod cluster
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListClusterEvents.html
     */
    toListClusterEvents() {
        return this.to('ListClusterEvents');
    }
    /**
     * Grants permission to list nodes within a SageMaker HyperPod cluster
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListClusterNodes.html
     */
    toListClusterNodes() {
        return this.to('ListClusterNodes');
    }
    /**
     * Grants permission to list cluster scheduler configs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListClusterSchedulerConfigs.html
     */
    toListClusterSchedulerConfigs() {
        return this.to('ListClusterSchedulerConfigs');
    }
    /**
     * Grants permission to list SageMaker HyperPod clusters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListClusters.html
     */
    toListClusters() {
        return this.to('ListClusters');
    }
    /**
     * Grants permission to list code repositories
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListCodeRepositories.html
     */
    toListCodeRepositories() {
        return this.to('ListCodeRepositories');
    }
    /**
     * Grants permission to list compilation jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListCompilationJobs.html
     */
    toListCompilationJobs() {
        return this.to('ListCompilationJobs');
    }
    /**
     * Grants permission to list compute quotas
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListComputeQuotas.html
     */
    toListComputeQuotas() {
        return this.to('ListComputeQuotas');
    }
    /**
     * Grants permission to list contexts
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListContexts.html
     */
    toListContexts() {
        return this.to('ListContexts');
    }
    /**
     * Grants permission to list data quality job definitions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListDataQualityJobDefinitions.html
     */
    toListDataQualityJobDefinitions() {
        return this.to('ListDataQualityJobDefinitions');
    }
    /**
     * Grants permission to list device fleets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListDeviceFleets.html
     */
    toListDeviceFleets() {
        return this.to('ListDeviceFleets');
    }
    /**
     * Grants permission to list devices
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListDevices.html
     */
    toListDevices() {
        return this.to('ListDevices');
    }
    /**
     * Grants permission to list the Domains in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListDomains.html
     */
    toListDomains() {
        return this.to('ListDomains');
    }
    /**
     * Grants permission to list edge deployment plans
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListEdgeDeploymentPlans.html
     */
    toListEdgeDeploymentPlans() {
        return this.to('ListEdgeDeploymentPlans');
    }
    /**
     * Grants permission to list edge packaging jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListEdgePackagingJobs.html
     */
    toListEdgePackagingJobs() {
        return this.to('ListEdgePackagingJobs');
    }
    /**
     * Grants permission to list endpoint configurations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListEndpointConfigs.html
     */
    toListEndpointConfigs() {
        return this.to('ListEndpointConfigs');
    }
    /**
     * Grants permission to list endpoints
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListEndpoints.html
     */
    toListEndpoints() {
        return this.to('ListEndpoints');
    }
    /**
     * Grants permission to list experiments
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListExperiments.html
     */
    toListExperiments() {
        return this.to('ListExperiments');
    }
    /**
     * Grants permission to list feature groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListFeatureGroups.html
     */
    toListFeatureGroups() {
        return this.to('ListFeatureGroups');
    }
    /**
     * Grants permission to return summary information about flow definitions, given the specified parameters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListFlowDefinitions.html
     */
    toListFlowDefinitions() {
        return this.to('ListFlowDefinitions');
    }
    /**
     * Grants permission to list all versions of hub content
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListHubContentVersions.html
     */
    toListHubContentVersions() {
        return this.to('ListHubContentVersions');
    }
    /**
     * Grants permission to list newest versions of hub content
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListHubContents.html
     */
    toListHubContents() {
        return this.to('ListHubContents');
    }
    /**
     * Grants permission to list hubs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListHubs.html
     */
    toListHubs() {
        return this.to('ListHubs');
    }
    /**
     * Grants permission to return summary information about human loops, given the specified parameters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListHumanLoops.html
     */
    toListHumanLoops() {
        return this.to('ListHumanLoops');
    }
    /**
     * Grants permission to return summary information about human review workflow user interfaces, given the specified parameters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListHumanTaskUis.html
     */
    toListHumanTaskUis() {
        return this.to('ListHumanTaskUis');
    }
    /**
     * Grants permission to list hyper parameter tuning jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListHyperParameterTuningJobs.html
     */
    toListHyperParameterTuningJobs() {
        return this.to('ListHyperParameterTuningJobs');
    }
    /**
     * Grants permission to list ImageVersions that belong to a SageMaker Image
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListImageVersions.html
     */
    toListImageVersions() {
        return this.to('ListImageVersions');
    }
    /**
     * Grants permission to list SageMaker Images in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListImages.html
     */
    toListImages() {
        return this.to('ListImages');
    }
    /**
     * Grants permission to list inference components
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListInferenceComponents.html
     */
    toListInferenceComponents() {
        return this.to('ListInferenceComponents');
    }
    /**
     * Grants permission to list inference experiments
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListInferenceExperiments.html
     */
    toListInferenceExperiments() {
        return this.to('ListInferenceExperiments');
    }
    /**
     * Grants permission to list inference recommendations job steps
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListInferenceRecommendationsJobSteps.html
     */
    toListInferenceRecommendationsJobSteps() {
        return this.to('ListInferenceRecommendationsJobSteps');
    }
    /**
     * Grants permission to list inference recommendations jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListInferenceRecommendationsJobs.html
     */
    toListInferenceRecommendationsJobs() {
        return this.to('ListInferenceRecommendationsJobs');
    }
    /**
     * Grants permission to list labeling jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListLabelingJobs.html
     */
    toListLabelingJobs() {
        return this.to('ListLabelingJobs');
    }
    /**
     * Grants permission to list labeling jobs for workteam
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListLabelingJobs.html
     */
    toListLabelingJobsForWorkteam() {
        return this.to('ListLabelingJobsForWorkteam');
    }
    /**
     * Grants permission to list lineage groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListLineageGroups.html
     */
    toListLineageGroups() {
        return this.to('ListLineageGroups');
    }
    /**
     * Grants permission to list MLflow tracking servers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListMlflowTrackingServers.html
     */
    toListMlflowTrackingServers() {
        return this.to('ListMlflowTrackingServers');
    }
    /**
     * Grants permission to list model bias job definitions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListModelBiasJobDefinitions.html
     */
    toListModelBiasJobDefinitions() {
        return this.to('ListModelBiasJobDefinitions');
    }
    /**
     * Grants permission to list export jobs for a model card
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListModelCardExportJobs.html
     */
    toListModelCardExportJobs() {
        return this.to('ListModelCardExportJobs');
    }
    /**
     * Grants permission to list versions of a model card
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListModelCardVersions.html
     */
    toListModelCardVersions() {
        return this.to('ListModelCardVersions');
    }
    /**
     * Grants permission to list model cards
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListModelCards.html
     */
    toListModelCards() {
        return this.to('ListModelCards');
    }
    /**
     * Grants permission to list model explainability job definitions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListModelExplainabilityJobDefinitions.html
     */
    toListModelExplainabilityJobDefinitions() {
        return this.to('ListModelExplainabilityJobDefinitions');
    }
    /**
     * Grants permission to list model metadata for inference recommendations jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListModelMetadata.html
     */
    toListModelMetadata() {
        return this.to('ListModelMetadata');
    }
    /**
     * Grants permission to list ModelPackageGroups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListModelPackageGroups.html
     */
    toListModelPackageGroups() {
        return this.to('ListModelPackageGroups');
    }
    /**
     * Grants permission to list ModelPackages
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListModelPackages.html
     */
    toListModelPackages() {
        return this.to('ListModelPackages');
    }
    /**
     * Grants permission to list model quality job definitions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListModelQualityJobDefinitions.html
     */
    toListModelQualityJobDefinitions() {
        return this.to('ListModelQualityJobDefinitions');
    }
    /**
     * Grants permission to list the models created with the CreateModel API
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListModels.html
     */
    toListModels() {
        return this.to('ListModels');
    }
    /**
     * Grants permission to list the history of a monitoring alert
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListMonitoringHistory.html
     */
    toListMonitoringAlertHistory() {
        return this.to('ListMonitoringAlertHistory');
    }
    /**
     * Grants permission to list monitoring alerts
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListMonitoringAlerts.html
     */
    toListMonitoringAlerts() {
        return this.to('ListMonitoringAlerts');
    }
    /**
     * Grants permission to list monitoring executions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListMonitoringExecutions.html
     */
    toListMonitoringExecutions() {
        return this.to('ListMonitoringExecutions');
    }
    /**
     * Grants permission to list monitoring schedules
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListMonitoringSchedules.html
     */
    toListMonitoringSchedules() {
        return this.to('ListMonitoringSchedules');
    }
    /**
     * Grants permission to list the notebook instance lifecycle configurations that can be deployed using Amazon SageMaker
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListNotebookInstanceLifecycleConfigs.html
     */
    toListNotebookInstanceLifecycleConfigs() {
        return this.to('ListNotebookInstanceLifecycleConfigs');
    }
    /**
     * Grants permission to list the Amazon SageMaker notebook instances in the requester's account in an AWS Region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListNotebookInstances.html
     */
    toListNotebookInstances() {
        return this.to('ListNotebookInstances');
    }
    /**
     * Grants permission to list optimization jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListOptimizationJobs.html
     */
    toListOptimizationJobs() {
        return this.to('ListOptimizationJobs');
    }
    /**
     * Grants permission to list the Amazon SageMaker Partner AI Apps in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListPartnerApps.html
     */
    toListPartnerApps() {
        return this.to('ListPartnerApps');
    }
    /**
     * Grants permission to list steps for a pipeline execution
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListPipelineExecutionSteps.html
     */
    toListPipelineExecutionSteps() {
        return this.to('ListPipelineExecutionSteps');
    }
    /**
     * Grants permission to list executions for a pipeline
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListPipelineExecutions.html
     */
    toListPipelineExecutions() {
        return this.to('ListPipelineExecutions');
    }
    /**
     * Grants permission to list parameters for a pipeline execution
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListPipelineParametersForExecution.html
     */
    toListPipelineParametersForExecution() {
        return this.to('ListPipelineParametersForExecution');
    }
    /**
     * Grants permission to list versions of a pipeline
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListPipelineVersions.html
     */
    toListPipelineVersions() {
        return this.to('ListPipelineVersions');
    }
    /**
     * Grants permission to list pipelines
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListPipelines.html
     */
    toListPipelines() {
        return this.to('ListPipelines');
    }
    /**
     * Grants permission to list processing jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListProcessingJobs.html
     */
    toListProcessingJobs() {
        return this.to('ListProcessingJobs');
    }
    /**
     * Grants permission to list Projects
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListProjects.html
     */
    toListProjects() {
        return this.to('ListProjects');
    }
    /**
     * Grants permission to list resource catalogs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListResourceCatalogs.html
     */
    toListResourceCatalogs() {
        return this.to('ListResourceCatalogs');
    }
    /**
     * Grants permission to list shared model events
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/canvas-collaborate-permissions.html
     */
    toListSharedModelEvents() {
        return this.to('ListSharedModelEvents');
    }
    /**
     * Grants permission to list shared model versions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/canvas-collaborate-permissions.html
     */
    toListSharedModelVersions() {
        return this.to('ListSharedModelVersions');
    }
    /**
     * Grants permission to list shared models
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/canvas-collaborate-permissions.html
     */
    toListSharedModels() {
        return this.to('ListSharedModels');
    }
    /**
     * Grants permission to list the Spaces in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListSpaces.html
     */
    toListSpaces() {
        return this.to('ListSpaces');
    }
    /**
     * Grants permission to list stage devices
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListStageDevices.html
     */
    toListStageDevices() {
        return this.to('ListStageDevices');
    }
    /**
     * Grants permission to list the Studio Lifecycle Configurations that can be deployed using Amazon SageMaker
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListStudioLifecycleConfigs.html
     */
    toListStudioLifecycleConfigs() {
        return this.to('ListStudioLifecycleConfigs');
    }
    /**
     * Grants permission to list subscribed workteams
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListSubscribedWorkteams.html
     */
    toListSubscribedWorkteams() {
        return this.to('ListSubscribedWorkteams');
    }
    /**
     * Grants permission to list the tag set associated with the specified resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListTags.html
     */
    toListTags() {
        return this.to('ListTags');
    }
    /**
     * Grants permission to list training jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListTrainingJobs.html
     */
    toListTrainingJobs() {
        return this.to('ListTrainingJobs');
    }
    /**
     * Grants permission to list training jobs for a hyper parameter tuning job
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListTrainingJobsForHyperParameterTuningJob.html
     */
    toListTrainingJobsForHyperParameterTuningJob() {
        return this.to('ListTrainingJobsForHyperParameterTuningJob');
    }
    /**
     * Grants permission to list all the training plans that have been created in a specified account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListTrainingPlans.html
     */
    toListTrainingPlans() {
        return this.to('ListTrainingPlans');
    }
    /**
     * Grants permission to list transform jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListTransformJobs.html
     */
    toListTransformJobs() {
        return this.to('ListTransformJobs');
    }
    /**
     * Grants permission to list trial components
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListTrialComponents.html
     */
    toListTrialComponents() {
        return this.to('ListTrialComponents');
    }
    /**
     * Grants permission to list trials
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListTrials.html
     */
    toListTrials() {
        return this.to('ListTrials');
    }
    /**
     * Grants permission to list all UltraServers in a specified Reserved Capacity
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListUltraServersByReservedCapacity.html
     */
    toListUltraServersByReservedCapacity() {
        return this.to('ListUltraServersByReservedCapacity');
    }
    /**
     * Grants permission to list the UserProfiles in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListUserProfiles.html
     */
    toListUserProfiles() {
        return this.to('ListUserProfiles');
    }
    /**
     * Grants permission to list workforces
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListWorkforces.html
     */
    toListWorkforces() {
        return this.to('ListWorkforces');
    }
    /**
     * Grants permission to list workteams
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ListWorkteams.html
     */
    toListWorkteams() {
        return this.to('ListWorkteams');
    }
    /**
     * Grants permission to put a lineage group policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/Welcome.html
     */
    toPutLineageGroupPolicy() {
        return this.to('PutLineageGroupPolicy');
    }
    /**
     * Grants permission to put a ModelPackageGroup policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_PutModelPackageGroupPolicy.html
     */
    toPutModelPackageGroupPolicy() {
        return this.to('PutModelPackageGroupPolicy');
    }
    /**
     * Grants permission to put a record to a feature group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_feature_store_PutRecord.html
     */
    toPutRecord() {
        return this.to('PutRecord');
    }
    /**
     * Grants AWS Resource Access Manager permission to create a resource policy on a SageMaker resource that supports cross-account sharing
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_PutResourcePolicy.html
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to explore the lineage graph
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_QueryLineage.html
     */
    toQueryLineage() {
        return this.to('QueryLineage');
    }
    /**
     * Grants permission to register a set of devices
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_RegisterDevices.html
     */
    toRegisterDevices() {
        return this.to('RegisterDevices');
    }
    /**
     * Grants permission to render a UI template used for a human annotation task
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_RenderUiTemplate.html
     */
    toRenderUiTemplate() {
        return this.to('RenderUiTemplate');
    }
    /**
     * Grants permission to retry a pipeline execution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_RetryPipelineExecution.html
     */
    toRetryPipelineExecution() {
        return this.to('RetryPipelineExecution');
    }
    /**
     * Grants permission to search for SageMaker objects
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifSearchVisibilityCondition()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_Search.html
     */
    toSearch() {
        return this.to('Search');
    }
    /**
     * Grants permissions to search for the available training plan offerings that best match specified capacity requirements
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_SearchTrainingPlanOfferings.html
     */
    toSearchTrainingPlanOfferings() {
        return this.to('SearchTrainingPlanOfferings');
    }
    /**
     * Grants permission to publish heartbeat data from devices. After you deploy a model onto edge devices this api is used to report device status
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_edge_SendHeartbeat.html
     */
    toSendHeartbeat() {
        return this.to('SendHeartbeat');
    }
    /**
     * Grants permission to fail a pending callback step
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_SendPipelineExecutionStepFailure.html
     */
    toSendPipelineExecutionStepFailure() {
        return this.to('SendPipelineExecutionStepFailure');
    }
    /**
     * Grants permission to succeed a pending callback step
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_SendPipelineExecutionStepSuccess.html
     */
    toSendPipelineExecutionStepSuccess() {
        return this.to('SendPipelineExecutionStepSuccess');
    }
    /**
     * Grants permission to send a shared model event
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/canvas-collaborate-permissions.html
     */
    toSendSharedModelEvent() {
        return this.to('SendSharedModelEvent');
    }
    /**
     * Grants permission to start an edge deployment stage
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_StartEdgeDeploymentStage.html
     */
    toStartEdgeDeploymentStage() {
        return this.to('StartEdgeDeploymentStage');
    }
    /**
     * Grants permission to start a human loop
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_StartHumanLoop.html
     */
    toStartHumanLoop() {
        return this.to('StartHumanLoop');
    }
    /**
     * Grants permission to start an inference experiment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_StartInferenceExperiment.html
     */
    toStartInferenceExperiment() {
        return this.to('StartInferenceExperiment');
    }
    /**
     * Grants permission to start an MLfLow tracking server
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_StartMlflowTrackingServer.html
     */
    toStartMlflowTrackingServer() {
        return this.to('StartMlflowTrackingServer');
    }
    /**
     * Grants permission to start a monitoring schedule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_StartMonitoringSchedule.html
     */
    toStartMonitoringSchedule() {
        return this.to('StartMonitoringSchedule');
    }
    /**
     * Grants permission to start a notebook instance. This launches an EC2 instance with the latest version of the libraries and attaches your EBS volume
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_StartNotebookInstance.html
     */
    toStartNotebookInstance() {
        return this.to('StartNotebookInstance');
    }
    /**
     * Grants permission to start a pipeline execution
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPipelineVersionId()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_StartPipelineExecution.html
     */
    toStartPipelineExecution() {
        return this.to('StartPipelineExecution');
    }
    /**
     * Grants permission to start a remote session for a SageMaker space
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_StartSession.html
     */
    toStartSession() {
        return this.to('StartSession');
    }
    /**
     * Grants permission to stop a running AutoML job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_StopAutoMLJob.html
     */
    toStopAutoMLJob() {
        return this.to('StopAutoMLJob');
    }
    /**
     * Grants permission to stop a compilation job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_StopCompilationJob.html
     */
    toStopCompilationJob() {
        return this.to('StopCompilationJob');
    }
    /**
     * Grants permission to stop an edge deployment stage
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_StopEdgeDeploymentStage.html
     */
    toStopEdgeDeploymentStage() {
        return this.to('StopEdgeDeploymentStage');
    }
    /**
     * Grants permission to stop an edge packaging job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_StopEdgePackagingJob.html
     */
    toStopEdgePackagingJob() {
        return this.to('StopEdgePackagingJob');
    }
    /**
     * Grants permission to stop a specified human loop
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_StopHumanLoop.html
     */
    toStopHumanLoop() {
        return this.to('StopHumanLoop');
    }
    /**
     * Grants permission to stop a running hyper parameter tuning job create via the CreateHyperParameterTuningJob
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_StopHyperParameterTuningJob.html
     */
    toStopHyperParameterTuningJob() {
        return this.to('StopHyperParameterTuningJob');
    }
    /**
     * Grants permission to stop an inference experiment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_StopInferenceExperiment.html
     */
    toStopInferenceExperiment() {
        return this.to('StopInferenceExperiment');
    }
    /**
     * Grants permission to stop an inference recommendations job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_StopInferenceRecommendationsJob.html
     */
    toStopInferenceRecommendationsJob() {
        return this.to('StopInferenceRecommendationsJob');
    }
    /**
     * Grants permission to stop a labeling job. Any labels already generated will be exported before stopping
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_StopLabelingJob.html
     */
    toStopLabelingJob() {
        return this.to('StopLabelingJob');
    }
    /**
     * Grants permission to stop an MLflow tracking server
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_StopMlflowTrackingServer.html
     */
    toStopMlflowTrackingServer() {
        return this.to('StopMlflowTrackingServer');
    }
    /**
     * Grants permission to stop a monitoring schedule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_StopMonitoringSchedule.html
     */
    toStopMonitoringSchedule() {
        return this.to('StopMonitoringSchedule');
    }
    /**
     * Grants permission to stop a notebook instance. This terminates the EC2 instance. Before terminating the instance, Amazon SageMaker disconnects the EBS volume from it. Amazon SageMaker preserves the EBS volume
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_StopNotebookInstance.html
     */
    toStopNotebookInstance() {
        return this.to('StopNotebookInstance');
    }
    /**
     * Grants permission to stop an optimization job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_StopOptimizationJob.html
     */
    toStopOptimizationJob() {
        return this.to('StopOptimizationJob');
    }
    /**
     * Grants permission to stop a pipeline execution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_StopPipelineExecution.html
     */
    toStopPipelineExecution() {
        return this.to('StopPipelineExecution');
    }
    /**
     * Grants permission to stop a processing job. To stop a job, Amazon SageMaker sends the algorithm the SIGTERM signal, which delays job termination for 120 seconds
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_StopProcessingJob.html
     */
    toStopProcessingJob() {
        return this.to('StopProcessingJob');
    }
    /**
     * Grants permission to stop a training job. To stop a job, Amazon SageMaker sends the algorithm the SIGTERM signal, which delays job termination for 120 seconds
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_StopTrainingJob.html
     */
    toStopTrainingJob() {
        return this.to('StopTrainingJob');
    }
    /**
     * Grants permission to stop a transform job. When Amazon SageMaker receives a StopTransformJob request, the status of the job changes to Stopping. After Amazon SageMaker stops the job, the status is set to Stopped
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_StopTransformJob.html
     */
    toStopTransformJob() {
        return this.to('StopTransformJob');
    }
    /**
     * Grants permission to train a model in hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/jumpstart-curated-hubs-admin-guide.html
     */
    toTrainHubModel() {
        return this.to('TrainHubModel');
    }
    /**
     * Grants permission to update an action
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateAction.html
     */
    toUpdateAction() {
        return this.to('UpdateAction');
    }
    /**
     * Grants permission to update an AppImageConfig
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateAppImageConfig.html
     */
    toUpdateAppImageConfig() {
        return this.to('UpdateAppImageConfig');
    }
    /**
     * Grants permission to update an artifact
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateArtifact.html
     */
    toUpdateArtifact() {
        return this.to('UpdateArtifact');
    }
    /**
     * Grants permission to update a SageMaker HyperPod cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceTypes()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * Dependent actions:
     * - ec2:DescribeImages
     * - ec2:DescribeSnapshots
     * - ec2:ModifyImageAttribute
     * - ec2:ModifySnapshotAttribute
     * - eks:AssociateAccessPolicy
     * - eks:CreateAccessEntry
     * - eks:DeleteAccessEntry
     * - eks:DescribeAccessEntry
     * - eks:DescribeCluster
     * - iam:PassRole
     * - sagemaker:BatchAddClusterNodes
     * - sagemaker:BatchDeleteClusterNodes
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateCluster.html
     */
    toUpdateCluster() {
        return this.to('UpdateCluster');
    }
    /**
     * Grants permission to update the inference operator for a SageMaker HyperPod cluster
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - eks:AssociateAccessPolicy
     * - eks:DescribeCluster
     * - eks:ListAssociatedAccessPolicies
     * - iam:PassRole
     * - sagemaker:DescribeCluster
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/sagemaker-hyperpod-inference.html
     */
    toUpdateClusterInference() {
        return this.to('UpdateClusterInference');
    }
    /**
     * Grants permission to update a cluster scheduler config
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateClusterSchedulerConfig.html
     */
    toUpdateClusterSchedulerConfig() {
        return this.to('UpdateClusterSchedulerConfig');
    }
    /**
     * Grants permission to update platform software for a SageMaker HyperPod cluster
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:DescribeImages
     * - ec2:DescribeSnapshots
     * - ec2:ModifyImageAttribute
     * - ec2:ModifySnapshotAttribute
     * - eks:DescribeCluster
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateClusterSoftware.html
     */
    toUpdateClusterSoftware() {
        return this.to('UpdateClusterSoftware');
    }
    /**
     * Grants permission to update a CodeRepository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateCodeRepository.html
     */
    toUpdateCodeRepository() {
        return this.to('UpdateCodeRepository');
    }
    /**
     * Grants permission to update a compute quota
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateComputeQuota.html
     */
    toUpdateComputeQuota() {
        return this.to('UpdateComputeQuota');
    }
    /**
     * Grants permission to update a context
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateContext.html
     */
    toUpdateContext() {
        return this.to('UpdateContext');
    }
    /**
     * Grants permission to update a device fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateDeviceFleet.html
     */
    toUpdateDeviceFleet() {
        return this.to('UpdateDeviceFleet');
    }
    /**
     * Grants permission to update a set of devices
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateDevices.html
     */
    toUpdateDevices() {
        return this.to('UpdateDevices');
    }
    /**
     * Grants permission to update a Domain
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifVpcSecurityGroupIds()
     * - .ifInstanceTypes()
     * - .ifDomainSharingOutputKmsKey()
     * - .ifImageArns()
     * - .ifImageVersionArns()
     * - .ifAppNetworkAccessType()
     * - .ifVpcSubnets()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateDomain.html
     */
    toUpdateDomain() {
        return this.to('UpdateDomain');
    }
    /**
     * Grants permission to update an endpoint to use the endpoint configuration specified in the request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateEndpoint.html
     */
    toUpdateEndpoint() {
        return this.to('UpdateEndpoint');
    }
    /**
     * Grants permission to update variant weight, capacity, or both of one or more variants associated with an endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateEndpointWeightsAndCapacities.html
     */
    toUpdateEndpointWeightsAndCapacities() {
        return this.to('UpdateEndpointWeightsAndCapacities');
    }
    /**
     * Grants permission to update an experiment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateExperiment.html
     */
    toUpdateExperiment() {
        return this.to('UpdateExperiment');
    }
    /**
     * Grants permission to update a feature group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateFeatureGroup.html
     */
    toUpdateFeatureGroup() {
        return this.to('UpdateFeatureGroup');
    }
    /**
     * Grants permission to update a feature metadata
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateFeatureMetadata.html
     */
    toUpdateFeatureMetadata() {
        return this.to('UpdateFeatureMetadata');
    }
    /**
     * Grants permission to update hubs
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateHub.html
     */
    toUpdateHub() {
        return this.to('UpdateHub');
    }
    /**
     * Grants permission to update hub content
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateHubContent.html
     */
    toUpdateHubContent() {
        return this.to('UpdateHubContent');
    }
    /**
     * Grants permission to update hub content reference
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateHubContentReference.html
     */
    toUpdateHubContentReference() {
        return this.to('UpdateHubContentReference');
    }
    /**
     * Grants permission to update the properties of a SageMaker Image
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateImage.html
     */
    toUpdateImage() {
        return this.to('UpdateImage');
    }
    /**
     * Grants permission to update the properties of a SageMaker ImageVersion
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateImageVersion.html
     */
    toUpdateImageVersion() {
        return this.to('UpdateImageVersion');
    }
    /**
     * Grants permission to update an inference component to use the specification and configurations specified in the request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateInferenceComponent.html
     */
    toUpdateInferenceComponent() {
        return this.to('UpdateInferenceComponent');
    }
    /**
     * Grants permission to update the runtime config of a given inference component
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateInferenceComponentRuntimeConfig.html
     */
    toUpdateInferenceComponentRuntimeConfig() {
        return this.to('UpdateInferenceComponentRuntimeConfig');
    }
    /**
     * Grants permission to update an inference experiment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateInferenceExperiment.html
     */
    toUpdateInferenceExperiment() {
        return this.to('UpdateInferenceExperiment');
    }
    /**
     * Grants permission to update an MLflow tracking server
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateMlflowTrackingServer.html
     */
    toUpdateMlflowTrackingServer() {
        return this.to('UpdateMlflowTrackingServer');
    }
    /**
     * Grants permission to update a model card
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateModelCard.html
     */
    toUpdateModelCard() {
        return this.to('UpdateModelCard');
    }
    /**
     * Grants permission to update a ModelPackage
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifModelApprovalStatus()
     * - .ifCustomerMetadataProperties()
     * - .ifCustomerMetadataPropertiesToRemove()
     * - .ifModelLifeCycle()
     * - .ifModelLifeCycle()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateModelPackage.html
     */
    toUpdateModelPackage() {
        return this.to('UpdateModelPackage');
    }
    /**
     * Grants permission to update a monitoring alert
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateMonitoringAlert.html
     */
    toUpdateMonitoringAlert() {
        return this.to('UpdateMonitoringAlert');
    }
    /**
     * Grants permission to update a monitoring schedule
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceTypes()
     * - .ifMaxRuntimeInSeconds()
     * - .ifNetworkIsolation()
     * - .ifOutputKmsKey()
     * - .ifVolumeKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     * - .ifInterContainerTrafficEncryption()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateMonitoringSchedule.html
     */
    toUpdateMonitoringSchedule() {
        return this.to('UpdateMonitoringSchedule');
    }
    /**
     * Grants permission to update a notebook instance. Notebook instance updates include upgrading or downgrading the EC2 instance used for your notebook instance to accommodate changes in your workload requirements
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAcceleratorTypes()
     * - .ifInstanceTypes()
     * - .ifMinimumInstanceMetadataServiceVersion()
     * - .ifRootAccess()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateNotebookInstance.html
     */
    toUpdateNotebookInstance() {
        return this.to('UpdateNotebookInstance');
    }
    /**
     * Grants permission to updates a notebook instance lifecycle configuration created with the CreateNotebookInstanceLifecycleConfig API
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateNotebookInstanceLifecycleConfig.html
     */
    toUpdateNotebookInstanceLifecycleConfig() {
        return this.to('UpdateNotebookInstanceLifecycleConfig');
    }
    /**
     * Grants permission to update an Amazon SageMaker Partner AI App
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdatePartnerApp.html
     */
    toUpdatePartnerApp() {
        return this.to('UpdatePartnerApp');
    }
    /**
     * Grants permission to update a pipeline
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdatePipeline.html
     */
    toUpdatePipeline() {
        return this.to('UpdatePipeline');
    }
    /**
     * Grants permission to update a pipeline execution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdatePipelineExecution.html
     */
    toUpdatePipelineExecution() {
        return this.to('UpdatePipelineExecution');
    }
    /**
     * Grants permission to update a pipeline version
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPipelineVersionId()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdatePipelineVersion.html
     */
    toUpdatePipelineVersion() {
        return this.to('UpdatePipelineVersion');
    }
    /**
     * Grants permission to update a Project
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateProject.html
     */
    toUpdateProject() {
        return this.to('UpdateProject');
    }
    /**
     * Grants permission to update a shared model
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/canvas-collaborate-permissions.html
     */
    toUpdateSharedModel() {
        return this.to('UpdateSharedModel');
    }
    /**
     * Grants permission to update a Space
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceTypes()
     * - .ifImageArns()
     * - .ifImageVersionArns()
     * - .ifOwnerUserProfileArn()
     * - .ifRemoteAccess()
     * - .ifSpaceSharingType()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateSpace.html
     */
    toUpdateSpace() {
        return this.to('UpdateSpace');
    }
    /**
     * Grants permission to update a training job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceTypes()
     * - .ifKeepAlivePeriod()
     * - .ifEnableRemoteDebug()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateTrainingJob.html
     */
    toUpdateTrainingJob() {
        return this.to('UpdateTrainingJob');
    }
    /**
     * Grants permission to update a trial
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateTrial.html
     */
    toUpdateTrial() {
        return this.to('UpdateTrial');
    }
    /**
     * Grants permission to update a trial component
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateTrialComponent.html
     */
    toUpdateTrialComponent() {
        return this.to('UpdateTrialComponent');
    }
    /**
     * Grants permission to update a UserProfile
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceTypes()
     * - .ifVpcSecurityGroupIds()
     * - .ifInstanceTypes()
     * - .ifDomainSharingOutputKmsKey()
     * - .ifImageArns()
     * - .ifImageVersionArns()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateUserProfile.html
     */
    toUpdateUserProfile() {
        return this.to('UpdateUserProfile');
    }
    /**
     * Grants permission to update a workforce
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateWorkforce.html
     */
    toUpdateWorkforce() {
        return this.to('UpdateWorkforce');
    }
    /**
     * Grants permission to update a workteam
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateWorkteam.html
     */
    toUpdateWorkteam() {
        return this.to('UpdateWorkteam');
    }
    /**
     * Adds a resource of type device to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/neo-edge-devices.html
     *
     * @param deviceFleetName - Identifier for the deviceFleetName.
     * @param deviceName - Identifier for the deviceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onDevice(deviceFleetName, deviceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:device-fleet/${deviceFleetName}/device/${deviceName}`);
    }
    /**
     * Adds a resource of type device-fleet to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/edge-device-fleet.html
     *
     * @param deviceFleetName - Identifier for the deviceFleetName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onDeviceFleet(deviceFleetName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:device-fleet/${deviceFleetName}`);
    }
    /**
     * Adds a resource of type edge-packaging-job to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/edge-packaging-job.html
     *
     * @param edgePackagingJobName - Identifier for the edgePackagingJobName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onEdgePackagingJob(edgePackagingJobName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:edge-packaging-job/${edgePackagingJobName}`);
    }
    /**
     * Adds a resource of type edge-deployment-plan to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/edge.html
     *
     * @param edgeDeploymentPlanName - Identifier for the edgeDeploymentPlanName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onEdgeDeploymentPlan(edgeDeploymentPlanName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:edge-deployment/${edgeDeploymentPlanName}`);
    }
    /**
     * Adds a resource of type human-loop to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/a2i-start-human-loop.html
     *
     * @param humanLoopName - Identifier for the humanLoopName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onHumanLoop(humanLoopName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:human-loop/${humanLoopName}`);
    }
    /**
     * Adds a resource of type flow-definition to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/a2i-create-flow-definition.html
     *
     * @param flowDefinitionName - Identifier for the flowDefinitionName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onFlowDefinition(flowDefinitionName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:flow-definition/${flowDefinitionName}`);
    }
    /**
     * Adds a resource of type human-task-ui to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/a2i-instructions-overview.html
     *
     * @param humanTaskUiName - Identifier for the humanTaskUiName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onHumanTaskUi(humanTaskUiName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:human-task-ui/${humanTaskUiName}`);
    }
    /**
     * Adds a resource of type hub to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/jumpstart-content-sharing.html
     *
     * @param hubName - Identifier for the hubName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onHub(hubName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:hub/${hubName}`);
    }
    /**
     * Adds a resource of type hub-content to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/jumpstart-content-sharing.html
     *
     * @param hubName - Identifier for the hubName.
     * @param hubContentType - Identifier for the hubContentType.
     * @param hubContentName - Identifier for the hubContentName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onHubContent(hubName, hubContentType, hubContentName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:hub-content/${hubName}/${hubContentType}/${hubContentName}`);
    }
    /**
     * Adds a resource of type inference-recommendations-job to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/inference-recommender-recommendation-jobs.html
     *
     * @param inferenceRecommendationsJobName - Identifier for the inferenceRecommendationsJobName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onInferenceRecommendationsJob(inferenceRecommendationsJobName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:inference-recommendations-job/${inferenceRecommendationsJobName}`);
    }
    /**
     * Adds a resource of type inference-experiment to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/inference-experiment.html
     *
     * @param inferenceExperimentName - Identifier for the inferenceExperimentName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onInferenceExperiment(inferenceExperimentName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:inference-experiment/${inferenceExperimentName}`);
    }
    /**
     * Adds a resource of type labeling-job to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/sms.html
     *
     * @param labelingJobName - Identifier for the labelingJobName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onLabelingJob(labelingJobName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:labeling-job/${labelingJobName}`);
    }
    /**
     * Adds a resource of type workteam to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/sms-workforce-management.html
     *
     * @param workteamName - Identifier for the workteamName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onWorkteam(workteamName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:workteam/${workteamName}`);
    }
    /**
     * Adds a resource of type workforce to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/sms-workforce-management.html
     *
     * @param workforceName - Identifier for the workforceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onWorkforce(workforceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:workforce/${workforceName}`);
    }
    /**
     * Adds a resource of type domain to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/studio-entity-status.html
     *
     * @param domainId - Identifier for the domainId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onDomain(domainId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:domain/${domainId}`);
    }
    /**
     * Adds a resource of type user-profile to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/studio-entity-status.html
     *
     * @param domainId - Identifier for the domainId.
     * @param userProfileName - Identifier for the userProfileName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onUserProfile(domainId, userProfileName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:user-profile/${domainId}/${userProfileName}`);
    }
    /**
     * Adds a resource of type space to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/studio-entity-status.html
     *
     * @param domainId - Identifier for the domainId.
     * @param spaceName - Identifier for the spaceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onSpace(domainId, spaceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:space/${domainId}/${spaceName}`);
    }
    /**
     * Adds a resource of type app to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/studio-entity-status.html
     *
     * @param domainId - Identifier for the domainId.
     * @param userProfileName - Identifier for the userProfileName.
     * @param appType - Identifier for the appType.
     * @param appName - Identifier for the appName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onApp(domainId, userProfileName, appType, appName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:app/${domainId}/${userProfileName}/${appType}/${appName}`);
    }
    /**
     * Adds a resource of type app-image-config to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/studio-byoi-create.html
     *
     * @param appImageConfigName - Identifier for the appImageConfigName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onAppImageConfig(appImageConfigName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:app-image-config/${appImageConfigName}`);
    }
    /**
     * Adds a resource of type studio-lifecycle-config to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/studio-lcc.html
     *
     * @param studioLifecycleConfigName - Identifier for the studioLifecycleConfigName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onStudioLifecycleConfig(studioLifecycleConfigName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:studio-lifecycle-config/${studioLifecycleConfigName}`);
    }
    /**
     * Adds a resource of type notebook-instance to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/nbi.html
     *
     * @param notebookInstanceName - Identifier for the notebookInstanceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onNotebookInstance(notebookInstanceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:notebook-instance/${notebookInstanceName}`);
    }
    /**
     * Adds a resource of type notebook-instance-lifecycle-config to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/notebook-lifecycle-config.html
     *
     * @param notebookInstanceLifecycleConfigName - Identifier for the notebookInstanceLifecycleConfigName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onNotebookInstanceLifecycleConfig(notebookInstanceLifecycleConfigName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:notebook-instance-lifecycle-config/${notebookInstanceLifecycleConfigName}`);
    }
    /**
     * Adds a resource of type code-repository to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/nbi-git-repo.html
     *
     * @param codeRepositoryName - Identifier for the codeRepositoryName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onCodeRepository(codeRepositoryName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:code-repository/${codeRepositoryName}`);
    }
    /**
     * Adds a resource of type image to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/studio-byoi.html
     *
     * @param imageName - Identifier for the imageName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onImage(imageName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:image/${imageName}`);
    }
    /**
     * Adds a resource of type image-version to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/studio-byoi.html
     *
     * @param imageName - Identifier for the imageName.
     * @param version - Identifier for the version.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onImageVersion(imageName, version, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:image-version/${imageName}/${version}`);
    }
    /**
     * Adds a resource of type algorithm to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/algorithms-choose.html
     *
     * @param algorithmName - Identifier for the algorithmName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onAlgorithm(algorithmName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:algorithm/${algorithmName}`);
    }
    /**
     * Adds a resource of type cluster to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/sagemaker-cluster.html
     *
     * @param clusterId - Identifier for the clusterId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onCluster(clusterId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:cluster/${clusterId}`);
    }
    /**
     * Adds a resource of type training-job to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/how-it-works-training.html
     *
     * @param trainingJobName - Identifier for the trainingJobName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onTrainingJob(trainingJobName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:training-job/${trainingJobName}`);
    }
    /**
     * Adds a resource of type processing-job to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/processing-job.html
     *
     * @param processingJobName - Identifier for the processingJobName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onProcessingJob(processingJobName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:processing-job/${processingJobName}`);
    }
    /**
     * Adds a resource of type hyper-parameter-tuning-job to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/automatic-model-tuning-how-it-works.html
     *
     * @param hyperParameterTuningJobName - Identifier for the hyperParameterTuningJobName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onHyperParameterTuningJob(hyperParameterTuningJobName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:hyper-parameter-tuning-job/${hyperParameterTuningJobName}`);
    }
    /**
     * Adds a resource of type training-plan to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/reserve-capacity-with-training-plans.html
     *
     * @param trainingPlanName - Identifier for the trainingPlanName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onTrainingPlan(trainingPlanName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:training-plan/${trainingPlanName}`);
    }
    /**
     * Adds a resource of type reserved-capacity to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/reserve-capacity-with-training-plans.html
     *
     * @param randomString - Identifier for the randomString.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onReservedCapacity(randomString, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:reserved-capacity/${randomString}`);
    }
    /**
     * Adds a resource of type project to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/sagemaker-projects-whatis.html
     *
     * @param projectName - Identifier for the projectName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onProject(projectName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:project/${projectName}`);
    }
    /**
     * Adds a resource of type model-package to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ModelPackage.html
     *
     * @param modelPackageName - Identifier for the modelPackageName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onModelPackage(modelPackageName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:model-package/${modelPackageName}`);
    }
    /**
     * Adds a resource of type model-package-group to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/model-registry-model-group.html
     *
     * @param modelPackageGroupName - Identifier for the modelPackageGroupName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onModelPackageGroup(modelPackageGroupName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:model-package-group/${modelPackageGroupName}`);
    }
    /**
     * Adds a resource of type model to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/realtime-endpoints.html
     *
     * @param modelName - Identifier for the modelName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onModel(modelName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:model/${modelName}`);
    }
    /**
     * Adds a resource of type endpoint-config to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/realtime-endpoints.html
     *
     * @param endpointConfigName - Identifier for the endpointConfigName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onEndpointConfig(endpointConfigName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:endpoint-config/${endpointConfigName}`);
    }
    /**
     * Adds a resource of type endpoint to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/realtime-endpoints.html
     *
     * @param endpointName - Identifier for the endpointName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onEndpoint(endpointName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:endpoint/${endpointName}`);
    }
    /**
     * Adds a resource of type inference-component to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/realtime-endpoints.html
     *
     * @param inferenceComponentName - Identifier for the inferenceComponentName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onInferenceComponent(inferenceComponentName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:inference-component/${inferenceComponentName}`);
    }
    /**
     * Adds a resource of type transform-job to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_TransformJob.html.html
     *
     * @param transformJobName - Identifier for the transformJobName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onTransformJob(transformJobName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:transform-job/${transformJobName}`);
    }
    /**
     * Adds a resource of type compilation-job to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CompilationJobSummary.html
     *
     * @param compilationJobName - Identifier for the compilationJobName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onCompilationJob(compilationJobName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:compilation-job/${compilationJobName}`);
    }
    /**
     * Adds a resource of type optimization-job to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_OptimizationJobSummary.html
     *
     * @param optimizationJobName - Identifier for the optimizationJobName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onOptimizationJob(optimizationJobName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:optimization-job/${optimizationJobName}`);
    }
    /**
     * Adds a resource of type automl-job to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/autopilot-automate-model-development.html
     *
     * @param autoMLJobJobName - Identifier for the autoMLJobJobName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onAutomlJob(autoMLJobJobName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:automl-job/${autoMLJobJobName}`);
    }
    /**
     * Adds a resource of type monitoring-schedule to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/model-monitor-scheduling.html
     *
     * @param monitoringScheduleName - Identifier for the monitoringScheduleName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onMonitoringSchedule(monitoringScheduleName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:monitoring-schedule/${monitoringScheduleName}`);
    }
    /**
     * Adds a resource of type monitoring-schedule-alert to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/model-monitor-scheduling.html
     *
     * @param monitoringScheduleName - Identifier for the monitoringScheduleName.
     * @param monitoringScheduleAlertName - Identifier for the monitoringScheduleAlertName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onMonitoringScheduleAlert(monitoringScheduleName, monitoringScheduleAlertName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:monitoring-schedule/${monitoringScheduleName}/alert/${monitoringScheduleAlertName}`);
    }
    /**
     * Adds a resource of type data-quality-job-definition to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/model-monitor-data-quality.html
     *
     * @param dataQualityJobDefinitionName - Identifier for the dataQualityJobDefinitionName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onDataQualityJobDefinition(dataQualityJobDefinitionName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:data-quality-job-definition/${dataQualityJobDefinitionName}`);
    }
    /**
     * Adds a resource of type model-quality-job-definition to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/model-monitor-model-quality.html
     *
     * @param modelQualityJobDefinitionName - Identifier for the modelQualityJobDefinitionName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onModelQualityJobDefinition(modelQualityJobDefinitionName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:model-quality-job-definition/${modelQualityJobDefinitionName}`);
    }
    /**
     * Adds a resource of type model-bias-job-definition to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/clarify-detect-post-training-bias.html
     *
     * @param modelBiasJobDefinitionName - Identifier for the modelBiasJobDefinitionName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onModelBiasJobDefinition(modelBiasJobDefinitionName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:model-bias-job-definition/${modelBiasJobDefinitionName}`);
    }
    /**
     * Adds a resource of type model-explainability-job-definition to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/clarify-model-explainability.html
     *
     * @param modelExplainabilityJobDefinitionName - Identifier for the modelExplainabilityJobDefinitionName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onModelExplainabilityJobDefinition(modelExplainabilityJobDefinitionName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:model-explainability-job-definition/${modelExplainabilityJobDefinitionName}`);
    }
    /**
     * Adds a resource of type experiment to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_Experiment.html
     *
     * @param experimentName - Identifier for the experimentName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onExperiment(experimentName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:experiment/${experimentName}`);
    }
    /**
     * Adds a resource of type experiment-trial to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_Trial.html
     *
     * @param trialName - Identifier for the trialName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onExperimentTrial(trialName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:experiment-trial/${trialName}`);
    }
    /**
     * Adds a resource of type experiment-trial-component to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_TrialComponent.html
     *
     * @param trialComponentName - Identifier for the trialComponentName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onExperimentTrialComponent(trialComponentName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:experiment-trial-component/${trialComponentName}`);
    }
    /**
     * Adds a resource of type feature-group to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/feature-store.html
     *
     * @param featureGroupName - Identifier for the featureGroupName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onFeatureGroup(featureGroupName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:feature-group/${featureGroupName}`);
    }
    /**
     * Adds a resource of type pipeline to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_Pipeline.html
     *
     * @param pipelineName - Identifier for the pipelineName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onPipeline(pipelineName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:pipeline/${pipelineName}`);
    }
    /**
     * Adds a resource of type pipeline-execution to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_PipelineExecution.html
     *
     * @param pipelineName - Identifier for the pipelineName.
     * @param randomString - Identifier for the randomString.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onPipelineExecution(pipelineName, randomString, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:pipeline/${pipelineName}/execution/${randomString}`);
    }
    /**
     * Adds a resource of type artifact to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ArtifactSummary.html
     *
     * @param hashOfArtifactSource - Identifier for the hashOfArtifactSource.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onArtifact(hashOfArtifactSource, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:artifact/${hashOfArtifactSource}`);
    }
    /**
     * Adds a resource of type context to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ContextSummary.html
     *
     * @param contextName - Identifier for the contextName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onContext(contextName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:context/${contextName}`);
    }
    /**
     * Adds a resource of type action to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ActionSummary.html
     *
     * @param actionName - Identifier for the actionName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onAction(actionName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:action/${actionName}`);
    }
    /**
     * Adds a resource of type lineage-group to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_LineageGroupSummary.html
     *
     * @param lineageGroupName - Identifier for the lineageGroupName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onLineageGroup(lineageGroupName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:lineage-group/${lineageGroupName}`);
    }
    /**
     * Adds a resource of type model-card to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ModelCard.html
     *
     * @param modelCardName - Identifier for the modelCardName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onModelCard(modelCardName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:model-card/${modelCardName}`);
    }
    /**
     * Adds a resource of type model-card-export-job to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ModelCardExportJob.html
     *
     * @param modelCardName - Identifier for the modelCardName.
     * @param exportJobName - Identifier for the exportJobName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onModelCardExportJob(modelCardName, exportJobName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:model-card/${modelCardName}/export-job/${exportJobName}`);
    }
    /**
     * Adds a resource of type shared-model to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/canvas-collaborate-permissions.html
     *
     * @param sharedModelId - Identifier for the sharedModelId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onSharedModel(sharedModelId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:shared-model/${sharedModelId}`);
    }
    /**
     * Adds a resource of type shared-model-event to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/canvas-collaborate-permissions.html
     *
     * @param eventId - Identifier for the eventId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onSharedModelEvent(eventId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:shared-model-event/${eventId}`);
    }
    /**
     * Adds a resource of type sagemaker-catalog to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ResourceCatalog.html
     *
     * @param resourceCatalogName - Identifier for the resourceCatalogName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onSagemakerCatalog(resourceCatalogName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:sagemaker-catalog/${resourceCatalogName}`);
    }
    /**
     * Adds a resource of type mlflow-tracking-server to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_MlflowTrackingServer.html
     *
     * @param mlflowTrackingServerName - Identifier for the mlflowTrackingServerName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onMlflowTrackingServer(mlflowTrackingServerName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:mlflow-tracking-server/${mlflowTrackingServerName}`);
    }
    /**
     * Adds a resource of type compute-quota to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/sagemaker-hyperpod-eks-operate-console-ui-governance.html
     *
     * @param computeQuotaId - Identifier for the computeQuotaId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onComputeQuota(computeQuotaId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:compute-quota/${computeQuotaId}`);
    }
    /**
     * Adds a resource of type cluster-scheduler-config to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/sagemaker-hyperpod-eks-operate-console-ui-governance.html
     *
     * @param clusterSchedulerConfigId - Identifier for the clusterSchedulerConfigId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onClusterSchedulerConfig(clusterSchedulerConfigId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:cluster-scheduler-config/${clusterSchedulerConfigId}`);
    }
    /**
     * Adds a resource of type partner-app to the statement
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/partner-apps.html
     *
     * @param appId - Identifier for the appId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onPartnerApp(appId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sagemaker:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:partner-app/${appId}`);
    }
    /**
     * Filters access by a key that is present in the request the user makes to the SageMaker service
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toAddTags()
     * - .toCreateAction()
     * - .toCreateAlgorithm()
     * - .toCreateApp()
     * - .toCreateAppImageConfig()
     * - .toCreateArtifact()
     * - .toCreateAutoMLJob()
     * - .toCreateAutoMLJobV2()
     * - .toCreateCluster()
     * - .toCreateClusterSchedulerConfig()
     * - .toCreateCodeRepository()
     * - .toCreateCompilationJob()
     * - .toCreateComputeQuota()
     * - .toCreateContext()
     * - .toCreateDataQualityJobDefinition()
     * - .toCreateDeviceFleet()
     * - .toCreateDomain()
     * - .toCreateEdgeDeploymentPlan()
     * - .toCreateEdgeDeploymentStage()
     * - .toCreateEdgePackagingJob()
     * - .toCreateEndpoint()
     * - .toCreateEndpointConfig()
     * - .toCreateExperiment()
     * - .toCreateFeatureGroup()
     * - .toCreateFlowDefinition()
     * - .toCreateHub()
     * - .toCreateHubContentReference()
     * - .toCreateHumanTaskUi()
     * - .toCreateHyperParameterTuningJob()
     * - .toCreateImage()
     * - .toCreateInferenceComponent()
     * - .toCreateInferenceExperiment()
     * - .toCreateInferenceRecommendationsJob()
     * - .toCreateLabelingJob()
     * - .toCreateMlflowTrackingServer()
     * - .toCreateModel()
     * - .toCreateModelBiasJobDefinition()
     * - .toCreateModelCard()
     * - .toCreateModelExplainabilityJobDefinition()
     * - .toCreateModelPackage()
     * - .toCreateModelPackageGroup()
     * - .toCreateModelQualityJobDefinition()
     * - .toCreateMonitoringSchedule()
     * - .toCreateNotebookInstance()
     * - .toCreateNotebookInstanceLifecycleConfig()
     * - .toCreateOptimizationJob()
     * - .toCreatePartnerApp()
     * - .toCreatePipeline()
     * - .toCreateProcessingJob()
     * - .toCreateProject()
     * - .toCreateReservedCapacity()
     * - .toCreateSpace()
     * - .toCreateStudioLifecycleConfig()
     * - .toCreateTrainingJob()
     * - .toCreateTrainingPlan()
     * - .toCreateTransformJob()
     * - .toCreateTrial()
     * - .toCreateTrialComponent()
     * - .toCreateUserProfile()
     * - .toCreateWorkforce()
     * - .toCreateWorkteam()
     * - .toDeleteFeatureGroup()
     * - .toImportHubContent()
     * - .toRegisterDevices()
     * - .toUpdateMonitoringSchedule()
     * - .toUpdateProject()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a tag key and value pair
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to resource types:
     * - device
     * - device-fleet
     * - edge-packaging-job
     * - edge-deployment-plan
     * - flow-definition
     * - human-task-ui
     * - hub
     * - hub-content
     * - inference-recommendations-job
     * - inference-experiment
     * - labeling-job
     * - workteam
     * - workforce
     * - domain
     * - user-profile
     * - space
     * - app
     * - app-image-config
     * - studio-lifecycle-config
     * - notebook-instance
     * - notebook-instance-lifecycle-config
     * - code-repository
     * - image
     * - algorithm
     * - cluster
     * - training-job
     * - processing-job
     * - hyper-parameter-tuning-job
     * - training-plan
     * - reserved-capacity
     * - project
     * - model-package
     * - model-package-group
     * - model
     * - endpoint-config
     * - endpoint
     * - inference-component
     * - transform-job
     * - compilation-job
     * - optimization-job
     * - automl-job
     * - monitoring-schedule
     * - data-quality-job-definition
     * - model-quality-job-definition
     * - model-bias-job-definition
     * - model-explainability-job-definition
     * - experiment
     * - experiment-trial
     * - experiment-trial-component
     * - feature-group
     * - pipeline
     * - pipeline-execution
     * - artifact
     * - context
     * - action
     * - lineage-group
     * - model-card
     * - model-card-export-job
     * - mlflow-tracking-server
     * - compute-quota
     * - cluster-scheduler-config
     * - partner-app
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the list of all the tag key names associated with the resource in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toAddTags()
     * - .toCreateAction()
     * - .toCreateAlgorithm()
     * - .toCreateApp()
     * - .toCreateAppImageConfig()
     * - .toCreateArtifact()
     * - .toCreateAutoMLJob()
     * - .toCreateAutoMLJobV2()
     * - .toCreateCluster()
     * - .toCreateClusterSchedulerConfig()
     * - .toCreateCodeRepository()
     * - .toCreateCompilationJob()
     * - .toCreateComputeQuota()
     * - .toCreateContext()
     * - .toCreateDataQualityJobDefinition()
     * - .toCreateDeviceFleet()
     * - .toCreateDomain()
     * - .toCreateEdgeDeploymentPlan()
     * - .toCreateEdgeDeploymentStage()
     * - .toCreateEdgePackagingJob()
     * - .toCreateEndpoint()
     * - .toCreateEndpointConfig()
     * - .toCreateExperiment()
     * - .toCreateFeatureGroup()
     * - .toCreateFlowDefinition()
     * - .toCreateHub()
     * - .toCreateHubContentReference()
     * - .toCreateHumanTaskUi()
     * - .toCreateHyperParameterTuningJob()
     * - .toCreateImage()
     * - .toCreateInferenceComponent()
     * - .toCreateInferenceExperiment()
     * - .toCreateInferenceRecommendationsJob()
     * - .toCreateLabelingJob()
     * - .toCreateMlflowTrackingServer()
     * - .toCreateModel()
     * - .toCreateModelBiasJobDefinition()
     * - .toCreateModelCard()
     * - .toCreateModelExplainabilityJobDefinition()
     * - .toCreateModelPackage()
     * - .toCreateModelPackageGroup()
     * - .toCreateModelQualityJobDefinition()
     * - .toCreateMonitoringSchedule()
     * - .toCreateNotebookInstance()
     * - .toCreateNotebookInstanceLifecycleConfig()
     * - .toCreateOptimizationJob()
     * - .toCreatePartnerApp()
     * - .toCreatePipeline()
     * - .toCreateProcessingJob()
     * - .toCreateProject()
     * - .toCreateReservedCapacity()
     * - .toCreateSpace()
     * - .toCreateStudioLifecycleConfig()
     * - .toCreateTrainingJob()
     * - .toCreateTrainingPlan()
     * - .toCreateTransformJob()
     * - .toCreateTrial()
     * - .toCreateTrialComponent()
     * - .toCreateUserProfile()
     * - .toCreateWorkforce()
     * - .toCreateWorkteam()
     * - .toDeleteTags()
     * - .toImportHubContent()
     * - .toRegisterDevices()
     * - .toUpdateMonitoringSchedule()
     * - .toUpdateProject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the list of all accelerator types associated with the resource in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateEndpointConfig()
     * - .toCreateNotebookInstance()
     * - .toUpdateNotebookInstance()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAcceleratorTypes(value, operator) {
        return this.if(`AcceleratorTypes`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the app network access type associated with the resource in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateDomain()
     * - .toUpdateDomain()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAppNetworkAccessType(value, operator) {
        return this.if(`AppNetworkAccessType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a metadata key and value pair
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateModelPackage()
     * - .toUpdateModelPackage()
     *
     * @param metadataKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCustomerMetadataProperties(metadataKey, value, operator) {
        return this.if(`CustomerMetadataProperties/${metadataKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the list of metadata properties associated with the model-package resource in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toUpdateModelPackage()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCustomerMetadataPropertiesToRemove(value, operator) {
        return this.if(`CustomerMetadataPropertiesToRemove`, value, operator ?? 'StringLike');
    }
    /**
     * Used to deny direct access to SageMaker gated ModelReferences
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateModel()
     * - .toCreateTrainingJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDirectGatedModelAccess(value, operator) {
        return this.if(`DirectGatedModelAccess`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the direct internet access associated with the resource in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateNotebookInstance()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDirectInternetAccess(value, operator) {
        return this.if(`DirectInternetAccess`, value, operator ?? 'StringLike');
    }
    /**
     * You can use the domainId as a policy variable to filter requests from specific SageMaker Domains
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDomainId(value, operator) {
        return this.if(`DomainId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the Domain sharing output KMS key associated with the resource in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateDomain()
     * - .toCreateUserProfile()
     * - .toUpdateDomain()
     * - .toUpdateUserProfile()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifDomainSharingOutputKmsKey(value, operator) {
        return this.if(`DomainSharingOutputKmsKey`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the remote debug config in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateTrainingJob()
     * - .toUpdateTrainingJob()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifEnableRemoteDebug(value) {
        return this.if(`EnableRemoteDebug`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the DisableGlueTableCreation flag associated with the feature group resource in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateFeatureGroup()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifFeatureGroupDisableGlueTableCreation(value) {
        return this.if(`FeatureGroupDisableGlueTableCreation`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the EnableOnlineStore flag associated with feature group in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateFeatureGroup()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifFeatureGroupEnableOnlineStore(value) {
        return this.if(`FeatureGroupEnableOnlineStore`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the presence of an OfflineStoreConfig in the feature group resource in the request. This access filter only supports the null-conditional operator
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateFeatureGroup()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifFeatureGroupOfflineStoreConfig(value) {
        return this.if(`FeatureGroupOfflineStoreConfig`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the offline store kms key associated with the feature group resource in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateFeatureGroup()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifFeatureGroupOfflineStoreKmsKey(value, operator) {
        return this.if(`FeatureGroupOfflineStoreKmsKey`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the offline store s3 uri associated with the feature group resource in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateFeatureGroup()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFeatureGroupOfflineStoreS3Uri(value, operator) {
        return this.if(`FeatureGroupOfflineStoreS3Uri`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the online store kms key associated with the feature group resource in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateFeatureGroup()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifFeatureGroupOnlineStoreKmsKey(value, operator) {
        return this.if(`FeatureGroupOnlineStoreKmsKey`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by a file system access mode associated with the resource in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateHyperParameterTuningJob()
     * - .toCreateTrainingJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFileSystemAccessMode(value, operator) {
        return this.if(`FileSystemAccessMode`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a file system directory path associated with the resource in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateHyperParameterTuningJob()
     * - .toCreateTrainingJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFileSystemDirectoryPath(value, operator) {
        return this.if(`FileSystemDirectoryPath`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a file system ID associated with the resource in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateHyperParameterTuningJob()
     * - .toCreateTrainingJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFileSystemId(value, operator) {
        return this.if(`FileSystemId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a file system type associated with the resource in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateHyperParameterTuningJob()
     * - .toCreateTrainingJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFileSystemType(value, operator) {
        return this.if(`FileSystemType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a key that is present in the request the user makes to the SageMaker service. This key is deprecated. It has been replaced by sagemaker:VolumeKmsKey
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifHomeEfsFileSystemKmsKey(value, operator) {
        return this.if(`HomeEfsFileSystemKmsKey`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the list of all image arns associated with the resource in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateApp()
     * - .toCreateDomain()
     * - .toCreateSpace()
     * - .toCreateUserProfile()
     * - .toUpdateDomain()
     * - .toUpdateSpace()
     * - .toUpdateUserProfile()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifImageArns(value, operator) {
        return this.if(`ImageArns`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the list of all image version arns associated with the resource in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateApp()
     * - .toCreateDomain()
     * - .toCreateSpace()
     * - .toCreateUserProfile()
     * - .toUpdateDomain()
     * - .toUpdateSpace()
     * - .toUpdateUserProfile()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifImageVersionArns(value, operator) {
        return this.if(`ImageVersionArns`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the list of all instance types associated with the resource in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateApp()
     * - .toCreateCluster()
     * - .toCreateDataQualityJobDefinition()
     * - .toCreateDomain()
     * - .toCreateEndpointConfig()
     * - .toCreateHyperParameterTuningJob()
     * - .toCreateModelBiasJobDefinition()
     * - .toCreateModelExplainabilityJobDefinition()
     * - .toCreateModelQualityJobDefinition()
     * - .toCreateMonitoringSchedule()
     * - .toCreateNotebookInstance()
     * - .toCreateProcessingJob()
     * - .toCreateSpace()
     * - .toCreateTrainingJob()
     * - .toCreateTransformJob()
     * - .toCreateUserProfile()
     * - .toUpdateCluster()
     * - .toUpdateDomain()
     * - .toUpdateMonitoringSchedule()
     * - .toUpdateNotebookInstance()
     * - .toUpdateSpace()
     * - .toUpdateTrainingJob()
     * - .toUpdateUserProfile()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifInstanceTypes(value, operator) {
        return this.if(`InstanceTypes`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the inter container traffic encryption associated with the resource in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateAutoMLJob()
     * - .toCreateAutoMLJobV2()
     * - .toCreateDataQualityJobDefinition()
     * - .toCreateHyperParameterTuningJob()
     * - .toCreateModelBiasJobDefinition()
     * - .toCreateModelExplainabilityJobDefinition()
     * - .toCreateModelQualityJobDefinition()
     * - .toCreateMonitoringSchedule()
     * - .toCreateProcessingJob()
     * - .toCreateTrainingJob()
     * - .toUpdateMonitoringSchedule()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifInterContainerTrafficEncryption(value) {
        return this.if(`InterContainerTrafficEncryption`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the keep-alive period associated with the resource in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateTrainingJob()
     * - .toUpdateTrainingJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifKeepAlivePeriod(value, operator) {
        return this.if(`KeepAlivePeriod`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access by the max runtime in seconds associated with the resource in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateDataQualityJobDefinition()
     * - .toCreateHyperParameterTuningJob()
     * - .toCreateModelBiasJobDefinition()
     * - .toCreateModelExplainabilityJobDefinition()
     * - .toCreateModelQualityJobDefinition()
     * - .toCreateMonitoringSchedule()
     * - .toCreateProcessingJob()
     * - .toCreateTrainingJob()
     * - .toUpdateMonitoringSchedule()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifMaxRuntimeInSeconds(value, operator) {
        return this.if(`MaxRuntimeInSeconds`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access by the minimum instance metadata service version used by the resource in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateNotebookInstance()
     * - .toUpdateNotebookInstance()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifMinimumInstanceMetadataServiceVersion(value, operator) {
        return this.if(`MinimumInstanceMetadataServiceVersion`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the model approval status with the model-package in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateModelPackage()
     * - .toUpdateModelPackage()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifModelApprovalStatus(value, operator) {
        return this.if(`ModelApprovalStatus`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the model arn associated with the resource in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateEndpointConfig()
     * - .toCreateInferenceComponent()
     * - .toCreateTransformJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifModelArn(value, operator) {
        return this.if(`ModelArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by stageStatus field in the model life cycle object associated with the model-package resource in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateModelPackage()
     * - .toUpdateModelPackage()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifModelLifeCycle(value, operator) {
        return this.if(`ModelLifeCycle`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the network isolation associated with the resource in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateDataQualityJobDefinition()
     * - .toCreateEndpointConfig()
     * - .toCreateHyperParameterTuningJob()
     * - .toCreateModel()
     * - .toCreateModelBiasJobDefinition()
     * - .toCreateModelExplainabilityJobDefinition()
     * - .toCreateModelQualityJobDefinition()
     * - .toCreateMonitoringSchedule()
     * - .toCreateProcessingJob()
     * - .toCreateTrainingJob()
     * - .toUpdateMonitoringSchedule()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifNetworkIsolation(value) {
        return this.if(`NetworkIsolation`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the output kms key associated with the resource in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateAutoMLJob()
     * - .toCreateAutoMLJobV2()
     * - .toCreateDataQualityJobDefinition()
     * - .toCreateHyperParameterTuningJob()
     * - .toCreateLabelingJob()
     * - .toCreateModelBiasJobDefinition()
     * - .toCreateModelExplainabilityJobDefinition()
     * - .toCreateModelQualityJobDefinition()
     * - .toCreateMonitoringSchedule()
     * - .toCreateProcessingJob()
     * - .toCreateTrainingJob()
     * - .toCreateTransformJob()
     * - .toUpdateMonitoringSchedule()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifOutputKmsKey(value, operator) {
        return this.if(`OutputKmsKey`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the OwnerUserProfile arn associated with the space in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateApp()
     * - .toCreateSpace()
     * - .toDeleteApp()
     * - .toDeleteSpace()
     * - .toUpdateSpace()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifOwnerUserProfileArn(value, operator) {
        return this.if(`OwnerUserProfileArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access to specific version IDs of a Sagemaker pipeline
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toDescribePipeline()
     * - .toStartPipelineExecution()
     * - .toUpdatePipelineVersion()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPipelineVersionId(value, operator) {
        return this.if(`PipelineVersionId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the remote access flag associated with the space in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateSpace()
     * - .toUpdateSpace()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRemoteAccess(value, operator) {
        return this.if(`RemoteAccess`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the preface string for a tag key and value pair attached to a resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTagExists(value, operator) {
        return this.if(`ResourceTag/`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a tag key and value pair
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to resource types:
     * - device
     * - device-fleet
     * - edge-packaging-job
     * - edge-deployment-plan
     * - flow-definition
     * - human-task-ui
     * - hub
     * - hub-content
     * - inference-recommendations-job
     * - inference-experiment
     * - labeling-job
     * - workteam
     * - workforce
     * - domain
     * - user-profile
     * - space
     * - app
     * - app-image-config
     * - studio-lifecycle-config
     * - notebook-instance
     * - notebook-instance-lifecycle-config
     * - code-repository
     * - image
     * - algorithm
     * - cluster
     * - training-job
     * - processing-job
     * - hyper-parameter-tuning-job
     * - training-plan
     * - reserved-capacity
     * - project
     * - model-package
     * - model-package-group
     * - model
     * - endpoint-config
     * - endpoint
     * - inference-component
     * - transform-job
     * - compilation-job
     * - optimization-job
     * - automl-job
     * - monitoring-schedule
     * - data-quality-job-definition
     * - model-quality-job-definition
     * - model-bias-job-definition
     * - model-explainability-job-definition
     * - experiment
     * - experiment-trial
     * - experiment-trial-component
     * - feature-group
     * - pipeline
     * - pipeline-execution
     * - artifact
     * - context
     * - action
     * - lineage-group
     * - model-card
     * - model-card-export-job
     * - mlflow-tracking-server
     * - compute-quota
     * - cluster-scheduler-config
     * - partner-app
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTagTagKey(tagKey, value, operator) {
        return this.if(`ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the root access associated with the resource in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateNotebookInstance()
     * - .toUpdateNotebookInstance()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRootAccess(value, operator) {
        return this.if(`RootAccess`, value, operator ?? 'StringLike');
    }
    /**
     * Limits the results of your search request to the resources that you can access. ${FilterKey} is a key that the VisibilityConditions configuration presents in the Search request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toSearch()
     *
     * @param filterKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSearchVisibilityCondition(filterKey, value, operator) {
        return this.if(`SearchVisibilityCondition/${filterKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by limiting maximum concurrency used for Serverless inference in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateEndpointConfig()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifServerlessMaxConcurrency(value, operator) {
        return this.if(`ServerlessMaxConcurrency`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access by limiting memory size used for Serverless inference in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateEndpointConfig()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifServerlessMemorySize(value, operator) {
        return this.if(`ServerlessMemorySize`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access by the sharing type associated with the space in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateApp()
     * - .toCreateSpace()
     * - .toDeleteApp()
     * - .toDeleteSpace()
     * - .toUpdateSpace()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSpaceSharingType(value, operator) {
        return this.if(`SpaceSharingType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the API actions to which a user can apply tags. Uses the name of the API operation that creates a taggable resource to filter access
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toAddTags()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTaggingAction(value, operator) {
        return this.if(`TaggingAction`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the target model associated with the Multi-Model Endpoint in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toInvokeEndpoint()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTargetModel(value, operator) {
        return this.if(`TargetModel`, value, operator ?? 'StringLike');
    }
    /**
     * You can use the UserProfileName as a policy variable to filter requests from specific user profiles within a SageMaker Domain. This context key is not applicable to user profiles within shared spaces
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifUserProfileName(value, operator) {
        return this.if(`UserProfileName`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the volume kms key associated with the resource in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateAutoMLJob()
     * - .toCreateAutoMLJobV2()
     * - .toCreateDataQualityJobDefinition()
     * - .toCreateDomain()
     * - .toCreateEndpointConfig()
     * - .toCreateHyperParameterTuningJob()
     * - .toCreateLabelingJob()
     * - .toCreateModelBiasJobDefinition()
     * - .toCreateModelExplainabilityJobDefinition()
     * - .toCreateModelQualityJobDefinition()
     * - .toCreateMonitoringSchedule()
     * - .toCreateNotebookInstance()
     * - .toCreateProcessingJob()
     * - .toCreateTrainingJob()
     * - .toCreateTransformJob()
     * - .toUpdateMonitoringSchedule()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifVolumeKmsKey(value, operator) {
        return this.if(`VolumeKmsKey`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the list of all VPC security group ids associated with the resource in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateAutoMLJob()
     * - .toCreateAutoMLJobV2()
     * - .toCreateCluster()
     * - .toCreateDataQualityJobDefinition()
     * - .toCreateDomain()
     * - .toCreateEndpointConfig()
     * - .toCreateHyperParameterTuningJob()
     * - .toCreateModel()
     * - .toCreateModelBiasJobDefinition()
     * - .toCreateModelExplainabilityJobDefinition()
     * - .toCreateModelQualityJobDefinition()
     * - .toCreateMonitoringSchedule()
     * - .toCreateNotebookInstance()
     * - .toCreateProcessingJob()
     * - .toCreateTrainingJob()
     * - .toCreateUserProfile()
     * - .toUpdateCluster()
     * - .toUpdateDomain()
     * - .toUpdateMonitoringSchedule()
     * - .toUpdateUserProfile()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVpcSecurityGroupIds(value, operator) {
        return this.if(`VpcSecurityGroupIds`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the list of all VPC subnets associated with the resource in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateAutoMLJob()
     * - .toCreateAutoMLJobV2()
     * - .toCreateCluster()
     * - .toCreateDataQualityJobDefinition()
     * - .toCreateDomain()
     * - .toCreateEndpointConfig()
     * - .toCreateHyperParameterTuningJob()
     * - .toCreateModel()
     * - .toCreateModelBiasJobDefinition()
     * - .toCreateModelExplainabilityJobDefinition()
     * - .toCreateModelQualityJobDefinition()
     * - .toCreateMonitoringSchedule()
     * - .toCreateNotebookInstance()
     * - .toCreateProcessingJob()
     * - .toCreateTrainingJob()
     * - .toUpdateCluster()
     * - .toUpdateDomain()
     * - .toUpdateMonitoringSchedule()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVpcSubnets(value, operator) {
        return this.if(`VpcSubnets`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the workteam arn associated to the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateFlowDefinition()
     * - .toCreateLabelingJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifWorkteamArn(value, operator) {
        return this.if(`WorkteamArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the workteam type associated to the request. This can be public-crowd, private-crowd or vendor-crowd
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateFlowDefinition()
     * - .toCreateLabelingJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifWorkteamType(value, operator) {
        return this.if(`WorkteamType`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [sagemaker](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonsagemaker.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'sagemaker';
        this.accessLevelList = {
            Write: [
                'AddAssociation',
                'AssociateTrialComponent',
                'AttachClusterNodeVolume',
                'BatchAddClusterNodes',
                'BatchDeleteClusterNodes',
                'BatchPutMetrics',
                'CallPartnerAppApi',
                'CreateAction',
                'CreateAlgorithm',
                'CreateApp',
                'CreateAppImageConfig',
                'CreateArtifact',
                'CreateAutoMLJob',
                'CreateAutoMLJobV2',
                'CreateCluster',
                'CreateClusterSchedulerConfig',
                'CreateCodeRepository',
                'CreateCompilationJob',
                'CreateComputeQuota',
                'CreateContext',
                'CreateDataQualityJobDefinition',
                'CreateDeviceFleet',
                'CreateDomain',
                'CreateEdgeDeploymentPlan',
                'CreateEdgeDeploymentStage',
                'CreateEdgePackagingJob',
                'CreateEndpoint',
                'CreateEndpointConfig',
                'CreateExperiment',
                'CreateFeatureGroup',
                'CreateFlowDefinition',
                'CreateHub',
                'CreateHubContentReference',
                'CreateHumanTaskUi',
                'CreateHyperParameterTuningJob',
                'CreateImage',
                'CreateImageVersion',
                'CreateInferenceComponent',
                'CreateInferenceExperiment',
                'CreateInferenceRecommendationsJob',
                'CreateLabelingJob',
                'CreateLineageGroupPolicy',
                'CreateMlflowTrackingServer',
                'CreateModel',
                'CreateModelBiasJobDefinition',
                'CreateModelCard',
                'CreateModelCardExportJob',
                'CreateModelExplainabilityJobDefinition',
                'CreateModelPackage',
                'CreateModelPackageGroup',
                'CreateModelQualityJobDefinition',
                'CreateMonitoringSchedule',
                'CreateNotebookInstance',
                'CreateNotebookInstanceLifecycleConfig',
                'CreateOptimizationJob',
                'CreatePartnerApp',
                'CreatePartnerAppPresignedUrl',
                'CreatePipeline',
                'CreatePresignedDomainUrl',
                'CreatePresignedMlflowTrackingServerUrl',
                'CreatePresignedNotebookInstanceUrl',
                'CreateProcessingJob',
                'CreateProject',
                'CreateReservedCapacity',
                'CreateSharedModel',
                'CreateSpace',
                'CreateStudioLifecycleConfig',
                'CreateTrainingJob',
                'CreateTrainingPlan',
                'CreateTransformJob',
                'CreateTrial',
                'CreateTrialComponent',
                'CreateUserProfile',
                'CreateWorkforce',
                'CreateWorkteam',
                'DeleteAction',
                'DeleteAlgorithm',
                'DeleteApp',
                'DeleteAppImageConfig',
                'DeleteArtifact',
                'DeleteAssociation',
                'DeleteCluster',
                'DeleteClusterSchedulerConfig',
                'DeleteCodeRepository',
                'DeleteCompilationJob',
                'DeleteComputeQuota',
                'DeleteContext',
                'DeleteDataQualityJobDefinition',
                'DeleteDeviceFleet',
                'DeleteDomain',
                'DeleteEdgeDeploymentPlan',
                'DeleteEdgeDeploymentStage',
                'DeleteEndpoint',
                'DeleteEndpointConfig',
                'DeleteExperiment',
                'DeleteFeatureGroup',
                'DeleteFlowDefinition',
                'DeleteHub',
                'DeleteHubContent',
                'DeleteHubContentReference',
                'DeleteHumanLoop',
                'DeleteHumanTaskUi',
                'DeleteHyperParameterTuningJob',
                'DeleteImage',
                'DeleteImageVersion',
                'DeleteInferenceComponent',
                'DeleteInferenceExperiment',
                'DeleteLineageGroupPolicy',
                'DeleteMlflowTrackingServer',
                'DeleteModel',
                'DeleteModelBiasJobDefinition',
                'DeleteModelCard',
                'DeleteModelExplainabilityJobDefinition',
                'DeleteModelPackage',
                'DeleteModelPackageGroup',
                'DeleteModelPackageGroupPolicy',
                'DeleteModelQualityJobDefinition',
                'DeleteMonitoringSchedule',
                'DeleteNotebookInstance',
                'DeleteNotebookInstanceLifecycleConfig',
                'DeleteOptimizationJob',
                'DeletePartnerApp',
                'DeletePipeline',
                'DeleteProject',
                'DeleteRecord',
                'DeleteResourcePolicy',
                'DeleteSpace',
                'DeleteStudioLifecycleConfig',
                'DeleteTrial',
                'DeleteTrialComponent',
                'DeleteUserProfile',
                'DeleteWorkforce',
                'DeleteWorkteam',
                'DeployHubModel',
                'DeregisterDevices',
                'DetachClusterNodeVolume',
                'DisableSagemakerServicecatalogPortfolio',
                'DisassociateTrialComponent',
                'EnableSagemakerServicecatalogPortfolio',
                'ImportHubContent',
                'PutLineageGroupPolicy',
                'PutModelPackageGroupPolicy',
                'PutRecord',
                'PutResourcePolicy',
                'RegisterDevices',
                'RetryPipelineExecution',
                'SendHeartbeat',
                'SendPipelineExecutionStepFailure',
                'SendPipelineExecutionStepSuccess',
                'SendSharedModelEvent',
                'StartEdgeDeploymentStage',
                'StartHumanLoop',
                'StartInferenceExperiment',
                'StartMlflowTrackingServer',
                'StartMonitoringSchedule',
                'StartNotebookInstance',
                'StartPipelineExecution',
                'StartSession',
                'StopAutoMLJob',
                'StopCompilationJob',
                'StopEdgeDeploymentStage',
                'StopEdgePackagingJob',
                'StopHumanLoop',
                'StopHyperParameterTuningJob',
                'StopInferenceExperiment',
                'StopInferenceRecommendationsJob',
                'StopLabelingJob',
                'StopMlflowTrackingServer',
                'StopMonitoringSchedule',
                'StopNotebookInstance',
                'StopOptimizationJob',
                'StopPipelineExecution',
                'StopProcessingJob',
                'StopTrainingJob',
                'StopTransformJob',
                'TrainHubModel',
                'UpdateAction',
                'UpdateAppImageConfig',
                'UpdateArtifact',
                'UpdateCluster',
                'UpdateClusterInference',
                'UpdateClusterSchedulerConfig',
                'UpdateClusterSoftware',
                'UpdateCodeRepository',
                'UpdateComputeQuota',
                'UpdateContext',
                'UpdateDeviceFleet',
                'UpdateDevices',
                'UpdateDomain',
                'UpdateEndpoint',
                'UpdateEndpointWeightsAndCapacities',
                'UpdateExperiment',
                'UpdateFeatureGroup',
                'UpdateFeatureMetadata',
                'UpdateHub',
                'UpdateHubContent',
                'UpdateHubContentReference',
                'UpdateImage',
                'UpdateImageVersion',
                'UpdateInferenceComponent',
                'UpdateInferenceComponentRuntimeConfig',
                'UpdateInferenceExperiment',
                'UpdateMlflowTrackingServer',
                'UpdateModelCard',
                'UpdateModelPackage',
                'UpdateMonitoringAlert',
                'UpdateMonitoringSchedule',
                'UpdateNotebookInstance',
                'UpdateNotebookInstanceLifecycleConfig',
                'UpdatePartnerApp',
                'UpdatePipeline',
                'UpdatePipelineExecution',
                'UpdatePipelineVersion',
                'UpdateProject',
                'UpdateSharedModel',
                'UpdateSpace',
                'UpdateTrainingJob',
                'UpdateTrial',
                'UpdateTrialComponent',
                'UpdateUserProfile',
                'UpdateWorkforce',
                'UpdateWorkteam'
            ],
            Tagging: [
                'AddTags',
                'DeleteTags'
            ],
            Read: [
                'BatchDescribeModelPackage',
                'BatchGetMetrics',
                'BatchGetRecord',
                'CreateHubContentPresignedUrls',
                'DescribeAction',
                'DescribeAlgorithm',
                'DescribeApp',
                'DescribeAppImageConfig',
                'DescribeArtifact',
                'DescribeAutoMLJob',
                'DescribeAutoMLJobV2',
                'DescribeCluster',
                'DescribeClusterEvent',
                'DescribeClusterInference',
                'DescribeClusterNode',
                'DescribeClusterSchedulerConfig',
                'DescribeCodeRepository',
                'DescribeCompilationJob',
                'DescribeComputeQuota',
                'DescribeContext',
                'DescribeDataQualityJobDefinition',
                'DescribeDevice',
                'DescribeDeviceFleet',
                'DescribeDomain',
                'DescribeEdgeDeploymentPlan',
                'DescribeEdgePackagingJob',
                'DescribeEndpoint',
                'DescribeEndpointConfig',
                'DescribeExperiment',
                'DescribeFeatureGroup',
                'DescribeFeatureMetadata',
                'DescribeFlowDefinition',
                'DescribeHub',
                'DescribeHubContent',
                'DescribeHumanLoop',
                'DescribeHumanTaskUi',
                'DescribeHyperParameterTuningJob',
                'DescribeImage',
                'DescribeImageVersion',
                'DescribeInferenceComponent',
                'DescribeInferenceExperiment',
                'DescribeInferenceRecommendationsJob',
                'DescribeLabelingJob',
                'DescribeLineageGroup',
                'DescribeMlflowTrackingServer',
                'DescribeModel',
                'DescribeModelBiasJobDefinition',
                'DescribeModelCard',
                'DescribeModelCardExportJob',
                'DescribeModelExplainabilityJobDefinition',
                'DescribeModelPackage',
                'DescribeModelPackageGroup',
                'DescribeModelQualityJobDefinition',
                'DescribeMonitoringSchedule',
                'DescribeNotebookInstance',
                'DescribeNotebookInstanceLifecycleConfig',
                'DescribeOptimizationJob',
                'DescribePartnerApp',
                'DescribePipeline',
                'DescribePipelineDefinitionForExecution',
                'DescribePipelineExecution',
                'DescribeProcessingJob',
                'DescribeProject',
                'DescribeReservedCapacity',
                'DescribeSharedModel',
                'DescribeSpace',
                'DescribeStudioLifecycleConfig',
                'DescribeSubscribedWorkteam',
                'DescribeTrainingJob',
                'DescribeTrainingPlan',
                'DescribeTransformJob',
                'DescribeTrial',
                'DescribeTrialComponent',
                'DescribeUserProfile',
                'DescribeWorkforce',
                'DescribeWorkteam',
                'GetDeployments',
                'GetDeviceFleetReport',
                'GetDeviceRegistration',
                'GetLineageGroupPolicy',
                'GetModelPackageGroupPolicy',
                'GetRecord',
                'GetResourcePolicy',
                'GetSagemakerServicecatalogPortfolioStatus',
                'GetScalingConfigurationRecommendation',
                'GetSearchSuggestions',
                'InvokeEndpoint',
                'InvokeEndpointAsync',
                'InvokeEndpointWithResponseStream',
                'RenderUiTemplate',
                'Search',
                'SearchTrainingPlanOfferings'
            ],
            List: [
                'ListActions',
                'ListAlgorithms',
                'ListAliases',
                'ListAppImageConfigs',
                'ListApps',
                'ListArtifacts',
                'ListAssociations',
                'ListAutoMLJobs',
                'ListCandidatesForAutoMLJob',
                'ListClusterEvents',
                'ListClusterNodes',
                'ListClusterSchedulerConfigs',
                'ListClusters',
                'ListCodeRepositories',
                'ListCompilationJobs',
                'ListComputeQuotas',
                'ListContexts',
                'ListDataQualityJobDefinitions',
                'ListDeviceFleets',
                'ListDevices',
                'ListDomains',
                'ListEdgeDeploymentPlans',
                'ListEdgePackagingJobs',
                'ListEndpointConfigs',
                'ListEndpoints',
                'ListExperiments',
                'ListFeatureGroups',
                'ListFlowDefinitions',
                'ListHubContentVersions',
                'ListHubContents',
                'ListHubs',
                'ListHumanLoops',
                'ListHumanTaskUis',
                'ListHyperParameterTuningJobs',
                'ListImageVersions',
                'ListImages',
                'ListInferenceComponents',
                'ListInferenceExperiments',
                'ListInferenceRecommendationsJobSteps',
                'ListInferenceRecommendationsJobs',
                'ListLabelingJobs',
                'ListLabelingJobsForWorkteam',
                'ListLineageGroups',
                'ListMlflowTrackingServers',
                'ListModelBiasJobDefinitions',
                'ListModelCardExportJobs',
                'ListModelCardVersions',
                'ListModelCards',
                'ListModelExplainabilityJobDefinitions',
                'ListModelMetadata',
                'ListModelPackageGroups',
                'ListModelPackages',
                'ListModelQualityJobDefinitions',
                'ListModels',
                'ListMonitoringAlertHistory',
                'ListMonitoringAlerts',
                'ListMonitoringExecutions',
                'ListMonitoringSchedules',
                'ListNotebookInstanceLifecycleConfigs',
                'ListNotebookInstances',
                'ListOptimizationJobs',
                'ListPartnerApps',
                'ListPipelineExecutionSteps',
                'ListPipelineExecutions',
                'ListPipelineParametersForExecution',
                'ListPipelineVersions',
                'ListPipelines',
                'ListProcessingJobs',
                'ListProjects',
                'ListResourceCatalogs',
                'ListSharedModelEvents',
                'ListSharedModelVersions',
                'ListSharedModels',
                'ListSpaces',
                'ListStageDevices',
                'ListStudioLifecycleConfigs',
                'ListSubscribedWorkteams',
                'ListTags',
                'ListTrainingJobs',
                'ListTrainingJobsForHyperParameterTuningJob',
                'ListTrainingPlans',
                'ListTransformJobs',
                'ListTrialComponents',
                'ListTrials',
                'ListUltraServersByReservedCapacity',
                'ListUserProfiles',
                'ListWorkforces',
                'ListWorkteams',
                'QueryLineage'
            ]
        };
    }
}
exports.Sagemaker = Sagemaker;
//# sourceMappingURL=data:application/json;base64,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