"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Securityhub = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [securityhub](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssecurityhub.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Securityhub extends shared_1.PolicyStatement {
    /**
     * Grants permission to accept Security Hub invitations to become a member account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_AcceptAdministratorInvitation.html
     */
    toAcceptAdministratorInvitation() {
        return this.to('AcceptAdministratorInvitation');
    }
    /**
     * Grants permission to accept Security Hub invitations to become a member account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_AcceptInvitation.html
     */
    toAcceptInvitation() {
        return this.to('AcceptInvitation');
    }
    /**
     * Grants permission to delete one or more automation rules in Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/automation-rules
     */
    toBatchDeleteAutomationRules() {
        return this.to('BatchDeleteAutomationRules');
    }
    /**
     * Grants permission to disable standards in Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchDisableStandards.html
     */
    toBatchDisableStandards() {
        return this.to('BatchDisableStandards');
    }
    /**
     * Grants permission to enable standards in Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchEnableStandards.html
     */
    toBatchEnableStandards() {
        return this.to('BatchEnableStandards');
    }
    /**
     * Grants permission to retrieve a list of details for automation rules from Security Hub based on rule Amazon Resource Names (ARNs)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/automation-rules
     */
    toBatchGetAutomationRules() {
        return this.to('BatchGetAutomationRules');
    }
    /**
     * Grants permission to retrieve information about configuration policies associated with a specific list of member accounts and organizational units of the calling account's organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchGetConfigurationPolicyAssociations.html
     */
    toBatchGetConfigurationPolicyAssociations() {
        return this.to('BatchGetConfigurationPolicyAssociations');
    }
    /**
     * Grants permission to get the enablement and compliance status of controls, the findings count for controls, and the overall security score for controls on the Security Hub console
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/iam-permissions-controls-standards.html
     */
    toBatchGetControlEvaluations() {
        return this.to('BatchGetControlEvaluations');
    }
    /**
     * Grants permission to get details about specific security controls identified by ID or ARN
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - securityhub:DescribeStandardsControls
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchGetSecurityControls.html
     */
    toBatchGetSecurityControls() {
        return this.to('BatchGetSecurityControls');
    }
    /**
     * Grants permission to get the enablement status of a batch of security controls in standards
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - securityhub:DescribeStandardsControls
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchGetStandardsControlAssociations.html
     */
    toBatchGetStandardsControlAssociations() {
        return this.to('BatchGetStandardsControlAssociations');
    }
    /**
     * Grants permission to import findings into Security Hub from an integrated product
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTargetAccount()
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchImportFindings.html
     */
    toBatchImportFindings() {
        return this.to('BatchImportFindings');
    }
    /**
     * Grants permission to update one or more automation rules from Security Hub based on rule Amazon Resource Names (ARNs) and input parameters
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/automation-rules
     */
    toBatchUpdateAutomationRules() {
        return this.to('BatchUpdateAutomationRules');
    }
    /**
     * Grants permission to update customer-controlled fields for a selected set of Security Hub findings
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifASFFSyntaxPath()
     * - .ifOCSFSyntaxPath()
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchUpdateFindingsV2.html
     */
    toBatchUpdateFindings() {
        return this.to('BatchUpdateFindings');
    }
    /**
     * Grants permission to update the enablement status of a batch of security controls in standards
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - securityhub:UpdateStandardsControl
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchUpdateStandardsControlAssociations.html
     */
    toBatchUpdateStandardsControlAssociations() {
        return this.to('BatchUpdateStandardsControlAssociations');
    }
    /**
     * Grants permission to complete the OAuth 2.0 authorization code flow based on input parameters
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ConnectorRegistrationsV2.html
     */
    toConnectorRegistrationsV2() {
        return this.to('ConnectorRegistrationsV2');
    }
    /**
     * Grants permission to create custom actions in Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_CreateActionTarget.html
     */
    toCreateActionTarget() {
        return this.to('CreateActionTarget');
    }
    /**
     * Grants permission to create an aggregatorV2, which configures data aggregation across Regions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_CreateAggregatorV2.html
     */
    toCreateAggregatorV2() {
        return this.to('CreateAggregatorV2');
    }
    /**
     * Grants permission to create an automation rule based on input parameters
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/automation-rules
     */
    toCreateAutomationRule() {
        return this.to('CreateAutomationRule');
    }
    /**
     * Grants permission to create an automation rule V2 based on input parameters
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/automation-rules
     */
    toCreateAutomationRuleV2() {
        return this.to('CreateAutomationRuleV2');
    }
    /**
     * Grants permission to create a configuration policy to manage organization member settings in Security Hub
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_CreateConfigurationPolicy.html
     */
    toCreateConfigurationPolicy() {
        return this.to('CreateConfigurationPolicy');
    }
    /**
     * Grants permission to create a connector V2 based on input parameters
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_CreateConnectorV2.html
     */
    toCreateConnectorV2() {
        return this.to('CreateConnectorV2');
    }
    /**
     * Grants permission to create a finding aggregator, which contains the cross-Region finding aggregation configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_CreateFindingAggregator.html
     */
    toCreateFindingAggregator() {
        return this.to('CreateFindingAggregator');
    }
    /**
     * Grants permission to create insights in Security Hub. Insights are collections of related findings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_CreateInsight.html
     */
    toCreateInsight() {
        return this.to('CreateInsight');
    }
    /**
     * Grants permission to create member accounts in Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_CreateMembers.html
     */
    toCreateMembers() {
        return this.to('CreateMembers');
    }
    /**
     * Grants permission to create ticket for a selected OCSF finding
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_CreateTicketV2.html
     */
    toCreateTicketV2() {
        return this.to('CreateTicketV2');
    }
    /**
     * Grants permission to decline Security Hub invitations to become a member account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeclineInvitations.html
     */
    toDeclineInvitations() {
        return this.to('DeclineInvitations');
    }
    /**
     * Grants permission to delete custom actions in Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeleteActionTarget.html
     */
    toDeleteActionTarget() {
        return this.to('DeleteActionTarget');
    }
    /**
     * Grants permission to delete an aggregatorV2, which configures data aggregation across Regions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeleteAggregatorV2.html
     */
    toDeleteAggregatorV2() {
        return this.to('DeleteAggregatorV2');
    }
    /**
     * Grants permission to delete an automation rule V2 in Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/automation-rules
     */
    toDeleteAutomationRuleV2() {
        return this.to('DeleteAutomationRuleV2');
    }
    /**
     * Grants permission to delete an existing configuration policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeleteConfigurationPolicy.html
     */
    toDeleteConfigurationPolicy() {
        return this.to('DeleteConfigurationPolicy');
    }
    /**
     * Grants permission to delete a connector V2 in Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeleteConnectorV2.html
     */
    toDeleteConnectorV2() {
        return this.to('DeleteConnectorV2');
    }
    /**
     * Grants permission to delete a finding aggregator, which disables finding aggregation across Regions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeleteFindingAggregator.html
     */
    toDeleteFindingAggregator() {
        return this.to('DeleteFindingAggregator');
    }
    /**
     * Grants permission to delete insights from Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeleteInsight.html
     */
    toDeleteInsight() {
        return this.to('DeleteInsight');
    }
    /**
     * Grants permission to delete Security Hub invitations to become a member account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeleteInvitations.html
     */
    toDeleteInvitations() {
        return this.to('DeleteInvitations');
    }
    /**
     * Grants permission to delete Security Hub member accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeleteMembers.html
     */
    toDeleteMembers() {
        return this.to('DeleteMembers');
    }
    /**
     * Grants permission to retrieve a list of custom actions using the API
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DescribeActionTargets.html
     */
    toDescribeActionTargets() {
        return this.to('DescribeActionTargets');
    }
    /**
     * Grants permission to retrieve information about the hub resource in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DescribeHub.html
     */
    toDescribeHub() {
        return this.to('DescribeHub');
    }
    /**
     * Grants permission to describe the organization configuration for Security Hub
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DescribeOrganizationConfiguration.html
     */
    toDescribeOrganizationConfiguration() {
        return this.to('DescribeOrganizationConfiguration');
    }
    /**
     * Grants permission to retrieve information about the available Security Hub product integrations
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DescribeProducts.html
     */
    toDescribeProducts() {
        return this.to('DescribeProducts');
    }
    /**
     * Grants permission to retrieve information about the available Security Hub V2 product integrations
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DescribeProductsV2.html
     */
    toDescribeProductsV2() {
        return this.to('DescribeProductsV2');
    }
    /**
     * Grants permission to retrieve information about the hub V2 resource in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DescribeSecurityHubV2.html
     */
    toDescribeSecurityHubV2() {
        return this.to('DescribeSecurityHubV2');
    }
    /**
     * Grants permission to retrieve information about Security Hub standards
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DescribeStandards.html
     */
    toDescribeStandards() {
        return this.to('DescribeStandards');
    }
    /**
     * Grants permission to retrieve information about Security Hub standards controls
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DescribeStandardsControls.html
     */
    toDescribeStandardsControls() {
        return this.to('DescribeStandardsControls');
    }
    /**
     * Grants permission to disable the findings importing for a Security Hub integrated product
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DisableImportFindingsForProduct.html
     */
    toDisableImportFindingsForProduct() {
        return this.to('DisableImportFindingsForProduct');
    }
    /**
     * Grants permission to remove the Security Hub administrator account for your organization
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - organizations:DeregisterDelegatedAdministrator
     * - organizations:DescribeOrganization
     * - organizations:ListDelegatedAdministrators
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DisableOrganizationAdminAccount.html
     */
    toDisableOrganizationAdminAccount() {
        return this.to('DisableOrganizationAdminAccount');
    }
    /**
     * Grants permission to disable Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DisableSecurityHub.html
     */
    toDisableSecurityHub() {
        return this.to('DisableSecurityHub');
    }
    /**
     * Grants permission to disable Security Hub V2
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DisableSecurityHubV2.html
     */
    toDisableSecurityHubV2() {
        return this.to('DisableSecurityHubV2');
    }
    /**
     * Grants permission to a Security Hub member account to disassociate from the associated administrator account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DisassociateFromAdministratorAccount.html
     */
    toDisassociateFromAdministratorAccount() {
        return this.to('DisassociateFromAdministratorAccount');
    }
    /**
     * Grants permission to a Security Hub member account to disassociate from the associated master account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DisassociateFromMasterAccount.html
     */
    toDisassociateFromMasterAccount() {
        return this.to('DisassociateFromMasterAccount');
    }
    /**
     * Grants permission to disassociate Security Hub member accounts from the associated administrator account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DisassociateMembers.html
     */
    toDisassociateMembers() {
        return this.to('DisassociateMembers');
    }
    /**
     * Grants permission to enable the findings importing for a Security Hub integrated product
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_EnableImportFindingsForProduct.html
     */
    toEnableImportFindingsForProduct() {
        return this.to('EnableImportFindingsForProduct');
    }
    /**
     * Grants permission to designate a Security Hub administrator account for your organization
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - organizations:DescribeOrganization
     * - organizations:EnableAWSServiceAccess
     * - organizations:ListAWSServiceAccessForOrganization
     * - organizations:ListDelegatedAdministrators
     * - organizations:RegisterDelegatedAdministrator
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_EnableOrganizationAdminAccount.html
     */
    toEnableOrganizationAdminAccount() {
        return this.to('EnableOrganizationAdminAccount');
    }
    /**
     * Grants permission to enable Security Hub
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_EnableSecurityHub.html
     */
    toEnableSecurityHub() {
        return this.to('EnableSecurityHub');
    }
    /**
     * Grants permission to enable Security Hub V2
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_EnableSecurityHubV2.html
     */
    toEnableSecurityHubV2() {
        return this.to('EnableSecurityHubV2');
    }
    /**
     * Grants permission to retrieve aggregated statistical data about the findings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetFindingStatisticsV2.html
     */
    toGetAdhocInsightResults() {
        return this.to('GetAdhocInsightResults');
    }
    /**
     * Grants permission to retrieve details about the Security Hub administrator account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetAdministratorAccount.html
     */
    toGetAdministratorAccount() {
        return this.to('GetAdministratorAccount');
    }
    /**
     * Grants permission to retrieve details for an aggregatorV2, which configures data aggregation across Regions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetAggregatorV2.html
     */
    toGetAggregatorV2() {
        return this.to('GetAggregatorV2');
    }
    /**
     * Grants permission to retrieve details for an automation rule V2 from Security Hub based on rule Amazon Resource Name (ARN)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/automation-rules
     */
    toGetAutomationRuleV2() {
        return this.to('GetAutomationRuleV2');
    }
    /**
     * Grants permission to get a complete overview of one configuration policy created by the calling account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetConfigurationPolicy.html
     */
    toGetConfigurationPolicy() {
        return this.to('GetConfigurationPolicy');
    }
    /**
     * Grants permission to retrieve information about a configuration policy associated with a member account or organizational unit of the calling account's organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetConfigurationPolicyAssociation.html
     */
    toGetConfigurationPolicyAssociation() {
        return this.to('GetConfigurationPolicyAssociation');
    }
    /**
     * Grants permission to retrieve details for a connector V2 from Security Hub based on connector id
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetConnectorV2.html
     */
    toGetConnectorV2() {
        return this.to('GetConnectorV2');
    }
    /**
     * Grants permission to retrieve a security score and counts of finding and control statuses for a security standard
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetControlFindingSummary.html
     */
    toGetControlFindingSummary() {
        return this.to('GetControlFindingSummary');
    }
    /**
     * Grants permission to retrieve a list of the standards that are enabled in Security Hub
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetEnabledStandards.html
     */
    toGetEnabledStandards() {
        return this.to('GetEnabledStandards');
    }
    /**
     * Grants permission to retrieve details for a finding aggregator, which configures finding aggregation across Regions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetFindingAggregator.html
     */
    toGetFindingAggregator() {
        return this.to('GetFindingAggregator');
    }
    /**
     * Grants permission to retrieve a list of finding history from Security Hub
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetFindingHistory.html
     */
    toGetFindingHistory() {
        return this.to('GetFindingHistory');
    }
    /**
     * Grants permission to retrieve a list of findings from Security Hub
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetFindingsV2.html
     */
    toGetFindings() {
        return this.to('GetFindings');
    }
    /**
     * Grants permission to retrieve the end date for an account's free trial of Security Hub
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetFreeTrialEndDate.html
     */
    toGetFreeTrialEndDate() {
        return this.to('GetFreeTrialEndDate');
    }
    /**
     * Grants permission to retrieve information about Security Hub usage during the free trial period
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetFreeTrialUsage.html
     */
    toGetFreeTrialUsage() {
        return this.to('GetFreeTrialUsage');
    }
    /**
     * Grants permission to retrieve an insight finding trend from Security Hub in order to generate a graph
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetInsightFindingTrend.html
     */
    toGetInsightFindingTrend() {
        return this.to('GetInsightFindingTrend');
    }
    /**
     * Grants permission to retrieve insight results from Security Hub
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetInsightResults.html
     */
    toGetInsightResults() {
        return this.to('GetInsightResults');
    }
    /**
     * Grants permission to retrieve Security Hub insights
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetInsights.html
     */
    toGetInsights() {
        return this.to('GetInsights');
    }
    /**
     * Grants permission to retrieve the count of Security Hub membership invitations sent to the account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetInvitationsCount.html
     */
    toGetInvitationsCount() {
        return this.to('GetInvitationsCount');
    }
    /**
     * Grants permission to retrieve details about the Security Hub master account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetMasterAccount.html
     */
    toGetMasterAccount() {
        return this.to('GetMasterAccount');
    }
    /**
     * Grants permission to retrieve the details of Security Hub member accounts
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetMembers.html
     */
    toGetMembers() {
        return this.to('GetMembers');
    }
    /**
     * Grants permission to retrieve aggregate statistics about resources
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetResourcesStatisticsV2.html
     */
    toGetResourcesStatisticsV2() {
        return this.to('GetResourcesStatisticsV2');
    }
    /**
     * Grants permission to retrieve a list of resources
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetResourcesV2.html
     */
    toGetResourcesV2() {
        return this.to('GetResourcesV2');
    }
    /**
     * Grants permission to get the definition details of a specific security control identified by ID
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - securityhub:DescribeStandardsControls
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetSecurityControlDefinition.html
     */
    toGetSecurityControlDefinition() {
        return this.to('GetSecurityControlDefinition');
    }
    /**
     * Grants permission to retrieve information about Security Hub usage by accounts
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetUsage.html
     */
    toGetUsage() {
        return this.to('GetUsage');
    }
    /**
     * Grants permission to invite other AWS accounts to become Security Hub member accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_InviteMembers.html
     */
    toInviteMembers() {
        return this.to('InviteMembers');
    }
    /**
     * Grants permission to retrieve a list of aggregatorsV2, which configures data aggregation across Regions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListAggregatorsV2.html
     */
    toListAggregatorsV2() {
        return this.to('ListAggregatorsV2');
    }
    /**
     * Grants permission to retrieve a list of automation rules and their metadata for the calling account from Security Hub
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/automation-rules
     */
    toListAutomationRules() {
        return this.to('ListAutomationRules');
    }
    /**
     * Grants permission to retrieve a list of automation rules V2 and their metadata for the calling account from Security Hub
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/automation-rules
     */
    toListAutomationRulesV2() {
        return this.to('ListAutomationRulesV2');
    }
    /**
     * Grants permission to list the summaries of all configuration policies created by the calling account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListConfigurationPolicies.html
     */
    toListConfigurationPolicies() {
        return this.to('ListConfigurationPolicies');
    }
    /**
     * Grants permission to retrieve information about all configuration policies associationed with all member accounts and organizational units of the calling account's organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListConfigurationPolicyAssociations.html
     */
    toListConfigurationPolicyAssociations() {
        return this.to('ListConfigurationPolicyAssociations');
    }
    /**
     * Grants permission to retrieve a list of connectors V2 and their metadata for the calling account from Security Hub
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListConnectorsV2.html
     */
    toListConnectorsV2() {
        return this.to('ListConnectorsV2');
    }
    /**
     * Grants permission to retrieve a list of controls for a standard, including the control IDs, statuses and finding counts
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListControlEvaluationSummaries.html
     */
    toListControlEvaluationSummaries() {
        return this.to('ListControlEvaluationSummaries');
    }
    /**
     * Grants permission to retrieve the Security Hub integrated products that are currently enabled
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListEnabledProductsForImport.html
     */
    toListEnabledProductsForImport() {
        return this.to('ListEnabledProductsForImport');
    }
    /**
     * Grants permission to retrieve a list of finding aggregators, which contain the cross-Region finding aggregation configuration
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListFindingAggregators.html
     */
    toListFindingAggregators() {
        return this.to('ListFindingAggregators');
    }
    /**
     * Grants permission to retrieve the Security Hub invitations sent to the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListInvitations.html
     */
    toListInvitations() {
        return this.to('ListInvitations');
    }
    /**
     * Grants permission to retrieve details about Security Hub member accounts associated with the administrator account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListMembers.html
     */
    toListMembers() {
        return this.to('ListMembers');
    }
    /**
     * Grants permission to list the Security Hub administrator accounts for your organization
     *
     * Access Level: List
     *
     * Dependent actions:
     * - organizations:DescribeOrganization
     * - organizations:ListDelegatedAdministrators
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListOrganizationAdminAccounts.html
     */
    toListOrganizationAdminAccounts() {
        return this.to('ListOrganizationAdminAccounts');
    }
    /**
     * Grants permission to retrieve a list of security control definitions, which contain details for security controls in the current region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListSecurityControlDefinitions.html
     */
    toListSecurityControlDefinitions() {
        return this.to('ListSecurityControlDefinitions');
    }
    /**
     * Grants permission to list the enablement status of a security control in standards
     *
     * Access Level: List
     *
     * Dependent actions:
     * - securityhub:DescribeStandardsControls
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListStandardsControlAssociations.html
     */
    toListStandardsControlAssociations() {
        return this.to('ListStandardsControlAssociations');
    }
    /**
     * Grants permission to list of tags associated with a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to use a custom action to send Security Hub findings to Amazon EventBridge
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_SendFindingEvents.html
     */
    toSendFindingEvents() {
        return this.to('SendFindingEvents');
    }
    /**
     * Grants permission to use a custom action to send Security Hub insights to Amazon EventBridge
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_SendInsightEvents.html
     */
    toSendInsightEvents() {
        return this.to('SendInsightEvents');
    }
    /**
     * Grants permission to associate a configuration policy with a member account or organizational unit in the calling account's organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_StartConfigurationPolicyAssociation.html
     */
    toStartConfigurationPolicyAssociation() {
        return this.to('StartConfigurationPolicyAssociation');
    }
    /**
     * Grants permission to remove a configuration policy association from a member account or organizational unit in the calling account's organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_StartConfigurationPolicyDisassociation.html
     */
    toStartConfigurationPolicyDisassociation() {
        return this.to('StartConfigurationPolicyDisassociation');
    }
    /**
     * Grants permission to add tags to a Security Hub resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from a Security Hub resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update custom actions in Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateActionTarget.html
     */
    toUpdateActionTarget() {
        return this.to('UpdateActionTarget');
    }
    /**
     * Grants permission to update an aggregatorV2, which configures data aggregation across Regions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateAggregatorV2.html
     */
    toUpdateAggregatorV2() {
        return this.to('UpdateAggregatorV2');
    }
    /**
     * Grants permission to update an automation rule V2 in Security Hub based on rule Amazon Resource Name (ARN) and input parameters
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/automation-rules
     */
    toUpdateAutomationRuleV2() {
        return this.to('UpdateAutomationRuleV2');
    }
    /**
     * Grants permission to update an existing configuration policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateConfigurationPolicy.html
     */
    toUpdateConfigurationPolicy() {
        return this.to('UpdateConfigurationPolicy');
    }
    /**
     * Grants permission to update a connector V2 in Security Hub based on connector id and input parameters
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateConnectorV2.html
     */
    toUpdateConnectorV2() {
        return this.to('UpdateConnectorV2');
    }
    /**
     * Grants permission to update a finding aggregator, which contains the cross-Region finding aggregation configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateFindingAggregator.html
     */
    toUpdateFindingAggregator() {
        return this.to('UpdateFindingAggregator');
    }
    /**
     * Grants permission to update Security Hub findings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateFindings.html
     */
    toUpdateFindings() {
        return this.to('UpdateFindings');
    }
    /**
     * Grants permission to update insights in Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateInsight.html
     */
    toUpdateInsight() {
        return this.to('UpdateInsight');
    }
    /**
     * Grants permission to update the organization configuration for Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateOrganizationConfiguration.html
     */
    toUpdateOrganizationConfiguration() {
        return this.to('UpdateOrganizationConfiguration');
    }
    /**
     * Grants permission to update properties of a specific security control identified by ID or ARN
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - securityhub:UpdateStandardsControl
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateSecurityControl.html
     */
    toUpdateSecurityControl() {
        return this.to('UpdateSecurityControl');
    }
    /**
     * Grants permission to update Security Hub configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateSecurityHubConfiguration.html
     */
    toUpdateSecurityHubConfiguration() {
        return this.to('UpdateSecurityHubConfiguration');
    }
    /**
     * Grants permission to update Security Hub standards controls
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateStandardsControl.html
     */
    toUpdateStandardsControl() {
        return this.to('UpdateStandardsControl');
    }
    /**
     * Adds a resource of type hub to the statement
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/securityhub-access.html#resources
     *
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onHub(account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:securityhub:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:hub/default`);
    }
    /**
     * Adds a resource of type hubv2 to the statement
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/securityhub-access.html#resources
     *
     * @param hubV2Id - Identifier for the hubV2Id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onHubv2(hubV2Id, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:securityhub:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:hubv2/${hubV2Id}`);
    }
    /**
     * Adds a resource of type product to the statement
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/securityhub-access.html#resources
     *
     * @param company - Identifier for the company.
     * @param productId - Identifier for the productId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onProduct(company, productId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:securityhub:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:product/${company}/${productId}`);
    }
    /**
     * Adds a resource of type finding-aggregator to the statement
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/securityhub-access.html#resources
     *
     * @param findingAggregatorId - Identifier for the findingAggregatorId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onFindingAggregator(findingAggregatorId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:securityhub:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:finding-aggregator/${findingAggregatorId}`);
    }
    /**
     * Adds a resource of type aggregatorv2 to the statement
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/securityhub-access.html#resources
     *
     * @param aggregatorV2Id - Identifier for the aggregatorV2Id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAggregatorv2(aggregatorV2Id, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:securityhub:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:aggregatorv2/${aggregatorV2Id}`);
    }
    /**
     * Adds a resource of type automation-rule to the statement
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/automation-rules
     *
     * @param automationRuleId - Identifier for the automationRuleId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAutomationRule(automationRuleId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:securityhub:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:automation-rule/${automationRuleId}`);
    }
    /**
     * Adds a resource of type automation-rulev2 to the statement
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/automation-rules
     *
     * @param automationRuleV2Id - Identifier for the automationRuleV2Id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAutomationRulev2(automationRuleV2Id, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:securityhub:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:automation-rulev2/${automationRuleV2Id}`);
    }
    /**
     * Adds a resource of type configuration-policy to the statement
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/central-configuration-intro.html
     *
     * @param configurationPolicyId - Identifier for the configurationPolicyId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConfigurationPolicy(configurationPolicyId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:securityhub:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:configuration-policy/${configurationPolicyId}`);
    }
    /**
     * Adds a resource of type connectorv2 to the statement
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/securityhub-access.html#resources
     *
     * @param connectorV2Id - Identifier for the connectorV2Id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConnectorv2(connectorV2Id, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:securityhub:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:connectorv2/${connectorV2Id}`);
    }
    /**
     * Filters access by actions based on the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateAutomationRule()
     * - .toCreateAutomationRuleV2()
     * - .toCreateConfigurationPolicy()
     * - .toCreateConnectorV2()
     * - .toEnableSecurityHub()
     * - .toEnableSecurityHubV2()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by actions based on tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - hub
     * - hubv2
     * - aggregatorv2
     * - automation-rule
     * - automation-rulev2
     * - configuration-policy
     * - connectorv2
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by actions based on the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateAutomationRule()
     * - .toCreateAutomationRuleV2()
     * - .toCreateConfigurationPolicy()
     * - .toCreateConnectorV2()
     * - .toEnableSecurityHub()
     * - .toEnableSecurityHubV2()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the specified fields and values in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-asffsyntaxpath
     *
     * Applies to actions:
     * - .toBatchUpdateFindings()
     *
     * @param aSFFSyntaxPath The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifASFFSyntaxPath(aSFFSyntaxPath, value, operator) {
        return this.if(`ASFFSyntaxPath/${aSFFSyntaxPath}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the specified fields and values in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-ocsfsyntaxpath
     *
     * Applies to actions:
     * - .toBatchUpdateFindings()
     *
     * @param oCSFSyntaxPath The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifOCSFSyntaxPath(oCSFSyntaxPath, value, operator) {
        return this.if(`OCSFSyntaxPath/${oCSFSyntaxPath}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the AwsAccountId field that is specified in the request
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/securityhub-access.html#conditions
     *
     * Applies to actions:
     * - .toBatchImportFindings()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTargetAccount(value, operator) {
        return this.if(`TargetAccount`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [securityhub](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssecurityhub.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'securityhub';
        this.accessLevelList = {
            Write: [
                'AcceptAdministratorInvitation',
                'AcceptInvitation',
                'BatchDeleteAutomationRules',
                'BatchDisableStandards',
                'BatchEnableStandards',
                'BatchImportFindings',
                'BatchUpdateAutomationRules',
                'BatchUpdateFindings',
                'BatchUpdateStandardsControlAssociations',
                'ConnectorRegistrationsV2',
                'CreateActionTarget',
                'CreateAggregatorV2',
                'CreateAutomationRule',
                'CreateAutomationRuleV2',
                'CreateConfigurationPolicy',
                'CreateConnectorV2',
                'CreateFindingAggregator',
                'CreateInsight',
                'CreateMembers',
                'CreateTicketV2',
                'DeclineInvitations',
                'DeleteActionTarget',
                'DeleteAggregatorV2',
                'DeleteAutomationRuleV2',
                'DeleteConfigurationPolicy',
                'DeleteConnectorV2',
                'DeleteFindingAggregator',
                'DeleteInsight',
                'DeleteInvitations',
                'DeleteMembers',
                'DisableImportFindingsForProduct',
                'DisableOrganizationAdminAccount',
                'DisableSecurityHub',
                'DisableSecurityHubV2',
                'DisassociateFromAdministratorAccount',
                'DisassociateFromMasterAccount',
                'DisassociateMembers',
                'EnableImportFindingsForProduct',
                'EnableOrganizationAdminAccount',
                'EnableSecurityHub',
                'EnableSecurityHubV2',
                'InviteMembers',
                'StartConfigurationPolicyAssociation',
                'StartConfigurationPolicyDisassociation',
                'UpdateActionTarget',
                'UpdateAggregatorV2',
                'UpdateAutomationRuleV2',
                'UpdateConfigurationPolicy',
                'UpdateConnectorV2',
                'UpdateFindingAggregator',
                'UpdateFindings',
                'UpdateInsight',
                'UpdateOrganizationConfiguration',
                'UpdateSecurityControl',
                'UpdateSecurityHubConfiguration',
                'UpdateStandardsControl'
            ],
            Read: [
                'BatchGetAutomationRules',
                'BatchGetConfigurationPolicyAssociations',
                'BatchGetControlEvaluations',
                'BatchGetSecurityControls',
                'BatchGetStandardsControlAssociations',
                'DescribeActionTargets',
                'DescribeHub',
                'DescribeOrganizationConfiguration',
                'DescribeProducts',
                'DescribeProductsV2',
                'DescribeSecurityHubV2',
                'DescribeStandards',
                'DescribeStandardsControls',
                'GetAdhocInsightResults',
                'GetAdministratorAccount',
                'GetAggregatorV2',
                'GetAutomationRuleV2',
                'GetConfigurationPolicy',
                'GetConfigurationPolicyAssociation',
                'GetConnectorV2',
                'GetControlFindingSummary',
                'GetFindingAggregator',
                'GetFindingHistory',
                'GetFindings',
                'GetFreeTrialEndDate',
                'GetFreeTrialUsage',
                'GetInsightFindingTrend',
                'GetInsightResults',
                'GetInvitationsCount',
                'GetMasterAccount',
                'GetMembers',
                'GetResourcesStatisticsV2',
                'GetResourcesV2',
                'GetSecurityControlDefinition',
                'GetUsage',
                'ListControlEvaluationSummaries',
                'ListTagsForResource',
                'SendFindingEvents',
                'SendInsightEvents'
            ],
            List: [
                'GetEnabledStandards',
                'GetInsights',
                'ListAggregatorsV2',
                'ListAutomationRules',
                'ListAutomationRulesV2',
                'ListConfigurationPolicies',
                'ListConfigurationPolicyAssociations',
                'ListConnectorsV2',
                'ListEnabledProductsForImport',
                'ListFindingAggregators',
                'ListInvitations',
                'ListMembers',
                'ListOrganizationAdminAccounts',
                'ListSecurityControlDefinitions',
                'ListStandardsControlAssociations'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Securityhub = Securityhub;
//# sourceMappingURL=data:application/json;base64,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