"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecurityIr = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [security-ir](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssecurityincidentresponse.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class SecurityIr extends shared_1.PolicyStatement {
    /**
     * Grants permission to get member account details in batch
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_BatchGetMemberAccountDetails.html
     */
    toBatchGetMemberAccountDetails() {
        return this.to('BatchGetMemberAccountDetails');
    }
    /**
     * Grants permission to cancel a membership
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_CancelMembership.html
     */
    toCancelMembership() {
        return this.to('CancelMembership');
    }
    /**
     * Grants permission to close a case
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_CloseCase.html
     */
    toCloseCase() {
        return this.to('CloseCase');
    }
    /**
     * Grants permission to create a case
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_CreateCase.html
     */
    toCreateCase() {
        return this.to('CreateCase');
    }
    /**
     * Grants permission to create a case comment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_CreateCaseComment.html
     */
    toCreateCaseComment() {
        return this.to('CreateCaseComment');
    }
    /**
     * Grants permission to create a membership
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - organizations:DescribeOrganization
     * - organizations:ListDelegatedAdministrators
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_CreateMembership.html
     */
    toCreateMembership() {
        return this.to('CreateMembership');
    }
    /**
     * Grants permission to get a case
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_GetCase.html
     */
    toGetCase() {
        return this.to('GetCase');
    }
    /**
     * Grants permission to get a case attachment download URL
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_GetCaseAttachmentDownloadUrl.html
     */
    toGetCaseAttachmentDownloadUrl() {
        return this.to('GetCaseAttachmentDownloadUrl');
    }
    /**
     * Grants permission to get a case attachment upload URL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_GetCaseAttachmentUploadUrl.html
     */
    toGetCaseAttachmentUploadUrl() {
        return this.to('GetCaseAttachmentUploadUrl');
    }
    /**
     * Grants permission to get a membership
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_GetMembership.html
     */
    toGetMembership() {
        return this.to('GetMembership');
    }
    /**
     * Grants permission to list case edits
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_ListCaseEdits.html
     */
    toListCaseEdits() {
        return this.to('ListCaseEdits');
    }
    /**
     * Grants permission to list cases
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_ListCases.html
     */
    toListCases() {
        return this.to('ListCases');
    }
    /**
     * Grants permission to list case comments
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_ListComments.html
     */
    toListComments() {
        return this.to('ListComments');
    }
    /**
     * Grants permission to list memberships
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_ListMemberships.html
     */
    toListMemberships() {
        return this.to('ListMemberships');
    }
    /**
     * Grants permission to list the tags attached to the specified resource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to add tags to the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a case
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_UpdateCase.html
     */
    toUpdateCase() {
        return this.to('UpdateCase');
    }
    /**
     * Grants permission to update a case comment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_UpdateCaseComment.html
     */
    toUpdateCaseComment() {
        return this.to('UpdateCaseComment');
    }
    /**
     * Grants permission to update a case status
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_UpdateCaseStatus.html
     */
    toUpdateCaseStatus() {
        return this.to('UpdateCaseStatus');
    }
    /**
     * Grants permission to update memberships
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - organizations:DescribeOrganizationalUnit
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_UpdateMembership.html
     */
    toUpdateMembership() {
        return this.to('UpdateMembership');
    }
    /**
     * Grants permission to update case resolver type
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/security-ir/latest/APIReference/API_UpdateResolverType.html
     */
    toUpdateResolverType() {
        return this.to('UpdateResolverType');
    }
    /**
     * Adds a resource of type case to the statement
     *
     * https://docs.aws.amazon.com/security-ir/latest/userguide/cases.html
     *
     * @param caseId - Identifier for the caseId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCase(caseId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:security-ir:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:case/${caseId}`);
    }
    /**
     * Adds a resource of type membership to the statement
     *
     * https://docs.aws.amazon.com/security-ir/latest/userguide/select-a-membership-account.html
     *
     * @param membershipId - Identifier for the membershipId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMembership(membershipId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:security-ir:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:membership/${membershipId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateCase()
     * - .toCreateMembership()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toListTagsForResource()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * Applies to resource types:
     * - case
     * - membership
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateCase()
     * - .toCreateMembership()
     * - .toListTagsForResource()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [security-ir](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssecurityincidentresponse.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'security-ir';
        this.accessLevelList = {
            Read: [
                'BatchGetMemberAccountDetails',
                'GetCase',
                'GetCaseAttachmentDownloadUrl',
                'GetMembership',
                'ListCaseEdits',
                'ListComments',
                'ListTagsForResource'
            ],
            Write: [
                'CancelMembership',
                'CloseCase',
                'CreateCase',
                'CreateCaseComment',
                'CreateMembership',
                'GetCaseAttachmentUploadUrl',
                'UpdateCase',
                'UpdateCaseComment',
                'UpdateCaseStatus',
                'UpdateMembership',
                'UpdateResolverType'
            ],
            List: [
                'ListCases',
                'ListMemberships'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.SecurityIr = SecurityIr;
//# sourceMappingURL=data:application/json;base64,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