"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Securitylake = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [securitylake](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonsecuritylake.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Securitylake extends shared_1.PolicyStatement {
    /**
     * Grants permission to enable any source type in any region for accounts that are either part of a trusted organization or standalone account
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - glue:CreateDatabase
     * - glue:CreateTable
     * - glue:GetDatabase
     * - glue:GetTable
     * - iam:CreateServiceLinkedRole
     * - kms:CreateGrant
     * - kms:DescribeKey
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_CreateAwsLogSource.html
     */
    toCreateAwsLogSource() {
        return this.to('CreateAwsLogSource');
    }
    /**
     * Grants permission to add a custom source
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - glue:CreateCrawler
     * - glue:CreateDatabase
     * - glue:CreateTable
     * - glue:StartCrawlerSchedule
     * - iam:DeleteRolePolicy
     * - iam:GetRole
     * - iam:PassRole
     * - iam:PutRolePolicy
     * - kms:CreateGrant
     * - kms:DescribeKey
     * - kms:GenerateDataKey
     * - lakeformation:GrantPermissions
     * - lakeformation:RegisterResource
     * - s3:ListBucket
     * - s3:PutObject
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_CreateCustomLogSource.html
     */
    toCreateCustomLogSource() {
        return this.to('CreateCustomLogSource');
    }
    /**
     * Grants permission to create a new security data lake
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - events:PutRule
     * - events:PutTargets
     * - iam:CreateServiceLinkedRole
     * - iam:DeleteRolePolicy
     * - iam:GetRole
     * - iam:ListAttachedRolePolicies
     * - iam:PassRole
     * - iam:PutRolePolicy
     * - kms:CreateGrant
     * - kms:DescribeKey
     * - lakeformation:GetDataLakeSettings
     * - lakeformation:PutDataLakeSettings
     * - lambda:AddPermission
     * - lambda:CreateEventSourceMapping
     * - lambda:CreateFunction
     * - organizations:DescribeOrganization
     * - organizations:ListAccounts
     * - organizations:ListDelegatedServicesForAccount
     * - s3:CreateBucket
     * - s3:GetObject
     * - s3:GetObjectVersion
     * - s3:ListBucket
     * - s3:PutBucketPolicy
     * - s3:PutBucketPublicAccessBlock
     * - s3:PutBucketVersioning
     * - sqs:CreateQueue
     * - sqs:GetQueueAttributes
     * - sqs:SetQueueAttributes
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_CreateDataLake.html
     */
    toCreateDataLake() {
        return this.to('CreateDataLake');
    }
    /**
     * Grants permission to get instant notifications about exceptions. Subscribes to the SNS topics for exception notifications
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_CreateDataLakeExceptionSubscription.html
     */
    toCreateDataLakeExceptionSubscription() {
        return this.to('CreateDataLakeExceptionSubscription');
    }
    /**
     * Grants permission to automatically enable Amazon Security Lake for new member accounts in your organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_CreateDataLakeOrganizationConfiguration.html
     */
    toCreateDataLakeOrganizationConfiguration() {
        return this.to('CreateDataLakeOrganizationConfiguration');
    }
    /**
     * Grants permission to create a subscriber
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:CreateRole
     * - iam:DeleteRolePolicy
     * - iam:GetRole
     * - iam:PutRolePolicy
     * - lakeformation:GrantPermissions
     * - lakeformation:ListPermissions
     * - lakeformation:RegisterResource
     * - lakeformation:RevokePermissions
     * - ram:GetResourceShareAssociations
     * - ram:GetResourceShares
     * - ram:UpdateResourceShare
     * - s3:PutObject
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_CreateSubscriber.html
     */
    toCreateSubscriber() {
        return this.to('CreateSubscriber');
    }
    /**
     * Grants permission to create a webhook invocation to notify a client when there is new data in the data lake
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - events:CreateApiDestination
     * - events:CreateConnection
     * - events:DescribeRule
     * - events:ListApiDestinations
     * - events:ListConnections
     * - events:PutRule
     * - events:PutTargets
     * - iam:DeleteRolePolicy
     * - iam:GetRole
     * - iam:PassRole
     * - s3:GetBucketNotification
     * - s3:PutBucketNotification
     * - sqs:CreateQueue
     * - sqs:DeleteQueue
     * - sqs:GetQueueAttributes
     * - sqs:GetQueueUrl
     * - sqs:SetQueueAttributes
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_CreateSubscriberNotification.html
     */
    toCreateSubscriberNotification() {
        return this.to('CreateSubscriberNotification');
    }
    /**
     * Grants permission to disable any source type in any region for accounts that are part of a trusted organization or standalone accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_DeleteAwsLogSource.html
     */
    toDeleteAwsLogSource() {
        return this.to('DeleteAwsLogSource');
    }
    /**
     * Grants permission to remove a custom source
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - glue:StopCrawlerSchedule
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_DeleteCustomLogSource.html
     */
    toDeleteCustomLogSource() {
        return this.to('DeleteCustomLogSource');
    }
    /**
     * Grants permission to delete security data lake
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - organizations:DescribeOrganization
     * - organizations:ListDelegatedAdministrators
     * - organizations:ListDelegatedServicesForAccount
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_DeleteDataLake.html
     */
    toDeleteDataLake() {
        return this.to('DeleteDataLake');
    }
    /**
     * Grants permission to unsubscribe from SNS topics for exception notifications. Removes exception notifications for the SNS topic
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_DeleteDataLakeExceptionSubscription.html
     */
    toDeleteDataLakeExceptionSubscription() {
        return this.to('DeleteDataLakeExceptionSubscription');
    }
    /**
     * Grants permission to remove the automatic enablement of Amazon Security Lake access for new organization accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_DeleteDataLakeOrganizationConfiguration.html
     */
    toDeleteDataLakeOrganizationConfiguration() {
        return this.to('DeleteDataLakeOrganizationConfiguration');
    }
    /**
     * Grants permission to delete the specified subscriber
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - events:DeleteApiDestination
     * - events:DeleteConnection
     * - events:DeleteRule
     * - events:DescribeRule
     * - events:ListApiDestinations
     * - events:ListTargetsByRule
     * - events:RemoveTargets
     * - iam:DeleteRole
     * - iam:DeleteRolePolicy
     * - iam:GetRole
     * - iam:ListRolePolicies
     * - lakeformation:ListPermissions
     * - lakeformation:RevokePermissions
     * - sqs:DeleteQueue
     * - sqs:GetQueueUrl
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_DeleteSubscriber.html
     */
    toDeleteSubscriber() {
        return this.to('DeleteSubscriber');
    }
    /**
     * Grants permission to remove a webhook invocation to notify a client when there is new data in the data lake
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - events:DeleteApiDestination
     * - events:DeleteConnection
     * - events:DeleteRule
     * - events:DescribeRule
     * - events:ListApiDestinations
     * - events:ListTargetsByRule
     * - events:RemoveTargets
     * - iam:DeleteRole
     * - iam:DeleteRolePolicy
     * - iam:GetRole
     * - iam:ListRolePolicies
     * - lakeformation:RevokePermissions
     * - sqs:DeleteQueue
     * - sqs:GetQueueUrl
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_DeleteSubscriberNotification.html
     */
    toDeleteSubscriberNotification() {
        return this.to('DeleteSubscriberNotification');
    }
    /**
     * Grants permission to remove the Delegated Administrator account and disable Amazon Security Lake as a service for this organization
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - organizations:DeregisterDelegatedAdministrator
     * - organizations:DescribeOrganization
     * - organizations:ListDelegatedServicesForAccount
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_DeregisterDataLakeDelegatedAdministrator.html
     */
    toDeregisterDataLakeDelegatedAdministrator() {
        return this.to('DeregisterDataLakeDelegatedAdministrator');
    }
    /**
     * Grants permission to query the protocol and endpoint that were provided when subscribing to SNS topics for exception notifications
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_GetDataLakeExceptionSubscription.html
     */
    toGetDataLakeExceptionSubscription() {
        return this.to('GetDataLakeExceptionSubscription');
    }
    /**
     * Grants permission to get an organization's configuration setting for automatically enabling Amazon Security Lake access for new organization accounts
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - organizations:DescribeOrganization
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_GetDataLakeOrganizationConfiguration.html
     */
    toGetDataLakeOrganizationConfiguration() {
        return this.to('GetDataLakeOrganizationConfiguration');
    }
    /**
     * Grants permission to get a static snapshot of the security data lake in the current region. The snapshot includes enabled accounts and log sources
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_GetDataLakeSources.html
     */
    toGetDataLakeSources() {
        return this.to('GetDataLakeSources');
    }
    /**
     * Grants permission to get information about subscriber that is already created
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_GetSubscriber.html
     */
    toGetSubscriber() {
        return this.to('GetSubscriber');
    }
    /**
     * Grants permission to get the list of all non-retryable failures
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_ListDataLakeExceptions.html
     */
    toListDataLakeExceptions() {
        return this.to('ListDataLakeExceptions');
    }
    /**
     * Grants permission to list information about the security data lakes
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_ListDataLakes.html
     */
    toListDataLakes() {
        return this.to('ListDataLakes');
    }
    /**
     * Grants permission to view the enabled accounts. You can view the enabled sources in the enabled regions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_ListLogSources.html
     */
    toListLogSources() {
        return this.to('ListLogSources');
    }
    /**
     * Grants permission to list all subscribers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_ListSubscribers.html
     */
    toListSubscribers() {
        return this.to('ListSubscribers');
    }
    /**
     * Grants permission to list all tags for the resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to designate an account as the Amazon Security Lake administrator account for the organization
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - organizations:DescribeOrganization
     * - organizations:EnableAWSServiceAccess
     * - organizations:ListDelegatedAdministrators
     * - organizations:ListDelegatedServicesForAccount
     * - organizations:RegisterDelegatedAdministrator
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_RegisterDataLakeDelegatedAdministrator.html
     */
    toRegisterDataLakeDelegatedAdministrator() {
        return this.to('RegisterDataLakeDelegatedAdministrator');
    }
    /**
     * Grants permission to add tags to the resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from the resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a security data lake
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - events:PutRule
     * - events:PutTargets
     * - iam:CreateServiceLinkedRole
     * - iam:DeleteRolePolicy
     * - iam:GetRole
     * - iam:ListAttachedRolePolicies
     * - iam:PutRolePolicy
     * - kms:CreateGrant
     * - kms:DescribeKey
     * - lakeformation:GetDataLakeSettings
     * - lakeformation:PutDataLakeSettings
     * - lambda:AddPermission
     * - lambda:CreateEventSourceMapping
     * - lambda:CreateFunction
     * - organizations:DescribeOrganization
     * - organizations:ListDelegatedServicesForAccount
     * - s3:CreateBucket
     * - s3:GetObject
     * - s3:GetObjectVersion
     * - s3:ListBucket
     * - s3:PutBucketPolicy
     * - s3:PutBucketPublicAccessBlock
     * - s3:PutBucketVersioning
     * - sqs:CreateQueue
     * - sqs:GetQueueAttributes
     * - sqs:SetQueueAttributes
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_UpdateDataLake.html
     */
    toUpdateDataLake() {
        return this.to('UpdateDataLake');
    }
    /**
     * Grants permission to update subscriptions to the SNS topics for exception notifications
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_UpdateDataLakeExceptionSubscription.html
     */
    toUpdateDataLakeExceptionSubscription() {
        return this.to('UpdateDataLakeExceptionSubscription');
    }
    /**
     * Grants permission to update subscriber
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - events:CreateApiDestination
     * - events:CreateConnection
     * - events:DescribeRule
     * - events:ListApiDestinations
     * - events:ListConnections
     * - events:PutRule
     * - events:PutTargets
     * - iam:DeleteRolePolicy
     * - iam:GetRole
     * - iam:PutRolePolicy
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_UpdateSubscriber.html
     */
    toUpdateSubscriber() {
        return this.to('UpdateSubscriber');
    }
    /**
     * Grants permission to update a webhook invocation to notify a client when there is new data in the data lake
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - events:CreateApiDestination
     * - events:CreateConnection
     * - events:DescribeRule
     * - events:ListApiDestinations
     * - events:ListConnections
     * - events:PutRule
     * - events:PutTargets
     * - iam:CreateServiceLinkedRole
     * - iam:DeleteRolePolicy
     * - iam:GetRole
     * - iam:PassRole
     * - iam:PutRolePolicy
     * - s3:CreateBucket
     * - s3:GetBucketNotification
     * - s3:ListBucket
     * - s3:PutBucketNotification
     * - s3:PutBucketPolicy
     * - s3:PutBucketPublicAccessBlock
     * - s3:PutBucketVersioning
     * - s3:PutLifecycleConfiguration
     * - sqs:CreateQueue
     * - sqs:DeleteQueue
     * - sqs:GetQueueAttributes
     * - sqs:GetQueueUrl
     * - sqs:SetQueueAttributes
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_UpdateSubscriberNotification.html
     */
    toUpdateSubscriberNotification() {
        return this.to('UpdateSubscriberNotification');
    }
    /**
     * Adds a resource of type data-lake to the statement
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_DataLakeResource.html
     *
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     */
    onDataLake(account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:securitylake:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:data-lake/default`);
    }
    /**
     * Adds a resource of type subscriber to the statement
     *
     * https://docs.aws.amazon.com/security-lake/latest/APIReference/API_SubscriberResource.html
     *
     * @param subscriberId - Identifier for the subscriberId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     */
    onSubscriber(subscriberId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:securitylake:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:subscriber/${subscriberId}`);
    }
    /**
     * Filters access by tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateDataLake()
     * - .toCreateSubscriber()
     * - .toTagResource()
     *
     * Applies to resource types:
     * - data-lake
     * - subscriber
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a tag key and value pair of a resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toCreateSubscriber()
     *
     * Applies to resource types:
     * - data-lake
     * - subscriber
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateDataLake()
     * - .toCreateSubscriber()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [securitylake](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonsecuritylake.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'securitylake';
        this.accessLevelList = {
            Write: [
                'CreateAwsLogSource',
                'CreateCustomLogSource',
                'CreateDataLake',
                'CreateDataLakeExceptionSubscription',
                'CreateDataLakeOrganizationConfiguration',
                'CreateSubscriber',
                'CreateSubscriberNotification',
                'DeleteAwsLogSource',
                'DeleteCustomLogSource',
                'DeleteDataLake',
                'DeleteDataLakeExceptionSubscription',
                'DeleteDataLakeOrganizationConfiguration',
                'DeleteSubscriber',
                'DeleteSubscriberNotification',
                'DeregisterDataLakeDelegatedAdministrator',
                'RegisterDataLakeDelegatedAdministrator',
                'UpdateDataLake',
                'UpdateDataLakeExceptionSubscription',
                'UpdateSubscriber',
                'UpdateSubscriberNotification'
            ],
            Read: [
                'GetDataLakeExceptionSubscription',
                'GetDataLakeOrganizationConfiguration',
                'GetDataLakeSources',
                'GetSubscriber'
            ],
            List: [
                'ListDataLakeExceptions',
                'ListDataLakes',
                'ListLogSources',
                'ListSubscribers',
                'ListTagsForResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Securitylake = Securitylake;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic2VjdXJpdHlsYWtlLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsic2VjdXJpdHlsYWtlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUNBLHlDQUF5RDtBQUd6RDs7OztHQUlHO0FBQ0gsTUFBYSxZQUFhLFNBQVEsd0JBQWU7SUFHL0M7Ozs7Ozs7Ozs7Ozs7OztPQWVHO0lBQ0ksb0JBQW9CO1FBQ3pCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO0lBQ3ZDLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7T0F1Qkc7SUFDSSx1QkFBdUI7UUFDNUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7SUFDMUMsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O09Bd0NHO0lBQ0ksZ0JBQWdCO1FBQ3JCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO0lBQ25DLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQ0FBcUM7UUFDMUMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFDQUFxQyxDQUFDLENBQUM7SUFDeEQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHlDQUF5QztRQUM5QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMseUNBQXlDLENBQUMsQ0FBQztJQUM1RCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7T0F5Qkc7SUFDSSxrQkFBa0I7UUFDdkIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFrQixDQUFDLENBQUM7SUFDckMsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O09BeUJHO0lBQ0ksOEJBQThCO1FBQ25DLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO0lBQ2pELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQkFBb0I7UUFDekIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG9CQUFvQixDQUFDLENBQUM7SUFDdkMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLHVCQUF1QjtRQUM1QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsdUJBQXVCLENBQUMsQ0FBQztJQUMxQyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxnQkFBZ0I7UUFDckIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdCQUFnQixDQUFDLENBQUM7SUFDbkMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFDQUFxQztRQUMxQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUNBQXFDLENBQUMsQ0FBQztJQUN4RCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kseUNBQXlDO1FBQzlDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx5Q0FBeUMsQ0FBQyxDQUFDO0lBQzVELENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7T0F1Qkc7SUFDSSxrQkFBa0I7UUFDdkIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFrQixDQUFDLENBQUM7SUFDckMsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O09Bc0JHO0lBQ0ksOEJBQThCO1FBQ25DLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO0lBQ2pELENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLDBDQUEwQztRQUMvQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMENBQTBDLENBQUMsQ0FBQztJQUM3RCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0NBQWtDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQ0FBa0MsQ0FBQyxDQUFDO0lBQ3JELENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxzQ0FBc0M7UUFDM0MsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHNDQUFzQyxDQUFDLENBQUM7SUFDekQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9CQUFvQjtRQUN6QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0JBQW9CLENBQUMsQ0FBQztJQUN2QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZTtRQUNwQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZSxDQUFDLENBQUM7SUFDbEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHdCQUF3QjtRQUM3QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLENBQUMsQ0FBQztJQUMzQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZTtRQUNwQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZSxDQUFDLENBQUM7SUFDbEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztJQUNuQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksaUJBQWlCO1FBQ3RCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO0lBQ3BDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7OztPQWNHO0lBQ0ksd0NBQXdDO1FBQzdDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx3Q0FBd0MsQ0FBQyxDQUFDO0lBQzNELENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksYUFBYTtRQUNsQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsYUFBYSxDQUFDLENBQUM7SUFDaEMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLGVBQWU7UUFDcEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxDQUFDO0lBQ2xDLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztPQWtDRztJQUNJLGdCQUFnQjtRQUNyQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztJQUNuQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUNBQXFDO1FBQzFDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQ0FBcUMsQ0FBQyxDQUFDO0lBQ3hELENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7O09Ba0JHO0lBQ0ksa0JBQWtCO1FBQ3ZCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO0lBQ3JDLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O09BaUNHO0lBQ0ksOEJBQThCO1FBQ25DLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO0lBQ2pELENBQUM7SUE0Q0Q7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksVUFBVSxDQUFDLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3JFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGlCQUFrQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsb0JBQW9CLENBQUMsQ0FBQztJQUNySyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLFlBQVksQ0FBQyxZQUFvQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQzdGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGlCQUFrQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsZUFBZ0IsWUFBYSxFQUFFLENBQUMsQ0FBQztJQUNoTCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7O09BaUJHO0lBQ0ksZUFBZSxDQUFDLE1BQWMsRUFBRSxLQUF3QixFQUFFLFFBQTRCO1FBQzNGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBbUIsTUFBTyxFQUFFLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNoRixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7OztPQWVHO0lBQ0ksZ0JBQWdCLENBQUMsTUFBYyxFQUFFLEtBQXdCLEVBQUUsUUFBNEI7UUFDNUYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFvQixNQUFPLEVBQUUsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2pGLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0ksWUFBWSxDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDeEUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGFBQWEsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2pFLENBQUM7SUFFRDs7O09BR0c7SUFDSCxZQUFZLEtBQWdDO1FBQzFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQztRQTFzQlIsa0JBQWEsR0FBRyxjQUFjLENBQUM7UUE0akI1QixvQkFBZSxHQUFvQjtZQUMzQyxLQUFLLEVBQUU7Z0JBQ0wsb0JBQW9CO2dCQUNwQix1QkFBdUI7Z0JBQ3ZCLGdCQUFnQjtnQkFDaEIscUNBQXFDO2dCQUNyQyx5Q0FBeUM7Z0JBQ3pDLGtCQUFrQjtnQkFDbEIsOEJBQThCO2dCQUM5QixvQkFBb0I7Z0JBQ3BCLHVCQUF1QjtnQkFDdkIsZ0JBQWdCO2dCQUNoQixxQ0FBcUM7Z0JBQ3JDLHlDQUF5QztnQkFDekMsa0JBQWtCO2dCQUNsQiw4QkFBOEI7Z0JBQzlCLDBDQUEwQztnQkFDMUMsd0NBQXdDO2dCQUN4QyxnQkFBZ0I7Z0JBQ2hCLHFDQUFxQztnQkFDckMsa0JBQWtCO2dCQUNsQiw4QkFBOEI7YUFDL0I7WUFDRCxJQUFJLEVBQUU7Z0JBQ0osa0NBQWtDO2dCQUNsQyxzQ0FBc0M7Z0JBQ3RDLG9CQUFvQjtnQkFDcEIsZUFBZTthQUNoQjtZQUNELElBQUksRUFBRTtnQkFDSix3QkFBd0I7Z0JBQ3hCLGVBQWU7Z0JBQ2YsZ0JBQWdCO2dCQUNoQixpQkFBaUI7Z0JBQ2pCLHFCQUFxQjthQUN0QjtZQUNELE9BQU8sRUFBRTtnQkFDUCxhQUFhO2dCQUNiLGVBQWU7YUFDaEI7U0FDRixDQUFDO0lBdUdGLENBQUM7Q0FDRjtBQTdzQkQsb0NBNnNCQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IEFjY2Vzc0xldmVsTGlzdCB9IGZyb20gJy4uLy4uL3NoYXJlZC9hY2Nlc3MtbGV2ZWwnO1xuaW1wb3J0IHsgUG9saWN5U3RhdGVtZW50LCBPcGVyYXRvciB9IGZyb20gJy4uLy4uL3NoYXJlZCc7XG5pbXBvcnQgeyBhd3NfaWFtIGFzIGlhbSB9IGZyb20gXCJhd3MtY2RrLWxpYlwiO1xuXG4vKipcbiAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbc2VjdXJpdHlsYWtlXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZS1hdXRob3JpemF0aW9uL2xhdGVzdC9yZWZlcmVuY2UvbGlzdF9hbWF6b25zZWN1cml0eWxha2UuaHRtbCkuXG4gKlxuICogQHBhcmFtIHNpZCBbU0lEXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX3NpZC5odG1sKSBvZiB0aGUgc3RhdGVtZW50XG4gKi9cbmV4cG9ydCBjbGFzcyBTZWN1cml0eWxha2UgZXh0ZW5kcyBQb2xpY3lTdGF0ZW1lbnQge1xuICBwdWJsaWMgc2VydmljZVByZWZpeCA9ICdzZWN1cml0eWxha2UnO1xuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBlbmFibGUgYW55IHNvdXJjZSB0eXBlIGluIGFueSByZWdpb24gZm9yIGFjY291bnRzIHRoYXQgYXJlIGVpdGhlciBwYXJ0IG9mIGEgdHJ1c3RlZCBvcmdhbml6YXRpb24gb3Igc3RhbmRhbG9uZSBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gZ2x1ZTpDcmVhdGVEYXRhYmFzZVxuICAgKiAtIGdsdWU6Q3JlYXRlVGFibGVcbiAgICogLSBnbHVlOkdldERhdGFiYXNlXG4gICAqIC0gZ2x1ZTpHZXRUYWJsZVxuICAgKiAtIGlhbTpDcmVhdGVTZXJ2aWNlTGlua2VkUm9sZVxuICAgKiAtIGttczpDcmVhdGVHcmFudFxuICAgKiAtIGttczpEZXNjcmliZUtleVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VjdXJpdHktbGFrZS9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVBd3NMb2dTb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlQXdzTG9nU291cmNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVBd3NMb2dTb3VyY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBhZGQgYSBjdXN0b20gc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gZ2x1ZTpDcmVhdGVDcmF3bGVyXG4gICAqIC0gZ2x1ZTpDcmVhdGVEYXRhYmFzZVxuICAgKiAtIGdsdWU6Q3JlYXRlVGFibGVcbiAgICogLSBnbHVlOlN0YXJ0Q3Jhd2xlclNjaGVkdWxlXG4gICAqIC0gaWFtOkRlbGV0ZVJvbGVQb2xpY3lcbiAgICogLSBpYW06R2V0Um9sZVxuICAgKiAtIGlhbTpQYXNzUm9sZVxuICAgKiAtIGlhbTpQdXRSb2xlUG9saWN5XG4gICAqIC0ga21zOkNyZWF0ZUdyYW50XG4gICAqIC0ga21zOkRlc2NyaWJlS2V5XG4gICAqIC0ga21zOkdlbmVyYXRlRGF0YUtleVxuICAgKiAtIGxha2Vmb3JtYXRpb246R3JhbnRQZXJtaXNzaW9uc1xuICAgKiAtIGxha2Vmb3JtYXRpb246UmVnaXN0ZXJSZXNvdXJjZVxuICAgKiAtIHMzOkxpc3RCdWNrZXRcbiAgICogLSBzMzpQdXRPYmplY3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlY3VyaXR5LWxha2UvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlQ3VzdG9tTG9nU291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUN1c3RvbUxvZ1NvdXJjZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlQ3VzdG9tTG9nU291cmNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgbmV3IHNlY3VyaXR5IGRhdGEgbGFrZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIGV2ZW50czpQdXRSdWxlXG4gICAqIC0gZXZlbnRzOlB1dFRhcmdldHNcbiAgICogLSBpYW06Q3JlYXRlU2VydmljZUxpbmtlZFJvbGVcbiAgICogLSBpYW06RGVsZXRlUm9sZVBvbGljeVxuICAgKiAtIGlhbTpHZXRSb2xlXG4gICAqIC0gaWFtOkxpc3RBdHRhY2hlZFJvbGVQb2xpY2llc1xuICAgKiAtIGlhbTpQYXNzUm9sZVxuICAgKiAtIGlhbTpQdXRSb2xlUG9saWN5XG4gICAqIC0ga21zOkNyZWF0ZUdyYW50XG4gICAqIC0ga21zOkRlc2NyaWJlS2V5XG4gICAqIC0gbGFrZWZvcm1hdGlvbjpHZXREYXRhTGFrZVNldHRpbmdzXG4gICAqIC0gbGFrZWZvcm1hdGlvbjpQdXREYXRhTGFrZVNldHRpbmdzXG4gICAqIC0gbGFtYmRhOkFkZFBlcm1pc3Npb25cbiAgICogLSBsYW1iZGE6Q3JlYXRlRXZlbnRTb3VyY2VNYXBwaW5nXG4gICAqIC0gbGFtYmRhOkNyZWF0ZUZ1bmN0aW9uXG4gICAqIC0gb3JnYW5pemF0aW9uczpEZXNjcmliZU9yZ2FuaXphdGlvblxuICAgKiAtIG9yZ2FuaXphdGlvbnM6TGlzdEFjY291bnRzXG4gICAqIC0gb3JnYW5pemF0aW9uczpMaXN0RGVsZWdhdGVkU2VydmljZXNGb3JBY2NvdW50XG4gICAqIC0gczM6Q3JlYXRlQnVja2V0XG4gICAqIC0gczM6R2V0T2JqZWN0XG4gICAqIC0gczM6R2V0T2JqZWN0VmVyc2lvblxuICAgKiAtIHMzOkxpc3RCdWNrZXRcbiAgICogLSBzMzpQdXRCdWNrZXRQb2xpY3lcbiAgICogLSBzMzpQdXRCdWNrZXRQdWJsaWNBY2Nlc3NCbG9ja1xuICAgKiAtIHMzOlB1dEJ1Y2tldFZlcnNpb25pbmdcbiAgICogLSBzcXM6Q3JlYXRlUXVldWVcbiAgICogLSBzcXM6R2V0UXVldWVBdHRyaWJ1dGVzXG4gICAqIC0gc3FzOlNldFF1ZXVlQXR0cmlidXRlc1xuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VjdXJpdHktbGFrZS9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVEYXRhTGFrZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVEYXRhTGFrZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlRGF0YUxha2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgaW5zdGFudCBub3RpZmljYXRpb25zIGFib3V0IGV4Y2VwdGlvbnMuIFN1YnNjcmliZXMgdG8gdGhlIFNOUyB0b3BpY3MgZm9yIGV4Y2VwdGlvbiBub3RpZmljYXRpb25zXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlY3VyaXR5LWxha2UvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlRGF0YUxha2VFeGNlcHRpb25TdWJzY3JpcHRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlRGF0YUxha2VFeGNlcHRpb25TdWJzY3JpcHRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZURhdGFMYWtlRXhjZXB0aW9uU3Vic2NyaXB0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYXV0b21hdGljYWxseSBlbmFibGUgQW1hem9uIFNlY3VyaXR5IExha2UgZm9yIG5ldyBtZW1iZXIgYWNjb3VudHMgaW4geW91ciBvcmdhbml6YXRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VjdXJpdHktbGFrZS9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVEYXRhTGFrZU9yZ2FuaXphdGlvbkNvbmZpZ3VyYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlRGF0YUxha2VPcmdhbml6YXRpb25Db25maWd1cmF0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVEYXRhTGFrZU9yZ2FuaXphdGlvbkNvbmZpZ3VyYXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBzdWJzY3JpYmVyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIGlhbTpDcmVhdGVSb2xlXG4gICAqIC0gaWFtOkRlbGV0ZVJvbGVQb2xpY3lcbiAgICogLSBpYW06R2V0Um9sZVxuICAgKiAtIGlhbTpQdXRSb2xlUG9saWN5XG4gICAqIC0gbGFrZWZvcm1hdGlvbjpHcmFudFBlcm1pc3Npb25zXG4gICAqIC0gbGFrZWZvcm1hdGlvbjpMaXN0UGVybWlzc2lvbnNcbiAgICogLSBsYWtlZm9ybWF0aW9uOlJlZ2lzdGVyUmVzb3VyY2VcbiAgICogLSBsYWtlZm9ybWF0aW9uOlJldm9rZVBlcm1pc3Npb25zXG4gICAqIC0gcmFtOkdldFJlc291cmNlU2hhcmVBc3NvY2lhdGlvbnNcbiAgICogLSByYW06R2V0UmVzb3VyY2VTaGFyZXNcbiAgICogLSByYW06VXBkYXRlUmVzb3VyY2VTaGFyZVxuICAgKiAtIHMzOlB1dE9iamVjdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VjdXJpdHktbGFrZS9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVTdWJzY3JpYmVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZVN1YnNjcmliZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZVN1YnNjcmliZXInKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSB3ZWJob29rIGludm9jYXRpb24gdG8gbm90aWZ5IGEgY2xpZW50IHdoZW4gdGhlcmUgaXMgbmV3IGRhdGEgaW4gdGhlIGRhdGEgbGFrZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIGV2ZW50czpDcmVhdGVBcGlEZXN0aW5hdGlvblxuICAgKiAtIGV2ZW50czpDcmVhdGVDb25uZWN0aW9uXG4gICAqIC0gZXZlbnRzOkRlc2NyaWJlUnVsZVxuICAgKiAtIGV2ZW50czpMaXN0QXBpRGVzdGluYXRpb25zXG4gICAqIC0gZXZlbnRzOkxpc3RDb25uZWN0aW9uc1xuICAgKiAtIGV2ZW50czpQdXRSdWxlXG4gICAqIC0gZXZlbnRzOlB1dFRhcmdldHNcbiAgICogLSBpYW06RGVsZXRlUm9sZVBvbGljeVxuICAgKiAtIGlhbTpHZXRSb2xlXG4gICAqIC0gaWFtOlBhc3NSb2xlXG4gICAqIC0gczM6R2V0QnVja2V0Tm90aWZpY2F0aW9uXG4gICAqIC0gczM6UHV0QnVja2V0Tm90aWZpY2F0aW9uXG4gICAqIC0gc3FzOkNyZWF0ZVF1ZXVlXG4gICAqIC0gc3FzOkRlbGV0ZVF1ZXVlXG4gICAqIC0gc3FzOkdldFF1ZXVlQXR0cmlidXRlc1xuICAgKiAtIHNxczpHZXRRdWV1ZVVybFxuICAgKiAtIHNxczpTZXRRdWV1ZUF0dHJpYnV0ZXNcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlY3VyaXR5LWxha2UvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlU3Vic2NyaWJlck5vdGlmaWNhdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVTdWJzY3JpYmVyTm90aWZpY2F0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVTdWJzY3JpYmVyTm90aWZpY2F0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGlzYWJsZSBhbnkgc291cmNlIHR5cGUgaW4gYW55IHJlZ2lvbiBmb3IgYWNjb3VudHMgdGhhdCBhcmUgcGFydCBvZiBhIHRydXN0ZWQgb3JnYW5pemF0aW9uIG9yIHN0YW5kYWxvbmUgYWNjb3VudHNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VjdXJpdHktbGFrZS9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVBd3NMb2dTb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlQXdzTG9nU291cmNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVBd3NMb2dTb3VyY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZW1vdmUgYSBjdXN0b20gc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gZ2x1ZTpTdG9wQ3Jhd2xlclNjaGVkdWxlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZWN1cml0eS1sYWtlL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZUN1c3RvbUxvZ1NvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVDdXN0b21Mb2dTb3VyY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZUN1c3RvbUxvZ1NvdXJjZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBzZWN1cml0eSBkYXRhIGxha2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBvcmdhbml6YXRpb25zOkRlc2NyaWJlT3JnYW5pemF0aW9uXG4gICAqIC0gb3JnYW5pemF0aW9uczpMaXN0RGVsZWdhdGVkQWRtaW5pc3RyYXRvcnNcbiAgICogLSBvcmdhbml6YXRpb25zOkxpc3REZWxlZ2F0ZWRTZXJ2aWNlc0ZvckFjY291bnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlY3VyaXR5LWxha2UvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlRGF0YUxha2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlRGF0YUxha2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZURhdGFMYWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdW5zdWJzY3JpYmUgZnJvbSBTTlMgdG9waWNzIGZvciBleGNlcHRpb24gbm90aWZpY2F0aW9ucy4gUmVtb3ZlcyBleGNlcHRpb24gbm90aWZpY2F0aW9ucyBmb3IgdGhlIFNOUyB0b3BpY1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZWN1cml0eS1sYWtlL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZURhdGFMYWtlRXhjZXB0aW9uU3Vic2NyaXB0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZURhdGFMYWtlRXhjZXB0aW9uU3Vic2NyaXB0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVEYXRhTGFrZUV4Y2VwdGlvblN1YnNjcmlwdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJlbW92ZSB0aGUgYXV0b21hdGljIGVuYWJsZW1lbnQgb2YgQW1hem9uIFNlY3VyaXR5IExha2UgYWNjZXNzIGZvciBuZXcgb3JnYW5pemF0aW9uIGFjY291bnRzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlY3VyaXR5LWxha2UvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlRGF0YUxha2VPcmdhbml6YXRpb25Db25maWd1cmF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZURhdGFMYWtlT3JnYW5pemF0aW9uQ29uZmlndXJhdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlRGF0YUxha2VPcmdhbml6YXRpb25Db25maWd1cmF0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIHRoZSBzcGVjaWZpZWQgc3Vic2NyaWJlclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIGV2ZW50czpEZWxldGVBcGlEZXN0aW5hdGlvblxuICAgKiAtIGV2ZW50czpEZWxldGVDb25uZWN0aW9uXG4gICAqIC0gZXZlbnRzOkRlbGV0ZVJ1bGVcbiAgICogLSBldmVudHM6RGVzY3JpYmVSdWxlXG4gICAqIC0gZXZlbnRzOkxpc3RBcGlEZXN0aW5hdGlvbnNcbiAgICogLSBldmVudHM6TGlzdFRhcmdldHNCeVJ1bGVcbiAgICogLSBldmVudHM6UmVtb3ZlVGFyZ2V0c1xuICAgKiAtIGlhbTpEZWxldGVSb2xlXG4gICAqIC0gaWFtOkRlbGV0ZVJvbGVQb2xpY3lcbiAgICogLSBpYW06R2V0Um9sZVxuICAgKiAtIGlhbTpMaXN0Um9sZVBvbGljaWVzXG4gICAqIC0gbGFrZWZvcm1hdGlvbjpMaXN0UGVybWlzc2lvbnNcbiAgICogLSBsYWtlZm9ybWF0aW9uOlJldm9rZVBlcm1pc3Npb25zXG4gICAqIC0gc3FzOkRlbGV0ZVF1ZXVlXG4gICAqIC0gc3FzOkdldFF1ZXVlVXJsXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZWN1cml0eS1sYWtlL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZVN1YnNjcmliZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlU3Vic2NyaWJlcigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlU3Vic2NyaWJlcicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJlbW92ZSBhIHdlYmhvb2sgaW52b2NhdGlvbiB0byBub3RpZnkgYSBjbGllbnQgd2hlbiB0aGVyZSBpcyBuZXcgZGF0YSBpbiB0aGUgZGF0YSBsYWtlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gZXZlbnRzOkRlbGV0ZUFwaURlc3RpbmF0aW9uXG4gICAqIC0gZXZlbnRzOkRlbGV0ZUNvbm5lY3Rpb25cbiAgICogLSBldmVudHM6RGVsZXRlUnVsZVxuICAgKiAtIGV2ZW50czpEZXNjcmliZVJ1bGVcbiAgICogLSBldmVudHM6TGlzdEFwaURlc3RpbmF0aW9uc1xuICAgKiAtIGV2ZW50czpMaXN0VGFyZ2V0c0J5UnVsZVxuICAgKiAtIGV2ZW50czpSZW1vdmVUYXJnZXRzXG4gICAqIC0gaWFtOkRlbGV0ZVJvbGVcbiAgICogLSBpYW06RGVsZXRlUm9sZVBvbGljeVxuICAgKiAtIGlhbTpHZXRSb2xlXG4gICAqIC0gaWFtOkxpc3RSb2xlUG9saWNpZXNcbiAgICogLSBsYWtlZm9ybWF0aW9uOlJldm9rZVBlcm1pc3Npb25zXG4gICAqIC0gc3FzOkRlbGV0ZVF1ZXVlXG4gICAqIC0gc3FzOkdldFF1ZXVlVXJsXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZWN1cml0eS1sYWtlL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZVN1YnNjcmliZXJOb3RpZmljYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlU3Vic2NyaWJlck5vdGlmaWNhdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlU3Vic2NyaWJlck5vdGlmaWNhdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJlbW92ZSB0aGUgRGVsZWdhdGVkIEFkbWluaXN0cmF0b3IgYWNjb3VudCBhbmQgZGlzYWJsZSBBbWF6b24gU2VjdXJpdHkgTGFrZSBhcyBhIHNlcnZpY2UgZm9yIHRoaXMgb3JnYW5pemF0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gb3JnYW5pemF0aW9uczpEZXJlZ2lzdGVyRGVsZWdhdGVkQWRtaW5pc3RyYXRvclxuICAgKiAtIG9yZ2FuaXphdGlvbnM6RGVzY3JpYmVPcmdhbml6YXRpb25cbiAgICogLSBvcmdhbml6YXRpb25zOkxpc3REZWxlZ2F0ZWRTZXJ2aWNlc0ZvckFjY291bnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlY3VyaXR5LWxha2UvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVyZWdpc3RlckRhdGFMYWtlRGVsZWdhdGVkQWRtaW5pc3RyYXRvci5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXJlZ2lzdGVyRGF0YUxha2VEZWxlZ2F0ZWRBZG1pbmlzdHJhdG9yKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXJlZ2lzdGVyRGF0YUxha2VEZWxlZ2F0ZWRBZG1pbmlzdHJhdG9yJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcXVlcnkgdGhlIHByb3RvY29sIGFuZCBlbmRwb2ludCB0aGF0IHdlcmUgcHJvdmlkZWQgd2hlbiBzdWJzY3JpYmluZyB0byBTTlMgdG9waWNzIGZvciBleGNlcHRpb24gbm90aWZpY2F0aW9uc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlY3VyaXR5LWxha2UvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0RGF0YUxha2VFeGNlcHRpb25TdWJzY3JpcHRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0RGF0YUxha2VFeGNlcHRpb25TdWJzY3JpcHRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldERhdGFMYWtlRXhjZXB0aW9uU3Vic2NyaXB0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IGFuIG9yZ2FuaXphdGlvbidzIGNvbmZpZ3VyYXRpb24gc2V0dGluZyBmb3IgYXV0b21hdGljYWxseSBlbmFibGluZyBBbWF6b24gU2VjdXJpdHkgTGFrZSBhY2Nlc3MgZm9yIG5ldyBvcmdhbml6YXRpb24gYWNjb3VudHNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIG9yZ2FuaXphdGlvbnM6RGVzY3JpYmVPcmdhbml6YXRpb25cbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlY3VyaXR5LWxha2UvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0RGF0YUxha2VPcmdhbml6YXRpb25Db25maWd1cmF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldERhdGFMYWtlT3JnYW5pemF0aW9uQ29uZmlndXJhdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0RGF0YUxha2VPcmdhbml6YXRpb25Db25maWd1cmF0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IGEgc3RhdGljIHNuYXBzaG90IG9mIHRoZSBzZWN1cml0eSBkYXRhIGxha2UgaW4gdGhlIGN1cnJlbnQgcmVnaW9uLiBUaGUgc25hcHNob3QgaW5jbHVkZXMgZW5hYmxlZCBhY2NvdW50cyBhbmQgbG9nIHNvdXJjZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZWN1cml0eS1sYWtlL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldERhdGFMYWtlU291cmNlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXREYXRhTGFrZVNvdXJjZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldERhdGFMYWtlU291cmNlcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBpbmZvcm1hdGlvbiBhYm91dCBzdWJzY3JpYmVyIHRoYXQgaXMgYWxyZWFkeSBjcmVhdGVkXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VjdXJpdHktbGFrZS9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9HZXRTdWJzY3JpYmVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFN1YnNjcmliZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldFN1YnNjcmliZXInKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgdGhlIGxpc3Qgb2YgYWxsIG5vbi1yZXRyeWFibGUgZmFpbHVyZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZWN1cml0eS1sYWtlL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3REYXRhTGFrZUV4Y2VwdGlvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdERhdGFMYWtlRXhjZXB0aW9ucygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdERhdGFMYWtlRXhjZXB0aW9ucycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgaW5mb3JtYXRpb24gYWJvdXQgdGhlIHNlY3VyaXR5IGRhdGEgbGFrZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZWN1cml0eS1sYWtlL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3REYXRhTGFrZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdERhdGFMYWtlcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdERhdGFMYWtlcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHZpZXcgdGhlIGVuYWJsZWQgYWNjb3VudHMuIFlvdSBjYW4gdmlldyB0aGUgZW5hYmxlZCBzb3VyY2VzIGluIHRoZSBlbmFibGVkIHJlZ2lvbnNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZWN1cml0eS1sYWtlL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RMb2dTb3VyY2VzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RMb2dTb3VyY2VzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0TG9nU291cmNlcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgYWxsIHN1YnNjcmliZXJzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VjdXJpdHktbGFrZS9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0U3Vic2NyaWJlcnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFN1YnNjcmliZXJzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0U3Vic2NyaWJlcnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IGFsbCB0YWdzIGZvciB0aGUgcmVzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZWN1cml0eS1sYWtlL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RUYWdzRm9yUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFRhZ3NGb3JSZXNvdXJjZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdFRhZ3NGb3JSZXNvdXJjZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlc2lnbmF0ZSBhbiBhY2NvdW50IGFzIHRoZSBBbWF6b24gU2VjdXJpdHkgTGFrZSBhZG1pbmlzdHJhdG9yIGFjY291bnQgZm9yIHRoZSBvcmdhbml6YXRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBpYW06Q3JlYXRlU2VydmljZUxpbmtlZFJvbGVcbiAgICogLSBvcmdhbml6YXRpb25zOkRlc2NyaWJlT3JnYW5pemF0aW9uXG4gICAqIC0gb3JnYW5pemF0aW9uczpFbmFibGVBV1NTZXJ2aWNlQWNjZXNzXG4gICAqIC0gb3JnYW5pemF0aW9uczpMaXN0RGVsZWdhdGVkQWRtaW5pc3RyYXRvcnNcbiAgICogLSBvcmdhbml6YXRpb25zOkxpc3REZWxlZ2F0ZWRTZXJ2aWNlc0ZvckFjY291bnRcbiAgICogLSBvcmdhbml6YXRpb25zOlJlZ2lzdGVyRGVsZWdhdGVkQWRtaW5pc3RyYXRvclxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VjdXJpdHktbGFrZS9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9SZWdpc3RlckRhdGFMYWtlRGVsZWdhdGVkQWRtaW5pc3RyYXRvci5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9SZWdpc3RlckRhdGFMYWtlRGVsZWdhdGVkQWRtaW5pc3RyYXRvcigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnUmVnaXN0ZXJEYXRhTGFrZURlbGVnYXRlZEFkbWluaXN0cmF0b3InKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBhZGQgdGFncyB0byB0aGUgcmVzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZWN1cml0eS1sYWtlL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1RhZ1Jlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1RhZ1Jlc291cmNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdUYWdSZXNvdXJjZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJlbW92ZSB0YWdzIGZyb20gdGhlIHJlc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogVGFnZ2luZ1xuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VjdXJpdHktbGFrZS9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VbnRhZ1Jlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VudGFnUmVzb3VyY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VudGFnUmVzb3VyY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgYSBzZWN1cml0eSBkYXRhIGxha2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBldmVudHM6UHV0UnVsZVxuICAgKiAtIGV2ZW50czpQdXRUYXJnZXRzXG4gICAqIC0gaWFtOkNyZWF0ZVNlcnZpY2VMaW5rZWRSb2xlXG4gICAqIC0gaWFtOkRlbGV0ZVJvbGVQb2xpY3lcbiAgICogLSBpYW06R2V0Um9sZVxuICAgKiAtIGlhbTpMaXN0QXR0YWNoZWRSb2xlUG9saWNpZXNcbiAgICogLSBpYW06UHV0Um9sZVBvbGljeVxuICAgKiAtIGttczpDcmVhdGVHcmFudFxuICAgKiAtIGttczpEZXNjcmliZUtleVxuICAgKiAtIGxha2Vmb3JtYXRpb246R2V0RGF0YUxha2VTZXR0aW5nc1xuICAgKiAtIGxha2Vmb3JtYXRpb246UHV0RGF0YUxha2VTZXR0aW5nc1xuICAgKiAtIGxhbWJkYTpBZGRQZXJtaXNzaW9uXG4gICAqIC0gbGFtYmRhOkNyZWF0ZUV2ZW50U291cmNlTWFwcGluZ1xuICAgKiAtIGxhbWJkYTpDcmVhdGVGdW5jdGlvblxuICAgKiAtIG9yZ2FuaXphdGlvbnM6RGVzY3JpYmVPcmdhbml6YXRpb25cbiAgICogLSBvcmdhbml6YXRpb25zOkxpc3REZWxlZ2F0ZWRTZXJ2aWNlc0ZvckFjY291bnRcbiAgICogLSBzMzpDcmVhdGVCdWNrZXRcbiAgICogLSBzMzpHZXRPYmplY3RcbiAgICogLSBzMzpHZXRPYmplY3RWZXJzaW9uXG4gICAqIC0gczM6TGlzdEJ1Y2tldFxuICAgKiAtIHMzOlB1dEJ1Y2tldFBvbGljeVxuICAgKiAtIHMzOlB1dEJ1Y2tldFB1YmxpY0FjY2Vzc0Jsb2NrXG4gICAqIC0gczM6UHV0QnVja2V0VmVyc2lvbmluZ1xuICAgKiAtIHNxczpDcmVhdGVRdWV1ZVxuICAgKiAtIHNxczpHZXRRdWV1ZUF0dHJpYnV0ZXNcbiAgICogLSBzcXM6U2V0UXVldWVBdHRyaWJ1dGVzXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZWN1cml0eS1sYWtlL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZURhdGFMYWtlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZURhdGFMYWtlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVEYXRhTGFrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBzdWJzY3JpcHRpb25zIHRvIHRoZSBTTlMgdG9waWNzIGZvciBleGNlcHRpb24gbm90aWZpY2F0aW9uc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZWN1cml0eS1sYWtlL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZURhdGFMYWtlRXhjZXB0aW9uU3Vic2NyaXB0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZURhdGFMYWtlRXhjZXB0aW9uU3Vic2NyaXB0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVEYXRhTGFrZUV4Y2VwdGlvblN1YnNjcmlwdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBzdWJzY3JpYmVyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gZXZlbnRzOkNyZWF0ZUFwaURlc3RpbmF0aW9uXG4gICAqIC0gZXZlbnRzOkNyZWF0ZUNvbm5lY3Rpb25cbiAgICogLSBldmVudHM6RGVzY3JpYmVSdWxlXG4gICAqIC0gZXZlbnRzOkxpc3RBcGlEZXN0aW5hdGlvbnNcbiAgICogLSBldmVudHM6TGlzdENvbm5lY3Rpb25zXG4gICAqIC0gZXZlbnRzOlB1dFJ1bGVcbiAgICogLSBldmVudHM6UHV0VGFyZ2V0c1xuICAgKiAtIGlhbTpEZWxldGVSb2xlUG9saWN5XG4gICAqIC0gaWFtOkdldFJvbGVcbiAgICogLSBpYW06UHV0Um9sZVBvbGljeVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VjdXJpdHktbGFrZS9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVTdWJzY3JpYmVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZVN1YnNjcmliZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZVN1YnNjcmliZXInKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgYSB3ZWJob29rIGludm9jYXRpb24gdG8gbm90aWZ5IGEgY2xpZW50IHdoZW4gdGhlcmUgaXMgbmV3IGRhdGEgaW4gdGhlIGRhdGEgbGFrZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIGV2ZW50czpDcmVhdGVBcGlEZXN0aW5hdGlvblxuICAgKiAtIGV2ZW50czpDcmVhdGVDb25uZWN0aW9uXG4gICAqIC0gZXZlbnRzOkRlc2NyaWJlUnVsZVxuICAgKiAtIGV2ZW50czpMaXN0QXBpRGVzdGluYXRpb25zXG4gICAqIC0gZXZlbnRzOkxpc3RDb25uZWN0aW9uc1xuICAgKiAtIGV2ZW50czpQdXRSdWxlXG4gICAqIC0gZXZlbnRzOlB1dFRhcmdldHNcbiAgICogLSBpYW06Q3JlYXRlU2VydmljZUxpbmtlZFJvbGVcbiAgICogLSBpYW06RGVsZXRlUm9sZVBvbGljeVxuICAgKiAtIGlhbTpHZXRSb2xlXG4gICAqIC0gaWFtOlBhc3NSb2xlXG4gICAqIC0gaWFtOlB1dFJvbGVQb2xpY3lcbiAgICogLSBzMzpDcmVhdGVCdWNrZXRcbiAgICogLSBzMzpHZXRCdWNrZXROb3RpZmljYXRpb25cbiAgICogLSBzMzpMaXN0QnVja2V0XG4gICAqIC0gczM6UHV0QnVja2V0Tm90aWZpY2F0aW9uXG4gICAqIC0gczM6UHV0QnVja2V0UG9saWN5XG4gICAqIC0gczM6UHV0QnVja2V0UHVibGljQWNjZXNzQmxvY2tcbiAgICogLSBzMzpQdXRCdWNrZXRWZXJzaW9uaW5nXG4gICAqIC0gczM6UHV0TGlmZWN5Y2xlQ29uZmlndXJhdGlvblxuICAgKiAtIHNxczpDcmVhdGVRdWV1ZVxuICAgKiAtIHNxczpEZWxldGVRdWV1ZVxuICAgKiAtIHNxczpHZXRRdWV1ZUF0dHJpYnV0ZXNcbiAgICogLSBzcXM6R2V0UXVldWVVcmxcbiAgICogLSBzcXM6U2V0UXVldWVBdHRyaWJ1dGVzXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZWN1cml0eS1sYWtlL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZVN1YnNjcmliZXJOb3RpZmljYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlU3Vic2NyaWJlck5vdGlmaWNhdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlU3Vic2NyaWJlck5vdGlmaWNhdGlvbicpO1xuICB9XG5cbiAgcHJvdGVjdGVkIGFjY2Vzc0xldmVsTGlzdDogQWNjZXNzTGV2ZWxMaXN0ID0ge1xuICAgIFdyaXRlOiBbXG4gICAgICAnQ3JlYXRlQXdzTG9nU291cmNlJyxcbiAgICAgICdDcmVhdGVDdXN0b21Mb2dTb3VyY2UnLFxuICAgICAgJ0NyZWF0ZURhdGFMYWtlJyxcbiAgICAgICdDcmVhdGVEYXRhTGFrZUV4Y2VwdGlvblN1YnNjcmlwdGlvbicsXG4gICAgICAnQ3JlYXRlRGF0YUxha2VPcmdhbml6YXRpb25Db25maWd1cmF0aW9uJyxcbiAgICAgICdDcmVhdGVTdWJzY3JpYmVyJyxcbiAgICAgICdDcmVhdGVTdWJzY3JpYmVyTm90aWZpY2F0aW9uJyxcbiAgICAgICdEZWxldGVBd3NMb2dTb3VyY2UnLFxuICAgICAgJ0RlbGV0ZUN1c3RvbUxvZ1NvdXJjZScsXG4gICAgICAnRGVsZXRlRGF0YUxha2UnLFxuICAgICAgJ0RlbGV0ZURhdGFMYWtlRXhjZXB0aW9uU3Vic2NyaXB0aW9uJyxcbiAgICAgICdEZWxldGVEYXRhTGFrZU9yZ2FuaXphdGlvbkNvbmZpZ3VyYXRpb24nLFxuICAgICAgJ0RlbGV0ZVN1YnNjcmliZXInLFxuICAgICAgJ0RlbGV0ZVN1YnNjcmliZXJOb3RpZmljYXRpb24nLFxuICAgICAgJ0RlcmVnaXN0ZXJEYXRhTGFrZURlbGVnYXRlZEFkbWluaXN0cmF0b3InLFxuICAgICAgJ1JlZ2lzdGVyRGF0YUxha2VEZWxlZ2F0ZWRBZG1pbmlzdHJhdG9yJyxcbiAgICAgICdVcGRhdGVEYXRhTGFrZScsXG4gICAgICAnVXBkYXRlRGF0YUxha2VFeGNlcHRpb25TdWJzY3JpcHRpb24nLFxuICAgICAgJ1VwZGF0ZVN1YnNjcmliZXInLFxuICAgICAgJ1VwZGF0ZVN1YnNjcmliZXJOb3RpZmljYXRpb24nXG4gICAgXSxcbiAgICBSZWFkOiBbXG4gICAgICAnR2V0RGF0YUxha2VFeGNlcHRpb25TdWJzY3JpcHRpb24nLFxuICAgICAgJ0dldERhdGFMYWtlT3JnYW5pemF0aW9uQ29uZmlndXJhdGlvbicsXG4gICAgICAnR2V0RGF0YUxha2VTb3VyY2VzJyxcbiAgICAgICdHZXRTdWJzY3JpYmVyJ1xuICAgIF0sXG4gICAgTGlzdDogW1xuICAgICAgJ0xpc3REYXRhTGFrZUV4Y2VwdGlvbnMnLFxuICAgICAgJ0xpc3REYXRhTGFrZXMnLFxuICAgICAgJ0xpc3RMb2dTb3VyY2VzJyxcbiAgICAgICdMaXN0U3Vic2NyaWJlcnMnLFxuICAgICAgJ0xpc3RUYWdzRm9yUmVzb3VyY2UnXG4gICAgXSxcbiAgICBUYWdnaW5nOiBbXG4gICAgICAnVGFnUmVzb3VyY2UnLFxuICAgICAgJ1VudGFnUmVzb3VyY2UnXG4gICAgXVxuICB9O1xuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBkYXRhLWxha2UgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VjdXJpdHktbGFrZS9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EYXRhTGFrZVJlc291cmNlLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uRGF0YUxha2UoYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTpzZWN1cml0eWxha2U6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OmRhdGEtbGFrZS9kZWZhdWx0YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgc3Vic2NyaWJlciB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZWN1cml0eS1sYWtlL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1N1YnNjcmliZXJSZXNvdXJjZS5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBzdWJzY3JpYmVySWQgLSBJZGVudGlmaWVyIGZvciB0aGUgc3Vic2NyaWJlcklkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25TdWJzY3JpYmVyKHN1YnNjcmliZXJJZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OnNlY3VyaXR5bGFrZTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06c3Vic2NyaWJlci8keyBzdWJzY3JpYmVySWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRhZ3MgdGhhdCBhcmUgcGFzc2VkIGluIHRoZSByZXF1ZXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfY29uZGl0aW9uLWtleXMuaHRtbCNjb25kaXRpb24ta2V5cy1yZXF1ZXN0dGFnXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9DcmVhdGVEYXRhTGFrZSgpXG4gICAqIC0gLnRvQ3JlYXRlU3Vic2NyaWJlcigpXG4gICAqIC0gLnRvVGFnUmVzb3VyY2UoKVxuICAgKlxuICAgKiBBcHBsaWVzIHRvIHJlc291cmNlIHR5cGVzOlxuICAgKiAtIGRhdGEtbGFrZVxuICAgKiAtIHN1YnNjcmliZXJcbiAgICpcbiAgICogQHBhcmFtIHRhZ0tleSBUaGUgdGFnIGtleSB0byBjaGVja1xuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1JlcXVlc3RUYWcodGFnS2V5OiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBhd3M6UmVxdWVzdFRhZy8keyB0YWdLZXkgfWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IGEgdGFnIGtleSBhbmQgdmFsdWUgcGFpciBvZiBhIHJlc291cmNlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfY29uZGl0aW9uLWtleXMuaHRtbCNjb25kaXRpb24ta2V5cy1yZXNvdXJjZXRhZ1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvQ3JlYXRlU3Vic2NyaWJlcigpXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gZGF0YS1sYWtlXG4gICAqIC0gc3Vic2NyaWJlclxuICAgKlxuICAgKiBAcGFyYW0gdGFnS2V5IFRoZSB0YWcga2V5IHRvIGNoZWNrXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQXdzUmVzb3VyY2VUYWcodGFnS2V5OiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBhd3M6UmVzb3VyY2VUYWcvJHsgdGFnS2V5IH1gLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0YWcga2V5cyB0aGF0IGFyZSBwYXNzZWQgaW4gdGhlIHJlcXVlc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19jb25kaXRpb24ta2V5cy5odG1sI2NvbmRpdGlvbi1rZXlzLXRhZ2tleXNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0NyZWF0ZURhdGFMYWtlKClcbiAgICogLSAudG9DcmVhdGVTdWJzY3JpYmVyKClcbiAgICogLSAudG9UYWdSZXNvdXJjZSgpXG4gICAqIC0gLnRvVW50YWdSZXNvdXJjZSgpXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQXdzVGFnS2V5cyh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgYXdzOlRhZ0tleXNgLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW3NlY3VyaXR5bGFrZV0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2UtYXV0aG9yaXphdGlvbi9sYXRlc3QvcmVmZXJlbmNlL2xpc3RfYW1hem9uc2VjdXJpdHlsYWtlLmh0bWwpLlxuICAgKlxuICAgKi9cbiAgY29uc3RydWN0b3IocHJvcHM/OiBpYW0uUG9saWN5U3RhdGVtZW50UHJvcHMpIHtcbiAgICBzdXBlcihwcm9wcyk7XG4gIH1cbn1cbiJdfQ==