"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Odb = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [odb](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsservice-oracledatabase_aws.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Odb extends shared_1.PolicyStatement {
    /**
     * Grants permission to register the Amazon Web Services Marketplace token for your Amazon Web Services account to activate your Oracle Database@Amazon Web Services subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_AcceptMarketplaceRegistration.html
     */
    toAcceptMarketplaceRegistration() {
        return this.to('AcceptMarketplaceRegistration');
    }
    /**
     * Grants permission to create a new Autonomous VM cluster in the specified Exadata infrastructure
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_CreateCloudAutonomousVmCluster.html
     */
    toCreateCloudAutonomousVmCluster() {
        return this.to('CreateCloudAutonomousVmCluster');
    }
    /**
     * Grants permission to create an Exadata infrastructure
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_CreateCloudExadataInfrastructure.html
     */
    toCreateCloudExadataInfrastructure() {
        return this.to('CreateCloudExadataInfrastructure');
    }
    /**
     * Grants permission to create a VM cluster on the specified Exadata infrastructure
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_CreateCloudVmCluster.html
     */
    toCreateCloudVmCluster() {
        return this.to('CreateCloudVmCluster');
    }
    /**
     * Grants permission to create a DB Node
     *
     * Access Level: Write
     */
    toCreateDbNode() {
        return this.to('CreateDbNode');
    }
    /**
     * Grants permission to create an ODB network
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_CreateOdbNetwork.html
     */
    toCreateOdbNetwork() {
        return this.to('CreateOdbNetwork');
    }
    /**
     * Grants permission to create an ODB Peering Connection
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_CreateOdbPeeringConnection.html
     */
    toCreateOdbPeeringConnection() {
        return this.to('CreateOdbPeeringConnection');
    }
    /**
     * Grants permission to create an Outbound Integration
     *
     * Access Level: Write
     */
    toCreateOutboundIntegration() {
        return this.to('CreateOutboundIntegration');
    }
    /**
     * Grants permission to Deletes an Autonomous VM cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_DeleteCloudAutonomousVmCluster.html
     */
    toDeleteCloudAutonomousVmCluster() {
        return this.to('DeleteCloudAutonomousVmCluster');
    }
    /**
     * Grants permission to delete a specified Exadata infrastructure. Before you use this operation, make sure to delete all of the VM clusters that are hosted on this Exadata infrastructure
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_DeleteCloudExadataInfrastructure.html
     */
    toDeleteCloudExadataInfrastructure() {
        return this.to('DeleteCloudExadataInfrastructure');
    }
    /**
     * Grants permission to delete a specified VM cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_DeleteCloudVmCluster.html
     */
    toDeleteCloudVmCluster() {
        return this.to('DeleteCloudVmCluster');
    }
    /**
     * Grants permission to delete a DB Node
     *
     * Access Level: Write
     */
    toDeleteDbNode() {
        return this.to('DeleteDbNode');
    }
    /**
     * Grants permission to delete the specified ODB network
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_DeleteOdbNetwork.html
     */
    toDeleteOdbNetwork() {
        return this.to('DeleteOdbNetwork');
    }
    /**
     * Grants permission to delete the specified ODB Peering Connection. When you delete an ODB peering connection, the underlying VPC peering connection is also deleted
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_DeleteOdbPeeringConnection.html
     */
    toDeleteOdbPeeringConnection() {
        return this.to('DeleteOdbPeeringConnection');
    }
    /**
     * Grants permission to delete a resource policy
     *
     * Access Level: Write
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to get information about a specific Autonomous VM cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_GetCloudAutonomousVmCluster.html
     */
    toGetCloudAutonomousVmCluster() {
        return this.to('GetCloudAutonomousVmCluster');
    }
    /**
     * Grants permission to get information about the specified Exadata infrastructure
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_GetCloudExadataInfrastructure.html
     */
    toGetCloudExadataInfrastructure() {
        return this.to('GetCloudExadataInfrastructure');
    }
    /**
     * Grants permission to retrieve information about unallocated resources in a specified Cloud Exadata Infrastructure
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_GetCloudExadataInfrastructureUnallocatedResources.html
     */
    toGetCloudExadataInfrastructureUnallocatedResources() {
        return this.to('GetCloudExadataInfrastructureUnallocatedResources');
    }
    /**
     * Grants permission to get information about the specified VM cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_GetCloudVmCluster.html
     */
    toGetCloudVmCluster() {
        return this.to('GetCloudVmCluster');
    }
    /**
     * Grants permission to get information about the specified DB node
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_GetDbNode.html
     */
    toGetDbNode() {
        return this.to('GetDbNode');
    }
    /**
     * Grants permission to get information about the specified database server
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_GetDbServer.html
     */
    toGetDbServer() {
        return this.to('GetDbServer');
    }
    /**
     * Grants permission to get the tenancy activation link and onboarding status for your Amazon Web Services account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_GetOciOnboardingStatus.html
     */
    toGetOciOnboardingStatus() {
        return this.to('GetOciOnboardingStatus');
    }
    /**
     * Grants permission to get information about the specified ODB network
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_GetOdbNetwork.html
     */
    toGetOdbNetwork() {
        return this.to('GetOdbNetwork');
    }
    /**
     * Grants permission to get information about the specified ODB Peering connection
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_GetOdbPeeringConnection.html
     */
    toGetOdbPeeringConnection() {
        return this.to('GetOdbPeeringConnection');
    }
    /**
     * Grants permission to get a resource policy
     *
     * Access Level: Read
     */
    toGetResourcePolicy() {
        return this.to('GetResourcePolicy');
    }
    /**
     * Grants permission to initialize the ODB service for the first time in an account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_InitializeService.html
     */
    toInitializeService() {
        return this.to('InitializeService');
    }
    /**
     * Grants permission to list all Autonomous VMs in an Autonomous VM cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_ListAutonomousVirtualMachines.html
     */
    toListAutonomousVirtualMachines() {
        return this.to('ListAutonomousVirtualMachines');
    }
    /**
     * Grants permission to list all Autonomous VM clusters in a specified Cloud Exadata infrastructure
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_ListCloudAutonomousVmClusters.html
     */
    toListCloudAutonomousVmClusters() {
        return this.to('ListCloudAutonomousVmClusters');
    }
    /**
     * Grants permission to list information about the Exadata infrastructures owned by your Amazon Web Services account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_ListCloudExadataInfrastructures.html
     */
    toListCloudExadataInfrastructures() {
        return this.to('ListCloudExadataInfrastructures');
    }
    /**
     * Grants permission to list information about the VM clusters owned by your Amazon Web Services account or only the ones on the specified Exadata infrastructure
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_ListCloudVmClusters.html
     */
    toListCloudVmClusters() {
        return this.to('ListCloudVmClusters');
    }
    /**
     * Grants permission to list information about the DB nodes for the specified VM cluster
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_ListDbNodes.html
     */
    toListDbNodes() {
        return this.to('ListDbNodes');
    }
    /**
     * Grants permission to list information about the database servers that belong to the specified Exadata infrastructure
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_ListDbServers.html
     */
    toListDbServers() {
        return this.to('ListDbServers');
    }
    /**
     * Grants permission to list information about the shapes that are available for an Exadata infrastructure
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_ListDbSystemShapes.html
     */
    toListDbSystemShapes() {
        return this.to('ListDbSystemShapes');
    }
    /**
     * Grants permission to list information about Oracle Grid Infrastructure (GI) software versions that are available for a VM cluster for the specified shape
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_ListGiVersions.html
     */
    toListGiVersions() {
        return this.to('ListGiVersions');
    }
    /**
     * Grants permission to list information about the ODB networks owned by your Amazon Web Services account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_ListOdbNetworks.html
     */
    toListOdbNetworks() {
        return this.to('ListOdbNetworks');
    }
    /**
     * Grants permission to list all ODB peering connections or those associated with a specific ODB network
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_ListOdbPeeringConnections.html
     */
    toListOdbPeeringConnections() {
        return this.to('ListOdbPeeringConnections');
    }
    /**
     * Grants permission to list information about the system versions that are available for a VM cluster for the specified giVersion and shape
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_ListSystemVersions.html
     */
    toListSystemVersions() {
        return this.to('ListSystemVersions');
    }
    /**
     * Grants permission to list information about the tags applied to this resource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to update a resource policy
     *
     * Access Level: Write
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to reboot the specified DB node in a VM cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_RebootDbNode.html
     */
    toRebootDbNode() {
        return this.to('RebootDbNode');
    }
    /**
     * Grants permission to start the specified DB node in a VM cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_StartDbNode.html
     */
    toStartDbNode() {
        return this.to('StartDbNode');
    }
    /**
     * Grants permission to stop the specified DB node in a VM cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_StopDbNode.html
     */
    toStopDbNode() {
        return this.to('StopDbNode');
    }
    /**
     * Grants permission to apply tags to the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update the properties of an Exadata infrastructure resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_UpdateCloudExadataInfrastructure.html
     */
    toUpdateCloudExadataInfrastructure() {
        return this.to('UpdateCloudExadataInfrastructure');
    }
    /**
     * Grants permission to update properties of a specified ODB network
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_UpdateOdbNetwork.html
     */
    toUpdateOdbNetwork() {
        return this.to('UpdateOdbNetwork');
    }
    /**
     * Adds a resource of type cloud-autonomous-vm-cluster to the statement
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_CloudAutonomousVmCluster.html
     *
     * @param cloudAutonomousVmClusterId - Identifier for the cloudAutonomousVmClusterId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCloudAutonomousVmCluster(cloudAutonomousVmClusterId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:odb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:cloud-autonomous-vm-cluster/${cloudAutonomousVmClusterId}`);
    }
    /**
     * Adds a resource of type cloud-exadata-infrastructure to the statement
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_CloudExadataInfrastructure.html
     *
     * @param cloudExadataInfrastructureId - Identifier for the cloudExadataInfrastructureId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCloudExadataInfrastructure(cloudExadataInfrastructureId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:odb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:cloud-exadata-infrastructure/${cloudExadataInfrastructureId}`);
    }
    /**
     * Adds a resource of type cloud-vm-cluster to the statement
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_CloudVmCluster.html
     *
     * @param cloudVmClusterId - Identifier for the cloudVmClusterId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCloudVmCluster(cloudVmClusterId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:odb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:cloud-vm-cluster/${cloudVmClusterId}`);
    }
    /**
     * Adds a resource of type db-node to the statement
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_DbNode.html
     *
     * @param dbNodeId - Identifier for the dbNodeId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDbNode(dbNodeId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:odb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:db-node/${dbNodeId}`);
    }
    /**
     * Adds a resource of type odb-network to the statement
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_OdbNetwork.html
     *
     * @param odbNetworkId - Identifier for the odbNetworkId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onOdbNetwork(odbNetworkId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:odb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:odb-network/${odbNetworkId}`);
    }
    /**
     * Adds a resource of type odb-peering-connection to the statement
     *
     * https://docs.aws.amazon.com/odb/latest/APIReference/API_OdbPeeringConnection.html
     *
     * @param odbPeeringConnectionId - Identifier for the odbPeeringConnectionId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onOdbPeeringConnection(odbPeeringConnectionId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:odb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:odb-peering-connection/${odbPeeringConnectionId}`);
    }
    /**
     * Filters access by a tag key and value pair that is allowed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateCloudAutonomousVmCluster()
     * - .toCreateCloudExadataInfrastructure()
     * - .toCreateCloudVmCluster()
     * - .toCreateOdbNetwork()
     * - .toCreateOdbPeeringConnection()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a tag key and value pair of a resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toListTagsForResource()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * Applies to resource types:
     * - cloud-autonomous-vm-cluster
     * - cloud-exadata-infrastructure
     * - cloud-vm-cluster
     * - db-node
     * - odb-network
     * - odb-peering-connection
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a list of tag keys that are allowed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateCloudAutonomousVmCluster()
     * - .toCreateCloudExadataInfrastructure()
     * - .toCreateCloudVmCluster()
     * - .toCreateOdbNetwork()
     * - .toCreateOdbPeeringConnection()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [odb](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsservice-oracledatabase_aws.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'odb';
        this.accessLevelList = {
            Write: [
                'AcceptMarketplaceRegistration',
                'CreateCloudAutonomousVmCluster',
                'CreateCloudExadataInfrastructure',
                'CreateCloudVmCluster',
                'CreateDbNode',
                'CreateOdbNetwork',
                'CreateOdbPeeringConnection',
                'CreateOutboundIntegration',
                'DeleteCloudAutonomousVmCluster',
                'DeleteCloudExadataInfrastructure',
                'DeleteCloudVmCluster',
                'DeleteDbNode',
                'DeleteOdbNetwork',
                'DeleteOdbPeeringConnection',
                'DeleteResourcePolicy',
                'InitializeService',
                'PutResourcePolicy',
                'RebootDbNode',
                'StartDbNode',
                'StopDbNode',
                'UpdateCloudExadataInfrastructure',
                'UpdateOdbNetwork'
            ],
            Read: [
                'GetCloudAutonomousVmCluster',
                'GetCloudExadataInfrastructure',
                'GetCloudExadataInfrastructureUnallocatedResources',
                'GetCloudVmCluster',
                'GetDbNode',
                'GetDbServer',
                'GetOciOnboardingStatus',
                'GetOdbNetwork',
                'GetOdbPeeringConnection',
                'GetResourcePolicy',
                'ListAutonomousVirtualMachines',
                'ListDbServers',
                'ListDbSystemShapes',
                'ListGiVersions',
                'ListSystemVersions',
                'ListTagsForResource'
            ],
            List: [
                'ListCloudAutonomousVmClusters',
                'ListCloudExadataInfrastructures',
                'ListCloudVmClusters',
                'ListDbNodes',
                'ListOdbNetworks',
                'ListOdbPeeringConnections'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Odb = Odb;
//# sourceMappingURL=data:application/json;base64,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