"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Servicequotas = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [servicequotas](https://docs.aws.amazon.com/service-authorization/latest/reference/list_servicequotas.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Servicequotas extends shared_1.PolicyStatement {
    /**
     * Grants permission to associate the Service Quotas template with your organization
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - organizations:DescribeOrganization
     * - organizations:EnableAWSServiceAccess
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_AssociateServiceQuotaTemplate.html
     */
    toAssociateServiceQuotaTemplate() {
        return this.to('AssociateServiceQuotaTemplate');
    }
    /**
     * Grants permission to submit a request to create a support case for an existing quota increase request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_CreateSupportCase.html
     */
    toCreateSupportCase() {
        return this.to('CreateSupportCase');
    }
    /**
     * Grants permission to remove the specified service quota from the service quota template
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - organizations:DescribeOrganization
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_DeleteServiceQuotaIncreaseRequestFromTemplate.html
     */
    toDeleteServiceQuotaIncreaseRequestFromTemplate() {
        return this.to('DeleteServiceQuotaIncreaseRequestFromTemplate');
    }
    /**
     * Grants permission to disassociate the Service Quotas template from your organization
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - organizations:DescribeOrganization
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_DisassociateServiceQuotaTemplate.html
     */
    toDisassociateServiceQuotaTemplate() {
        return this.to('DisassociateServiceQuotaTemplate');
    }
    /**
     * Grants permission to return the details for the specified service quota, including the AWS default value
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_GetAWSDefaultServiceQuota.html
     */
    toGetAWSDefaultServiceQuota() {
        return this.to('GetAWSDefaultServiceQuota');
    }
    /**
     * Grants permission to retrieve the ServiceQuotaTemplateAssociationStatus value, which tells you if the Service Quotas template is associated with an organization
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - organizations:DescribeOrganization
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_GetAssociationForServiceQuotaTemplate.html
     */
    toGetAssociationForServiceQuotaTemplate() {
        return this.to('GetAssociationForServiceQuotaTemplate');
    }
    /**
     * Grants permission to retrieve the details for a particular service quota increase request
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_GetRequestedServiceQuotaChange.html
     */
    toGetRequestedServiceQuotaChange() {
        return this.to('GetRequestedServiceQuotaChange');
    }
    /**
     * Grants permission to return the details for the specified service quota, including the applied value
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - autoscaling:DescribeAccountLimits
     * - cloudformation:DescribeAccountLimits
     * - dynamodb:DescribeLimits
     * - elasticloadbalancing:DescribeAccountLimits
     * - iam:GetAccountSummary
     * - kinesis:DescribeLimits
     * - rds:DescribeAccountAttributes
     * - route53:GetAccountLimit
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_GetServiceQuota.html
     */
    toGetServiceQuota() {
        return this.to('GetServiceQuota');
    }
    /**
     * Grants permission to retrieve the details for a service quota increase request from the service quota template
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - organizations:DescribeOrganization
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_GetServiceQuotaIncreaseRequestFromTemplate.html
     */
    toGetServiceQuotaIncreaseRequestFromTemplate() {
        return this.to('GetServiceQuotaIncreaseRequestFromTemplate');
    }
    /**
     * Grants permission to list all default service quotas for the specified AWS service
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListAWSDefaultServiceQuotas.html
     */
    toListAWSDefaultServiceQuotas() {
        return this.to('ListAWSDefaultServiceQuotas');
    }
    /**
     * Grants permission to request a list of the changes to quotas for a service
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListRequestedServiceQuotaChangeHistory.html
     */
    toListRequestedServiceQuotaChangeHistory() {
        return this.to('ListRequestedServiceQuotaChangeHistory');
    }
    /**
     * Grants permission to request a list of the changes to specific service quotas
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListRequestedServiceQuotaChangeHistoryByQuota.html
     */
    toListRequestedServiceQuotaChangeHistoryByQuota() {
        return this.to('ListRequestedServiceQuotaChangeHistoryByQuota');
    }
    /**
     * Grants permission to return a list of the service quota increase requests from the service quota template
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - organizations:DescribeOrganization
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListServiceQuotaIncreaseRequestsInTemplate.html
     */
    toListServiceQuotaIncreaseRequestsInTemplate() {
        return this.to('ListServiceQuotaIncreaseRequestsInTemplate');
    }
    /**
     * Grants permission to list all service quotas for the specified AWS service, in that account, in that Region
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - autoscaling:DescribeAccountLimits
     * - cloudformation:DescribeAccountLimits
     * - dynamodb:DescribeLimits
     * - elasticloadbalancing:DescribeAccountLimits
     * - iam:GetAccountSummary
     * - kinesis:DescribeLimits
     * - rds:DescribeAccountAttributes
     * - route53:GetAccountLimit
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListServiceQuotas.html
     */
    toListServiceQuotas() {
        return this.to('ListServiceQuotas');
    }
    /**
     * Grants permission to list the AWS services available in Service Quotas
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListServices.html
     */
    toListServices() {
        return this.to('ListServices');
    }
    /**
     * Grants permission to view the existing tags on a SQ resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to define and add a quota to the service quota template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifService()
     *
     * Dependent actions:
     * - organizations:DescribeOrganization
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_PutServiceQuotaIncreaseRequestIntoTemplate.html
     */
    toPutServiceQuotaIncreaseRequestIntoTemplate() {
        return this.to('PutServiceQuotaIncreaseRequestIntoTemplate');
    }
    /**
     * Grants permission to submit the request for a service quota increase
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifService()
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_RequestServiceQuotaIncrease.html
     */
    toRequestServiceQuotaIncrease() {
        return this.to('RequestServiceQuotaIncrease');
    }
    /**
     * Grants permission to associate a set of tags with an existing SQ resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove a set of tags from a SQ resource, where tags to be removed match a set of customer-supplied tag keys
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/servicequotas/2019-06-24/apireference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Adds a resource of type quota to the statement
     *
     * https://docs.aws.amazon.com/servicequotas/latest/userguide/identity-access-management.html#resources
     *
     * @param serviceCode - Identifier for the serviceCode.
     * @param quotaCode - Identifier for the quotaCode.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onQuota(serviceCode, quotaCode, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:servicequotas:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:${serviceCode}/${quotaCode}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the specified AWS service
     *
     * https://docs.aws.amazon.com/servicequotas/latest/userguide/identity-access-management.html#condition-keys
     *
     * Applies to actions:
     * - .toPutServiceQuotaIncreaseRequestIntoTemplate()
     * - .toRequestServiceQuotaIncrease()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifService(value, operator) {
        return this.if(`service`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [servicequotas](https://docs.aws.amazon.com/service-authorization/latest/reference/list_servicequotas.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'servicequotas';
        this.accessLevelList = {
            Write: [
                'AssociateServiceQuotaTemplate',
                'CreateSupportCase',
                'DeleteServiceQuotaIncreaseRequestFromTemplate',
                'DisassociateServiceQuotaTemplate',
                'PutServiceQuotaIncreaseRequestIntoTemplate',
                'RequestServiceQuotaIncrease'
            ],
            Read: [
                'GetAWSDefaultServiceQuota',
                'GetAssociationForServiceQuotaTemplate',
                'GetRequestedServiceQuotaChange',
                'GetServiceQuota',
                'GetServiceQuotaIncreaseRequestFromTemplate',
                'ListAWSDefaultServiceQuotas',
                'ListRequestedServiceQuotaChangeHistory',
                'ListRequestedServiceQuotaChangeHistoryByQuota',
                'ListServiceQuotaIncreaseRequestsInTemplate',
                'ListServiceQuotas',
                'ListServices',
                'ListTagsForResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Servicequotas = Servicequotas;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic2VydmljZXF1b3Rhcy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInNlcnZpY2VxdW90YXMudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQ0EseUNBQXlEO0FBR3pEOzs7O0dBSUc7QUFDSCxNQUFhLGFBQWMsU0FBUSx3QkFBZTtJQUdoRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksK0JBQStCO1FBQ3BDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO0lBQ2xELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFtQixDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLCtDQUErQztRQUNwRCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsK0NBQStDLENBQUMsQ0FBQztJQUNsRSxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksa0NBQWtDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQ0FBa0MsQ0FBQyxDQUFDO0lBQ3JELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwyQkFBMkI7UUFDaEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDJCQUEyQixDQUFDLENBQUM7SUFDOUMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLHVDQUF1QztRQUM1QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsdUNBQXVDLENBQUMsQ0FBQztJQUMxRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0NBQWdDO1FBQ3JDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO0lBQ25ELENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7OztPQWdCRztJQUNJLGlCQUFpQjtRQUN0QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsaUJBQWlCLENBQUMsQ0FBQztJQUNwQyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksNENBQTRDO1FBQ2pELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw0Q0FBNEMsQ0FBQyxDQUFDO0lBQy9ELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw2QkFBNkI7UUFDbEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDZCQUE2QixDQUFDLENBQUM7SUFDaEQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHdDQUF3QztRQUM3QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsd0NBQXdDLENBQUMsQ0FBQztJQUMzRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksK0NBQStDO1FBQ3BELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywrQ0FBK0MsQ0FBQyxDQUFDO0lBQ2xFLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSw0Q0FBNEM7UUFDakQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDRDQUE0QyxDQUFDLENBQUM7SUFDL0QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7O09BZ0JHO0lBQ0ksbUJBQW1CO1FBQ3hCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO0lBQ3RDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxjQUFjO1FBQ25CLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxjQUFjLENBQUMsQ0FBQztJQUNqQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSw0Q0FBNEM7UUFDakQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDRDQUE0QyxDQUFDLENBQUM7SUFDL0QsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLDZCQUE2QjtRQUNsQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsNkJBQTZCLENBQUMsQ0FBQztJQUNoRCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGFBQWE7UUFDbEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBQ2hDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxlQUFlO1FBQ3BCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBK0JEOzs7Ozs7Ozs7O09BVUc7SUFDSSxPQUFPLENBQUMsV0FBbUIsRUFBRSxTQUFpQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQzFHLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGtCQUFtQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsSUFBSyxXQUFZLElBQUssU0FBVSxFQUFFLENBQUMsQ0FBQztJQUNwTCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxlQUFlLENBQUMsTUFBYyxFQUFFLEtBQXdCLEVBQUUsUUFBNEI7UUFDM0YsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFtQixNQUFPLEVBQUUsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2hGLENBQUM7SUFFRDs7Ozs7Ozs7T0FRRztJQUNJLGdCQUFnQixDQUFDLE1BQWMsRUFBRSxLQUF3QixFQUFFLFFBQTRCO1FBQzVGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBb0IsTUFBTyxFQUFFLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNqRixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxZQUFZLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUN4RSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsYUFBYSxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDakUsQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksU0FBUyxDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDckUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFNBQVMsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQzdELENBQUM7SUFFRDs7O09BR0c7SUFDSCxZQUFZLEtBQWdDO1FBQzFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQztRQW5ZUixrQkFBYSxHQUFHLGVBQWUsQ0FBQztRQXFSN0Isb0JBQWUsR0FBb0I7WUFDM0MsS0FBSyxFQUFFO2dCQUNMLCtCQUErQjtnQkFDL0IsbUJBQW1CO2dCQUNuQiwrQ0FBK0M7Z0JBQy9DLGtDQUFrQztnQkFDbEMsNENBQTRDO2dCQUM1Qyw2QkFBNkI7YUFDOUI7WUFDRCxJQUFJLEVBQUU7Z0JBQ0osMkJBQTJCO2dCQUMzQix1Q0FBdUM7Z0JBQ3ZDLGdDQUFnQztnQkFDaEMsaUJBQWlCO2dCQUNqQiw0Q0FBNEM7Z0JBQzVDLDZCQUE2QjtnQkFDN0Isd0NBQXdDO2dCQUN4QywrQ0FBK0M7Z0JBQy9DLDRDQUE0QztnQkFDNUMsbUJBQW1CO2dCQUNuQixjQUFjO2dCQUNkLHFCQUFxQjthQUN0QjtZQUNELE9BQU8sRUFBRTtnQkFDUCxhQUFhO2dCQUNiLGVBQWU7YUFDaEI7U0FDRixDQUFDO0lBb0ZGLENBQUM7Q0FDRjtBQXRZRCxzQ0FzWUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBBY2Nlc3NMZXZlbExpc3QgfSBmcm9tICcuLi8uLi9zaGFyZWQvYWNjZXNzLWxldmVsJztcbmltcG9ydCB7IFBvbGljeVN0YXRlbWVudCwgT3BlcmF0b3IgfSBmcm9tICcuLi8uLi9zaGFyZWQnO1xuaW1wb3J0IHsgYXdzX2lhbSBhcyBpYW0gfSBmcm9tIFwiYXdzLWNkay1saWJcIjtcblxuLyoqXG4gKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW3NlcnZpY2VxdW90YXNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXJ2aWNlLWF1dGhvcml6YXRpb24vbGF0ZXN0L3JlZmVyZW5jZS9saXN0X3NlcnZpY2VxdW90YXMuaHRtbCkuXG4gKlxuICogQHBhcmFtIHNpZCBbU0lEXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX3NpZC5odG1sKSBvZiB0aGUgc3RhdGVtZW50XG4gKi9cbmV4cG9ydCBjbGFzcyBTZXJ2aWNlcXVvdGFzIGV4dGVuZHMgUG9saWN5U3RhdGVtZW50IHtcbiAgcHVibGljIHNlcnZpY2VQcmVmaXggPSAnc2VydmljZXF1b3Rhcyc7XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGFzc29jaWF0ZSB0aGUgU2VydmljZSBRdW90YXMgdGVtcGxhdGUgd2l0aCB5b3VyIG9yZ2FuaXphdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIG9yZ2FuaXphdGlvbnM6RGVzY3JpYmVPcmdhbml6YXRpb25cbiAgICogLSBvcmdhbml6YXRpb25zOkVuYWJsZUFXU1NlcnZpY2VBY2Nlc3NcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2VxdW90YXMvMjAxOS0wNi0yNC9hcGlyZWZlcmVuY2UvQVBJX0Fzc29jaWF0ZVNlcnZpY2VRdW90YVRlbXBsYXRlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Fzc29jaWF0ZVNlcnZpY2VRdW90YVRlbXBsYXRlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdBc3NvY2lhdGVTZXJ2aWNlUXVvdGFUZW1wbGF0ZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHN1Ym1pdCBhIHJlcXVlc3QgdG8gY3JlYXRlIGEgc3VwcG9ydCBjYXNlIGZvciBhbiBleGlzdGluZyBxdW90YSBpbmNyZWFzZSByZXF1ZXN0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2VxdW90YXMvMjAxOS0wNi0yNC9hcGlyZWZlcmVuY2UvQVBJX0NyZWF0ZVN1cHBvcnRDYXNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZVN1cHBvcnRDYXNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVTdXBwb3J0Q2FzZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJlbW92ZSB0aGUgc3BlY2lmaWVkIHNlcnZpY2UgcXVvdGEgZnJvbSB0aGUgc2VydmljZSBxdW90YSB0ZW1wbGF0ZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIG9yZ2FuaXphdGlvbnM6RGVzY3JpYmVPcmdhbml6YXRpb25cbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2VxdW90YXMvMjAxOS0wNi0yNC9hcGlyZWZlcmVuY2UvQVBJX0RlbGV0ZVNlcnZpY2VRdW90YUluY3JlYXNlUmVxdWVzdEZyb21UZW1wbGF0ZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVTZXJ2aWNlUXVvdGFJbmNyZWFzZVJlcXVlc3RGcm9tVGVtcGxhdGUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZVNlcnZpY2VRdW90YUluY3JlYXNlUmVxdWVzdEZyb21UZW1wbGF0ZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRpc2Fzc29jaWF0ZSB0aGUgU2VydmljZSBRdW90YXMgdGVtcGxhdGUgZnJvbSB5b3VyIG9yZ2FuaXphdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIG9yZ2FuaXphdGlvbnM6RGVzY3JpYmVPcmdhbml6YXRpb25cbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2VxdW90YXMvMjAxOS0wNi0yNC9hcGlyZWZlcmVuY2UvQVBJX0Rpc2Fzc29jaWF0ZVNlcnZpY2VRdW90YVRlbXBsYXRlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rpc2Fzc29jaWF0ZVNlcnZpY2VRdW90YVRlbXBsYXRlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEaXNhc3NvY2lhdGVTZXJ2aWNlUXVvdGFUZW1wbGF0ZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHVybiB0aGUgZGV0YWlscyBmb3IgdGhlIHNwZWNpZmllZCBzZXJ2aWNlIHF1b3RhLCBpbmNsdWRpbmcgdGhlIEFXUyBkZWZhdWx0IHZhbHVlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZXF1b3Rhcy8yMDE5LTA2LTI0L2FwaXJlZmVyZW5jZS9BUElfR2V0QVdTRGVmYXVsdFNlcnZpY2VRdW90YS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRBV1NEZWZhdWx0U2VydmljZVF1b3RhKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRBV1NEZWZhdWx0U2VydmljZVF1b3RhJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgdGhlIFNlcnZpY2VRdW90YVRlbXBsYXRlQXNzb2NpYXRpb25TdGF0dXMgdmFsdWUsIHdoaWNoIHRlbGxzIHlvdSBpZiB0aGUgU2VydmljZSBRdW90YXMgdGVtcGxhdGUgaXMgYXNzb2NpYXRlZCB3aXRoIGFuIG9yZ2FuaXphdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gb3JnYW5pemF0aW9uczpEZXNjcmliZU9yZ2FuaXphdGlvblxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZXF1b3Rhcy8yMDE5LTA2LTI0L2FwaXJlZmVyZW5jZS9BUElfR2V0QXNzb2NpYXRpb25Gb3JTZXJ2aWNlUXVvdGFUZW1wbGF0ZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRBc3NvY2lhdGlvbkZvclNlcnZpY2VRdW90YVRlbXBsYXRlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRBc3NvY2lhdGlvbkZvclNlcnZpY2VRdW90YVRlbXBsYXRlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgdGhlIGRldGFpbHMgZm9yIGEgcGFydGljdWxhciBzZXJ2aWNlIHF1b3RhIGluY3JlYXNlIHJlcXVlc3RcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXJ2aWNlcXVvdGFzLzIwMTktMDYtMjQvYXBpcmVmZXJlbmNlL0FQSV9HZXRSZXF1ZXN0ZWRTZXJ2aWNlUXVvdGFDaGFuZ2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0UmVxdWVzdGVkU2VydmljZVF1b3RhQ2hhbmdlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRSZXF1ZXN0ZWRTZXJ2aWNlUXVvdGFDaGFuZ2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gdGhlIGRldGFpbHMgZm9yIHRoZSBzcGVjaWZpZWQgc2VydmljZSBxdW90YSwgaW5jbHVkaW5nIHRoZSBhcHBsaWVkIHZhbHVlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBhdXRvc2NhbGluZzpEZXNjcmliZUFjY291bnRMaW1pdHNcbiAgICogLSBjbG91ZGZvcm1hdGlvbjpEZXNjcmliZUFjY291bnRMaW1pdHNcbiAgICogLSBkeW5hbW9kYjpEZXNjcmliZUxpbWl0c1xuICAgKiAtIGVsYXN0aWNsb2FkYmFsYW5jaW5nOkRlc2NyaWJlQWNjb3VudExpbWl0c1xuICAgKiAtIGlhbTpHZXRBY2NvdW50U3VtbWFyeVxuICAgKiAtIGtpbmVzaXM6RGVzY3JpYmVMaW1pdHNcbiAgICogLSByZHM6RGVzY3JpYmVBY2NvdW50QXR0cmlidXRlc1xuICAgKiAtIHJvdXRlNTM6R2V0QWNjb3VudExpbWl0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXJ2aWNlcXVvdGFzLzIwMTktMDYtMjQvYXBpcmVmZXJlbmNlL0FQSV9HZXRTZXJ2aWNlUXVvdGEuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0U2VydmljZVF1b3RhKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRTZXJ2aWNlUXVvdGEnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSB0aGUgZGV0YWlscyBmb3IgYSBzZXJ2aWNlIHF1b3RhIGluY3JlYXNlIHJlcXVlc3QgZnJvbSB0aGUgc2VydmljZSBxdW90YSB0ZW1wbGF0ZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gb3JnYW5pemF0aW9uczpEZXNjcmliZU9yZ2FuaXphdGlvblxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZXF1b3Rhcy8yMDE5LTA2LTI0L2FwaXJlZmVyZW5jZS9BUElfR2V0U2VydmljZVF1b3RhSW5jcmVhc2VSZXF1ZXN0RnJvbVRlbXBsYXRlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFNlcnZpY2VRdW90YUluY3JlYXNlUmVxdWVzdEZyb21UZW1wbGF0ZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0U2VydmljZVF1b3RhSW5jcmVhc2VSZXF1ZXN0RnJvbVRlbXBsYXRlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBhbGwgZGVmYXVsdCBzZXJ2aWNlIHF1b3RhcyBmb3IgdGhlIHNwZWNpZmllZCBBV1Mgc2VydmljZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2VxdW90YXMvMjAxOS0wNi0yNC9hcGlyZWZlcmVuY2UvQVBJX0xpc3RBV1NEZWZhdWx0U2VydmljZVF1b3Rhcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0QVdTRGVmYXVsdFNlcnZpY2VRdW90YXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RBV1NEZWZhdWx0U2VydmljZVF1b3RhcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJlcXVlc3QgYSBsaXN0IG9mIHRoZSBjaGFuZ2VzIHRvIHF1b3RhcyBmb3IgYSBzZXJ2aWNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZXF1b3Rhcy8yMDE5LTA2LTI0L2FwaXJlZmVyZW5jZS9BUElfTGlzdFJlcXVlc3RlZFNlcnZpY2VRdW90YUNoYW5nZUhpc3RvcnkuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFJlcXVlc3RlZFNlcnZpY2VRdW90YUNoYW5nZUhpc3RvcnkoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RSZXF1ZXN0ZWRTZXJ2aWNlUXVvdGFDaGFuZ2VIaXN0b3J5Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmVxdWVzdCBhIGxpc3Qgb2YgdGhlIGNoYW5nZXMgdG8gc3BlY2lmaWMgc2VydmljZSBxdW90YXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXJ2aWNlcXVvdGFzLzIwMTktMDYtMjQvYXBpcmVmZXJlbmNlL0FQSV9MaXN0UmVxdWVzdGVkU2VydmljZVF1b3RhQ2hhbmdlSGlzdG9yeUJ5UXVvdGEuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFJlcXVlc3RlZFNlcnZpY2VRdW90YUNoYW5nZUhpc3RvcnlCeVF1b3RhKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0UmVxdWVzdGVkU2VydmljZVF1b3RhQ2hhbmdlSGlzdG9yeUJ5UXVvdGEnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4gYSBsaXN0IG9mIHRoZSBzZXJ2aWNlIHF1b3RhIGluY3JlYXNlIHJlcXVlc3RzIGZyb20gdGhlIHNlcnZpY2UgcXVvdGEgdGVtcGxhdGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIG9yZ2FuaXphdGlvbnM6RGVzY3JpYmVPcmdhbml6YXRpb25cbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2VxdW90YXMvMjAxOS0wNi0yNC9hcGlyZWZlcmVuY2UvQVBJX0xpc3RTZXJ2aWNlUXVvdGFJbmNyZWFzZVJlcXVlc3RzSW5UZW1wbGF0ZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0U2VydmljZVF1b3RhSW5jcmVhc2VSZXF1ZXN0c0luVGVtcGxhdGUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RTZXJ2aWNlUXVvdGFJbmNyZWFzZVJlcXVlc3RzSW5UZW1wbGF0ZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgYWxsIHNlcnZpY2UgcXVvdGFzIGZvciB0aGUgc3BlY2lmaWVkIEFXUyBzZXJ2aWNlLCBpbiB0aGF0IGFjY291bnQsIGluIHRoYXQgUmVnaW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBhdXRvc2NhbGluZzpEZXNjcmliZUFjY291bnRMaW1pdHNcbiAgICogLSBjbG91ZGZvcm1hdGlvbjpEZXNjcmliZUFjY291bnRMaW1pdHNcbiAgICogLSBkeW5hbW9kYjpEZXNjcmliZUxpbWl0c1xuICAgKiAtIGVsYXN0aWNsb2FkYmFsYW5jaW5nOkRlc2NyaWJlQWNjb3VudExpbWl0c1xuICAgKiAtIGlhbTpHZXRBY2NvdW50U3VtbWFyeVxuICAgKiAtIGtpbmVzaXM6RGVzY3JpYmVMaW1pdHNcbiAgICogLSByZHM6RGVzY3JpYmVBY2NvdW50QXR0cmlidXRlc1xuICAgKiAtIHJvdXRlNTM6R2V0QWNjb3VudExpbWl0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXJ2aWNlcXVvdGFzLzIwMTktMDYtMjQvYXBpcmVmZXJlbmNlL0FQSV9MaXN0U2VydmljZVF1b3Rhcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0U2VydmljZVF1b3RhcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdFNlcnZpY2VRdW90YXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IHRoZSBBV1Mgc2VydmljZXMgYXZhaWxhYmxlIGluIFNlcnZpY2UgUXVvdGFzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZXF1b3Rhcy8yMDE5LTA2LTI0L2FwaXJlZmVyZW5jZS9BUElfTGlzdFNlcnZpY2VzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RTZXJ2aWNlcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdFNlcnZpY2VzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdmlldyB0aGUgZXhpc3RpbmcgdGFncyBvbiBhIFNRIHJlc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZXF1b3Rhcy8yMDE5LTA2LTI0L2FwaXJlZmVyZW5jZS9BUElfTGlzdFRhZ3NGb3JSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0VGFnc0ZvclJlc291cmNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0VGFnc0ZvclJlc291cmNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVmaW5lIGFuZCBhZGQgYSBxdW90YSB0byB0aGUgc2VydmljZSBxdW90YSB0ZW1wbGF0ZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmU2VydmljZSgpXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIG9yZ2FuaXphdGlvbnM6RGVzY3JpYmVPcmdhbml6YXRpb25cbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2VxdW90YXMvMjAxOS0wNi0yNC9hcGlyZWZlcmVuY2UvQVBJX1B1dFNlcnZpY2VRdW90YUluY3JlYXNlUmVxdWVzdEludG9UZW1wbGF0ZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9QdXRTZXJ2aWNlUXVvdGFJbmNyZWFzZVJlcXVlc3RJbnRvVGVtcGxhdGUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1B1dFNlcnZpY2VRdW90YUluY3JlYXNlUmVxdWVzdEludG9UZW1wbGF0ZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHN1Ym1pdCB0aGUgcmVxdWVzdCBmb3IgYSBzZXJ2aWNlIHF1b3RhIGluY3JlYXNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZTZXJ2aWNlKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2VxdW90YXMvMjAxOS0wNi0yNC9hcGlyZWZlcmVuY2UvQVBJX1JlcXVlc3RTZXJ2aWNlUXVvdGFJbmNyZWFzZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9SZXF1ZXN0U2VydmljZVF1b3RhSW5jcmVhc2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1JlcXVlc3RTZXJ2aWNlUXVvdGFJbmNyZWFzZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGFzc29jaWF0ZSBhIHNldCBvZiB0YWdzIHdpdGggYW4gZXhpc3RpbmcgU1EgcmVzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXJ2aWNlcXVvdGFzLzIwMTktMDYtMjQvYXBpcmVmZXJlbmNlL0FQSV9UYWdSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9UYWdSZXNvdXJjZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVGFnUmVzb3VyY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZW1vdmUgYSBzZXQgb2YgdGFncyBmcm9tIGEgU1EgcmVzb3VyY2UsIHdoZXJlIHRhZ3MgdG8gYmUgcmVtb3ZlZCBtYXRjaCBhIHNldCBvZiBjdXN0b21lci1zdXBwbGllZCB0YWcga2V5c1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFRhZ2dpbmdcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2VxdW90YXMvMjAxOS0wNi0yNC9hcGlyZWZlcmVuY2UvQVBJX1VudGFnUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVW50YWdSZXNvdXJjZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVW50YWdSZXNvdXJjZScpO1xuICB9XG5cbiAgcHJvdGVjdGVkIGFjY2Vzc0xldmVsTGlzdDogQWNjZXNzTGV2ZWxMaXN0ID0ge1xuICAgIFdyaXRlOiBbXG4gICAgICAnQXNzb2NpYXRlU2VydmljZVF1b3RhVGVtcGxhdGUnLFxuICAgICAgJ0NyZWF0ZVN1cHBvcnRDYXNlJyxcbiAgICAgICdEZWxldGVTZXJ2aWNlUXVvdGFJbmNyZWFzZVJlcXVlc3RGcm9tVGVtcGxhdGUnLFxuICAgICAgJ0Rpc2Fzc29jaWF0ZVNlcnZpY2VRdW90YVRlbXBsYXRlJyxcbiAgICAgICdQdXRTZXJ2aWNlUXVvdGFJbmNyZWFzZVJlcXVlc3RJbnRvVGVtcGxhdGUnLFxuICAgICAgJ1JlcXVlc3RTZXJ2aWNlUXVvdGFJbmNyZWFzZSdcbiAgICBdLFxuICAgIFJlYWQ6IFtcbiAgICAgICdHZXRBV1NEZWZhdWx0U2VydmljZVF1b3RhJyxcbiAgICAgICdHZXRBc3NvY2lhdGlvbkZvclNlcnZpY2VRdW90YVRlbXBsYXRlJyxcbiAgICAgICdHZXRSZXF1ZXN0ZWRTZXJ2aWNlUXVvdGFDaGFuZ2UnLFxuICAgICAgJ0dldFNlcnZpY2VRdW90YScsXG4gICAgICAnR2V0U2VydmljZVF1b3RhSW5jcmVhc2VSZXF1ZXN0RnJvbVRlbXBsYXRlJyxcbiAgICAgICdMaXN0QVdTRGVmYXVsdFNlcnZpY2VRdW90YXMnLFxuICAgICAgJ0xpc3RSZXF1ZXN0ZWRTZXJ2aWNlUXVvdGFDaGFuZ2VIaXN0b3J5JyxcbiAgICAgICdMaXN0UmVxdWVzdGVkU2VydmljZVF1b3RhQ2hhbmdlSGlzdG9yeUJ5UXVvdGEnLFxuICAgICAgJ0xpc3RTZXJ2aWNlUXVvdGFJbmNyZWFzZVJlcXVlc3RzSW5UZW1wbGF0ZScsXG4gICAgICAnTGlzdFNlcnZpY2VRdW90YXMnLFxuICAgICAgJ0xpc3RTZXJ2aWNlcycsXG4gICAgICAnTGlzdFRhZ3NGb3JSZXNvdXJjZSdcbiAgICBdLFxuICAgIFRhZ2dpbmc6IFtcbiAgICAgICdUYWdSZXNvdXJjZScsXG4gICAgICAnVW50YWdSZXNvdXJjZSdcbiAgICBdXG4gIH07XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIHF1b3RhIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2VxdW90YXMvbGF0ZXN0L3VzZXJndWlkZS9pZGVudGl0eS1hY2Nlc3MtbWFuYWdlbWVudC5odG1sI3Jlc291cmNlc1xuICAgKlxuICAgKiBAcGFyYW0gc2VydmljZUNvZGUgLSBJZGVudGlmaWVyIGZvciB0aGUgc2VydmljZUNvZGUuXG4gICAqIEBwYXJhbSBxdW90YUNvZGUgLSBJZGVudGlmaWVyIGZvciB0aGUgcXVvdGFDb2RlLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICovXG4gIHB1YmxpYyBvblF1b3RhKHNlcnZpY2VDb2RlOiBzdHJpbmcsIHF1b3RhQ29kZTogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OnNlcnZpY2VxdW90YXM6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OiR7IHNlcnZpY2VDb2RlIH0vJHsgcXVvdGFDb2RlIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0aGUgdGFncyB0aGF0IGFyZSBwYXNzZWQgaW4gdGhlIHJlcXVlc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19jb25kaXRpb24ta2V5cy5odG1sI2NvbmRpdGlvbi1rZXlzLXJlcXVlc3R0YWdcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b1RhZ1Jlc291cmNlKClcbiAgICpcbiAgICogQHBhcmFtIHRhZ0tleSBUaGUgdGFnIGtleSB0byBjaGVja1xuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1JlcXVlc3RUYWcodGFnS2V5OiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBhd3M6UmVxdWVzdFRhZy8keyB0YWdLZXkgfWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRoZSB0YWdzIGFzc29jaWF0ZWQgd2l0aCB0aGUgcmVzb3VyY2VcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19jb25kaXRpb24ta2V5cy5odG1sI2NvbmRpdGlvbi1rZXlzLXJlc291cmNldGFnXG4gICAqXG4gICAqIEBwYXJhbSB0YWdLZXkgVGhlIHRhZyBrZXkgdG8gY2hlY2tcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NSZXNvdXJjZVRhZyh0YWdLZXk6IHN0cmluZywgdmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGF3czpSZXNvdXJjZVRhZy8keyB0YWdLZXkgfWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRoZSB0YWcga2V5cyB0aGF0IGFyZSBwYXNzZWQgaW4gdGhlIHJlcXVlc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19jb25kaXRpb24ta2V5cy5odG1sI2NvbmRpdGlvbi1rZXlzLXRhZ2tleXNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b1RhZ1Jlc291cmNlKClcbiAgICogLSAudG9VbnRhZ1Jlc291cmNlKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NUYWdLZXlzKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBhd3M6VGFnS2V5c2AsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRoZSBzcGVjaWZpZWQgQVdTIHNlcnZpY2VcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2VxdW90YXMvbGF0ZXN0L3VzZXJndWlkZS9pZGVudGl0eS1hY2Nlc3MtbWFuYWdlbWVudC5odG1sI2NvbmRpdGlvbi1rZXlzXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9QdXRTZXJ2aWNlUXVvdGFJbmNyZWFzZVJlcXVlc3RJbnRvVGVtcGxhdGUoKVxuICAgKiAtIC50b1JlcXVlc3RTZXJ2aWNlUXVvdGFJbmNyZWFzZSgpXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmU2VydmljZSh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgc2VydmljZWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbc2VydmljZXF1b3Rhc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2UtYXV0aG9yaXphdGlvbi9sYXRlc3QvcmVmZXJlbmNlL2xpc3Rfc2VydmljZXF1b3Rhcy5odG1sKS5cbiAgICpcbiAgICovXG4gIGNvbnN0cnVjdG9yKHByb3BzPzogaWFtLlBvbGljeVN0YXRlbWVudFByb3BzKSB7XG4gICAgc3VwZXIocHJvcHMpO1xuICB9XG59XG4iXX0=