import { AccessLevelList } from '../../shared/access-level';
import { PolicyStatement, Operator } from '../../shared';
import { aws_iam as iam } from "aws-cdk-lib";
/**
 * Statement provider for service [ses-v2](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonsimpleemailservicev2.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class SesV2 extends PolicyStatement {
    servicePrefix: string;
    /**
     * Grants permission to get metric data on your activity
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_BatchGetMetricData.html
     */
    toBatchGetMetricData(): this;
    /**
     * Grants permission to cancel an export job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifExportSourceType()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CancelExportJob.html
     */
    toCancelExportJob(): this;
    /**
     * Grants permission to create a new configuration set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateConfigurationSet.html
     */
    toCreateConfigurationSet(): this;
    /**
     * Grants permission to create a configuration set event destination
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateConfigurationSetEventDestination.html
     */
    toCreateConfigurationSetEventDestination(): this;
    /**
     * Grants permission to create a contact
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateContact.html
     */
    toCreateContact(): this;
    /**
     * Grants permission to create a contact list
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateContactList.html
     */
    toCreateContactList(): this;
    /**
     * Grants permission to create a new custom verification email template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateCustomVerificationEmailTemplate.html
     */
    toCreateCustomVerificationEmailTemplate(): this;
    /**
     * Grants permission to create a new pool of dedicated IP addresses
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateDedicatedIpPool.html
     */
    toCreateDedicatedIpPool(): this;
    /**
     * Grants permission to create a new predictive inbox placement test
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateDeliverabilityTestReport.html
     */
    toCreateDeliverabilityTestReport(): this;
    /**
     * Grants permission to start the process of verifying an email identity
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateEmailIdentity.html
     */
    toCreateEmailIdentity(): this;
    /**
     * Grants permission to create the specified sending authorization policy for the given identity
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateEmailIdentityPolicy.html
     */
    toCreateEmailIdentityPolicy(): this;
    /**
     * Grants permission to create an email template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateEmailTemplate.html
     */
    toCreateEmailTemplate(): this;
    /**
     * Grants permission to create an export job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifExportSourceType()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateExportJob.html
     */
    toCreateExportJob(): this;
    /**
     * Grants permission to creates an import job for a data destination
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateImportJob.html
     */
    toCreateImportJob(): this;
    /**
     * Grants permission to create a new multi-region endpoint
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateMultiRegionEndpoint.html
     */
    toCreateMultiRegionEndpoint(): this;
    /**
     * Grants permission to create a new tenant
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateTenant.html
     */
    toCreateTenant(): this;
    /**
     * Grants permission to associate a SES resource to a tenant
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateTenantResourceAssociation.html
     */
    toCreateTenantResourceAssociation(): this;
    /**
     * Grants permission to delete an existing configuration set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DeleteConfigurationSet.html
     */
    toDeleteConfigurationSet(): this;
    /**
     * Grants permission to delete an event destination
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DeleteConfigurationSetEventDestination.html
     */
    toDeleteConfigurationSetEventDestination(): this;
    /**
     * Grants permission to delete a contact from a contact list
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DeleteContact.html
     */
    toDeleteContact(): this;
    /**
     * Grants permission to delete a contact list with all of its contacts
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DeleteContactList.html
     */
    toDeleteContactList(): this;
    /**
     * Grants permission to delete an existing custom verification email template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DeleteCustomVerificationEmailTemplate.html
     */
    toDeleteCustomVerificationEmailTemplate(): this;
    /**
     * Grants permission to delete a dedicated IP pool
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DeleteDedicatedIpPool.html
     */
    toDeleteDedicatedIpPool(): this;
    /**
     * Grants permission to delete an email identity
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DeleteEmailIdentity.html
     */
    toDeleteEmailIdentity(): this;
    /**
     * Grants permission to delete the specified sending authorization policy for the given identity (an email address or a domain)
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DeleteEmailIdentityPolicy.html
     */
    toDeleteEmailIdentityPolicy(): this;
    /**
     * Grants permission to delete an email template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DeleteEmailTemplate.html
     */
    toDeleteEmailTemplate(): this;
    /**
     * Grants permission to delete a multi-region endpoint
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DeleteMultiRegionEndpoint.html
     */
    toDeleteMultiRegionEndpoint(): this;
    /**
     * Grants permission to remove an email address from the suppression list for your account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DeleteSuppressedDestination.html
     */
    toDeleteSuppressedDestination(): this;
    /**
     * Grants permission to delete a tenant
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DeleteTenant.html
     */
    toDeleteTenant(): this;
    /**
     * Grants permission to remove an associated SES resource from a tenant
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DeleteTenantResourceAssociation.html
     */
    toDeleteTenantResourceAssociation(): this;
    /**
     * Grants permission to get information about the email-sending status and capabilities for your account
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetAccount.html
     */
    toGetAccount(): this;
    /**
     * Grants permission to retrieve a list of the deny lists on which your dedicated IP addresses or tracked domains appear
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetBlacklistReports.html
     */
    toGetBlacklistReports(): this;
    /**
     * Grants permission to get information about an existing configuration set
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetConfigurationSet.html
     */
    toGetConfigurationSet(): this;
    /**
     * Grants permission to retrieve a list of event destinations that are associated with a configuration set
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetConfigurationSetEventDestinations.html
     */
    toGetConfigurationSetEventDestinations(): this;
    /**
     * Grants permission to return a contact from a contact list
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetContact.html
     */
    toGetContact(): this;
    /**
     * Grants permission to return contact list metadata
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetContactList.html
     */
    toGetContactList(): this;
    /**
     * Grants permission to return the custom email verification template for the template name you specify
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetCustomVerificationEmailTemplate.html
     */
    toGetCustomVerificationEmailTemplate(): this;
    /**
     * Grants permission to get information about a dedicated IP address
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetDedicatedIp.html
     */
    toGetDedicatedIp(): this;
    /**
     * Grants permission to get information about a dedicated IP pool
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetDedicatedIpPool.html
     */
    toGetDedicatedIpPool(): this;
    /**
     * Grants permission to list the dedicated IP addresses a dedicated IP pool
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetDedicatedIps.html
     */
    toGetDedicatedIps(): this;
    /**
     * Grants permission to get the status of the Deliverability dashboard
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetDeliverabilityDashboardOptions.html
     */
    toGetDeliverabilityDashboardOptions(): this;
    /**
     * Grants permission to retrieve the results of a predictive inbox placement test
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetDeliverabilityTestReport.html
     */
    toGetDeliverabilityTestReport(): this;
    /**
     * Grants permission to retrieve all the deliverability data for a specific campaign
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetDomainDeliverabilityCampaign.html
     */
    toGetDomainDeliverabilityCampaign(): this;
    /**
     * Grants permission to retrieve inbox placement and engagement rates for the domains that you use to send email
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetDomainStatisticsReport.html
     */
    toGetDomainStatisticsReport(): this;
    /**
     * Grants permission to get information about a specific identity
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetEmailIdentity.html
     */
    toGetEmailIdentity(): this;
    /**
     * Grants permission to return the requested sending authorization policies for the given identity (an email address or a domain)
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetEmailIdentityPolicies.html
     */
    toGetEmailIdentityPolicies(): this;
    /**
     * Grants permission to return the template object, which includes the subject line, HTML part, and text part for the template you specify
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetEmailTemplate.html
     */
    toGetEmailTemplate(): this;
    /**
     * Grants permission to get information about an export job
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifExportSourceType()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetExportJob.html
     */
    toGetExportJob(): this;
    /**
     * Grants permission to provide information about an import job
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetImportJob.html
     */
    toGetImportJob(): this;
    /**
     * Grants permission to provide insights about a message
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetMessageInsights.html
     */
    toGetMessageInsights(): this;
    /**
     * Grants permission to get information about a multi-region endpoint
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetMultiRegionEndpoint.html
     */
    toGetMultiRegionEndpoint(): this;
    /**
     * Grants permission to retrieve information about a reputation entity's status
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetReputationEntity.html
     */
    toGetReputationEntity(): this;
    /**
     * Grants permission to retrieve information about a specific email address that's on the suppression list for your account
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetSuppressedDestination.html
     */
    toGetSuppressedDestination(): this;
    /**
     * Grants permission to get information about a tenant
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetTenant.html
     */
    toGetTenant(): this;
    /**
     * Grants permission to list all of the configuration sets for your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListConfigurationSets.html
     */
    toListConfigurationSets(): this;
    /**
     * Grants permission to list all of the contact lists available for your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListContactLists.html
     */
    toListContactLists(): this;
    /**
     * Grants permission to list the contacts present in a specific contact list
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListContacts.html
     */
    toListContacts(): this;
    /**
     * Grants permission to list all of the existing custom verification email templates for your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListCustomVerificationEmailTemplates.html
     */
    toListCustomVerificationEmailTemplates(): this;
    /**
     * Grants permission to list all of the dedicated IP pools for your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListDedicatedIpPools.html
     */
    toListDedicatedIpPools(): this;
    /**
     * Grants permission to retrieve the list of the predictive inbox placement tests that you've performed, regardless of their statuses, for your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListDeliverabilityTestReports.html
     */
    toListDeliverabilityTestReports(): this;
    /**
     * Grants permission to list deliverability data for campaigns that used a specific domain to send email during a specified time range
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListDomainDeliverabilityCampaigns.html
     */
    toListDomainDeliverabilityCampaigns(): this;
    /**
     * Grants permission to list the email identities for your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListEmailIdentities.html
     */
    toListEmailIdentities(): this;
    /**
     * Grants permission to list all of the email templates for your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListEmailTemplates.html
     */
    toListEmailTemplates(): this;
    /**
     * Grants permission to list all the exports jobs for your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifExportSourceType()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListExportJobs.html
     */
    toListExportJobs(): this;
    /**
     * Grants permission to list all of the import jobs for your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListImportJobs.html
     */
    toListImportJobs(): this;
    /**
     * Grants permission to list all of the multi-region endpoints for your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListMultiRegionEndpoints.html
     */
    toListMultiRegionEndpoints(): this;
    /**
     * Grants permission to list recommendations for your account
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListRecommendations.html
     */
    toListRecommendations(): this;
    /**
     * Grants permission to retrieve a list of reputation entities
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListReputationEntities.html
     */
    toListReputationEntities(): this;
    /**
     * Grants permission to list all the tenants associated to a SES resource
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListResourceTenants
     */
    toListResourceTenants(): this;
    /**
     * Grants permission to list email addresses that are on the suppression list for your account
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListSuppressedDestinations.html
     */
    toListSuppressedDestinations(): this;
    /**
     * Grants permission to retrieve a list of the tags (keys and values) that are associated with a specific resource for your account
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListTagsForResource.html
     */
    toListTagsForResource(): this;
    /**
     * Grants permission to list all the resources associated to a tenant
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListTenantResources
     */
    toListTenantResources(): this;
    /**
     * Grants permission to list all the tenants for your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListTenants
     */
    toListTenants(): this;
    /**
     * Grants permission to enable or disable the automatic warm-up feature for dedicated IP addresses
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutAccountDedicatedIpWarmupAttributes.html
     */
    toPutAccountDedicatedIpWarmupAttributes(): this;
    /**
     * Grants permission to update your account details
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutAccountDetails.html
     */
    toPutAccountDetails(): this;
    /**
     * Grants permission to enable or disable the ability to send email for your account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutAccountSendingAttributes.html
     */
    toPutAccountSendingAttributes(): this;
    /**
     * Grants permission to change the settings for the account-level suppression list
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutAccountSuppressionAttributes.html
     */
    toPutAccountSuppressionAttributes(): this;
    /**
     * Grants permission to change the settings for VDM for your account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutAccountVdmAttributes.html
     */
    toPutAccountVdmAttributes(): this;
    /**
     * Grants permission to associate a configuration set with a Mail Manager archive
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutConfigurationSetArchivingOptions.html
     */
    toPutConfigurationSetArchivingOptions(): this;
    /**
     * Grants permission to associate a configuration set with a dedicated IP pool
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutConfigurationSetDeliveryOptions.html
     */
    toPutConfigurationSetDeliveryOptions(): this;
    /**
     * Grants permission to enable or disable collection of reputation metrics for emails that you send using a particular configuration set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutConfigurationSetReputationOptions.html
     */
    toPutConfigurationSetReputationOptions(): this;
    /**
     * Grants permission to enable or disable email sending for messages that use a particular configuration set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutConfigurationSetSendingOptions.html
     */
    toPutConfigurationSetSendingOptions(): this;
    /**
     * Grants permission to specify the account suppression list preferences for a particular configuration set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutConfigurationSetSuppressionOptions.html
     */
    toPutConfigurationSetSuppressionOptions(): this;
    /**
     * Grants permission to specify a custom domain to use for open and click tracking elements in email that you send for a particular configuration set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutConfigurationSetTrackingOptions.html
     */
    toPutConfigurationSetTrackingOptions(): this;
    /**
     * Grants permission to override account-level VDM settings for a particular configuration set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutConfigurationSetVdmOptions.html
     */
    toPutConfigurationSetVdmOptions(): this;
    /**
     * Grants permission to move a dedicated IP address to an existing dedicated IP pool
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutDedicatedIpInPool.html
     */
    toPutDedicatedIpInPool(): this;
    /**
     * Grants permission to transition a dedicated IP pool from Standard to Managed
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutDedicatedIpPoolScalingAttributes.html
     */
    toPutDedicatedIpPoolScalingAttributes(): this;
    /**
     * Grants permission to put Dedicated IP warm up attributes
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutDedicatedIpWarmupAttributes.html
     */
    toPutDedicatedIpWarmupAttributes(): this;
    /**
     * Grants permission to enable or disable the Deliverability dashboard
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutDeliverabilityDashboardOption.html
     */
    toPutDeliverabilityDashboardOption(): this;
    /**
     * Grants permission to associate a configuration set with an email identity
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutEmailIdentityConfigurationSetAttributes.html
     */
    toPutEmailIdentityConfigurationSetAttributes(): this;
    /**
     * Grants permission to enable or disable DKIM authentication for an email identity
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutEmailIdentityDkimAttributes.html
     */
    toPutEmailIdentityDkimAttributes(): this;
    /**
     * Grants permission to configure or change the DKIM authentication settings for an email domain identity
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutEmailIdentityDkimSigningAttributes.html
     */
    toPutEmailIdentityDkimSigningAttributes(): this;
    /**
     * Grants permission to enable or disable feedback forwarding for an email identity
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutEmailIdentityFeedbackAttributes.html
     */
    toPutEmailIdentityFeedbackAttributes(): this;
    /**
     * Grants permission to enable or disable the custom MAIL FROM domain configuration for an email identity
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutEmailIdentityMailFromAttributes.html
     */
    toPutEmailIdentityMailFromAttributes(): this;
    /**
     * Grants permission to add an email address to the suppression list
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutSuppressedDestination.html
     */
    toPutSuppressedDestination(): this;
    /**
     * Grants permission to replicate email identity DKIM signing key
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifReplicaRegion()
     */
    toReplicateEmailIdentityDkimSigningKey(): this;
    /**
     * Grants permission to compose an email message to multiple destinations
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifMultiRegionEndpointId()
     * - .ifTenantName()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_SendBulkEmail.html
     */
    toSendBulkEmail(): this;
    /**
     * Grants permission to add an email address to the list of identities and attempts to verify it
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_SendCustomVerificationEmail.html
     */
    toSendCustomVerificationEmail(): this;
    /**
     * Grants permission to send an email message
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifFeedbackAddress()
     * - .ifFromAddress()
     * - .ifFromDisplayName()
     * - .ifRecipients()
     * - .ifMultiRegionEndpointId()
     * - .ifTenantName()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_SendEmail.html
     */
    toSendEmail(): this;
    /**
     * Grants permission to add one or more tags (keys and values) to a specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_TagResource.html
     */
    toTagResource(): this;
    /**
     * Grants permission to create a preview of the MIME content of an email when provided with a template and a set of replacement data
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_TestRenderEmailTemplate.html
     */
    toTestRenderEmailTemplate(): this;
    /**
     * Grants permission to remove one or more tags (keys and values) from a specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_UntagResource.html
     */
    toUntagResource(): this;
    /**
     * Grants permission to update the configuration of an event destination for a configuration set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_UpdateConfigurationSetEventDestination.html
     */
    toUpdateConfigurationSetEventDestination(): this;
    /**
     * Grants permission to update a contact's preferences for a list
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_UpdateContact.html
     */
    toUpdateContact(): this;
    /**
     * Grants permission to update contact list metadata
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_UpdateContactList.html
     */
    toUpdateContactList(): this;
    /**
     * Grants permission to update an existing custom verification email template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_UpdateCustomVerificationEmailTemplate.html
     */
    toUpdateCustomVerificationEmailTemplate(): this;
    /**
     * Grants permission to update the specified sending authorization policy for the given identity (an email address or a domain)
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_UpdateEmailIdentityPolicy.html
     */
    toUpdateEmailIdentityPolicy(): this;
    /**
     * Grants permission to update an email template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_UpdateEmailTemplate.html
     */
    toUpdateEmailTemplate(): this;
    /**
     * Grants permission to update the customer-managed sending status
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_UpdateReputationEntityCustomerManagedStatus.html
     */
    toUpdateReputationEntityCustomerManagedStatus(): this;
    /**
     * Grants permission to assign a reputation policy
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_UpdateReputationEntityPolicy.html
     */
    toUpdateReputationEntityPolicy(): this;
    protected accessLevelList: AccessLevelList;
    /**
     * Adds a resource of type reputation-policy to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ReputationPolicy.html
     *
     * @param reputationPolicyName - Identifier for the reputationPolicyName.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onReputationPolicy(reputationPolicyName: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type configuration-set to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ConfigurationSet.html
     *
     * @param configurationSetName - Identifier for the configurationSetName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConfigurationSet(configurationSetName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type contact-list to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ContactList.html
     *
     * @param contactListName - Identifier for the contactListName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onContactList(contactListName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type custom-verification-email-template to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CustomVerificationEmailTemplateMetadata.html
     *
     * @param templateName - Identifier for the templateName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onCustomVerificationEmailTemplate(templateName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type dedicated-ip-pool to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DedicatedIp.html
     *
     * @param dedicatedIPPool - Identifier for the dedicatedIPPool.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDedicatedIpPool(dedicatedIPPool: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type deliverability-test-report to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DeliverabilityTestReport.html
     *
     * @param reportId - Identifier for the reportId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDeliverabilityTestReport(reportId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type export-job to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ExportJobSummary.html
     *
     * @param exportJobId - Identifier for the exportJobId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onExportJob(exportJobId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type identity to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_IdentityInfo.html
     *
     * @param identityName - Identifier for the identityName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIdentity(identityName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type import-job to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ImportJobSummary.html
     *
     * @param importJobId - Identifier for the importJobId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onImportJob(importJobId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type template to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_Template.html
     *
     * @param templateName - Identifier for the templateName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onTemplate(templateName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type multi-region-endpoint to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_MultiRegionEndpoint.html
     *
     * @param endpointName - Identifier for the endpointName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onMultiRegionEndpoint(endpointName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type mailmanager-archive to the statement
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_Archive.html
     *
     * @param archiveId - Identifier for the archiveId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMailmanagerArchive(archiveId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type tenant to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_Tenant.html
     *
     * @param tenantName - Identifier for the tenantName.
     * @param tenantId - Identifier for the tenantId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTenant(tenantName: string, tenantId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateConfigurationSet()
     * - .toCreateContactList()
     * - .toCreateDedicatedIpPool()
     * - .toCreateDeliverabilityTestReport()
     * - .toCreateEmailIdentity()
     * - .toCreateMultiRegionEndpoint()
     * - .toCreateTenant()
     * - .toCreateTenantResourceAssociation()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toBatchGetMetricData()
     * - .toCreateConfigurationSetEventDestination()
     * - .toCreateContact()
     * - .toCreateEmailIdentityPolicy()
     * - .toDeleteConfigurationSet()
     * - .toDeleteConfigurationSetEventDestination()
     * - .toDeleteContact()
     * - .toDeleteContactList()
     * - .toDeleteDedicatedIpPool()
     * - .toDeleteEmailIdentity()
     * - .toDeleteEmailIdentityPolicy()
     * - .toDeleteMultiRegionEndpoint()
     * - .toDeleteTenant()
     * - .toDeleteTenantResourceAssociation()
     * - .toGetConfigurationSet()
     * - .toGetConfigurationSetEventDestinations()
     * - .toGetContact()
     * - .toGetDedicatedIpPool()
     * - .toGetDedicatedIps()
     * - .toGetDeliverabilityTestReport()
     * - .toGetDomainStatisticsReport()
     * - .toGetEmailIdentity()
     * - .toGetEmailIdentityPolicies()
     * - .toGetMultiRegionEndpoint()
     * - .toGetReputationEntity()
     * - .toGetTenant()
     * - .toListRecommendations()
     * - .toListReputationEntities()
     * - .toListResourceTenants()
     * - .toListTenantResources()
     * - .toPutConfigurationSetArchivingOptions()
     * - .toPutConfigurationSetDeliveryOptions()
     * - .toPutConfigurationSetReputationOptions()
     * - .toPutConfigurationSetSendingOptions()
     * - .toPutConfigurationSetSuppressionOptions()
     * - .toPutConfigurationSetTrackingOptions()
     * - .toPutConfigurationSetVdmOptions()
     * - .toPutDedicatedIpInPool()
     * - .toPutDedicatedIpPoolScalingAttributes()
     * - .toPutEmailIdentityConfigurationSetAttributes()
     * - .toPutEmailIdentityDkimAttributes()
     * - .toPutEmailIdentityDkimSigningAttributes()
     * - .toPutEmailIdentityFeedbackAttributes()
     * - .toPutEmailIdentityMailFromAttributes()
     * - .toUpdateConfigurationSetEventDestination()
     * - .toUpdateContact()
     * - .toUpdateContactList()
     * - .toUpdateEmailIdentityPolicy()
     * - .toUpdateReputationEntityCustomerManagedStatus()
     * - .toUpdateReputationEntityPolicy()
     *
     * Applies to resource types:
     * - configuration-set
     * - contact-list
     * - dedicated-ip-pool
     * - deliverability-test-report
     * - identity
     * - mailmanager-archive
     * - tenant
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateConfigurationSet()
     * - .toCreateContactList()
     * - .toCreateDedicatedIpPool()
     * - .toCreateDeliverabilityTestReport()
     * - .toCreateEmailIdentity()
     * - .toCreateMultiRegionEndpoint()
     * - .toCreateTenant()
     * - .toCreateTenantResourceAssociation()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the SES API version
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html#amazonses-policy-keys
     *
     * Applies to actions:
     * - .toBatchGetMetricData()
     * - .toCancelExportJob()
     * - .toCreateConfigurationSet()
     * - .toCreateConfigurationSetEventDestination()
     * - .toCreateContact()
     * - .toCreateContactList()
     * - .toCreateCustomVerificationEmailTemplate()
     * - .toCreateDedicatedIpPool()
     * - .toCreateDeliverabilityTestReport()
     * - .toCreateEmailIdentity()
     * - .toCreateEmailIdentityPolicy()
     * - .toCreateEmailTemplate()
     * - .toCreateExportJob()
     * - .toCreateImportJob()
     * - .toCreateMultiRegionEndpoint()
     * - .toCreateTenant()
     * - .toCreateTenantResourceAssociation()
     * - .toDeleteConfigurationSet()
     * - .toDeleteConfigurationSetEventDestination()
     * - .toDeleteContact()
     * - .toDeleteContactList()
     * - .toDeleteCustomVerificationEmailTemplate()
     * - .toDeleteDedicatedIpPool()
     * - .toDeleteEmailIdentity()
     * - .toDeleteEmailIdentityPolicy()
     * - .toDeleteEmailTemplate()
     * - .toDeleteMultiRegionEndpoint()
     * - .toDeleteSuppressedDestination()
     * - .toDeleteTenant()
     * - .toDeleteTenantResourceAssociation()
     * - .toGetAccount()
     * - .toGetBlacklistReports()
     * - .toGetConfigurationSet()
     * - .toGetConfigurationSetEventDestinations()
     * - .toGetContact()
     * - .toGetContactList()
     * - .toGetCustomVerificationEmailTemplate()
     * - .toGetDedicatedIp()
     * - .toGetDedicatedIpPool()
     * - .toGetDedicatedIps()
     * - .toGetDeliverabilityDashboardOptions()
     * - .toGetDeliverabilityTestReport()
     * - .toGetDomainDeliverabilityCampaign()
     * - .toGetDomainStatisticsReport()
     * - .toGetEmailIdentity()
     * - .toGetEmailIdentityPolicies()
     * - .toGetEmailTemplate()
     * - .toGetExportJob()
     * - .toGetImportJob()
     * - .toGetMessageInsights()
     * - .toGetMultiRegionEndpoint()
     * - .toGetReputationEntity()
     * - .toGetSuppressedDestination()
     * - .toGetTenant()
     * - .toListConfigurationSets()
     * - .toListContactLists()
     * - .toListContacts()
     * - .toListCustomVerificationEmailTemplates()
     * - .toListDedicatedIpPools()
     * - .toListDeliverabilityTestReports()
     * - .toListDomainDeliverabilityCampaigns()
     * - .toListEmailIdentities()
     * - .toListEmailTemplates()
     * - .toListExportJobs()
     * - .toListImportJobs()
     * - .toListMultiRegionEndpoints()
     * - .toListRecommendations()
     * - .toListReputationEntities()
     * - .toListResourceTenants()
     * - .toListSuppressedDestinations()
     * - .toListTagsForResource()
     * - .toListTenantResources()
     * - .toListTenants()
     * - .toPutAccountDedicatedIpWarmupAttributes()
     * - .toPutAccountDetails()
     * - .toPutAccountSendingAttributes()
     * - .toPutAccountSuppressionAttributes()
     * - .toPutAccountVdmAttributes()
     * - .toPutConfigurationSetArchivingOptions()
     * - .toPutConfigurationSetDeliveryOptions()
     * - .toPutConfigurationSetReputationOptions()
     * - .toPutConfigurationSetSendingOptions()
     * - .toPutConfigurationSetSuppressionOptions()
     * - .toPutConfigurationSetTrackingOptions()
     * - .toPutConfigurationSetVdmOptions()
     * - .toPutDedicatedIpInPool()
     * - .toPutDedicatedIpPoolScalingAttributes()
     * - .toPutDedicatedIpWarmupAttributes()
     * - .toPutDeliverabilityDashboardOption()
     * - .toPutEmailIdentityConfigurationSetAttributes()
     * - .toPutEmailIdentityDkimAttributes()
     * - .toPutEmailIdentityDkimSigningAttributes()
     * - .toPutEmailIdentityFeedbackAttributes()
     * - .toPutEmailIdentityMailFromAttributes()
     * - .toPutSuppressedDestination()
     * - .toSendBulkEmail()
     * - .toSendCustomVerificationEmail()
     * - .toSendEmail()
     * - .toTagResource()
     * - .toTestRenderEmailTemplate()
     * - .toUntagResource()
     * - .toUpdateConfigurationSetEventDestination()
     * - .toUpdateContact()
     * - .toUpdateContactList()
     * - .toUpdateCustomVerificationEmailTemplate()
     * - .toUpdateEmailIdentityPolicy()
     * - .toUpdateEmailTemplate()
     * - .toUpdateReputationEntityCustomerManagedStatus()
     * - .toUpdateReputationEntityPolicy()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifApiVersion(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the export source type
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html#amazonses-policy-keys
     *
     * Applies to actions:
     * - .toCancelExportJob()
     * - .toCreateExportJob()
     * - .toGetExportJob()
     * - .toListExportJobs()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifExportSourceType(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the "Return-Path" address, which specifies where bounces and complaints are sent by email feedback forwarding
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html#amazonses-policy-keys
     *
     * Applies to actions:
     * - .toSendEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFeedbackAddress(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the "From" address of a message
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html#amazonses-policy-keys
     *
     * Applies to actions:
     * - .toSendEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFromAddress(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the "From" address that is used as the display name of a message
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html#amazonses-policy-keys
     *
     * Applies to actions:
     * - .toSendEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFromDisplayName(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the multi-region endpoint ID that is used to send email
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html#amazonses-policy-keys
     *
     * Applies to actions:
     * - .toSendBulkEmail()
     * - .toSendEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifMultiRegionEndpointId(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the recipient addresses of a message, which include the "To", "CC", and "BCC" addresses
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html#amazonses-policy-keys
     *
     * Applies to actions:
     * - .toSendEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRecipients(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the replica regions for Replicating domain DKIM signing key
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html#amazonses-policy-keys
     *
     * Applies to actions:
     * - .toReplicateEmailIdentityDkimSigningKey()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifReplicaRegion(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the tenant name that is used to send email
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html#amazonses-policy-keys
     *
     * Applies to actions:
     * - .toSendBulkEmail()
     * - .toSendEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTenantName(value: string | string[], operator?: Operator | string): this;
    /**
     * Statement provider for service [ses-v2](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonsimpleemailservicev2.html).
     *
     */
    constructor(props?: iam.PolicyStatementProps);
}
