"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SesV2 = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [ses-v2](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonsimpleemailservicev2.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class SesV2 extends shared_1.PolicyStatement {
    /**
     * Grants permission to get metric data on your activity
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_BatchGetMetricData.html
     */
    toBatchGetMetricData() {
        return this.to('BatchGetMetricData');
    }
    /**
     * Grants permission to cancel an export job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifExportSourceType()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CancelExportJob.html
     */
    toCancelExportJob() {
        return this.to('CancelExportJob');
    }
    /**
     * Grants permission to create a new configuration set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateConfigurationSet.html
     */
    toCreateConfigurationSet() {
        return this.to('CreateConfigurationSet');
    }
    /**
     * Grants permission to create a configuration set event destination
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateConfigurationSetEventDestination.html
     */
    toCreateConfigurationSetEventDestination() {
        return this.to('CreateConfigurationSetEventDestination');
    }
    /**
     * Grants permission to create a contact
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateContact.html
     */
    toCreateContact() {
        return this.to('CreateContact');
    }
    /**
     * Grants permission to create a contact list
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateContactList.html
     */
    toCreateContactList() {
        return this.to('CreateContactList');
    }
    /**
     * Grants permission to create a new custom verification email template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateCustomVerificationEmailTemplate.html
     */
    toCreateCustomVerificationEmailTemplate() {
        return this.to('CreateCustomVerificationEmailTemplate');
    }
    /**
     * Grants permission to create a new pool of dedicated IP addresses
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateDedicatedIpPool.html
     */
    toCreateDedicatedIpPool() {
        return this.to('CreateDedicatedIpPool');
    }
    /**
     * Grants permission to create a new predictive inbox placement test
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateDeliverabilityTestReport.html
     */
    toCreateDeliverabilityTestReport() {
        return this.to('CreateDeliverabilityTestReport');
    }
    /**
     * Grants permission to start the process of verifying an email identity
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateEmailIdentity.html
     */
    toCreateEmailIdentity() {
        return this.to('CreateEmailIdentity');
    }
    /**
     * Grants permission to create the specified sending authorization policy for the given identity
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateEmailIdentityPolicy.html
     */
    toCreateEmailIdentityPolicy() {
        return this.to('CreateEmailIdentityPolicy');
    }
    /**
     * Grants permission to create an email template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateEmailTemplate.html
     */
    toCreateEmailTemplate() {
        return this.to('CreateEmailTemplate');
    }
    /**
     * Grants permission to create an export job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifExportSourceType()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateExportJob.html
     */
    toCreateExportJob() {
        return this.to('CreateExportJob');
    }
    /**
     * Grants permission to creates an import job for a data destination
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateImportJob.html
     */
    toCreateImportJob() {
        return this.to('CreateImportJob');
    }
    /**
     * Grants permission to create a new multi-region endpoint
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateMultiRegionEndpoint.html
     */
    toCreateMultiRegionEndpoint() {
        return this.to('CreateMultiRegionEndpoint');
    }
    /**
     * Grants permission to create a new tenant
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateTenant.html
     */
    toCreateTenant() {
        return this.to('CreateTenant');
    }
    /**
     * Grants permission to associate a SES resource to a tenant
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CreateTenantResourceAssociation.html
     */
    toCreateTenantResourceAssociation() {
        return this.to('CreateTenantResourceAssociation');
    }
    /**
     * Grants permission to delete an existing configuration set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DeleteConfigurationSet.html
     */
    toDeleteConfigurationSet() {
        return this.to('DeleteConfigurationSet');
    }
    /**
     * Grants permission to delete an event destination
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DeleteConfigurationSetEventDestination.html
     */
    toDeleteConfigurationSetEventDestination() {
        return this.to('DeleteConfigurationSetEventDestination');
    }
    /**
     * Grants permission to delete a contact from a contact list
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DeleteContact.html
     */
    toDeleteContact() {
        return this.to('DeleteContact');
    }
    /**
     * Grants permission to delete a contact list with all of its contacts
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DeleteContactList.html
     */
    toDeleteContactList() {
        return this.to('DeleteContactList');
    }
    /**
     * Grants permission to delete an existing custom verification email template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DeleteCustomVerificationEmailTemplate.html
     */
    toDeleteCustomVerificationEmailTemplate() {
        return this.to('DeleteCustomVerificationEmailTemplate');
    }
    /**
     * Grants permission to delete a dedicated IP pool
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DeleteDedicatedIpPool.html
     */
    toDeleteDedicatedIpPool() {
        return this.to('DeleteDedicatedIpPool');
    }
    /**
     * Grants permission to delete an email identity
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DeleteEmailIdentity.html
     */
    toDeleteEmailIdentity() {
        return this.to('DeleteEmailIdentity');
    }
    /**
     * Grants permission to delete the specified sending authorization policy for the given identity (an email address or a domain)
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DeleteEmailIdentityPolicy.html
     */
    toDeleteEmailIdentityPolicy() {
        return this.to('DeleteEmailIdentityPolicy');
    }
    /**
     * Grants permission to delete an email template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DeleteEmailTemplate.html
     */
    toDeleteEmailTemplate() {
        return this.to('DeleteEmailTemplate');
    }
    /**
     * Grants permission to delete a multi-region endpoint
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DeleteMultiRegionEndpoint.html
     */
    toDeleteMultiRegionEndpoint() {
        return this.to('DeleteMultiRegionEndpoint');
    }
    /**
     * Grants permission to remove an email address from the suppression list for your account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DeleteSuppressedDestination.html
     */
    toDeleteSuppressedDestination() {
        return this.to('DeleteSuppressedDestination');
    }
    /**
     * Grants permission to delete a tenant
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DeleteTenant.html
     */
    toDeleteTenant() {
        return this.to('DeleteTenant');
    }
    /**
     * Grants permission to remove an associated SES resource from a tenant
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DeleteTenantResourceAssociation.html
     */
    toDeleteTenantResourceAssociation() {
        return this.to('DeleteTenantResourceAssociation');
    }
    /**
     * Grants permission to get information about the email-sending status and capabilities for your account
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetAccount.html
     */
    toGetAccount() {
        return this.to('GetAccount');
    }
    /**
     * Grants permission to retrieve a list of the deny lists on which your dedicated IP addresses or tracked domains appear
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetBlacklistReports.html
     */
    toGetBlacklistReports() {
        return this.to('GetBlacklistReports');
    }
    /**
     * Grants permission to get information about an existing configuration set
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetConfigurationSet.html
     */
    toGetConfigurationSet() {
        return this.to('GetConfigurationSet');
    }
    /**
     * Grants permission to retrieve a list of event destinations that are associated with a configuration set
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetConfigurationSetEventDestinations.html
     */
    toGetConfigurationSetEventDestinations() {
        return this.to('GetConfigurationSetEventDestinations');
    }
    /**
     * Grants permission to return a contact from a contact list
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetContact.html
     */
    toGetContact() {
        return this.to('GetContact');
    }
    /**
     * Grants permission to return contact list metadata
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetContactList.html
     */
    toGetContactList() {
        return this.to('GetContactList');
    }
    /**
     * Grants permission to return the custom email verification template for the template name you specify
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetCustomVerificationEmailTemplate.html
     */
    toGetCustomVerificationEmailTemplate() {
        return this.to('GetCustomVerificationEmailTemplate');
    }
    /**
     * Grants permission to get information about a dedicated IP address
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetDedicatedIp.html
     */
    toGetDedicatedIp() {
        return this.to('GetDedicatedIp');
    }
    /**
     * Grants permission to get information about a dedicated IP pool
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetDedicatedIpPool.html
     */
    toGetDedicatedIpPool() {
        return this.to('GetDedicatedIpPool');
    }
    /**
     * Grants permission to list the dedicated IP addresses a dedicated IP pool
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetDedicatedIps.html
     */
    toGetDedicatedIps() {
        return this.to('GetDedicatedIps');
    }
    /**
     * Grants permission to get the status of the Deliverability dashboard
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetDeliverabilityDashboardOptions.html
     */
    toGetDeliverabilityDashboardOptions() {
        return this.to('GetDeliverabilityDashboardOptions');
    }
    /**
     * Grants permission to retrieve the results of a predictive inbox placement test
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetDeliverabilityTestReport.html
     */
    toGetDeliverabilityTestReport() {
        return this.to('GetDeliverabilityTestReport');
    }
    /**
     * Grants permission to retrieve all the deliverability data for a specific campaign
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetDomainDeliverabilityCampaign.html
     */
    toGetDomainDeliverabilityCampaign() {
        return this.to('GetDomainDeliverabilityCampaign');
    }
    /**
     * Grants permission to retrieve inbox placement and engagement rates for the domains that you use to send email
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetDomainStatisticsReport.html
     */
    toGetDomainStatisticsReport() {
        return this.to('GetDomainStatisticsReport');
    }
    /**
     * Grants permission to get information about a specific identity
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetEmailIdentity.html
     */
    toGetEmailIdentity() {
        return this.to('GetEmailIdentity');
    }
    /**
     * Grants permission to return the requested sending authorization policies for the given identity (an email address or a domain)
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetEmailIdentityPolicies.html
     */
    toGetEmailIdentityPolicies() {
        return this.to('GetEmailIdentityPolicies');
    }
    /**
     * Grants permission to return the template object, which includes the subject line, HTML part, and text part for the template you specify
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetEmailTemplate.html
     */
    toGetEmailTemplate() {
        return this.to('GetEmailTemplate');
    }
    /**
     * Grants permission to get information about an export job
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifExportSourceType()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetExportJob.html
     */
    toGetExportJob() {
        return this.to('GetExportJob');
    }
    /**
     * Grants permission to provide information about an import job
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetImportJob.html
     */
    toGetImportJob() {
        return this.to('GetImportJob');
    }
    /**
     * Grants permission to provide insights about a message
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetMessageInsights.html
     */
    toGetMessageInsights() {
        return this.to('GetMessageInsights');
    }
    /**
     * Grants permission to get information about a multi-region endpoint
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetMultiRegionEndpoint.html
     */
    toGetMultiRegionEndpoint() {
        return this.to('GetMultiRegionEndpoint');
    }
    /**
     * Grants permission to retrieve information about a reputation entity's status
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetReputationEntity.html
     */
    toGetReputationEntity() {
        return this.to('GetReputationEntity');
    }
    /**
     * Grants permission to retrieve information about a specific email address that's on the suppression list for your account
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetSuppressedDestination.html
     */
    toGetSuppressedDestination() {
        return this.to('GetSuppressedDestination');
    }
    /**
     * Grants permission to get information about a tenant
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_GetTenant.html
     */
    toGetTenant() {
        return this.to('GetTenant');
    }
    /**
     * Grants permission to list all of the configuration sets for your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListConfigurationSets.html
     */
    toListConfigurationSets() {
        return this.to('ListConfigurationSets');
    }
    /**
     * Grants permission to list all of the contact lists available for your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListContactLists.html
     */
    toListContactLists() {
        return this.to('ListContactLists');
    }
    /**
     * Grants permission to list the contacts present in a specific contact list
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListContacts.html
     */
    toListContacts() {
        return this.to('ListContacts');
    }
    /**
     * Grants permission to list all of the existing custom verification email templates for your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListCustomVerificationEmailTemplates.html
     */
    toListCustomVerificationEmailTemplates() {
        return this.to('ListCustomVerificationEmailTemplates');
    }
    /**
     * Grants permission to list all of the dedicated IP pools for your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListDedicatedIpPools.html
     */
    toListDedicatedIpPools() {
        return this.to('ListDedicatedIpPools');
    }
    /**
     * Grants permission to retrieve the list of the predictive inbox placement tests that you've performed, regardless of their statuses, for your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListDeliverabilityTestReports.html
     */
    toListDeliverabilityTestReports() {
        return this.to('ListDeliverabilityTestReports');
    }
    /**
     * Grants permission to list deliverability data for campaigns that used a specific domain to send email during a specified time range
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListDomainDeliverabilityCampaigns.html
     */
    toListDomainDeliverabilityCampaigns() {
        return this.to('ListDomainDeliverabilityCampaigns');
    }
    /**
     * Grants permission to list the email identities for your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListEmailIdentities.html
     */
    toListEmailIdentities() {
        return this.to('ListEmailIdentities');
    }
    /**
     * Grants permission to list all of the email templates for your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListEmailTemplates.html
     */
    toListEmailTemplates() {
        return this.to('ListEmailTemplates');
    }
    /**
     * Grants permission to list all the exports jobs for your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifExportSourceType()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListExportJobs.html
     */
    toListExportJobs() {
        return this.to('ListExportJobs');
    }
    /**
     * Grants permission to list all of the import jobs for your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListImportJobs.html
     */
    toListImportJobs() {
        return this.to('ListImportJobs');
    }
    /**
     * Grants permission to list all of the multi-region endpoints for your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListMultiRegionEndpoints.html
     */
    toListMultiRegionEndpoints() {
        return this.to('ListMultiRegionEndpoints');
    }
    /**
     * Grants permission to list recommendations for your account
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListRecommendations.html
     */
    toListRecommendations() {
        return this.to('ListRecommendations');
    }
    /**
     * Grants permission to retrieve a list of reputation entities
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListReputationEntities.html
     */
    toListReputationEntities() {
        return this.to('ListReputationEntities');
    }
    /**
     * Grants permission to list all the tenants associated to a SES resource
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListResourceTenants
     */
    toListResourceTenants() {
        return this.to('ListResourceTenants');
    }
    /**
     * Grants permission to list email addresses that are on the suppression list for your account
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListSuppressedDestinations.html
     */
    toListSuppressedDestinations() {
        return this.to('ListSuppressedDestinations');
    }
    /**
     * Grants permission to retrieve a list of the tags (keys and values) that are associated with a specific resource for your account
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list all the resources associated to a tenant
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListTenantResources
     */
    toListTenantResources() {
        return this.to('ListTenantResources');
    }
    /**
     * Grants permission to list all the tenants for your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ListTenants
     */
    toListTenants() {
        return this.to('ListTenants');
    }
    /**
     * Grants permission to enable or disable the automatic warm-up feature for dedicated IP addresses
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutAccountDedicatedIpWarmupAttributes.html
     */
    toPutAccountDedicatedIpWarmupAttributes() {
        return this.to('PutAccountDedicatedIpWarmupAttributes');
    }
    /**
     * Grants permission to update your account details
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutAccountDetails.html
     */
    toPutAccountDetails() {
        return this.to('PutAccountDetails');
    }
    /**
     * Grants permission to enable or disable the ability to send email for your account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutAccountSendingAttributes.html
     */
    toPutAccountSendingAttributes() {
        return this.to('PutAccountSendingAttributes');
    }
    /**
     * Grants permission to change the settings for the account-level suppression list
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutAccountSuppressionAttributes.html
     */
    toPutAccountSuppressionAttributes() {
        return this.to('PutAccountSuppressionAttributes');
    }
    /**
     * Grants permission to change the settings for VDM for your account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutAccountVdmAttributes.html
     */
    toPutAccountVdmAttributes() {
        return this.to('PutAccountVdmAttributes');
    }
    /**
     * Grants permission to associate a configuration set with a Mail Manager archive
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutConfigurationSetArchivingOptions.html
     */
    toPutConfigurationSetArchivingOptions() {
        return this.to('PutConfigurationSetArchivingOptions');
    }
    /**
     * Grants permission to associate a configuration set with a dedicated IP pool
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutConfigurationSetDeliveryOptions.html
     */
    toPutConfigurationSetDeliveryOptions() {
        return this.to('PutConfigurationSetDeliveryOptions');
    }
    /**
     * Grants permission to enable or disable collection of reputation metrics for emails that you send using a particular configuration set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutConfigurationSetReputationOptions.html
     */
    toPutConfigurationSetReputationOptions() {
        return this.to('PutConfigurationSetReputationOptions');
    }
    /**
     * Grants permission to enable or disable email sending for messages that use a particular configuration set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutConfigurationSetSendingOptions.html
     */
    toPutConfigurationSetSendingOptions() {
        return this.to('PutConfigurationSetSendingOptions');
    }
    /**
     * Grants permission to specify the account suppression list preferences for a particular configuration set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutConfigurationSetSuppressionOptions.html
     */
    toPutConfigurationSetSuppressionOptions() {
        return this.to('PutConfigurationSetSuppressionOptions');
    }
    /**
     * Grants permission to specify a custom domain to use for open and click tracking elements in email that you send for a particular configuration set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutConfigurationSetTrackingOptions.html
     */
    toPutConfigurationSetTrackingOptions() {
        return this.to('PutConfigurationSetTrackingOptions');
    }
    /**
     * Grants permission to override account-level VDM settings for a particular configuration set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutConfigurationSetVdmOptions.html
     */
    toPutConfigurationSetVdmOptions() {
        return this.to('PutConfigurationSetVdmOptions');
    }
    /**
     * Grants permission to move a dedicated IP address to an existing dedicated IP pool
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutDedicatedIpInPool.html
     */
    toPutDedicatedIpInPool() {
        return this.to('PutDedicatedIpInPool');
    }
    /**
     * Grants permission to transition a dedicated IP pool from Standard to Managed
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutDedicatedIpPoolScalingAttributes.html
     */
    toPutDedicatedIpPoolScalingAttributes() {
        return this.to('PutDedicatedIpPoolScalingAttributes');
    }
    /**
     * Grants permission to put Dedicated IP warm up attributes
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutDedicatedIpWarmupAttributes.html
     */
    toPutDedicatedIpWarmupAttributes() {
        return this.to('PutDedicatedIpWarmupAttributes');
    }
    /**
     * Grants permission to enable or disable the Deliverability dashboard
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutDeliverabilityDashboardOption.html
     */
    toPutDeliverabilityDashboardOption() {
        return this.to('PutDeliverabilityDashboardOption');
    }
    /**
     * Grants permission to associate a configuration set with an email identity
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutEmailIdentityConfigurationSetAttributes.html
     */
    toPutEmailIdentityConfigurationSetAttributes() {
        return this.to('PutEmailIdentityConfigurationSetAttributes');
    }
    /**
     * Grants permission to enable or disable DKIM authentication for an email identity
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutEmailIdentityDkimAttributes.html
     */
    toPutEmailIdentityDkimAttributes() {
        return this.to('PutEmailIdentityDkimAttributes');
    }
    /**
     * Grants permission to configure or change the DKIM authentication settings for an email domain identity
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutEmailIdentityDkimSigningAttributes.html
     */
    toPutEmailIdentityDkimSigningAttributes() {
        return this.to('PutEmailIdentityDkimSigningAttributes');
    }
    /**
     * Grants permission to enable or disable feedback forwarding for an email identity
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutEmailIdentityFeedbackAttributes.html
     */
    toPutEmailIdentityFeedbackAttributes() {
        return this.to('PutEmailIdentityFeedbackAttributes');
    }
    /**
     * Grants permission to enable or disable the custom MAIL FROM domain configuration for an email identity
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutEmailIdentityMailFromAttributes.html
     */
    toPutEmailIdentityMailFromAttributes() {
        return this.to('PutEmailIdentityMailFromAttributes');
    }
    /**
     * Grants permission to add an email address to the suppression list
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_PutSuppressedDestination.html
     */
    toPutSuppressedDestination() {
        return this.to('PutSuppressedDestination');
    }
    /**
     * Grants permission to replicate email identity DKIM signing key
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifReplicaRegion()
     */
    toReplicateEmailIdentityDkimSigningKey() {
        return this.to('ReplicateEmailIdentityDkimSigningKey');
    }
    /**
     * Grants permission to compose an email message to multiple destinations
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifMultiRegionEndpointId()
     * - .ifTenantName()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_SendBulkEmail.html
     */
    toSendBulkEmail() {
        return this.to('SendBulkEmail');
    }
    /**
     * Grants permission to add an email address to the list of identities and attempts to verify it
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_SendCustomVerificationEmail.html
     */
    toSendCustomVerificationEmail() {
        return this.to('SendCustomVerificationEmail');
    }
    /**
     * Grants permission to send an email message
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifFeedbackAddress()
     * - .ifFromAddress()
     * - .ifFromDisplayName()
     * - .ifRecipients()
     * - .ifMultiRegionEndpointId()
     * - .ifTenantName()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_SendEmail.html
     */
    toSendEmail() {
        return this.to('SendEmail');
    }
    /**
     * Grants permission to add one or more tags (keys and values) to a specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to create a preview of the MIME content of an email when provided with a template and a set of replacement data
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_TestRenderEmailTemplate.html
     */
    toTestRenderEmailTemplate() {
        return this.to('TestRenderEmailTemplate');
    }
    /**
     * Grants permission to remove one or more tags (keys and values) from a specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update the configuration of an event destination for a configuration set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_UpdateConfigurationSetEventDestination.html
     */
    toUpdateConfigurationSetEventDestination() {
        return this.to('UpdateConfigurationSetEventDestination');
    }
    /**
     * Grants permission to update a contact's preferences for a list
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_UpdateContact.html
     */
    toUpdateContact() {
        return this.to('UpdateContact');
    }
    /**
     * Grants permission to update contact list metadata
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_UpdateContactList.html
     */
    toUpdateContactList() {
        return this.to('UpdateContactList');
    }
    /**
     * Grants permission to update an existing custom verification email template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_UpdateCustomVerificationEmailTemplate.html
     */
    toUpdateCustomVerificationEmailTemplate() {
        return this.to('UpdateCustomVerificationEmailTemplate');
    }
    /**
     * Grants permission to update the specified sending authorization policy for the given identity (an email address or a domain)
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_UpdateEmailIdentityPolicy.html
     */
    toUpdateEmailIdentityPolicy() {
        return this.to('UpdateEmailIdentityPolicy');
    }
    /**
     * Grants permission to update an email template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_UpdateEmailTemplate.html
     */
    toUpdateEmailTemplate() {
        return this.to('UpdateEmailTemplate');
    }
    /**
     * Grants permission to update the customer-managed sending status
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_UpdateReputationEntityCustomerManagedStatus.html
     */
    toUpdateReputationEntityCustomerManagedStatus() {
        return this.to('UpdateReputationEntityCustomerManagedStatus');
    }
    /**
     * Grants permission to assign a reputation policy
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApiVersion()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_UpdateReputationEntityPolicy.html
     */
    toUpdateReputationEntityPolicy() {
        return this.to('UpdateReputationEntityPolicy');
    }
    /**
     * Adds a resource of type reputation-policy to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ReputationPolicy.html
     *
     * @param reputationPolicyName - Identifier for the reputationPolicyName.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onReputationPolicy(reputationPolicyName, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ses:${region ?? this.defaultRegion}:aws:reputation-policy/${reputationPolicyName}`);
    }
    /**
     * Adds a resource of type configuration-set to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ConfigurationSet.html
     *
     * @param configurationSetName - Identifier for the configurationSetName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConfigurationSet(configurationSetName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ses:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:configuration-set/${configurationSetName}`);
    }
    /**
     * Adds a resource of type contact-list to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ContactList.html
     *
     * @param contactListName - Identifier for the contactListName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onContactList(contactListName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ses:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:contact-list/${contactListName}`);
    }
    /**
     * Adds a resource of type custom-verification-email-template to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_CustomVerificationEmailTemplateMetadata.html
     *
     * @param templateName - Identifier for the templateName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onCustomVerificationEmailTemplate(templateName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ses:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:custom-verification-email-template/${templateName}`);
    }
    /**
     * Adds a resource of type dedicated-ip-pool to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DedicatedIp.html
     *
     * @param dedicatedIPPool - Identifier for the dedicatedIPPool.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDedicatedIpPool(dedicatedIPPool, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ses:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:dedicated-ip-pool/${dedicatedIPPool}`);
    }
    /**
     * Adds a resource of type deliverability-test-report to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_DeliverabilityTestReport.html
     *
     * @param reportId - Identifier for the reportId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDeliverabilityTestReport(reportId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ses:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:deliverability-test-report/${reportId}`);
    }
    /**
     * Adds a resource of type export-job to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ExportJobSummary.html
     *
     * @param exportJobId - Identifier for the exportJobId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onExportJob(exportJobId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ses:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:export-job/${exportJobId}`);
    }
    /**
     * Adds a resource of type identity to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_IdentityInfo.html
     *
     * @param identityName - Identifier for the identityName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIdentity(identityName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ses:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:identity/${identityName}`);
    }
    /**
     * Adds a resource of type import-job to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_ImportJobSummary.html
     *
     * @param importJobId - Identifier for the importJobId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onImportJob(importJobId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ses:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:import-job/${importJobId}`);
    }
    /**
     * Adds a resource of type template to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_Template.html
     *
     * @param templateName - Identifier for the templateName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onTemplate(templateName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ses:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:template/${templateName}`);
    }
    /**
     * Adds a resource of type multi-region-endpoint to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference-V2/API_MultiRegionEndpoint.html
     *
     * @param endpointName - Identifier for the endpointName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onMultiRegionEndpoint(endpointName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ses:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:multi-region-endpoint/${endpointName}`);
    }
    /**
     * Adds a resource of type mailmanager-archive to the statement
     *
     * https://docs.aws.amazon.com/sesmailmanager/latest/APIReference/API_Archive.html
     *
     * @param archiveId - Identifier for the archiveId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMailmanagerArchive(archiveId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ses:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:mailmanager-archive/${archiveId}`);
    }
    /**
     * Adds a resource of type tenant to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_Tenant.html
     *
     * @param tenantName - Identifier for the tenantName.
     * @param tenantId - Identifier for the tenantId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTenant(tenantName, tenantId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ses:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:tenant/${tenantName}/${tenantId}`);
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateConfigurationSet()
     * - .toCreateContactList()
     * - .toCreateDedicatedIpPool()
     * - .toCreateDeliverabilityTestReport()
     * - .toCreateEmailIdentity()
     * - .toCreateMultiRegionEndpoint()
     * - .toCreateTenant()
     * - .toCreateTenantResourceAssociation()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toBatchGetMetricData()
     * - .toCreateConfigurationSetEventDestination()
     * - .toCreateContact()
     * - .toCreateEmailIdentityPolicy()
     * - .toDeleteConfigurationSet()
     * - .toDeleteConfigurationSetEventDestination()
     * - .toDeleteContact()
     * - .toDeleteContactList()
     * - .toDeleteDedicatedIpPool()
     * - .toDeleteEmailIdentity()
     * - .toDeleteEmailIdentityPolicy()
     * - .toDeleteMultiRegionEndpoint()
     * - .toDeleteTenant()
     * - .toDeleteTenantResourceAssociation()
     * - .toGetConfigurationSet()
     * - .toGetConfigurationSetEventDestinations()
     * - .toGetContact()
     * - .toGetDedicatedIpPool()
     * - .toGetDedicatedIps()
     * - .toGetDeliverabilityTestReport()
     * - .toGetDomainStatisticsReport()
     * - .toGetEmailIdentity()
     * - .toGetEmailIdentityPolicies()
     * - .toGetMultiRegionEndpoint()
     * - .toGetReputationEntity()
     * - .toGetTenant()
     * - .toListRecommendations()
     * - .toListReputationEntities()
     * - .toListResourceTenants()
     * - .toListTenantResources()
     * - .toPutConfigurationSetArchivingOptions()
     * - .toPutConfigurationSetDeliveryOptions()
     * - .toPutConfigurationSetReputationOptions()
     * - .toPutConfigurationSetSendingOptions()
     * - .toPutConfigurationSetSuppressionOptions()
     * - .toPutConfigurationSetTrackingOptions()
     * - .toPutConfigurationSetVdmOptions()
     * - .toPutDedicatedIpInPool()
     * - .toPutDedicatedIpPoolScalingAttributes()
     * - .toPutEmailIdentityConfigurationSetAttributes()
     * - .toPutEmailIdentityDkimAttributes()
     * - .toPutEmailIdentityDkimSigningAttributes()
     * - .toPutEmailIdentityFeedbackAttributes()
     * - .toPutEmailIdentityMailFromAttributes()
     * - .toUpdateConfigurationSetEventDestination()
     * - .toUpdateContact()
     * - .toUpdateContactList()
     * - .toUpdateEmailIdentityPolicy()
     * - .toUpdateReputationEntityCustomerManagedStatus()
     * - .toUpdateReputationEntityPolicy()
     *
     * Applies to resource types:
     * - configuration-set
     * - contact-list
     * - dedicated-ip-pool
     * - deliverability-test-report
     * - identity
     * - mailmanager-archive
     * - tenant
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateConfigurationSet()
     * - .toCreateContactList()
     * - .toCreateDedicatedIpPool()
     * - .toCreateDeliverabilityTestReport()
     * - .toCreateEmailIdentity()
     * - .toCreateMultiRegionEndpoint()
     * - .toCreateTenant()
     * - .toCreateTenantResourceAssociation()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the SES API version
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html#amazonses-policy-keys
     *
     * Applies to actions:
     * - .toBatchGetMetricData()
     * - .toCancelExportJob()
     * - .toCreateConfigurationSet()
     * - .toCreateConfigurationSetEventDestination()
     * - .toCreateContact()
     * - .toCreateContactList()
     * - .toCreateCustomVerificationEmailTemplate()
     * - .toCreateDedicatedIpPool()
     * - .toCreateDeliverabilityTestReport()
     * - .toCreateEmailIdentity()
     * - .toCreateEmailIdentityPolicy()
     * - .toCreateEmailTemplate()
     * - .toCreateExportJob()
     * - .toCreateImportJob()
     * - .toCreateMultiRegionEndpoint()
     * - .toCreateTenant()
     * - .toCreateTenantResourceAssociation()
     * - .toDeleteConfigurationSet()
     * - .toDeleteConfigurationSetEventDestination()
     * - .toDeleteContact()
     * - .toDeleteContactList()
     * - .toDeleteCustomVerificationEmailTemplate()
     * - .toDeleteDedicatedIpPool()
     * - .toDeleteEmailIdentity()
     * - .toDeleteEmailIdentityPolicy()
     * - .toDeleteEmailTemplate()
     * - .toDeleteMultiRegionEndpoint()
     * - .toDeleteSuppressedDestination()
     * - .toDeleteTenant()
     * - .toDeleteTenantResourceAssociation()
     * - .toGetAccount()
     * - .toGetBlacklistReports()
     * - .toGetConfigurationSet()
     * - .toGetConfigurationSetEventDestinations()
     * - .toGetContact()
     * - .toGetContactList()
     * - .toGetCustomVerificationEmailTemplate()
     * - .toGetDedicatedIp()
     * - .toGetDedicatedIpPool()
     * - .toGetDedicatedIps()
     * - .toGetDeliverabilityDashboardOptions()
     * - .toGetDeliverabilityTestReport()
     * - .toGetDomainDeliverabilityCampaign()
     * - .toGetDomainStatisticsReport()
     * - .toGetEmailIdentity()
     * - .toGetEmailIdentityPolicies()
     * - .toGetEmailTemplate()
     * - .toGetExportJob()
     * - .toGetImportJob()
     * - .toGetMessageInsights()
     * - .toGetMultiRegionEndpoint()
     * - .toGetReputationEntity()
     * - .toGetSuppressedDestination()
     * - .toGetTenant()
     * - .toListConfigurationSets()
     * - .toListContactLists()
     * - .toListContacts()
     * - .toListCustomVerificationEmailTemplates()
     * - .toListDedicatedIpPools()
     * - .toListDeliverabilityTestReports()
     * - .toListDomainDeliverabilityCampaigns()
     * - .toListEmailIdentities()
     * - .toListEmailTemplates()
     * - .toListExportJobs()
     * - .toListImportJobs()
     * - .toListMultiRegionEndpoints()
     * - .toListRecommendations()
     * - .toListReputationEntities()
     * - .toListResourceTenants()
     * - .toListSuppressedDestinations()
     * - .toListTagsForResource()
     * - .toListTenantResources()
     * - .toListTenants()
     * - .toPutAccountDedicatedIpWarmupAttributes()
     * - .toPutAccountDetails()
     * - .toPutAccountSendingAttributes()
     * - .toPutAccountSuppressionAttributes()
     * - .toPutAccountVdmAttributes()
     * - .toPutConfigurationSetArchivingOptions()
     * - .toPutConfigurationSetDeliveryOptions()
     * - .toPutConfigurationSetReputationOptions()
     * - .toPutConfigurationSetSendingOptions()
     * - .toPutConfigurationSetSuppressionOptions()
     * - .toPutConfigurationSetTrackingOptions()
     * - .toPutConfigurationSetVdmOptions()
     * - .toPutDedicatedIpInPool()
     * - .toPutDedicatedIpPoolScalingAttributes()
     * - .toPutDedicatedIpWarmupAttributes()
     * - .toPutDeliverabilityDashboardOption()
     * - .toPutEmailIdentityConfigurationSetAttributes()
     * - .toPutEmailIdentityDkimAttributes()
     * - .toPutEmailIdentityDkimSigningAttributes()
     * - .toPutEmailIdentityFeedbackAttributes()
     * - .toPutEmailIdentityMailFromAttributes()
     * - .toPutSuppressedDestination()
     * - .toSendBulkEmail()
     * - .toSendCustomVerificationEmail()
     * - .toSendEmail()
     * - .toTagResource()
     * - .toTestRenderEmailTemplate()
     * - .toUntagResource()
     * - .toUpdateConfigurationSetEventDestination()
     * - .toUpdateContact()
     * - .toUpdateContactList()
     * - .toUpdateCustomVerificationEmailTemplate()
     * - .toUpdateEmailIdentityPolicy()
     * - .toUpdateEmailTemplate()
     * - .toUpdateReputationEntityCustomerManagedStatus()
     * - .toUpdateReputationEntityPolicy()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifApiVersion(value, operator) {
        return this.if(`ApiVersion`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the export source type
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html#amazonses-policy-keys
     *
     * Applies to actions:
     * - .toCancelExportJob()
     * - .toCreateExportJob()
     * - .toGetExportJob()
     * - .toListExportJobs()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifExportSourceType(value, operator) {
        return this.if(`ExportSourceType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the "Return-Path" address, which specifies where bounces and complaints are sent by email feedback forwarding
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html#amazonses-policy-keys
     *
     * Applies to actions:
     * - .toSendEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFeedbackAddress(value, operator) {
        return this.if(`FeedbackAddress`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the "From" address of a message
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html#amazonses-policy-keys
     *
     * Applies to actions:
     * - .toSendEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFromAddress(value, operator) {
        return this.if(`FromAddress`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the "From" address that is used as the display name of a message
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html#amazonses-policy-keys
     *
     * Applies to actions:
     * - .toSendEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFromDisplayName(value, operator) {
        return this.if(`FromDisplayName`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the multi-region endpoint ID that is used to send email
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html#amazonses-policy-keys
     *
     * Applies to actions:
     * - .toSendBulkEmail()
     * - .toSendEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifMultiRegionEndpointId(value, operator) {
        return this.if(`MultiRegionEndpointId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the recipient addresses of a message, which include the "To", "CC", and "BCC" addresses
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html#amazonses-policy-keys
     *
     * Applies to actions:
     * - .toSendEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRecipients(value, operator) {
        return this.if(`Recipients`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the replica regions for Replicating domain DKIM signing key
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html#amazonses-policy-keys
     *
     * Applies to actions:
     * - .toReplicateEmailIdentityDkimSigningKey()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifReplicaRegion(value, operator) {
        return this.if(`ReplicaRegion`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tenant name that is used to send email
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html#amazonses-policy-keys
     *
     * Applies to actions:
     * - .toSendBulkEmail()
     * - .toSendEmail()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTenantName(value, operator) {
        return this.if(`TenantName`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [ses-v2](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonsimpleemailservicev2.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'ses';
        this.accessLevelList = {
            Read: [
                'BatchGetMetricData',
                'GetAccount',
                'GetBlacklistReports',
                'GetConfigurationSet',
                'GetConfigurationSetEventDestinations',
                'GetContact',
                'GetContactList',
                'GetCustomVerificationEmailTemplate',
                'GetDedicatedIp',
                'GetDedicatedIpPool',
                'GetDedicatedIps',
                'GetDeliverabilityDashboardOptions',
                'GetDeliverabilityTestReport',
                'GetDomainDeliverabilityCampaign',
                'GetDomainStatisticsReport',
                'GetEmailIdentity',
                'GetEmailIdentityPolicies',
                'GetEmailTemplate',
                'GetExportJob',
                'GetImportJob',
                'GetMessageInsights',
                'GetMultiRegionEndpoint',
                'GetReputationEntity',
                'GetSuppressedDestination',
                'GetTenant',
                'ListDomainDeliverabilityCampaigns',
                'ListRecommendations',
                'ListSuppressedDestinations',
                'ListTagsForResource'
            ],
            Write: [
                'CancelExportJob',
                'CreateConfigurationSet',
                'CreateConfigurationSetEventDestination',
                'CreateContact',
                'CreateContactList',
                'CreateCustomVerificationEmailTemplate',
                'CreateDedicatedIpPool',
                'CreateDeliverabilityTestReport',
                'CreateEmailIdentity',
                'CreateEmailTemplate',
                'CreateExportJob',
                'CreateImportJob',
                'CreateMultiRegionEndpoint',
                'CreateTenant',
                'CreateTenantResourceAssociation',
                'DeleteConfigurationSet',
                'DeleteConfigurationSetEventDestination',
                'DeleteContact',
                'DeleteContactList',
                'DeleteCustomVerificationEmailTemplate',
                'DeleteDedicatedIpPool',
                'DeleteEmailIdentity',
                'DeleteEmailTemplate',
                'DeleteMultiRegionEndpoint',
                'DeleteSuppressedDestination',
                'DeleteTenant',
                'DeleteTenantResourceAssociation',
                'PutAccountDedicatedIpWarmupAttributes',
                'PutAccountDetails',
                'PutAccountSendingAttributes',
                'PutAccountSuppressionAttributes',
                'PutAccountVdmAttributes',
                'PutConfigurationSetArchivingOptions',
                'PutConfigurationSetDeliveryOptions',
                'PutConfigurationSetReputationOptions',
                'PutConfigurationSetSendingOptions',
                'PutConfigurationSetSuppressionOptions',
                'PutConfigurationSetTrackingOptions',
                'PutConfigurationSetVdmOptions',
                'PutDedicatedIpInPool',
                'PutDedicatedIpPoolScalingAttributes',
                'PutDedicatedIpWarmupAttributes',
                'PutDeliverabilityDashboardOption',
                'PutEmailIdentityConfigurationSetAttributes',
                'PutEmailIdentityDkimAttributes',
                'PutEmailIdentityDkimSigningAttributes',
                'PutEmailIdentityFeedbackAttributes',
                'PutEmailIdentityMailFromAttributes',
                'PutSuppressedDestination',
                'SendBulkEmail',
                'SendCustomVerificationEmail',
                'SendEmail',
                'TestRenderEmailTemplate',
                'UpdateConfigurationSetEventDestination',
                'UpdateContact',
                'UpdateContactList',
                'UpdateCustomVerificationEmailTemplate',
                'UpdateEmailTemplate',
                'UpdateReputationEntityCustomerManagedStatus',
                'UpdateReputationEntityPolicy'
            ],
            'Permissions management': [
                'CreateEmailIdentityPolicy',
                'DeleteEmailIdentityPolicy',
                'ReplicateEmailIdentityDkimSigningKey',
                'UpdateEmailIdentityPolicy'
            ],
            List: [
                'ListConfigurationSets',
                'ListContactLists',
                'ListContacts',
                'ListCustomVerificationEmailTemplates',
                'ListDedicatedIpPools',
                'ListDeliverabilityTestReports',
                'ListEmailIdentities',
                'ListEmailTemplates',
                'ListExportJobs',
                'ListImportJobs',
                'ListMultiRegionEndpoints',
                'ListReputationEntities',
                'ListResourceTenants',
                'ListTenantResources',
                'ListTenants'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.SesV2 = SesV2;
//# sourceMappingURL=data:application/json;base64,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