"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Swf = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [swf](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonsimpleworkflowservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Swf extends shared_1.PolicyStatement {
    /**
     * Grants permission to cancel a previously started timer and record a TimerCanceled event in the history
     *
     * Access Level: Write
     */
    toCancelTimer() {
        return this.to('CancelTimer');
    }
    /**
     * Grants permission to close the workflow execution and record a WorkflowExecutionCanceled event in the history
     *
     * Access Level: Write
     */
    toCancelWorkflowExecution() {
        return this.to('CancelWorkflowExecution');
    }
    /**
     * Grants permission to close the workflow execution and record a WorkflowExecutionCompleted event in the history
     *
     * Access Level: Write
     */
    toCompleteWorkflowExecution() {
        return this.to('CompleteWorkflowExecution');
    }
    /**
     * Grants permission to close the workflow execution and start a new workflow execution of the same type using the same workflow ID and a unique run Id
     *
     * Access Level: Write
     */
    toContinueAsNewWorkflowExecution() {
        return this.to('ContinueAsNewWorkflowExecution');
    }
    /**
     * Grants permission to return the number of closed workflow executions within the given domain that meet the specified filtering criteria
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifTagFilterTag()
     * - .ifTypeFilterName()
     * - .ifTypeFilterVersion()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CountClosedWorkflowExecutions.html
     */
    toCountClosedWorkflowExecutions() {
        return this.to('CountClosedWorkflowExecutions');
    }
    /**
     * Grants permission to return the number of open workflow executions within the given domain that meet the specified filtering criteria
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifTagFilterTag()
     * - .ifTypeFilterName()
     * - .ifTypeFilterVersion()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CountOpenWorkflowExecutions.html
     */
    toCountOpenWorkflowExecutions() {
        return this.to('CountOpenWorkflowExecutions');
    }
    /**
     * Grants permission to return the estimated number of activity tasks in the specified task list
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifTaskListName()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CountPendingActivityTasks.html
     */
    toCountPendingActivityTasks() {
        return this.to('CountPendingActivityTasks');
    }
    /**
     * Grants permission to return the estimated number of decision tasks in the specified task list
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifTaskListName()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CountPendingDecisionTasks.html
     */
    toCountPendingDecisionTasks() {
        return this.to('CountPendingDecisionTasks');
    }
    /**
     * Grants permission to delete the specified activity type
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifActivityTypeName()
     * - .ifActivityTypeVersion()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DeleteActivityType.html
     */
    toDeleteActivityType() {
        return this.to('DeleteActivityType');
    }
    /**
     * Grants permission to delete the specified workflow type
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifWorkflowTypeName()
     * - .ifWorkflowTypeVersion()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DeleteWorkflowType.html
     */
    toDeleteWorkflowType() {
        return this.to('DeleteWorkflowType');
    }
    /**
     * Grants permission to deprecate the specified activity type
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifActivityTypeName()
     * - .ifActivityTypeVersion()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DeprecateActivityType.html
     */
    toDeprecateActivityType() {
        return this.to('DeprecateActivityType');
    }
    /**
     * Grants permission to deprecate the specified domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DeprecateDomain.html
     */
    toDeprecateDomain() {
        return this.to('DeprecateDomain');
    }
    /**
     * Grants permission to deprecate the specified workflow type
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifWorkflowTypeName()
     * - .ifWorkflowTypeVersion()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DeprecateWorkflowType.html
     */
    toDeprecateWorkflowType() {
        return this.to('DeprecateWorkflowType');
    }
    /**
     * Grants permission to return information about the specified activity type
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifActivityTypeName()
     * - .ifActivityTypeVersion()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DescribeActivityType.html
     */
    toDescribeActivityType() {
        return this.to('DescribeActivityType');
    }
    /**
     * Grants permission to return information about the specified domain, including its description and status
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DescribeDomain.html
     */
    toDescribeDomain() {
        return this.to('DescribeDomain');
    }
    /**
     * Grants permission to return information about the specified workflow execution including its type and some statistics
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DescribeWorkflowExecution.html
     */
    toDescribeWorkflowExecution() {
        return this.to('DescribeWorkflowExecution');
    }
    /**
     * Grants permission to return information about the specified workflow type
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifWorkflowTypeName()
     * - .ifWorkflowTypeVersion()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DescribeWorkflowType.html
     */
    toDescribeWorkflowType() {
        return this.to('DescribeWorkflowType');
    }
    /**
     * Grants permission to close the workflow execution and record a WorkflowExecutionFailed event in the history
     *
     * Access Level: Write
     */
    toFailWorkflowExecution() {
        return this.to('FailWorkflowExecution');
    }
    /**
     * Grants permission to return the history of the specified workflow execution
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_GetWorkflowExecutionHistory.html
     */
    toGetWorkflowExecutionHistory() {
        return this.to('GetWorkflowExecutionHistory');
    }
    /**
     * Grants permission to return information about all activities registered in the specified domain that match the specified name and registration status
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListActivityTypes.html
     */
    toListActivityTypes() {
        return this.to('ListActivityTypes');
    }
    /**
     * Grants permission to return a list of closed workflow executions in the specified domain that meet the filtering criteria
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifTagFilterTag()
     * - .ifTypeFilterName()
     * - .ifTypeFilterVersion()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListClosedWorkflowExecutions.html
     */
    toListClosedWorkflowExecutions() {
        return this.to('ListClosedWorkflowExecutions');
    }
    /**
     * Grants permission to return the list of domains registered in the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListDomains.html
     */
    toListDomains() {
        return this.to('ListDomains');
    }
    /**
     * Grants permission to return a list of open workflow executions in the specified domain that meet the filtering criteria
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifTagFilterTag()
     * - .ifTypeFilterName()
     * - .ifTypeFilterVersion()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListOpenWorkflowExecutions.html
     */
    toListOpenWorkflowExecutions() {
        return this.to('ListOpenWorkflowExecutions');
    }
    /**
     * Grants permission to list tags for an AWS SWF resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to return information about workflow types in the specified domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListWorkflowTypes.html
     */
    toListWorkflowTypes() {
        return this.to('ListWorkflowTypes');
    }
    /**
     * Grants permission to workers to get an ActivityTask from the specified activity taskList
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTaskListName()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_PollForActivityTask.html
     */
    toPollForActivityTask() {
        return this.to('PollForActivityTask');
    }
    /**
     * Grants permission to deciders to get a DecisionTask from the specified decision taskList
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTaskListName()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_PollForDecisionTask.html
     */
    toPollForDecisionTask() {
        return this.to('PollForDecisionTask');
    }
    /**
     * Grants permission to workers to report to the service that the ActivityTask represented by the specified taskToken is still making progress
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RecordActivityTaskHeartbeat.html
     */
    toRecordActivityTaskHeartbeat() {
        return this.to('RecordActivityTaskHeartbeat');
    }
    /**
     * Grants permission to record a MarkerRecorded event in the history
     *
     * Access Level: Write
     */
    toRecordMarker() {
        return this.to('RecordMarker');
    }
    /**
     * Grants permission to register a new activity type along with its configuration settings in the specified domain
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDefaultTaskListName()
     * - .ifName()
     * - .ifVersion()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RegisterActivityType.html
     */
    toRegisterActivityType() {
        return this.to('RegisterActivityType');
    }
    /**
     * Grants permission to register a new domain
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RegisterDomain.html
     */
    toRegisterDomain() {
        return this.to('RegisterDomain');
    }
    /**
     * Grants permission to register a new workflow type and its configuration settings in the specified domain
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDefaultTaskListName()
     * - .ifName()
     * - .ifVersion()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RegisterWorkflowType.html
     */
    toRegisterWorkflowType() {
        return this.to('RegisterWorkflowType');
    }
    /**
     * Grants permission to attempt to cancel a previously scheduled activity task
     *
     * Access Level: Write
     */
    toRequestCancelActivityTask() {
        return this.to('RequestCancelActivityTask');
    }
    /**
     * Grants permission to request that a request be made to cancel the specified external workflow execution
     *
     * Access Level: Write
     */
    toRequestCancelExternalWorkflowExecution() {
        return this.to('RequestCancelExternalWorkflowExecution');
    }
    /**
     * Grants permission to record a WorkflowExecutionCancelRequested event in the currently running workflow execution identified by the given domain, workflowId, and runId
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RequestCancelWorkflowExecution.html
     */
    toRequestCancelWorkflowExecution() {
        return this.to('RequestCancelWorkflowExecution');
    }
    /**
     * Grants permission to workers to tell the service that the ActivityTask identified by the taskToken was successfully canceled
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RespondActivityTaskCanceled.html
     */
    toRespondActivityTaskCanceled() {
        return this.to('RespondActivityTaskCanceled');
    }
    /**
     * Grants permission to workers to tell the service that the ActivityTask identified by the taskToken completed successfully with a result (if provided)
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifActivityTypeName()
     * - .ifActivityTypeVersion()
     * - .ifTagListMember0()
     * - .ifTagListMember1()
     * - .ifTagListMember2()
     * - .ifTagListMember3()
     * - .ifTagListMember4()
     * - .ifTaskListName()
     * - .ifWorkflowTypeName()
     * - .ifWorkflowTypeVersion()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RespondActivityTaskCompleted.html
     */
    toRespondActivityTaskCompleted() {
        return this.to('RespondActivityTaskCompleted');
    }
    /**
     * Grants permission to workers to tell the service that the ActivityTask identified by the taskToken has failed with reason (if specified)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RespondActivityTaskFailed.html
     */
    toRespondActivityTaskFailed() {
        return this.to('RespondActivityTaskFailed');
    }
    /**
     * Grants permission to deciders to tell the service that the DecisionTask identified by the taskToken has successfully completed
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RespondDecisionTaskCompleted.html
     */
    toRespondDecisionTaskCompleted() {
        return this.to('RespondDecisionTaskCompleted');
    }
    /**
     * Grants permission to schedule an activity task
     *
     * Access Level: Write
     */
    toScheduleActivityTask() {
        return this.to('ScheduleActivityTask');
    }
    /**
     * Grants permission to request a signal to be delivered to the specified external workflow execution and records
     *
     * Access Level: Write
     */
    toSignalExternalWorkflowExecution() {
        return this.to('SignalExternalWorkflowExecution');
    }
    /**
     * Grants permission to record a WorkflowExecutionSignaled event in the workflow execution history and create a decision task for the workflow execution identified by the given domain, workflowId and runId
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_SignalWorkflowExecution.html
     */
    toSignalWorkflowExecution() {
        return this.to('SignalWorkflowExecution');
    }
    /**
     * Grants permission to request that a child workflow execution be started
     *
     * Access Level: Write
     */
    toStartChildWorkflowExecution() {
        return this.to('StartChildWorkflowExecution');
    }
    /**
     * Grants permission to start a timer for a workflow execution
     *
     * Access Level: Write
     */
    toStartTimer() {
        return this.to('StartTimer');
    }
    /**
     * Grants permission to start an execution of the workflow type in the specified domain using the provided workflowId and input data
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTagListMember0()
     * - .ifTagListMember1()
     * - .ifTagListMember2()
     * - .ifTagListMember3()
     * - .ifTagListMember4()
     * - .ifTaskListName()
     * - .ifWorkflowTypeName()
     * - .ifWorkflowTypeVersion()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_StartWorkflowExecution.html
     */
    toStartWorkflowExecution() {
        return this.to('StartWorkflowExecution');
    }
    /**
     * Grants permission to tag an AWS SWF resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to record a WorkflowExecutionTerminated event and force closure of the workflow execution identified by the given domain, runId, and workflowId
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_TerminateWorkflowExecution.html
     */
    toTerminateWorkflowExecution() {
        return this.to('TerminateWorkflowExecution');
    }
    /**
     * Grants permission to undeprecate a previously deprecated activity type
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifActivityTypeName()
     * - .ifActivityTypeVersion()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_UndeprecateActivityType.html
     */
    toUndeprecateActivityType() {
        return this.to('UndeprecateActivityType');
    }
    /**
     * Grants permission to undeprecate a previously deprecated domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_UndeprecateDomain.html
     */
    toUndeprecateDomain() {
        return this.to('UndeprecateDomain');
    }
    /**
     * Grants permission to undeprecate a previously deprecated workflow type
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifWorkflowTypeName()
     * - .ifWorkflowTypeVersion()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_UndeprecateWorkflowType.html
     */
    toUndeprecateWorkflowType() {
        return this.to('UndeprecateWorkflowType');
    }
    /**
     * Grants permission to remove a tag from an AWS SWF resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Adds a resource of type domain to the statement
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-domains.html
     *
     * @param domainName - Identifier for the domainName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDomain(domainName, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:swf::${account ?? this.defaultAccount}:/domain/${domainName}`);
    }
    /**
     * Filters access by tag of the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toRegisterDomain()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag of the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - domain
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag of the key
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toRegisterDomain()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the name of the activity type
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .toDeleteActivityType()
     * - .toDeprecateActivityType()
     * - .toDescribeActivityType()
     * - .toRespondActivityTaskCompleted()
     * - .toUndeprecateActivityType()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifActivityTypeName(value, operator) {
        return this.if(`activityType.name`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the version of the activity type
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .toDeleteActivityType()
     * - .toDeprecateActivityType()
     * - .toDescribeActivityType()
     * - .toRespondActivityTaskCompleted()
     * - .toUndeprecateActivityType()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifActivityTypeVersion(value, operator) {
        return this.if(`activityType.version`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the name of the default task list
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .toRegisterActivityType()
     * - .toRegisterWorkflowType()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDefaultTaskListName(value, operator) {
        return this.if(`defaultTaskList.name`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the name of activities or workflows
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .toRegisterActivityType()
     * - .toRegisterWorkflowType()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifName(value, operator) {
        return this.if(`name`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the value of tagFilter.tag
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .toCountClosedWorkflowExecutions()
     * - .toCountOpenWorkflowExecutions()
     * - .toListClosedWorkflowExecutions()
     * - .toListOpenWorkflowExecutions()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTagFilterTag(value, operator) {
        return this.if(`tagFilter.tag`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the specified tag
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .toRespondActivityTaskCompleted()
     * - .toStartWorkflowExecution()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTagListMember0(value, operator) {
        return this.if(`tagList.member.0`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the specified tag
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .toRespondActivityTaskCompleted()
     * - .toStartWorkflowExecution()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTagListMember1(value, operator) {
        return this.if(`tagList.member.1`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the specified tag
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .toRespondActivityTaskCompleted()
     * - .toStartWorkflowExecution()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTagListMember2(value, operator) {
        return this.if(`tagList.member.2`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the specified tag
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .toRespondActivityTaskCompleted()
     * - .toStartWorkflowExecution()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTagListMember3(value, operator) {
        return this.if(`tagList.member.3`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the specified tag
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .toRespondActivityTaskCompleted()
     * - .toStartWorkflowExecution()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTagListMember4(value, operator) {
        return this.if(`tagList.member.4`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the name of the tasklist
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .toCountPendingActivityTasks()
     * - .toCountPendingDecisionTasks()
     * - .toPollForActivityTask()
     * - .toPollForDecisionTask()
     * - .toRespondActivityTaskCompleted()
     * - .toStartWorkflowExecution()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTaskListName(value, operator) {
        return this.if(`taskList.name`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the name of the type filter
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .toCountClosedWorkflowExecutions()
     * - .toCountOpenWorkflowExecutions()
     * - .toListClosedWorkflowExecutions()
     * - .toListOpenWorkflowExecutions()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTypeFilterName(value, operator) {
        return this.if(`typeFilter.name`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the version of the type filter
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .toCountClosedWorkflowExecutions()
     * - .toCountOpenWorkflowExecutions()
     * - .toListClosedWorkflowExecutions()
     * - .toListOpenWorkflowExecutions()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTypeFilterVersion(value, operator) {
        return this.if(`typeFilter.version`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the version of activities or workflows
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .toRegisterActivityType()
     * - .toRegisterWorkflowType()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVersion(value, operator) {
        return this.if(`version`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the name of the workflow type
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .toDeleteWorkflowType()
     * - .toDeprecateWorkflowType()
     * - .toDescribeWorkflowType()
     * - .toRespondActivityTaskCompleted()
     * - .toStartWorkflowExecution()
     * - .toUndeprecateWorkflowType()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifWorkflowTypeName(value, operator) {
        return this.if(`workflowType.name`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the version of the workflow type
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * Applies to actions:
     * - .toDeleteWorkflowType()
     * - .toDeprecateWorkflowType()
     * - .toDescribeWorkflowType()
     * - .toRespondActivityTaskCompleted()
     * - .toStartWorkflowExecution()
     * - .toUndeprecateWorkflowType()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifWorkflowTypeVersion(value, operator) {
        return this.if(`workflowType.version`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [swf](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonsimpleworkflowservice.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'swf';
        this.accessLevelList = {
            Write: [
                'CancelTimer',
                'CancelWorkflowExecution',
                'CompleteWorkflowExecution',
                'ContinueAsNewWorkflowExecution',
                'DeleteActivityType',
                'DeleteWorkflowType',
                'DeprecateActivityType',
                'DeprecateDomain',
                'DeprecateWorkflowType',
                'FailWorkflowExecution',
                'PollForActivityTask',
                'PollForDecisionTask',
                'RecordActivityTaskHeartbeat',
                'RecordMarker',
                'RegisterActivityType',
                'RegisterDomain',
                'RegisterWorkflowType',
                'RequestCancelActivityTask',
                'RequestCancelExternalWorkflowExecution',
                'RequestCancelWorkflowExecution',
                'RespondActivityTaskCanceled',
                'RespondActivityTaskCompleted',
                'RespondActivityTaskFailed',
                'RespondDecisionTaskCompleted',
                'ScheduleActivityTask',
                'SignalExternalWorkflowExecution',
                'SignalWorkflowExecution',
                'StartChildWorkflowExecution',
                'StartTimer',
                'StartWorkflowExecution',
                'TerminateWorkflowExecution',
                'UndeprecateActivityType',
                'UndeprecateDomain',
                'UndeprecateWorkflowType'
            ],
            Read: [
                'CountClosedWorkflowExecutions',
                'CountOpenWorkflowExecutions',
                'CountPendingActivityTasks',
                'CountPendingDecisionTasks',
                'DescribeActivityType',
                'DescribeDomain',
                'DescribeWorkflowExecution',
                'DescribeWorkflowType',
                'GetWorkflowExecutionHistory'
            ],
            List: [
                'ListActivityTypes',
                'ListClosedWorkflowExecutions',
                'ListDomains',
                'ListOpenWorkflowExecutions',
                'ListTagsForResource',
                'ListWorkflowTypes'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Swf = Swf;
//# sourceMappingURL=data:application/json;base64,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