import { AccessLevelList } from '../../shared/access-level';
import { PolicyStatement, Operator } from '../../shared';
import { aws_iam as iam } from "aws-cdk-lib";
/**
 * Statement provider for service [ssm-sap](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssystemsmanagerforsap.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class SsmSap extends PolicyStatement {
    servicePrefix: string;
    /**
     * Grants permission to perform backup operation on a specified database
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toBackupDatabase(): this;
    /**
     * Grants permission to delete the SSM for SAP level resource permissions associated with a SSM for SAP database resource
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toDeleteResourcePermission(): this;
    /**
     * Grants permission to deregister an SAP application with SSM for SAP
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toDeregisterApplication(): this;
    /**
     * Grants permission to access information about an application registered with SSM for SAP by providing the application ID or application ARN
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toGetApplication(): this;
    /**
     * Grants permission to access information about a component registered with SSM for SAP by providing the application ID and component ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toGetComponent(): this;
    /**
     * Grants permission to get the details of a configuration check operation by specifying the operation ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toGetConfigurationCheckOperation(): this;
    /**
     * Grants permission to access information about a database registered with SSM for SAP by providing the application ID, component ID, and database ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toGetDatabase(): this;
    /**
     * Grants permission to access information about an operation by providing its operation ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toGetOperation(): this;
    /**
     * Grants permission to get the SSM for SAP level resource permissions associated with a SSM for SAP database resource
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toGetResourcePermission(): this;
    /**
     * Grants permission to retrieve a list of all applications registered with SSM for SAP under the customer AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toListApplications(): this;
    /**
     * Grants permission to retrieve a list of all components in the account of customer, or a specific application
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toListComponents(): this;
    /**
     * Grants permission to list all configuration check types supported by AWS Systems Manager for SAP
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toListConfigurationCheckDefinitions(): this;
    /**
     * Grants permission to list past configuration check operations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toListConfigurationCheckOperations(): this;
    /**
     * Grants permission to retrieve a list of all databases in the account of customer, or a specific application
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toListDatabases(): this;
    /**
     * Grants permission to retrieve a list of all operation events in a specified operation
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toListOperationEvents(): this;
    /**
     * Grants permission to retrieve a list of all operations in the account of customer, additional filters can be applied
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toListOperations(): this;
    /**
     * Grants permission to list the sub-check results of a specified configuration check operation
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toListSubCheckResults(): this;
    /**
     * Grants permission to list the rules of a specified sub-check belonging to a configuration check operation
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toListSubCheckRuleResults(): this;
    /**
     * Grants permission to list the tags on a specified resource ARN
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toListTagsForResource(): this;
    /**
     * Grants permission to add the SSM for SAP level resource permissions associated with a SSM for SAP database resource
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toPutResourcePermission(): this;
    /**
     * Grants permission to registers an SAP application with SSM for SAP
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toRegisterApplication(): this;
    /**
     * Grants permission to restore a database from another database
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toRestoreDatabase(): this;
    /**
     * Grants permission to start a registered SSM for SAP application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toStartApplication(): this;
    /**
     * Grants permission to start an on-demand discovery of a registered SSM for SAP application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toStartApplicationRefresh(): this;
    /**
     * Grants permission to iniitiate configuration check operations against a specified application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toStartConfigurationChecks(): this;
    /**
     * Grants permission to stop a registered SSM for SAP application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toStopApplication(): this;
    /**
     * Grants permission to tag a specified resource ARN
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toTagResource(): this;
    /**
     * Grants permission to remove tags from a specified resource ARN
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toUntagResource(): this;
    /**
     * Grants permission to update settings of a registered SSM for SAP application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toUpdateApplicationSettings(): this;
    /**
     * Grants permission to update the HANA backup settings of a specified database
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toUpdateHANABackupSettings(): this;
    protected accessLevelList: AccessLevelList;
    /**
     * Adds a resource of type application to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     *
     * @param applicationType - Identifier for the applicationType.
     * @param applicationId - Identifier for the applicationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onApplication(applicationType: string, applicationId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type component to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     *
     * @param applicationType - Identifier for the applicationType.
     * @param applicationId - Identifier for the applicationId.
     * @param componentId - Identifier for the componentId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onComponent(applicationType: string, applicationId: string, componentId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type database to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     *
     * @param applicationType - Identifier for the applicationType.
     * @param applicationId - Identifier for the applicationId.
     * @param databaseId - Identifier for the databaseId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDatabase(applicationType: string, applicationId: string, databaseId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     *
     * Applies to actions:
     * - .toRegisterApplication()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     *
     * Applies to resource types:
     * - application
     * - component
     * - database
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     *
     * Applies to actions:
     * - .toRegisterApplication()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value: string | string[], operator?: Operator | string): this;
    /**
     * Statement provider for service [ssm-sap](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssystemsmanagerforsap.html).
     *
     */
    constructor(props?: iam.PolicyStatementProps);
}
