"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SsmSap = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [ssm-sap](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssystemsmanagerforsap.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class SsmSap extends shared_1.PolicyStatement {
    /**
     * Grants permission to perform backup operation on a specified database
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toBackupDatabase() {
        return this.to('BackupDatabase');
    }
    /**
     * Grants permission to delete the SSM for SAP level resource permissions associated with a SSM for SAP database resource
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toDeleteResourcePermission() {
        return this.to('DeleteResourcePermission');
    }
    /**
     * Grants permission to deregister an SAP application with SSM for SAP
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toDeregisterApplication() {
        return this.to('DeregisterApplication');
    }
    /**
     * Grants permission to access information about an application registered with SSM for SAP by providing the application ID or application ARN
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toGetApplication() {
        return this.to('GetApplication');
    }
    /**
     * Grants permission to access information about a component registered with SSM for SAP by providing the application ID and component ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toGetComponent() {
        return this.to('GetComponent');
    }
    /**
     * Grants permission to get the details of a configuration check operation by specifying the operation ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toGetConfigurationCheckOperation() {
        return this.to('GetConfigurationCheckOperation');
    }
    /**
     * Grants permission to access information about a database registered with SSM for SAP by providing the application ID, component ID, and database ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toGetDatabase() {
        return this.to('GetDatabase');
    }
    /**
     * Grants permission to access information about an operation by providing its operation ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toGetOperation() {
        return this.to('GetOperation');
    }
    /**
     * Grants permission to get the SSM for SAP level resource permissions associated with a SSM for SAP database resource
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toGetResourcePermission() {
        return this.to('GetResourcePermission');
    }
    /**
     * Grants permission to retrieve a list of all applications registered with SSM for SAP under the customer AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toListApplications() {
        return this.to('ListApplications');
    }
    /**
     * Grants permission to retrieve a list of all components in the account of customer, or a specific application
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toListComponents() {
        return this.to('ListComponents');
    }
    /**
     * Grants permission to list all configuration check types supported by AWS Systems Manager for SAP
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toListConfigurationCheckDefinitions() {
        return this.to('ListConfigurationCheckDefinitions');
    }
    /**
     * Grants permission to list past configuration check operations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toListConfigurationCheckOperations() {
        return this.to('ListConfigurationCheckOperations');
    }
    /**
     * Grants permission to retrieve a list of all databases in the account of customer, or a specific application
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toListDatabases() {
        return this.to('ListDatabases');
    }
    /**
     * Grants permission to retrieve a list of all operation events in a specified operation
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toListOperationEvents() {
        return this.to('ListOperationEvents');
    }
    /**
     * Grants permission to retrieve a list of all operations in the account of customer, additional filters can be applied
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toListOperations() {
        return this.to('ListOperations');
    }
    /**
     * Grants permission to list the sub-check results of a specified configuration check operation
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toListSubCheckResults() {
        return this.to('ListSubCheckResults');
    }
    /**
     * Grants permission to list the rules of a specified sub-check belonging to a configuration check operation
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toListSubCheckRuleResults() {
        return this.to('ListSubCheckRuleResults');
    }
    /**
     * Grants permission to list the tags on a specified resource ARN
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to add the SSM for SAP level resource permissions associated with a SSM for SAP database resource
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toPutResourcePermission() {
        return this.to('PutResourcePermission');
    }
    /**
     * Grants permission to registers an SAP application with SSM for SAP
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toRegisterApplication() {
        return this.to('RegisterApplication');
    }
    /**
     * Grants permission to restore a database from another database
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toRestoreDatabase() {
        return this.to('RestoreDatabase');
    }
    /**
     * Grants permission to start a registered SSM for SAP application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toStartApplication() {
        return this.to('StartApplication');
    }
    /**
     * Grants permission to start an on-demand discovery of a registered SSM for SAP application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toStartApplicationRefresh() {
        return this.to('StartApplicationRefresh');
    }
    /**
     * Grants permission to iniitiate configuration check operations against a specified application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toStartConfigurationChecks() {
        return this.to('StartConfigurationChecks');
    }
    /**
     * Grants permission to stop a registered SSM for SAP application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toStopApplication() {
        return this.to('StopApplication');
    }
    /**
     * Grants permission to tag a specified resource ARN
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from a specified resource ARN
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update settings of a registered SSM for SAP application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toUpdateApplicationSettings() {
        return this.to('UpdateApplicationSettings');
    }
    /**
     * Grants permission to update the HANA backup settings of a specified database
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     */
    toUpdateHANABackupSettings() {
        return this.to('UpdateHANABackupSettings');
    }
    /**
     * Adds a resource of type application to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     *
     * @param applicationType - Identifier for the applicationType.
     * @param applicationId - Identifier for the applicationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onApplication(applicationType, applicationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ssm-sap:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:${applicationType}/${applicationId}`);
    }
    /**
     * Adds a resource of type component to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     *
     * @param applicationType - Identifier for the applicationType.
     * @param applicationId - Identifier for the applicationId.
     * @param componentId - Identifier for the componentId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onComponent(applicationType, applicationId, componentId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ssm-sap:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:${applicationType}/${applicationId}/COMPONENT/${componentId}`);
    }
    /**
     * Adds a resource of type database to the statement
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     *
     * @param applicationType - Identifier for the applicationType.
     * @param applicationId - Identifier for the applicationId.
     * @param databaseId - Identifier for the databaseId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDatabase(applicationType, applicationId, databaseId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ssm-sap:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:${applicationType}/${applicationId}/DB/${databaseId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     *
     * Applies to actions:
     * - .toRegisterApplication()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     *
     * Applies to resource types:
     * - application
     * - component
     * - database
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/systems-manager/index.html
     *
     * Applies to actions:
     * - .toRegisterApplication()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [ssm-sap](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssystemsmanagerforsap.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'ssm-sap';
        this.accessLevelList = {
            Write: [
                'BackupDatabase',
                'DeregisterApplication',
                'RegisterApplication',
                'RestoreDatabase',
                'StartApplication',
                'StartApplicationRefresh',
                'StartConfigurationChecks',
                'StopApplication',
                'UpdateApplicationSettings',
                'UpdateHANABackupSettings'
            ],
            'Permissions management': [
                'DeleteResourcePermission',
                'GetResourcePermission',
                'PutResourcePermission'
            ],
            Read: [
                'GetApplication',
                'GetComponent',
                'GetConfigurationCheckOperation',
                'GetDatabase',
                'GetOperation',
                'ListTagsForResource'
            ],
            List: [
                'ListApplications',
                'ListComponents',
                'ListConfigurationCheckDefinitions',
                'ListConfigurationCheckOperations',
                'ListDatabases',
                'ListOperationEvents',
                'ListOperations',
                'ListSubCheckResults',
                'ListSubCheckRuleResults'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.SsmSap = SsmSap;
//# sourceMappingURL=data:application/json;base64,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