"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SsmContacts = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [ssm-contacts](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssystemsmanagerincidentmanagercontacts.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class SsmContacts extends shared_1.PolicyStatement {
    /**
     * Grants permission to accept a page
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_SSMContacts_AcceptPage.html
     */
    toAcceptPage() {
        return this.to('AcceptPage');
    }
    /**
     * Grants permission to activate a contact's contact channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_SSMContacts_ActivateContactChannel.html
     */
    toActivateContactChannel() {
        return this.to('ActivateContactChannel');
    }
    /**
     * Grants permission to use a contact in an escalation plan
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/incident-manager/latest/userguide/contacts.html
     */
    toAssociateContact() {
        return this.to('AssociateContact');
    }
    /**
     * Grants permission to create a contact
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * Dependent actions:
     * - ssm-contacts:AssociateContact
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_SSMContacts_CreateContact.html
     */
    toCreateContact() {
        return this.to('CreateContact');
    }
    /**
     * Grants permission to create a contact channel for a contact
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_SSMContacts_CreateContactChannel.html
     */
    toCreateContactChannel() {
        return this.to('CreateContactChannel');
    }
    /**
     * Grants permission to create a rotation in an on-call schedule
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_SSMContacts_CreateRotation.html
     */
    toCreateRotation() {
        return this.to('CreateRotation');
    }
    /**
     * Grants permission to create an override for a rotation in an on-call schedule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_SSMContacts_CreateRotationOverride.html
     */
    toCreateRotationOverride() {
        return this.to('CreateRotationOverride');
    }
    /**
     * Grants permission to deactivate a contact's contact channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_SSMContacts_DeactivateContactChannel.html
     */
    toDeactivateContactChannel() {
        return this.to('DeactivateContactChannel');
    }
    /**
     * Grants permission to delete a contact
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_SSMContacts_DeleteContact.html
     */
    toDeleteContact() {
        return this.to('DeleteContact');
    }
    /**
     * Grants permission to delete a contact's contact channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_SSMContacts_DeleteContactChannel.html
     */
    toDeleteContactChannel() {
        return this.to('DeleteContactChannel');
    }
    /**
     * Grants permission to delete a rotation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_SSMContacts_DeleteRotation.html
     */
    toDeleteRotation() {
        return this.to('DeleteRotation');
    }
    /**
     * Grants permission to delete a rotation's rotation override
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_SSMContacts_DeleteRotationOverride.html
     */
    toDeleteRotationOverride() {
        return this.to('DeleteRotationOverride');
    }
    /**
     * Grants permission to describe an engagement
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_SSMContacts_DescribeEngagement.html
     */
    toDescribeEngagement() {
        return this.to('DescribeEngagement');
    }
    /**
     * Grants permission to describe a page
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_SSMContacts_DescribePage.html
     */
    toDescribePage() {
        return this.to('DescribePage');
    }
    /**
     * Grants permission to get a contact
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_SSMContacts_GetContact.html
     */
    toGetContact() {
        return this.to('GetContact');
    }
    /**
     * Grants permission to get a contact's contact channel
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_SSMContacts_GetContactChannel.html
     */
    toGetContactChannel() {
        return this.to('GetContactChannel');
    }
    /**
     * Grants permission to get a contact's resource policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_SSMContacts_GetContactPolicy.html
     */
    toGetContactPolicy() {
        return this.to('GetContactPolicy');
    }
    /**
     * Grants permission to retrieve information about an on-call rotation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_SSMContacts_GetRotation.html
     */
    toGetRotation() {
        return this.to('GetRotation');
    }
    /**
     * Grants permission to retrieve information about an override in an on-call rotation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_SSMContacts_GetRotationOverride.html
     */
    toGetRotationOverride() {
        return this.to('GetRotationOverride');
    }
    /**
     * Grants permission to list all of a contact's contact channels
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_SSMContacts_ListContactChannels.html
     */
    toListContactChannels() {
        return this.to('ListContactChannels');
    }
    /**
     * Grants permission to list all contacts
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_SSMContacts_ListContacts.html
     */
    toListContacts() {
        return this.to('ListContacts');
    }
    /**
     * Grants permission to list all engagements
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_SSMContacts_ListEngagements.html
     */
    toListEngagements() {
        return this.to('ListEngagements');
    }
    /**
     * Grants permission to list all receipts of a page
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_SSMContacts_ListPageReceipts.html
     */
    toListPageReceipts() {
        return this.to('ListPageReceipts');
    }
    /**
     * Grants permission to list the resolution path of an engagement
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_SSMContacts_ListPageResolutions.html
     */
    toListPageResolutions() {
        return this.to('ListPageResolutions');
    }
    /**
     * Grants permission to list all pages sent to a contact
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_SSMContacts_ListPagesByContact.html
     */
    toListPagesByContact() {
        return this.to('ListPagesByContact');
    }
    /**
     * Grants permission to list all pages created in an engagement
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_SSMContacts_ListPagesByEngagement.html
     */
    toListPagesByEngagement() {
        return this.to('ListPagesByEngagement');
    }
    /**
     * Grants permission to retrieve a list of shifts based on rotation configuration parameters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_SSMContacts_ListPreviewRotationShifts.html
     */
    toListPreviewRotationShifts() {
        return this.to('ListPreviewRotationShifts');
    }
    /**
     * Grants permission to retrieve a list of overrides currently specified for an on-call rotation
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_SSMContacts_ListRotationOverrides.html
     */
    toListRotationOverrides() {
        return this.to('ListRotationOverrides');
    }
    /**
     * Grants permission to retrieve a list of rotation shifts in an on-call schedule
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_SSMContacts_ListRotationShifts.html
     */
    toListRotationShifts() {
        return this.to('ListRotationShifts');
    }
    /**
     * Grants permission to retrieve a list of on-call rotations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_SSMContacts_ListRotations.html
     */
    toListRotations() {
        return this.to('ListRotations');
    }
    /**
     * Grants permission to view a list of resource tags for a specified resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_SSMContacts_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to add a resource policy to a contact
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_SSMContacts_PutContactPolicy.html
     */
    toPutContactPolicy() {
        return this.to('PutContactPolicy');
    }
    /**
     * Grants permission to send the activation code of a contact's contact channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_SSMContacts_SendActivationCode.html
     */
    toSendActivationCode() {
        return this.to('SendActivationCode');
    }
    /**
     * Grants permission to start an engagement
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_SSMContacts_StartEngagement.html
     */
    toStartEngagement() {
        return this.to('StartEngagement');
    }
    /**
     * Grants permission to stop an engagement
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_SSMContacts_StopEngagement.html
     */
    toStopEngagement() {
        return this.to('StopEngagement');
    }
    /**
     * Grants permission to add tags to the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_SSMContacts_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_SSMContacts_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a contact
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ssm-contacts:AssociateContact
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_SSMContacts_UpdateContact.html
     */
    toUpdateContact() {
        return this.to('UpdateContact');
    }
    /**
     * Grants permission to update a contact's contact channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_SSMContacts_UpdateContactChannel.html
     */
    toUpdateContactChannel() {
        return this.to('UpdateContactChannel');
    }
    /**
     * Grants permission to update the information specified for an on-call rotation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/incident-manager/latest/APIReference/API_SSMContacts_UpdateRotation.html
     */
    toUpdateRotation() {
        return this.to('UpdateRotation');
    }
    /**
     * Adds a resource of type contact to the statement
     *
     * https://docs.aws.amazon.com/incident-manager/latest/userguide/contacts.html
     *
     * @param contactAlias - Identifier for the contactAlias.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onContact(contactAlias, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ssm-contacts:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:contact/${contactAlias}`);
    }
    /**
     * Adds a resource of type contactchannel to the statement
     *
     * https://docs.aws.amazon.com/incident-manager/latest/userguide/contacts.html
     *
     * @param contactAlias - Identifier for the contactAlias.
     * @param contactChannelId - Identifier for the contactChannelId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onContactchannel(contactAlias, contactChannelId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ssm-contacts:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:contactchannel/${contactAlias}/${contactChannelId}`);
    }
    /**
     * Adds a resource of type engagement to the statement
     *
     * https://docs.aws.amazon.com/incident-manager/latest/userguide/escalation.html
     *
     * @param contactAlias - Identifier for the contactAlias.
     * @param engagementId - Identifier for the engagementId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onEngagement(contactAlias, engagementId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ssm-contacts:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:engagement/${contactAlias}/${engagementId}`);
    }
    /**
     * Adds a resource of type page to the statement
     *
     * https://docs.aws.amazon.com/incident-manager/latest/userguide/escalation.html
     *
     * @param contactAlias - Identifier for the contactAlias.
     * @param pageId - Identifier for the pageId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onPage(contactAlias, pageId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ssm-contacts:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:page/${contactAlias}/${pageId}`);
    }
    /**
     * Adds a resource of type rotation to the statement
     *
     * https://docs.aws.amazon.com/incident-manager/latest/userguide/incident-manager-on-call-schedule.html
     *
     * @param rotationId - Identifier for the rotationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRotation(rotationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ssm-contacts:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:rotation/${rotationId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateContact()
     * - .toCreateRotation()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - contact
     * - rotation
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateContact()
     * - .toCreateRotation()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [ssm-contacts](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssystemsmanagerincidentmanagercontacts.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'ssm-contacts';
        this.accessLevelList = {
            Write: [
                'AcceptPage',
                'ActivateContactChannel',
                'CreateContact',
                'CreateContactChannel',
                'CreateRotation',
                'CreateRotationOverride',
                'DeactivateContactChannel',
                'DeleteContact',
                'DeleteContactChannel',
                'DeleteRotation',
                'DeleteRotationOverride',
                'PutContactPolicy',
                'SendActivationCode',
                'StartEngagement',
                'StopEngagement',
                'UpdateContact',
                'UpdateContactChannel',
                'UpdateRotation'
            ],
            'Permissions management': [
                'AssociateContact'
            ],
            Read: [
                'DescribeEngagement',
                'DescribePage',
                'GetContact',
                'GetContactChannel',
                'GetContactPolicy',
                'GetRotation',
                'GetRotationOverride',
                'ListTagsForResource'
            ],
            List: [
                'ListContactChannels',
                'ListContacts',
                'ListEngagements',
                'ListPageReceipts',
                'ListPageResolutions',
                'ListPagesByContact',
                'ListPagesByEngagement',
                'ListPreviewRotationShifts',
                'ListRotationOverrides',
                'ListRotationShifts',
                'ListRotations'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.SsmContacts = SsmContacts;
//# sourceMappingURL=data:application/json;base64,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