"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Tnb = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [tnb](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awstelconetworkbuilder.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Tnb extends shared_1.PolicyStatement {
    /**
     * Grants permission to cancel a network operation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/tnb/latest/APIReference/API_CancelSolNetworkOperation.html
     */
    toCancelSolNetworkOperation() {
        return this.to('CancelSolNetworkOperation');
    }
    /**
     * Grants permission to create a function package
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/tnb/latest/APIReference/API_CreateSolFunctionPackage.html
     */
    toCreateSolFunctionPackage() {
        return this.to('CreateSolFunctionPackage');
    }
    /**
     * Grants permission to create a network instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/tnb/latest/APIReference/API_CreateSolNetworkInstance.html
     */
    toCreateSolNetworkInstance() {
        return this.to('CreateSolNetworkInstance');
    }
    /**
     * Grants permission to create a network package
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/tnb/latest/APIReference/API_CreateSolNetworkPackage.html
     */
    toCreateSolNetworkPackage() {
        return this.to('CreateSolNetworkPackage');
    }
    /**
     * Grants permission to delete a function package
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/tnb/latest/APIReference/API_DeleteSolFunctionPackage.html
     */
    toDeleteSolFunctionPackage() {
        return this.to('DeleteSolFunctionPackage');
    }
    /**
     * Grants permission to delete a network instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/tnb/latest/APIReference/API_DeleteSolNetworkInstance.html
     */
    toDeleteSolNetworkInstance() {
        return this.to('DeleteSolNetworkInstance');
    }
    /**
     * Grants permission to delete a network package
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/tnb/latest/APIReference/API_DeleteSolNetworkPackage.html
     */
    toDeleteSolNetworkPackage() {
        return this.to('DeleteSolNetworkPackage');
    }
    /**
     * Grants permission to get a function instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/tnb/latest/APIReference/API_GetSolFunctionInstance.html
     */
    toGetSolFunctionInstance() {
        return this.to('GetSolFunctionInstance');
    }
    /**
     * Grants permission to get a function package
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/tnb/latest/APIReference/API_GetSolFunctionPackage.html
     */
    toGetSolFunctionPackage() {
        return this.to('GetSolFunctionPackage');
    }
    /**
     * Grants permission to get a function package contents
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/tnb/latest/APIReference/API_GetSolFunctionPackageContent.html
     */
    toGetSolFunctionPackageContent() {
        return this.to('GetSolFunctionPackageContent');
    }
    /**
     * Grants permission to get a function package descriptor
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/tnb/latest/APIReference/API_GetSolFunctionPackageDescriptor.html
     */
    toGetSolFunctionPackageDescriptor() {
        return this.to('GetSolFunctionPackageDescriptor');
    }
    /**
     * Grants permission to get a network instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/tnb/latest/APIReference/API_GetSolNetworkInstance.html
     */
    toGetSolNetworkInstance() {
        return this.to('GetSolNetworkInstance');
    }
    /**
     * Grants permission to get a network operation
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/tnb/latest/APIReference/API_GetSolNetworkOperation.html
     */
    toGetSolNetworkOperation() {
        return this.to('GetSolNetworkOperation');
    }
    /**
     * Grants permission to get a network package
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/tnb/latest/APIReference/API_GetSolNetworkPackage.html
     */
    toGetSolNetworkPackage() {
        return this.to('GetSolNetworkPackage');
    }
    /**
     * Grants permission to get a network package contents
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/tnb/latest/APIReference/API_GetSolNetworkPackageContent.html
     */
    toGetSolNetworkPackageContent() {
        return this.to('GetSolNetworkPackageContent');
    }
    /**
     * Grants permission to get a network package descriptor
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/tnb/latest/APIReference/API_GetSolNetworkPackageDescriptor.html
     */
    toGetSolNetworkPackageDescriptor() {
        return this.to('GetSolNetworkPackageDescriptor');
    }
    /**
     * Grants permission to instantiate a network instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/tnb/latest/APIReference/API_InstantiateSolNetworkInstance.html
     */
    toInstantiateSolNetworkInstance() {
        return this.to('InstantiateSolNetworkInstance');
    }
    /**
     * Grants permission to list function instances
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/tnb/latest/APIReference/API_ListSolFunctionInstances.html
     */
    toListSolFunctionInstances() {
        return this.to('ListSolFunctionInstances');
    }
    /**
     * Grants permission to list function packages
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/tnb/latest/APIReference/API_ListSolFunctionPackages.html
     */
    toListSolFunctionPackages() {
        return this.to('ListSolFunctionPackages');
    }
    /**
     * Grants permission to list network instances
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/tnb/latest/APIReference/API_ListSolNetworkInstances.html
     */
    toListSolNetworkInstances() {
        return this.to('ListSolNetworkInstances');
    }
    /**
     * Grants permission to list network operations
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/tnb/latest/APIReference/API_ListSolNetworkOperations.html
     */
    toListSolNetworkOperations() {
        return this.to('ListSolNetworkOperations');
    }
    /**
     * Grants permission to list network packages
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/tnb/latest/APIReference/API_ListSolNetworkPackages.html
     */
    toListSolNetworkPackages() {
        return this.to('ListSolNetworkPackages');
    }
    /**
     * Grants permission to return a list of tags for a resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/tnb/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to upload function package content
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/tnb/latest/APIReference/API_PutSolFunctionPackageContent.html
     */
    toPutSolFunctionPackageContent() {
        return this.to('PutSolFunctionPackageContent');
    }
    /**
     * Grants permission to upload network package content
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/tnb/latest/APIReference/API_PutSolNetworkPackageContent.html
     */
    toPutSolNetworkPackageContent() {
        return this.to('PutSolNetworkPackageContent');
    }
    /**
     * Grants permission to add tags to the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/tnb/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to terminate a network instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/tnb/latest/APIReference/API_TerminateSolNetworkInstance.html
     */
    toTerminateSolNetworkInstance() {
        return this.to('TerminateSolNetworkInstance');
    }
    /**
     * Grants permission to remove tags from the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/tnb/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a function package
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/tnb/latest/APIReference/API_UpdateSolFunctionPackage.html
     */
    toUpdateSolFunctionPackage() {
        return this.to('UpdateSolFunctionPackage');
    }
    /**
     * Grants permission to update a network instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/tnb/latest/APIReference/API_UpdateSolNetworkInstance.html
     */
    toUpdateSolNetworkInstance() {
        return this.to('UpdateSolNetworkInstance');
    }
    /**
     * Grants permission to update a network package
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/tnb/latest/APIReference/API_UpdateSolNetworkPackage.html
     */
    toUpdateSolNetworkPackage() {
        return this.to('UpdateSolNetworkPackage');
    }
    /**
     * Grants permission to validate function package content
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/tnb/latest/APIReference/API_ValidateSolFunctionPackageContent.html
     */
    toValidateSolFunctionPackageContent() {
        return this.to('ValidateSolFunctionPackageContent');
    }
    /**
     * Grants permission to validate network package content
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/tnb/latest/APIReference/API_ValidateSolNetworkPackageContent.html
     */
    toValidateSolNetworkPackageContent() {
        return this.to('ValidateSolNetworkPackageContent');
    }
    /**
     * Adds a resource of type function-package to the statement
     *
     * https://docs.aws.amazon.com/tnb/latest/ug/function-packages.html
     *
     * @param functionPackageId - Identifier for the functionPackageId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFunctionPackage(functionPackageId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:tnb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:function-package/${functionPackageId}`);
    }
    /**
     * Adds a resource of type network-package to the statement
     *
     * https://docs.aws.amazon.com/tnb/latest/ug/network-packages.html
     *
     * @param networkPackageId - Identifier for the networkPackageId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onNetworkPackage(networkPackageId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:tnb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:network-package/${networkPackageId}`);
    }
    /**
     * Adds a resource of type network-instance to the statement
     *
     * https://docs.aws.amazon.com/tnb/latest/ug/network-instances.html
     *
     * @param networkInstanceId - Identifier for the networkInstanceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onNetworkInstance(networkInstanceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:tnb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:network-instance/${networkInstanceId}`);
    }
    /**
     * Adds a resource of type function-instance to the statement
     *
     * https://docs.aws.amazon.com/tnb/latest/ug/function-packages.html
     *
     * @param functionInstanceId - Identifier for the functionInstanceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFunctionInstance(functionInstanceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:tnb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:function-instance/${functionInstanceId}`);
    }
    /**
     * Adds a resource of type network-operation to the statement
     *
     * https://docs.aws.amazon.com/tnb/latest/ug/network-operations.html
     *
     * @param networkOperationId - Identifier for the networkOperationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onNetworkOperation(networkOperationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:tnb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:network-operation/${networkOperationId}`);
    }
    /**
     * Filters access by checking the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateSolFunctionPackage()
     * - .toCreateSolNetworkInstance()
     * - .toCreateSolNetworkPackage()
     * - .toInstantiateSolNetworkInstance()
     * - .toTagResource()
     * - .toTerminateSolNetworkInstance()
     * - .toUpdateSolNetworkInstance()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by checking tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toGetSolFunctionInstance()
     * - .toGetSolFunctionPackage()
     * - .toGetSolFunctionPackageContent()
     * - .toGetSolFunctionPackageDescriptor()
     * - .toGetSolNetworkInstance()
     * - .toGetSolNetworkOperation()
     * - .toGetSolNetworkPackage()
     * - .toGetSolNetworkPackageContent()
     * - .toGetSolNetworkPackageDescriptor()
     * - .toListSolFunctionInstances()
     * - .toListSolFunctionPackages()
     * - .toListSolNetworkInstances()
     * - .toListSolNetworkOperations()
     * - .toListSolNetworkPackages()
     *
     * Applies to resource types:
     * - function-package
     * - network-package
     * - network-instance
     * - function-instance
     * - network-operation
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateSolFunctionPackage()
     * - .toCreateSolNetworkInstance()
     * - .toCreateSolNetworkPackage()
     * - .toInstantiateSolNetworkInstance()
     * - .toTagResource()
     * - .toTerminateSolNetworkInstance()
     * - .toUntagResource()
     * - .toUpdateSolNetworkInstance()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [tnb](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awstelconetworkbuilder.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'tnb';
        this.accessLevelList = {
            Write: [
                'CancelSolNetworkOperation',
                'CreateSolFunctionPackage',
                'CreateSolNetworkInstance',
                'CreateSolNetworkPackage',
                'DeleteSolFunctionPackage',
                'DeleteSolNetworkInstance',
                'DeleteSolNetworkPackage',
                'InstantiateSolNetworkInstance',
                'PutSolFunctionPackageContent',
                'PutSolNetworkPackageContent',
                'TerminateSolNetworkInstance',
                'UpdateSolFunctionPackage',
                'UpdateSolNetworkInstance',
                'UpdateSolNetworkPackage',
                'ValidateSolFunctionPackageContent',
                'ValidateSolNetworkPackageContent'
            ],
            Read: [
                'GetSolFunctionInstance',
                'GetSolFunctionPackage',
                'GetSolFunctionPackageContent',
                'GetSolFunctionPackageDescriptor',
                'GetSolNetworkInstance',
                'GetSolNetworkOperation',
                'GetSolNetworkPackage',
                'GetSolNetworkPackageContent',
                'GetSolNetworkPackageDescriptor'
            ],
            List: [
                'ListSolFunctionInstances',
                'ListSolFunctionPackages',
                'ListSolNetworkInstances',
                'ListSolNetworkOperations',
                'ListSolNetworkPackages',
                'ListTagsForResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Tnb = Tnb;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVsY29uZXR3b3JrYnVpbGRlci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInRlbGNvbmV0d29ya2J1aWxkZXIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQ0EseUNBQXlEO0FBR3pEOzs7O0dBSUc7QUFDSCxNQUFhLEdBQUksU0FBUSx3QkFBZTtJQUd0Qzs7Ozs7O09BTUc7SUFDSSwyQkFBMkI7UUFDaEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDJCQUEyQixDQUFDLENBQUM7SUFDOUMsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSwwQkFBMEI7UUFDL0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDBCQUEwQixDQUFDLENBQUM7SUFDN0MsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSwwQkFBMEI7UUFDL0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDBCQUEwQixDQUFDLENBQUM7SUFDN0MsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSx5QkFBeUI7UUFDOUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHlCQUF5QixDQUFDLENBQUM7SUFDNUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDBCQUEwQjtRQUMvQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMEJBQTBCLENBQUMsQ0FBQztJQUM3QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMEJBQTBCO1FBQy9CLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO0lBQzdDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx5QkFBeUI7UUFDOUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHlCQUF5QixDQUFDLENBQUM7SUFDNUMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLHdCQUF3QjtRQUM3QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLENBQUMsQ0FBQztJQUMzQyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksdUJBQXVCO1FBQzVCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO0lBQzFDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSw4QkFBOEI7UUFDbkMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDhCQUE4QixDQUFDLENBQUM7SUFDakQsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLGlDQUFpQztRQUN0QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsaUNBQWlDLENBQUMsQ0FBQztJQUNwRCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksdUJBQXVCO1FBQzVCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO0lBQzFDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSx3QkFBd0I7UUFDN0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHdCQUF3QixDQUFDLENBQUM7SUFDM0MsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLHNCQUFzQjtRQUMzQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksNkJBQTZCO1FBQ2xDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO0lBQ2hELENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxnQ0FBZ0M7UUFDckMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdDQUFnQyxDQUFDLENBQUM7SUFDbkQsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSwrQkFBK0I7UUFDcEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLCtCQUErQixDQUFDLENBQUM7SUFDbEQsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLDBCQUEwQjtRQUMvQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMEJBQTBCLENBQUMsQ0FBQztJQUM3QyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0kseUJBQXlCO1FBQzlCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO0lBQzVDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSx5QkFBeUI7UUFDOUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHlCQUF5QixDQUFDLENBQUM7SUFDNUMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLDBCQUEwQjtRQUMvQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMEJBQTBCLENBQUMsQ0FBQztJQUM3QyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksd0JBQXdCO1FBQzdCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO0lBQzNDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDhCQUE4QjtRQUNuQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsOEJBQThCLENBQUMsQ0FBQztJQUNqRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksNkJBQTZCO1FBQ2xDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO0lBQ2hELENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksYUFBYTtRQUNsQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsYUFBYSxDQUFDLENBQUM7SUFDaEMsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSw2QkFBNkI7UUFDbEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDZCQUE2QixDQUFDLENBQUM7SUFDaEQsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLGVBQWU7UUFDcEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxDQUFDO0lBQ2xDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwwQkFBMEI7UUFDL0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDBCQUEwQixDQUFDLENBQUM7SUFDN0MsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSwwQkFBMEI7UUFDL0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDBCQUEwQixDQUFDLENBQUM7SUFDN0MsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHlCQUF5QjtRQUM5QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMseUJBQXlCLENBQUMsQ0FBQztJQUM1QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUNBQW1DO1FBQ3hDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQ0FBbUMsQ0FBQyxDQUFDO0lBQ3RELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQ0FBa0M7UUFDdkMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtDQUFrQyxDQUFDLENBQUM7SUFDckQsQ0FBQztJQThDRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxpQkFBaUIsQ0FBQyxpQkFBeUIsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUN2RyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixRQUFTLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxxQkFBc0IsaUJBQWtCLEVBQUUsQ0FBQyxDQUFDO0lBQ2xMLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxnQkFBZ0IsQ0FBQyxnQkFBd0IsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNyRyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixRQUFTLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxvQkFBcUIsZ0JBQWlCLEVBQUUsQ0FBQyxDQUFDO0lBQ2hMLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxpQkFBaUIsQ0FBQyxpQkFBeUIsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUN2RyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixRQUFTLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxxQkFBc0IsaUJBQWtCLEVBQUUsQ0FBQyxDQUFDO0lBQ2xMLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxrQkFBa0IsQ0FBQyxrQkFBMEIsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUN6RyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixRQUFTLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxzQkFBdUIsa0JBQW1CLEVBQUUsQ0FBQyxDQUFDO0lBQ3BMLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxrQkFBa0IsQ0FBQyxrQkFBMEIsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUN6RyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixRQUFTLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxzQkFBdUIsa0JBQW1CLEVBQUUsQ0FBQyxDQUFDO0lBQ3BMLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FpQkc7SUFDSSxlQUFlLENBQUMsTUFBYyxFQUFFLEtBQXdCLEVBQUUsUUFBNEI7UUFDM0YsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFtQixNQUFPLEVBQUUsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2hGLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztPQStCRztJQUNJLGdCQUFnQixDQUFDLE1BQWMsRUFBRSxLQUF3QixFQUFFLFFBQTRCO1FBQzVGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBb0IsTUFBTyxFQUFFLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNqRixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7O09BaUJHO0lBQ0ksWUFBWSxDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDeEUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGFBQWEsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2pFLENBQUM7SUFFRDs7O09BR0c7SUFDSCxZQUFZLEtBQWdDO1FBQzFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQztRQTVvQlIsa0JBQWEsR0FBRyxLQUFLLENBQUM7UUFzYm5CLG9CQUFlLEdBQW9CO1lBQzNDLEtBQUssRUFBRTtnQkFDTCwyQkFBMkI7Z0JBQzNCLDBCQUEwQjtnQkFDMUIsMEJBQTBCO2dCQUMxQix5QkFBeUI7Z0JBQ3pCLDBCQUEwQjtnQkFDMUIsMEJBQTBCO2dCQUMxQix5QkFBeUI7Z0JBQ3pCLCtCQUErQjtnQkFDL0IsOEJBQThCO2dCQUM5Qiw2QkFBNkI7Z0JBQzdCLDZCQUE2QjtnQkFDN0IsMEJBQTBCO2dCQUMxQiwwQkFBMEI7Z0JBQzFCLHlCQUF5QjtnQkFDekIsbUNBQW1DO2dCQUNuQyxrQ0FBa0M7YUFDbkM7WUFDRCxJQUFJLEVBQUU7Z0JBQ0osd0JBQXdCO2dCQUN4Qix1QkFBdUI7Z0JBQ3ZCLDhCQUE4QjtnQkFDOUIsaUNBQWlDO2dCQUNqQyx1QkFBdUI7Z0JBQ3ZCLHdCQUF3QjtnQkFDeEIsc0JBQXNCO2dCQUN0Qiw2QkFBNkI7Z0JBQzdCLGdDQUFnQzthQUNqQztZQUNELElBQUksRUFBRTtnQkFDSiwwQkFBMEI7Z0JBQzFCLHlCQUF5QjtnQkFDekIseUJBQXlCO2dCQUN6QiwwQkFBMEI7Z0JBQzFCLHdCQUF3QjtnQkFDeEIscUJBQXFCO2FBQ3RCO1lBQ0QsT0FBTyxFQUFFO2dCQUNQLGFBQWE7Z0JBQ2IsZUFBZTthQUNoQjtTQUNGLENBQUM7SUE2S0YsQ0FBQztDQUNGO0FBL29CRCxrQkErb0JDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQWNjZXNzTGV2ZWxMaXN0IH0gZnJvbSAnLi4vLi4vc2hhcmVkL2FjY2Vzcy1sZXZlbCc7XG5pbXBvcnQgeyBQb2xpY3lTdGF0ZW1lbnQsIE9wZXJhdG9yIH0gZnJvbSAnLi4vLi4vc2hhcmVkJztcbmltcG9ydCB7IGF3c19pYW0gYXMgaWFtIH0gZnJvbSBcImF3cy1jZGstbGliXCI7XG5cbi8qKlxuICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFt0bmJdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXJ2aWNlLWF1dGhvcml6YXRpb24vbGF0ZXN0L3JlZmVyZW5jZS9saXN0X2F3c3RlbGNvbmV0d29ya2J1aWxkZXIuaHRtbCkuXG4gKlxuICogQHBhcmFtIHNpZCBbU0lEXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX3NpZC5odG1sKSBvZiB0aGUgc3RhdGVtZW50XG4gKi9cbmV4cG9ydCBjbGFzcyBUbmIgZXh0ZW5kcyBQb2xpY3lTdGF0ZW1lbnQge1xuICBwdWJsaWMgc2VydmljZVByZWZpeCA9ICd0bmInO1xuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjYW5jZWwgYSBuZXR3b3JrIG9wZXJhdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS90bmIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ2FuY2VsU29sTmV0d29ya09wZXJhdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DYW5jZWxTb2xOZXR3b3JrT3BlcmF0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDYW5jZWxTb2xOZXR3b3JrT3BlcmF0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgZnVuY3Rpb24gcGFja2FnZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS90bmIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlU29sRnVuY3Rpb25QYWNrYWdlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZVNvbEZ1bmN0aW9uUGFja2FnZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlU29sRnVuY3Rpb25QYWNrYWdlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgbmV0d29yayBpbnN0YW5jZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS90bmIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlU29sTmV0d29ya0luc3RhbmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZVNvbE5ldHdvcmtJbnN0YW5jZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlU29sTmV0d29ya0luc3RhbmNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgbmV0d29yayBwYWNrYWdlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3RuYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVTb2xOZXR3b3JrUGFja2FnZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVTb2xOZXR3b3JrUGFja2FnZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlU29sTmV0d29ya1BhY2thZ2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYSBmdW5jdGlvbiBwYWNrYWdlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3RuYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVTb2xGdW5jdGlvblBhY2thZ2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlU29sRnVuY3Rpb25QYWNrYWdlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVTb2xGdW5jdGlvblBhY2thZ2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYSBuZXR3b3JrIGluc3RhbmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3RuYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVTb2xOZXR3b3JrSW5zdGFuY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlU29sTmV0d29ya0luc3RhbmNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVTb2xOZXR3b3JrSW5zdGFuY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYSBuZXR3b3JrIHBhY2thZ2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vdG5iL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZVNvbE5ldHdvcmtQYWNrYWdlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZVNvbE5ldHdvcmtQYWNrYWdlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVTb2xOZXR3b3JrUGFja2FnZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBhIGZ1bmN0aW9uIGluc3RhbmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3RuYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9HZXRTb2xGdW5jdGlvbkluc3RhbmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFNvbEZ1bmN0aW9uSW5zdGFuY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldFNvbEZ1bmN0aW9uSW5zdGFuY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgYSBmdW5jdGlvbiBwYWNrYWdlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3RuYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9HZXRTb2xGdW5jdGlvblBhY2thZ2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0U29sRnVuY3Rpb25QYWNrYWdlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRTb2xGdW5jdGlvblBhY2thZ2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgYSBmdW5jdGlvbiBwYWNrYWdlIGNvbnRlbnRzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3RuYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9HZXRTb2xGdW5jdGlvblBhY2thZ2VDb250ZW50Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFNvbEZ1bmN0aW9uUGFja2FnZUNvbnRlbnQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldFNvbEZ1bmN0aW9uUGFja2FnZUNvbnRlbnQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgYSBmdW5jdGlvbiBwYWNrYWdlIGRlc2NyaXB0b3JcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vdG5iL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldFNvbEZ1bmN0aW9uUGFja2FnZURlc2NyaXB0b3IuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0U29sRnVuY3Rpb25QYWNrYWdlRGVzY3JpcHRvcigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0U29sRnVuY3Rpb25QYWNrYWdlRGVzY3JpcHRvcicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBhIG5ldHdvcmsgaW5zdGFuY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vdG5iL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldFNvbE5ldHdvcmtJbnN0YW5jZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRTb2xOZXR3b3JrSW5zdGFuY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldFNvbE5ldHdvcmtJbnN0YW5jZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBhIG5ldHdvcmsgb3BlcmF0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3RuYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9HZXRTb2xOZXR3b3JrT3BlcmF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFNvbE5ldHdvcmtPcGVyYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldFNvbE5ldHdvcmtPcGVyYXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgYSBuZXR3b3JrIHBhY2thZ2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vdG5iL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldFNvbE5ldHdvcmtQYWNrYWdlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFNvbE5ldHdvcmtQYWNrYWdlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRTb2xOZXR3b3JrUGFja2FnZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBhIG5ldHdvcmsgcGFja2FnZSBjb250ZW50c1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS90bmIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0U29sTmV0d29ya1BhY2thZ2VDb250ZW50Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFNvbE5ldHdvcmtQYWNrYWdlQ29udGVudCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0U29sTmV0d29ya1BhY2thZ2VDb250ZW50Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IGEgbmV0d29yayBwYWNrYWdlIGRlc2NyaXB0b3JcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vdG5iL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldFNvbE5ldHdvcmtQYWNrYWdlRGVzY3JpcHRvci5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRTb2xOZXR3b3JrUGFja2FnZURlc2NyaXB0b3IoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldFNvbE5ldHdvcmtQYWNrYWdlRGVzY3JpcHRvcicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGluc3RhbnRpYXRlIGEgbmV0d29yayBpbnN0YW5jZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS90bmIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfSW5zdGFudGlhdGVTb2xOZXR3b3JrSW5zdGFuY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvSW5zdGFudGlhdGVTb2xOZXR3b3JrSW5zdGFuY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0luc3RhbnRpYXRlU29sTmV0d29ya0luc3RhbmNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBmdW5jdGlvbiBpbnN0YW5jZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vdG5iL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RTb2xGdW5jdGlvbkluc3RhbmNlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0U29sRnVuY3Rpb25JbnN0YW5jZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RTb2xGdW5jdGlvbkluc3RhbmNlcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgZnVuY3Rpb24gcGFja2FnZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vdG5iL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RTb2xGdW5jdGlvblBhY2thZ2VzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RTb2xGdW5jdGlvblBhY2thZ2VzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0U29sRnVuY3Rpb25QYWNrYWdlcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgbmV0d29yayBpbnN0YW5jZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vdG5iL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RTb2xOZXR3b3JrSW5zdGFuY2VzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RTb2xOZXR3b3JrSW5zdGFuY2VzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0U29sTmV0d29ya0luc3RhbmNlcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgbmV0d29yayBvcGVyYXRpb25zXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3RuYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0U29sTmV0d29ya09wZXJhdGlvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFNvbE5ldHdvcmtPcGVyYXRpb25zKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0U29sTmV0d29ya09wZXJhdGlvbnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IG5ldHdvcmsgcGFja2FnZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vdG5iL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RTb2xOZXR3b3JrUGFja2FnZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFNvbE5ldHdvcmtQYWNrYWdlcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdFNvbE5ldHdvcmtQYWNrYWdlcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHVybiBhIGxpc3Qgb2YgdGFncyBmb3IgYSByZXNvdXJjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3RuYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0VGFnc0ZvclJlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RUYWdzRm9yUmVzb3VyY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RUYWdzRm9yUmVzb3VyY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGxvYWQgZnVuY3Rpb24gcGFja2FnZSBjb250ZW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3RuYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9QdXRTb2xGdW5jdGlvblBhY2thZ2VDb250ZW50Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1B1dFNvbEZ1bmN0aW9uUGFja2FnZUNvbnRlbnQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1B1dFNvbEZ1bmN0aW9uUGFja2FnZUNvbnRlbnQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGxvYWQgbmV0d29yayBwYWNrYWdlIGNvbnRlbnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vdG5iL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1B1dFNvbE5ldHdvcmtQYWNrYWdlQ29udGVudC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9QdXRTb2xOZXR3b3JrUGFja2FnZUNvbnRlbnQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1B1dFNvbE5ldHdvcmtQYWNrYWdlQ29udGVudCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGFkZCB0YWdzIHRvIHRoZSBzcGVjaWZpZWQgcmVzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS90bmIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVGFnUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVGFnUmVzb3VyY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1RhZ1Jlc291cmNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdGVybWluYXRlIGEgbmV0d29yayBpbnN0YW5jZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS90bmIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVGVybWluYXRlU29sTmV0d29ya0luc3RhbmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1Rlcm1pbmF0ZVNvbE5ldHdvcmtJbnN0YW5jZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVGVybWluYXRlU29sTmV0d29ya0luc3RhbmNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmVtb3ZlIHRhZ3MgZnJvbSB0aGUgc3BlY2lmaWVkIHJlc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogVGFnZ2luZ1xuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vdG5iL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VudGFnUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVW50YWdSZXNvdXJjZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVW50YWdSZXNvdXJjZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBhIGZ1bmN0aW9uIHBhY2thZ2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vdG5iL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZVNvbEZ1bmN0aW9uUGFja2FnZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVTb2xGdW5jdGlvblBhY2thZ2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZVNvbEZ1bmN0aW9uUGFja2FnZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBhIG5ldHdvcmsgaW5zdGFuY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vdG5iL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZVNvbE5ldHdvcmtJbnN0YW5jZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVTb2xOZXR3b3JrSW5zdGFuY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZVNvbE5ldHdvcmtJbnN0YW5jZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBhIG5ldHdvcmsgcGFja2FnZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS90bmIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlU29sTmV0d29ya1BhY2thZ2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlU29sTmV0d29ya1BhY2thZ2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZVNvbE5ldHdvcmtQYWNrYWdlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdmFsaWRhdGUgZnVuY3Rpb24gcGFja2FnZSBjb250ZW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3RuYi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9WYWxpZGF0ZVNvbEZ1bmN0aW9uUGFja2FnZUNvbnRlbnQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVmFsaWRhdGVTb2xGdW5jdGlvblBhY2thZ2VDb250ZW50KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdWYWxpZGF0ZVNvbEZ1bmN0aW9uUGFja2FnZUNvbnRlbnQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB2YWxpZGF0ZSBuZXR3b3JrIHBhY2thZ2UgY29udGVudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS90bmIvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVmFsaWRhdGVTb2xOZXR3b3JrUGFja2FnZUNvbnRlbnQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVmFsaWRhdGVTb2xOZXR3b3JrUGFja2FnZUNvbnRlbnQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1ZhbGlkYXRlU29sTmV0d29ya1BhY2thZ2VDb250ZW50Jyk7XG4gIH1cblxuICBwcm90ZWN0ZWQgYWNjZXNzTGV2ZWxMaXN0OiBBY2Nlc3NMZXZlbExpc3QgPSB7XG4gICAgV3JpdGU6IFtcbiAgICAgICdDYW5jZWxTb2xOZXR3b3JrT3BlcmF0aW9uJyxcbiAgICAgICdDcmVhdGVTb2xGdW5jdGlvblBhY2thZ2UnLFxuICAgICAgJ0NyZWF0ZVNvbE5ldHdvcmtJbnN0YW5jZScsXG4gICAgICAnQ3JlYXRlU29sTmV0d29ya1BhY2thZ2UnLFxuICAgICAgJ0RlbGV0ZVNvbEZ1bmN0aW9uUGFja2FnZScsXG4gICAgICAnRGVsZXRlU29sTmV0d29ya0luc3RhbmNlJyxcbiAgICAgICdEZWxldGVTb2xOZXR3b3JrUGFja2FnZScsXG4gICAgICAnSW5zdGFudGlhdGVTb2xOZXR3b3JrSW5zdGFuY2UnLFxuICAgICAgJ1B1dFNvbEZ1bmN0aW9uUGFja2FnZUNvbnRlbnQnLFxuICAgICAgJ1B1dFNvbE5ldHdvcmtQYWNrYWdlQ29udGVudCcsXG4gICAgICAnVGVybWluYXRlU29sTmV0d29ya0luc3RhbmNlJyxcbiAgICAgICdVcGRhdGVTb2xGdW5jdGlvblBhY2thZ2UnLFxuICAgICAgJ1VwZGF0ZVNvbE5ldHdvcmtJbnN0YW5jZScsXG4gICAgICAnVXBkYXRlU29sTmV0d29ya1BhY2thZ2UnLFxuICAgICAgJ1ZhbGlkYXRlU29sRnVuY3Rpb25QYWNrYWdlQ29udGVudCcsXG4gICAgICAnVmFsaWRhdGVTb2xOZXR3b3JrUGFja2FnZUNvbnRlbnQnXG4gICAgXSxcbiAgICBSZWFkOiBbXG4gICAgICAnR2V0U29sRnVuY3Rpb25JbnN0YW5jZScsXG4gICAgICAnR2V0U29sRnVuY3Rpb25QYWNrYWdlJyxcbiAgICAgICdHZXRTb2xGdW5jdGlvblBhY2thZ2VDb250ZW50JyxcbiAgICAgICdHZXRTb2xGdW5jdGlvblBhY2thZ2VEZXNjcmlwdG9yJyxcbiAgICAgICdHZXRTb2xOZXR3b3JrSW5zdGFuY2UnLFxuICAgICAgJ0dldFNvbE5ldHdvcmtPcGVyYXRpb24nLFxuICAgICAgJ0dldFNvbE5ldHdvcmtQYWNrYWdlJyxcbiAgICAgICdHZXRTb2xOZXR3b3JrUGFja2FnZUNvbnRlbnQnLFxuICAgICAgJ0dldFNvbE5ldHdvcmtQYWNrYWdlRGVzY3JpcHRvcidcbiAgICBdLFxuICAgIExpc3Q6IFtcbiAgICAgICdMaXN0U29sRnVuY3Rpb25JbnN0YW5jZXMnLFxuICAgICAgJ0xpc3RTb2xGdW5jdGlvblBhY2thZ2VzJyxcbiAgICAgICdMaXN0U29sTmV0d29ya0luc3RhbmNlcycsXG4gICAgICAnTGlzdFNvbE5ldHdvcmtPcGVyYXRpb25zJyxcbiAgICAgICdMaXN0U29sTmV0d29ya1BhY2thZ2VzJyxcbiAgICAgICdMaXN0VGFnc0ZvclJlc291cmNlJ1xuICAgIF0sXG4gICAgVGFnZ2luZzogW1xuICAgICAgJ1RhZ1Jlc291cmNlJyxcbiAgICAgICdVbnRhZ1Jlc291cmNlJ1xuICAgIF1cbiAgfTtcblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgZnVuY3Rpb24tcGFja2FnZSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS90bmIvbGF0ZXN0L3VnL2Z1bmN0aW9uLXBhY2thZ2VzLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGZ1bmN0aW9uUGFja2FnZUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIGZ1bmN0aW9uUGFja2FnZUlkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25GdW5jdGlvblBhY2thZ2UoZnVuY3Rpb25QYWNrYWdlSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTp0bmI6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OmZ1bmN0aW9uLXBhY2thZ2UvJHsgZnVuY3Rpb25QYWNrYWdlSWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIG5ldHdvcmstcGFja2FnZSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS90bmIvbGF0ZXN0L3VnL25ldHdvcmstcGFja2FnZXMuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gbmV0d29ya1BhY2thZ2VJZCAtIElkZW50aWZpZXIgZm9yIHRoZSBuZXR3b3JrUGFja2FnZUlkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25OZXR3b3JrUGFja2FnZShuZXR3b3JrUGFja2FnZUlkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06dG5iOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpuZXR3b3JrLXBhY2thZ2UvJHsgbmV0d29ya1BhY2thZ2VJZCB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgbmV0d29yay1pbnN0YW5jZSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS90bmIvbGF0ZXN0L3VnL25ldHdvcmstaW5zdGFuY2VzLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIG5ldHdvcmtJbnN0YW5jZUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIG5ldHdvcmtJbnN0YW5jZUlkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25OZXR3b3JrSW5zdGFuY2UobmV0d29ya0luc3RhbmNlSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTp0bmI6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9Om5ldHdvcmstaW5zdGFuY2UvJHsgbmV0d29ya0luc3RhbmNlSWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGZ1bmN0aW9uLWluc3RhbmNlIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3RuYi9sYXRlc3QvdWcvZnVuY3Rpb24tcGFja2FnZXMuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gZnVuY3Rpb25JbnN0YW5jZUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIGZ1bmN0aW9uSW5zdGFuY2VJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uRnVuY3Rpb25JbnN0YW5jZShmdW5jdGlvbkluc3RhbmNlSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTp0bmI6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OmZ1bmN0aW9uLWluc3RhbmNlLyR7IGZ1bmN0aW9uSW5zdGFuY2VJZCB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgbmV0d29yay1vcGVyYXRpb24gdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vdG5iL2xhdGVzdC91Zy9uZXR3b3JrLW9wZXJhdGlvbnMuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gbmV0d29ya09wZXJhdGlvbklkIC0gSWRlbnRpZmllciBmb3IgdGhlIG5ldHdvcmtPcGVyYXRpb25JZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uTmV0d29ya09wZXJhdGlvbihuZXR3b3JrT3BlcmF0aW9uSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTp0bmI6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9Om5ldHdvcmstb3BlcmF0aW9uLyR7IG5ldHdvcmtPcGVyYXRpb25JZCB9YCk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgY2hlY2tpbmcgdGhlIHByZXNlbmNlIG9mIHRhZyBrZXktdmFsdWUgcGFpcnMgaW4gdGhlIHJlcXVlc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19jb25kaXRpb24ta2V5cy5odG1sI2NvbmRpdGlvbi1rZXlzLXJlcXVlc3R0YWdcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0NyZWF0ZVNvbEZ1bmN0aW9uUGFja2FnZSgpXG4gICAqIC0gLnRvQ3JlYXRlU29sTmV0d29ya0luc3RhbmNlKClcbiAgICogLSAudG9DcmVhdGVTb2xOZXR3b3JrUGFja2FnZSgpXG4gICAqIC0gLnRvSW5zdGFudGlhdGVTb2xOZXR3b3JrSW5zdGFuY2UoKVxuICAgKiAtIC50b1RhZ1Jlc291cmNlKClcbiAgICogLSAudG9UZXJtaW5hdGVTb2xOZXR3b3JrSW5zdGFuY2UoKVxuICAgKiAtIC50b1VwZGF0ZVNvbE5ldHdvcmtJbnN0YW5jZSgpXG4gICAqXG4gICAqIEBwYXJhbSB0YWdLZXkgVGhlIHRhZyBrZXkgdG8gY2hlY2tcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NSZXF1ZXN0VGFnKHRhZ0tleTogc3RyaW5nLCB2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgYXdzOlJlcXVlc3RUYWcvJHsgdGFnS2V5IH1gLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSBjaGVja2luZyB0YWcga2V5LXZhbHVlIHBhaXJzIGF0dGFjaGVkIHRvIHRoZSByZXNvdXJjZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2NvbmRpdGlvbi1rZXlzLmh0bWwjY29uZGl0aW9uLWtleXMtcmVzb3VyY2V0YWdcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0dldFNvbEZ1bmN0aW9uSW5zdGFuY2UoKVxuICAgKiAtIC50b0dldFNvbEZ1bmN0aW9uUGFja2FnZSgpXG4gICAqIC0gLnRvR2V0U29sRnVuY3Rpb25QYWNrYWdlQ29udGVudCgpXG4gICAqIC0gLnRvR2V0U29sRnVuY3Rpb25QYWNrYWdlRGVzY3JpcHRvcigpXG4gICAqIC0gLnRvR2V0U29sTmV0d29ya0luc3RhbmNlKClcbiAgICogLSAudG9HZXRTb2xOZXR3b3JrT3BlcmF0aW9uKClcbiAgICogLSAudG9HZXRTb2xOZXR3b3JrUGFja2FnZSgpXG4gICAqIC0gLnRvR2V0U29sTmV0d29ya1BhY2thZ2VDb250ZW50KClcbiAgICogLSAudG9HZXRTb2xOZXR3b3JrUGFja2FnZURlc2NyaXB0b3IoKVxuICAgKiAtIC50b0xpc3RTb2xGdW5jdGlvbkluc3RhbmNlcygpXG4gICAqIC0gLnRvTGlzdFNvbEZ1bmN0aW9uUGFja2FnZXMoKVxuICAgKiAtIC50b0xpc3RTb2xOZXR3b3JrSW5zdGFuY2VzKClcbiAgICogLSAudG9MaXN0U29sTmV0d29ya09wZXJhdGlvbnMoKVxuICAgKiAtIC50b0xpc3RTb2xOZXR3b3JrUGFja2FnZXMoKVxuICAgKlxuICAgKiBBcHBsaWVzIHRvIHJlc291cmNlIHR5cGVzOlxuICAgKiAtIGZ1bmN0aW9uLXBhY2thZ2VcbiAgICogLSBuZXR3b3JrLXBhY2thZ2VcbiAgICogLSBuZXR3b3JrLWluc3RhbmNlXG4gICAqIC0gZnVuY3Rpb24taW5zdGFuY2VcbiAgICogLSBuZXR3b3JrLW9wZXJhdGlvblxuICAgKlxuICAgKiBAcGFyYW0gdGFnS2V5IFRoZSB0YWcga2V5IHRvIGNoZWNrXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQXdzUmVzb3VyY2VUYWcodGFnS2V5OiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBhd3M6UmVzb3VyY2VUYWcvJHsgdGFnS2V5IH1gLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSBwcmVzZW5jZSBvZiB0YWcga2V5cyBpbiB0aGUgcmVxdWVzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2NvbmRpdGlvbi1rZXlzLmh0bWwjY29uZGl0aW9uLWtleXMtdGFna2V5c1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvQ3JlYXRlU29sRnVuY3Rpb25QYWNrYWdlKClcbiAgICogLSAudG9DcmVhdGVTb2xOZXR3b3JrSW5zdGFuY2UoKVxuICAgKiAtIC50b0NyZWF0ZVNvbE5ldHdvcmtQYWNrYWdlKClcbiAgICogLSAudG9JbnN0YW50aWF0ZVNvbE5ldHdvcmtJbnN0YW5jZSgpXG4gICAqIC0gLnRvVGFnUmVzb3VyY2UoKVxuICAgKiAtIC50b1Rlcm1pbmF0ZVNvbE5ldHdvcmtJbnN0YW5jZSgpXG4gICAqIC0gLnRvVW50YWdSZXNvdXJjZSgpXG4gICAqIC0gLnRvVXBkYXRlU29sTmV0d29ya0luc3RhbmNlKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NUYWdLZXlzKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBhd3M6VGFnS2V5c2AsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbdG5iXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZS1hdXRob3JpemF0aW9uL2xhdGVzdC9yZWZlcmVuY2UvbGlzdF9hd3N0ZWxjb25ldHdvcmtidWlsZGVyLmh0bWwpLlxuICAgKlxuICAgKi9cbiAgY29uc3RydWN0b3IocHJvcHM/OiBpYW0uUG9saWN5U3RhdGVtZW50UHJvcHMpIHtcbiAgICBzdXBlcihwcm9wcyk7XG4gIH1cbn1cbiJdfQ==