"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Transcribe = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [transcribe](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazontranscribe.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Transcribe extends shared_1.PolicyStatement {
    /**
     * Grants permission to create an analytics category. Amazon Transcribe applies the conditions specified by your analytics categories to your call analytics jobs
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_CreateCallAnalyticsCategory.html
     */
    toCreateCallAnalyticsCategory() {
        return this.to('CreateCallAnalyticsCategory');
    }
    /**
     * Grants permission to create a new custom language model
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - s3:GetObject
     * - s3:ListBucket
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_CreateLanguageModel.html
     */
    toCreateLanguageModel() {
        return this.to('CreateLanguageModel');
    }
    /**
     * Grants permission to create a new custom vocabulary that you can use to change the way Amazon Transcribe Medical handles transcription of an audio file
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_CreateMedicalVocabulary.html
     */
    toCreateMedicalVocabulary() {
        return this.to('CreateMedicalVocabulary');
    }
    /**
     * Grants permission to create a new custom vocabulary that you can use to change the way Amazon Transcribe handles transcription of an audio file
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_CreateVocabulary.html
     */
    toCreateVocabulary() {
        return this.to('CreateVocabulary');
    }
    /**
     * Grants permission to create a new vocabulary filter that you can use to filter out words from the transcription of an audio file generated by Amazon Transcribe
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_CreateVocabularyFilter.html
     */
    toCreateVocabularyFilter() {
        return this.to('CreateVocabularyFilter');
    }
    /**
     * Grants permission to delete a call analytics category using its name from Amazon Transcribe
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_DeleteCallAnalyticsCategory.html
     */
    toDeleteCallAnalyticsCategory() {
        return this.to('DeleteCallAnalyticsCategory');
    }
    /**
     * Grants permission to delete a previously submitted call analytics job along with any other generated results such as the transcription, models, and so on
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_DeleteCallAnalyticsJob.html
     */
    toDeleteCallAnalyticsJob() {
        return this.to('DeleteCallAnalyticsJob');
    }
    /**
     * Grants permission to delete a previously created custom language model
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_DeleteLanguageModel.html
     */
    toDeleteLanguageModel() {
        return this.to('DeleteLanguageModel');
    }
    /**
     * Grants permission to delete a previously submitted Medical Scribe job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_DeleteMedicalScribeJob.html
     */
    toDeleteMedicalScribeJob() {
        return this.to('DeleteMedicalScribeJob');
    }
    /**
     * Grants permission to delete a previously submitted medical transcription job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_DeleteMedicalTranscriptionJob.html
     */
    toDeleteMedicalTranscriptionJob() {
        return this.to('DeleteMedicalTranscriptionJob');
    }
    /**
     * Grants permission to delete a medical vocabulary from Amazon Transcribe
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_DeleteMedicalVocabulary.html
     */
    toDeleteMedicalVocabulary() {
        return this.to('DeleteMedicalVocabulary');
    }
    /**
     * Grants permission to delete a previously submitted transcription job along with any other generated results such as the transcription, models, and so on
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_DeleteTranscriptionJob.html
     */
    toDeleteTranscriptionJob() {
        return this.to('DeleteTranscriptionJob');
    }
    /**
     * Grants permission to delete a vocabulary from Amazon Transcribe
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_DeleteVocabulary.html
     */
    toDeleteVocabulary() {
        return this.to('DeleteVocabulary');
    }
    /**
     * Grants permission to delete a vocabulary filter from Amazon Transcribe
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_DeleteVocabularyFilter.html
     */
    toDeleteVocabularyFilter() {
        return this.to('DeleteVocabularyFilter');
    }
    /**
     * Grants permission to return information about a custom language model
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_DescribeLanguageModel.html
     */
    toDescribeLanguageModel() {
        return this.to('DescribeLanguageModel');
    }
    /**
     * Grants permission to retrieve information about a call analytics category
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_GetCallAnalyticsCategory.html
     */
    toGetCallAnalyticsCategory() {
        return this.to('GetCallAnalyticsCategory');
    }
    /**
     * Grants permission to return information about a call analytics job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_GetCallAnalyticsJob.html
     */
    toGetCallAnalyticsJob() {
        return this.to('GetCallAnalyticsJob');
    }
    /**
     * Grants permission to return information about a Medical Scribe job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_GetMedicalScribeJob.html
     */
    toGetMedicalScribeJob() {
        return this.to('GetMedicalScribeJob');
    }
    /**
     * Grants permission to get information about the specified AWS HealthScribe streaming session
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_streaming_GetMedicalScribeStream.html
     */
    toGetMedicalScribeStream() {
        return this.to('GetMedicalScribeStream');
    }
    /**
     * Grants permission to return information about a medical transcription job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_GetMedicalTranscriptionJob.html
     */
    toGetMedicalTranscriptionJob() {
        return this.to('GetMedicalTranscriptionJob');
    }
    /**
     * Grants permission to get information about a medical vocabulary
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_GetMedicalVocabulary.html
     */
    toGetMedicalVocabulary() {
        return this.to('GetMedicalVocabulary');
    }
    /**
     * Grants permission to return information about a transcription job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_GetTranscriptionJob.html
     */
    toGetTranscriptionJob() {
        return this.to('GetTranscriptionJob');
    }
    /**
     * Grants permission to to get information about a vocabulary
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_GetVocabulary.html
     */
    toGetVocabulary() {
        return this.to('GetVocabulary');
    }
    /**
     * Grants permission to get information about a vocabulary filter
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_GetVocabularyFilter.html
     */
    toGetVocabularyFilter() {
        return this.to('GetVocabularyFilter');
    }
    /**
     * Grants permission to list call analytics categories that has been created
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_ListCallAnalyticsCategories.html
     */
    toListCallAnalyticsCategories() {
        return this.to('ListCallAnalyticsCategories');
    }
    /**
     * Grants permission to list call analytics jobs with the specified status
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_ListCallAnalyticsJobs.html
     */
    toListCallAnalyticsJobs() {
        return this.to('ListCallAnalyticsJobs');
    }
    /**
     * Grants permission to list custom language models
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_ListLanguageModels.html
     */
    toListLanguageModels() {
        return this.to('ListLanguageModels');
    }
    /**
     * Grants permission to list Medical Scribe jobs with the specified status
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_ListMedicalScribeJobs.html
     */
    toListMedicalScribeJobs() {
        return this.to('ListMedicalScribeJobs');
    }
    /**
     * Grants permission to list medical transcription jobs with the specified status
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_ListMedicalTranscriptionJobs.html
     */
    toListMedicalTranscriptionJobs() {
        return this.to('ListMedicalTranscriptionJobs');
    }
    /**
     * Grants permission to return a list of medical vocabularies that match the specified criteria. If no criteria are specified, returns the entire list of vocabularies
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_ListMedicalVocabularies.html
     */
    toListMedicalVocabularies() {
        return this.to('ListMedicalVocabularies');
    }
    /**
     * Grants permission to list tags for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list transcription jobs with the specified status
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_ListTranscriptionJobs.html
     */
    toListTranscriptionJobs() {
        return this.to('ListTranscriptionJobs');
    }
    /**
     * Grants permission to return a list of vocabularies that match the specified criteria. If no criteria are specified, returns the entire list of vocabularies
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_ListVocabularies.html
     */
    toListVocabularies() {
        return this.to('ListVocabularies');
    }
    /**
     * Grants permission to return a list of vocabulary filters that match the specified criteria. If no criteria are specified, returns the at most 5 vocabulary filters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_ListVocabularyFilters.html
     */
    toListVocabularyFilters() {
        return this.to('ListVocabularyFilters');
    }
    /**
     * Grants permission to start an asynchronous analytics job that not only transcribes the audio recording of a caller and agent, but also returns additional insights
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifOutputEncryptionKMSKeyId()
     * - .ifOutputLocation()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_StartCallAnalyticsJob.html
     */
    toStartCallAnalyticsJob() {
        return this.to('StartCallAnalyticsJob');
    }
    /**
     * Grants permission to start a protocol where audio is streamed to Transcribe Call Analytics and the transcription results are streamed to your application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_streaming_StartCallAnalyticsStreamTranscription.html
     */
    toStartCallAnalyticsStreamTranscription() {
        return this.to('StartCallAnalyticsStreamTranscription');
    }
    /**
     * Grants permission to start a WebSocket where audio is streamed to Transcribe Call Analytics and the transcription results are streamed to your application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_streaming_StartCallAnalyticsStreamTranscriptionWebSocket.html
     */
    toStartCallAnalyticsStreamTranscriptionWebSocket() {
        return this.to('StartCallAnalyticsStreamTranscriptionWebSocket');
    }
    /**
     * Grants permission to start an asynchronous job to transcribe patient-clinician conversations and generates clinical notes
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifOutputBucketName()
     * - .ifOutputEncryptionKMSKeyId()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_StartMedicalScribeJob.html
     */
    toStartMedicalScribeJob() {
        return this.to('StartMedicalScribeJob');
    }
    /**
     * Grants permission to start a bidirectional HTTP2 stream where audio is streamed to AWS HealthScribe and the transcription results are streamed to your application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_streaming_StartMedicalScribeStream.html
     */
    toStartMedicalScribeStream() {
        return this.to('StartMedicalScribeStream');
    }
    /**
     * Grants permission to start a protocol where audio is streamed to Transcribe Medical and the transcription results are streamed to your application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_streaming_StartMedicalStreamTranscription.html
     */
    toStartMedicalStreamTranscription() {
        return this.to('StartMedicalStreamTranscription');
    }
    /**
     * Grants permission to start a WebSocket where audio is streamed to Transcribe Medical and the transcription results are streamed to your application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_streaming_StartMedicalStreamTranscriptionWebSocket.html
     */
    toStartMedicalStreamTranscriptionWebSocket() {
        return this.to('StartMedicalStreamTranscriptionWebSocket');
    }
    /**
     * Grants permission to start an asynchronous job to transcribe medical speech to text
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifOutputBucketName()
     * - .ifOutputEncryptionKMSKeyId()
     * - .ifOutputKey()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_StartMedicalTranscriptionJob.html
     */
    toStartMedicalTranscriptionJob() {
        return this.to('StartMedicalTranscriptionJob');
    }
    /**
     * Grants permission to start a bidirectional HTTP2 stream to transcribe speech to text in real time
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_streaming_StartStreamTranscription.html
     */
    toStartStreamTranscription() {
        return this.to('StartStreamTranscription');
    }
    /**
     * Grants permission to start a websocket stream to transcribe speech to text in real time
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_streaming_StartStreamTranscriptionWebSocket.html
     */
    toStartStreamTranscriptionWebSocket() {
        return this.to('StartStreamTranscriptionWebSocket');
    }
    /**
     * Grants permission to start an asynchronous job to transcribe speech to text
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifOutputBucketName()
     * - .ifOutputEncryptionKMSKeyId()
     * - .ifOutputKey()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_StartTranscriptionJob.html
     */
    toStartTranscriptionJob() {
        return this.to('StartTranscriptionJob');
    }
    /**
     * Grants permission to tag a resource with given key value pairs
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifOutputBucketName()
     * - .ifOutputEncryptionKMSKeyId()
     * - .ifOutputKey()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag a resource with given key
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update the call analytics category with new values. The UpdateCallAnalyticsCategory operation overwrites all of the existing information with the values that you provide in the request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_UpdateCallAnalyticsCategory.html
     */
    toUpdateCallAnalyticsCategory() {
        return this.to('UpdateCallAnalyticsCategory');
    }
    /**
     * Grants permission to update an existing medical vocabulary with new values. The UpdateMedicalVocabulary operation overwrites all of the existing information with the values that you provide in the request
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_UpdateMedicalVocabulary.html
     */
    toUpdateMedicalVocabulary() {
        return this.to('UpdateMedicalVocabulary');
    }
    /**
     * Grants permission to update an existing vocabulary with new values. The UpdateVocabulary operation overwrites all of the existing information with the values that you provide in the request
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_UpdateVocabulary.html
     */
    toUpdateVocabulary() {
        return this.to('UpdateVocabulary');
    }
    /**
     * Grants permission to update an existing vocabulary filter with new values. The UpdateVocabularyFilter operation overwrites all of the existing information with the values that you provide in the request
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_UpdateVocabularyFilter.html
     */
    toUpdateVocabularyFilter() {
        return this.to('UpdateVocabularyFilter');
    }
    /**
     * Adds a resource of type transcriptionjob to the statement
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_TranscriptionJob.html
     *
     * @param jobName - Identifier for the jobName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTranscriptionjob(jobName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:transcribe:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:transcription-job/${jobName}`);
    }
    /**
     * Adds a resource of type vocabulary to the statement
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_CreateVocabulary.html
     *
     * @param vocabularyName - Identifier for the vocabularyName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onVocabulary(vocabularyName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:transcribe:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:vocabulary/${vocabularyName}`);
    }
    /**
     * Adds a resource of type vocabularyfilter to the statement
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_CreateVocabularyFilter.html
     *
     * @param vocabularyFilterName - Identifier for the vocabularyFilterName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onVocabularyfilter(vocabularyFilterName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:transcribe:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:vocabulary-filter/${vocabularyFilterName}`);
    }
    /**
     * Adds a resource of type languagemodel to the statement
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_LanguageModel.html
     *
     * @param modelName - Identifier for the modelName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onLanguagemodel(modelName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:transcribe:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:language-model/${modelName}`);
    }
    /**
     * Adds a resource of type medicaltranscriptionjob to the statement
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_MedicalTranscriptionJob.html
     *
     * @param jobName - Identifier for the jobName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMedicaltranscriptionjob(jobName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:transcribe:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:medical-transcription-job/${jobName}`);
    }
    /**
     * Adds a resource of type medicalvocabulary to the statement
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_CreateMedicalVocabulary.html
     *
     * @param vocabularyName - Identifier for the vocabularyName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMedicalvocabulary(vocabularyName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:transcribe:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:medical-vocabulary/${vocabularyName}`);
    }
    /**
     * Adds a resource of type callanalyticsjob to the statement
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_CallAnalyticsJob.html
     *
     * @param jobName - Identifier for the jobName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCallanalyticsjob(jobName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:transcribe:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:analytics/${jobName}`);
    }
    /**
     * Adds a resource of type callanalyticscategory to the statement
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_CreateCallAnalyticsCategory.html
     *
     * @param categoryName - Identifier for the categoryName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCallanalyticscategory(categoryName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:transcribe:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:analytics-category/${categoryName}`);
    }
    /**
     * Adds a resource of type medicalscribejob to the statement
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_MedicalScribeJob.html
     *
     * @param jobName - Identifier for the jobName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMedicalscribejob(jobName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:transcribe:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:medical-scribe-job/${jobName}`);
    }
    /**
     * Filters access by requiring tag values present in a resource creation request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-globally-available
     *
     * Applies to actions:
     * - .toCreateCallAnalyticsCategory()
     * - .toCreateLanguageModel()
     * - .toCreateMedicalVocabulary()
     * - .toCreateVocabulary()
     * - .toCreateVocabularyFilter()
     * - .toStartCallAnalyticsJob()
     * - .toStartMedicalScribeJob()
     * - .toStartMedicalTranscriptionJob()
     * - .toStartTranscriptionJob()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by requiring tag value associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-globally-available
     *
     * Applies to actions:
     * - .toCreateCallAnalyticsCategory()
     * - .toCreateLanguageModel()
     * - .toCreateMedicalVocabulary()
     * - .toCreateVocabulary()
     * - .toCreateVocabularyFilter()
     * - .toStartMedicalScribeJob()
     * - .toStartMedicalTranscriptionJob()
     * - .toStartTranscriptionJob()
     * - .toTagResource()
     *
     * Applies to resource types:
     * - transcriptionjob
     * - vocabulary
     * - vocabularyfilter
     * - languagemodel
     * - medicaltranscriptionjob
     * - medicalvocabulary
     * - callanalyticsjob
     * - callanalyticscategory
     * - medicalscribejob
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by requiring the presence of mandatory tags in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-globally-available
     *
     * Applies to actions:
     * - .toCreateCallAnalyticsCategory()
     * - .toCreateLanguageModel()
     * - .toCreateMedicalVocabulary()
     * - .toCreateVocabulary()
     * - .toCreateVocabularyFilter()
     * - .toStartCallAnalyticsJob()
     * - .toStartMedicalScribeJob()
     * - .toStartMedicalTranscriptionJob()
     * - .toStartTranscriptionJob()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on the output bucket name included in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazontranscribe.html#amazontranscribe-policy-keys
     *
     * Applies to actions:
     * - .toStartMedicalScribeJob()
     * - .toStartMedicalTranscriptionJob()
     * - .toStartTranscriptionJob()
     * - .toTagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifOutputBucketName(value, operator) {
        return this.if(`OutputBucketName`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on the KMS key id included in the request, provided in the form of a KMS key ARN
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazontranscribe.html#amazontranscribe-policy-keys
     *
     * Applies to actions:
     * - .toStartCallAnalyticsJob()
     * - .toStartMedicalScribeJob()
     * - .toStartMedicalTranscriptionJob()
     * - .toStartTranscriptionJob()
     * - .toTagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifOutputEncryptionKMSKeyId(value, operator) {
        return this.if(`OutputEncryptionKMSKeyId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on the output key included in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazontranscribe.html#amazontranscribe-policy-keys
     *
     * Applies to actions:
     * - .toStartMedicalTranscriptionJob()
     * - .toStartTranscriptionJob()
     * - .toTagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifOutputKey(value, operator) {
        return this.if(`OutputKey`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on the output location included in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazontranscribe.html#amazontranscribe-policy-keys
     *
     * Applies to actions:
     * - .toStartCallAnalyticsJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifOutputLocation(value, operator) {
        return this.if(`OutputLocation`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [transcribe](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazontranscribe.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'transcribe';
        this.accessLevelList = {
            Write: [
                'CreateCallAnalyticsCategory',
                'CreateLanguageModel',
                'CreateMedicalVocabulary',
                'CreateVocabulary',
                'CreateVocabularyFilter',
                'DeleteCallAnalyticsCategory',
                'DeleteCallAnalyticsJob',
                'DeleteLanguageModel',
                'DeleteMedicalScribeJob',
                'DeleteMedicalTranscriptionJob',
                'DeleteMedicalVocabulary',
                'DeleteTranscriptionJob',
                'DeleteVocabulary',
                'DeleteVocabularyFilter',
                'StartCallAnalyticsJob',
                'StartCallAnalyticsStreamTranscription',
                'StartCallAnalyticsStreamTranscriptionWebSocket',
                'StartMedicalScribeJob',
                'StartMedicalScribeStream',
                'StartMedicalStreamTranscription',
                'StartMedicalStreamTranscriptionWebSocket',
                'StartMedicalTranscriptionJob',
                'StartStreamTranscription',
                'StartStreamTranscriptionWebSocket',
                'StartTranscriptionJob',
                'UpdateCallAnalyticsCategory',
                'UpdateMedicalVocabulary',
                'UpdateVocabulary',
                'UpdateVocabularyFilter'
            ],
            Read: [
                'DescribeLanguageModel',
                'GetCallAnalyticsCategory',
                'GetCallAnalyticsJob',
                'GetMedicalScribeJob',
                'GetMedicalScribeStream',
                'GetMedicalTranscriptionJob',
                'GetMedicalVocabulary',
                'GetTranscriptionJob',
                'GetVocabulary',
                'GetVocabularyFilter',
                'ListTagsForResource'
            ],
            List: [
                'ListCallAnalyticsCategories',
                'ListCallAnalyticsJobs',
                'ListLanguageModels',
                'ListMedicalScribeJobs',
                'ListMedicalTranscriptionJobs',
                'ListMedicalVocabularies',
                'ListTranscriptionJobs',
                'ListVocabularies',
                'ListVocabularyFilters'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Transcribe = Transcribe;
//# sourceMappingURL=data:application/json;base64,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