"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Transform = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [transform](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awstransform.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Transform extends shared_1.PolicyStatement {
    /**
     * Grants permission to invoke AssociateConnectorResource on AWS Transform
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transform/latest/userguide/security_iam_permissions.html/
     */
    toAssociateConnectorResource() {
        return this.to('AssociateConnectorResource');
    }
    /**
     * Grants permission to invoke CreateProfile on AWS Transform
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transform/latest/userguide/security_iam_permissions.html/
     */
    toCreateProfile() {
        return this.to('CreateProfile');
    }
    /**
     * Grants permission to invoke DeleteConnector on AWS Transform
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transform/latest/userguide/security_iam_permissions.html/
     */
    toDeleteConnector() {
        return this.to('DeleteConnector');
    }
    /**
     * Grants permission to invoke DeleteProfile on AWS Transform
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transform/latest/userguide/security_iam_permissions.html/
     */
    toDeleteProfile() {
        return this.to('DeleteProfile');
    }
    /**
     * Grants permission to invoke GetConnector on AWS Transform
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transform/latest/userguide/security_iam_permissions.html/
     */
    toGetConnector() {
        return this.to('GetConnector');
    }
    /**
     * Grants permission to invoke ListConnectors on AWS Transform
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transform/latest/userguide/security_iam_permissions.html/
     */
    toListConnectors() {
        return this.to('ListConnectors');
    }
    /**
     * Grants permission to invoke ListProfiles on AWS Transform
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transform/latest/userguide/security_iam_permissions.html/
     */
    toListProfiles() {
        return this.to('ListProfiles');
    }
    /**
     * Grants permission to invoke ListTagsForResource on AWS Transform
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transform/latest/userguide/security_iam_permissions.html/
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to invoke RejectConnector on AWS Transform
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transform/latest/userguide/security_iam_permissions.html/
     */
    toRejectConnector() {
        return this.to('RejectConnector');
    }
    /**
     * Grants permission to invoke TagResource on AWS Transform
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/transform/latest/userguide/security_iam_permissions.html/
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to invoke UntagResource on AWS Transform
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/transform/latest/userguide/security_iam_permissions.html/
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to invoke UpdateProfile on AWS Transform
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transform/latest/userguide/security_iam_permissions.html/
     */
    toUpdateProfile() {
        return this.to('UpdateProfile');
    }
    /**
     * Adds a resource of type profile to the statement
     *
     * https://docs.aws.amazon.com/transform/latest/userguide/security_iam_permissions.html
     *
     * @param identifier - Identifier for the identifier.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onProfile(identifier, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:transform:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:profile/${identifier}`);
    }
    /**
     * Adds a resource of type connector to the statement
     *
     * https://docs.aws.amazon.com/transform/latest/userguide/security_iam_permissions.html
     *
     * @param workspaceId - Identifier for the workspaceId.
     * @param connectorId - Identifier for the connectorId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onConnector(workspaceId, connectorId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:transform:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:connector/${workspaceId}/${connectorId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [transform](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awstransform.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'transform';
        this.accessLevelList = {
            Write: [
                'AssociateConnectorResource',
                'CreateProfile',
                'DeleteConnector',
                'DeleteProfile',
                'RejectConnector',
                'UpdateProfile'
            ],
            Read: [
                'GetConnector',
                'ListTagsForResource'
            ],
            List: [
                'ListConnectors',
                'ListProfiles'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Transform = Transform;
//# sourceMappingURL=data:application/json;base64,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