"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Notifications = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [notifications](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsusernotifications.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Notifications extends shared_1.PolicyStatement {
    /**
     * Grants permission to associate a new Channel with a particular NotificationConfiguration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_AssociateChannel.html
     */
    toAssociateChannel() {
        return this.to('AssociateChannel');
    }
    /**
     * Grants permission to associate an Account contact to a particular Managed Notification Configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_AssociateManagedNotificationAccountContact.html
     */
    toAssociateManagedNotificationAccountContact() {
        return this.to('AssociateManagedNotificationAccountContact');
    }
    /**
     * Grants permission to associate a Channel to a particular Managed Notification Configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_AssociateManagedNotificationAdditionalChannel.html
     */
    toAssociateManagedNotificationAdditionalChannel() {
        return this.to('AssociateManagedNotificationAdditionalChannel');
    }
    /**
     * Grants permission to associate an Organizational Unit to a particular Notification Configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_AssociateOrganizationalUnit.html
     */
    toAssociateOrganizationalUnit() {
        return this.to('AssociateOrganizationalUnit');
    }
    /**
     * Grants permission to create a new EventRule, associating it with a NotificationConfiguration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_CreateEventRule.html
     */
    toCreateEventRule() {
        return this.to('CreateEventRule');
    }
    /**
     * Grants permission to create a NotificationConfiguration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_CreateNotificationConfiguration.html
     */
    toCreateNotificationConfiguration() {
        return this.to('CreateNotificationConfiguration');
    }
    /**
     * Grants permission to delete an EventRule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_DeleteEventRule.html
     */
    toDeleteEventRule() {
        return this.to('DeleteEventRule');
    }
    /**
     * Grants permission to delete a NotificationConfiguration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_DeleteNotificationConfiguration.html
     */
    toDeleteNotificationConfiguration() {
        return this.to('DeleteNotificationConfiguration');
    }
    /**
     * Grants permission to deregister a NotificationHub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_DeregisterNotificationHub.html
     */
    toDeregisterNotificationHub() {
        return this.to('DeregisterNotificationHub');
    }
    /**
     * Grants permission to disable Service Trust for AWS User Notifications
     *
     * Access Level: Permissions management
     *
     * Dependent actions:
     * - organizations:DisableAWSServiceAccess
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_DisableNotificationsAccessForOrganization.html
     */
    toDisableNotificationsAccessForOrganization() {
        return this.to('DisableNotificationsAccessForOrganization');
    }
    /**
     * Grants permission to remove a Channel from a NotificationConfiguration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_DisassociateChannel.html
     */
    toDisassociateChannel() {
        return this.to('DisassociateChannel');
    }
    /**
     * Grants permission to remove an Account contact from a Managed Notification Configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_DisassociateManagedNotificationAccountContact.html
     */
    toDisassociateManagedNotificationAccountContact() {
        return this.to('DisassociateManagedNotificationAccountContact');
    }
    /**
     * Grants permission to remove a Channel from a Managed Notification Configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_DisassociateManagedNotificationAdditionalChannel.html
     */
    toDisassociateManagedNotificationAdditionalChannel() {
        return this.to('DisassociateManagedNotificationAdditionalChannel');
    }
    /**
     * Grants permission to disassociate an Organizational Unit to a particular Notification Configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_DisassociateOrganizationalUnit.html
     */
    toDisassociateOrganizationalUnit() {
        return this.to('DisassociateOrganizationalUnit');
    }
    /**
     * Grants permission to enable Service Trust for AWS User Notifications
     *
     * Access Level: Permissions management
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - organizations:EnableAWSServiceAccess
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_EnableNotificationsAccessForOrganization.html
     */
    toEnableNotificationsAccessForOrganization() {
        return this.to('EnableNotificationsAccessForOrganization');
    }
    /**
     * Grants permission to get an EventRule
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_GetEventRule.html
     */
    toGetEventRule() {
        return this.to('GetEventRule');
    }
    /**
     * Grants permission to read the opt-in status of an AWS User Notification Service feature
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/notifications/latest/userguide/managing-notification-features.html
     */
    toGetFeatureOptInStatus() {
        return this.to('GetFeatureOptInStatus');
    }
    /**
     * Grants permission to get a Managed Notification Child Event
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_GetManagedNotificationChildEvent.html
     */
    toGetManagedNotificationChildEvent() {
        return this.to('GetManagedNotificationChildEvent');
    }
    /**
     * Grants permission to get a Managed Notification Configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_GetManagedNotificationConfiguration.html
     */
    toGetManagedNotificationConfiguration() {
        return this.to('GetManagedNotificationConfiguration');
    }
    /**
     * Grants permission to get a Managed NotificationEvent
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_GetManagedNotificationEvent.html
     */
    toGetManagedNotificationEvent() {
        return this.to('GetManagedNotificationEvent');
    }
    /**
     * Grants permission to get a NotificationConfiguration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_GetNotificationConfiguration.html
     */
    toGetNotificationConfiguration() {
        return this.to('GetNotificationConfiguration');
    }
    /**
     * Grants permission to get a NotificationEvent
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_GetNotificationEvent.html
     */
    toGetNotificationEvent() {
        return this.to('GetNotificationEvent');
    }
    /**
     * Grants permission to read Service Trust for AWS User Notifications
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_GetNotificationsAccessForOrganization.html
     */
    toGetNotificationsAccessForOrganization() {
        return this.to('GetNotificationsAccessForOrganization');
    }
    /**
     * Grants permission to list Channels by NotificationConfiguration
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_ListChannels.html
     */
    toListChannels() {
        return this.to('ListChannels');
    }
    /**
     * Grants permission to list EventRules
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_ListEventRules.html
     */
    toListEventRules() {
        return this.to('ListEventRules');
    }
    /**
     * Grants permission to list Account contacts and Channels associated with a Managed Notification Configuration
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_ListManagedNotificationChannelAssociations.html
     */
    toListManagedNotificationChannelAssociations() {
        return this.to('ListManagedNotificationChannelAssociations');
    }
    /**
     * Grants permission to list Managed Notification Child Events
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_ListManagedNotificationChildEvents.html
     */
    toListManagedNotificationChildEvents() {
        return this.to('ListManagedNotificationChildEvents');
    }
    /**
     * Grants permission to list Managed Notification Configurations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_ListManagedNotificationConfigurations.html
     */
    toListManagedNotificationConfigurations() {
        return this.to('ListManagedNotificationConfigurations');
    }
    /**
     * Grants permission to list Managed Notification Events
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_ListManagedNotificationEvents.html
     */
    toListManagedNotificationEvents() {
        return this.to('ListManagedNotificationEvents');
    }
    /**
     * Grants permission to list Member Accounts for a Notification Configuration
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_ListMemberAccounts.html
     */
    toListMemberAccounts() {
        return this.to('ListMemberAccounts');
    }
    /**
     * Grants permission to list NotificationConfigurations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_ListNotificationConfigurations.html
     */
    toListNotificationConfigurations() {
        return this.to('ListNotificationConfigurations');
    }
    /**
     * Grants permission to list NotificationEvents
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_ListNotificationEvents.html
     */
    toListNotificationEvents() {
        return this.to('ListNotificationEvents');
    }
    /**
     * Grants permission to list NotificationHubs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_ListNotificationHubs.html
     */
    toListNotificationHubs() {
        return this.to('ListNotificationHubs');
    }
    /**
     * Grants permission to list Organizational Units for a Notification Configuration
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_ListOrganizationalUnits.html
     */
    toListOrganizationalUnits() {
        return this.to('ListOrganizationalUnits');
    }
    /**
     * Grants permission to get tags for a resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to update the opt-in status of an AWS User Notification Service feature
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/notifications/latest/userguide/managing-notification-features.html
     */
    toPutFeatureOptInStatus() {
        return this.to('PutFeatureOptInStatus');
    }
    /**
     * Grants permission to register a NotificationHub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_RegisterNotificationHub.html
     */
    toRegisterNotificationHub() {
        return this.to('RegisterNotificationHub');
    }
    /**
     * Grants permission to tag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update an EventRule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_UpdateEventRule.html
     */
    toUpdateEventRule() {
        return this.to('UpdateEventRule');
    }
    /**
     * Grants permission to update a NotificationConfiguration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/notifications/latest/APIReference/API_UpdateNotificationConfiguration.html
     */
    toUpdateNotificationConfiguration() {
        return this.to('UpdateNotificationConfiguration');
    }
    /**
     * Adds a resource of type EventRule to the statement
     *
     * https://docs.aws.amazon.com/notifications/latest/userguide/resource-level-permissions.html
     *
     * @param notificationConfigurationId - Identifier for the notificationConfigurationId.
     * @param eventRuleId - Identifier for the eventRuleId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onEventRule(notificationConfigurationId, eventRuleId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:notifications::${account ?? this.defaultAccount}:configuration/${notificationConfigurationId}/rule/${eventRuleId}`);
    }
    /**
     * Adds a resource of type NotificationConfiguration to the statement
     *
     * https://docs.aws.amazon.com/notifications/latest/userguide/resource-level-permissions.html
     *
     * @param notificationConfigurationId - Identifier for the notificationConfigurationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onNotificationConfiguration(notificationConfigurationId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:notifications::${account ?? this.defaultAccount}:configuration/${notificationConfigurationId}`);
    }
    /**
     * Adds a resource of type NotificationEvent to the statement
     *
     * https://docs.aws.amazon.com/notifications/latest/userguide/resource-level-permissions.html
     *
     * @param notificationConfigurationId - Identifier for the notificationConfigurationId.
     * @param notificationEventId - Identifier for the notificationEventId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onNotificationEvent(notificationConfigurationId, notificationEventId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:notifications:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:configuration/${notificationConfigurationId}/event/${notificationEventId}`);
    }
    /**
     * Adds a resource of type ManagedNotificationChildEvent to the statement
     *
     * https://docs.aws.amazon.com/notifications/latest/userguide/resource-level-permissions.html
     *
     * @param category - Identifier for the category.
     * @param subcategory - Identifier for the subcategory.
     * @param notificationEventId - Identifier for the notificationEventId.
     * @param notificationChildEventId - Identifier for the notificationChildEventId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onManagedNotificationChildEvent(category, subcategory, notificationEventId, notificationChildEventId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:notifications::${account ?? this.defaultAccount}:managed-notification-configuration/category/${category}/sub-category/${subcategory}/event/${notificationEventId}/child-event/${notificationChildEventId}`);
    }
    /**
     * Adds a resource of type ManagedNotificationConfiguration to the statement
     *
     * https://docs.aws.amazon.com/notifications/latest/userguide/resource-level-permissions.html
     *
     * @param category - Identifier for the category.
     * @param subcategory - Identifier for the subcategory.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onManagedNotificationConfiguration(category, subcategory, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:notifications::${account ?? this.defaultAccount}:managed-notification-configuration/category/${category}/sub-category/${subcategory}`);
    }
    /**
     * Adds a resource of type ManagedNotificationEvent to the statement
     *
     * https://docs.aws.amazon.com/notifications/latest/userguide/resource-level-permissions.html
     *
     * @param category - Identifier for the category.
     * @param subcategory - Identifier for the subcategory.
     * @param notificationEventId - Identifier for the notificationEventId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onManagedNotificationEvent(category, subcategory, notificationEventId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:notifications::${account ?? this.defaultAccount}:managed-notification-configuration/category/${category}/sub-category/${subcategory}/event/${notificationEventId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateNotificationConfiguration()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - NotificationConfiguration
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateNotificationConfiguration()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [notifications](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsusernotifications.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'notifications';
        this.accessLevelList = {
            Write: [
                'AssociateChannel',
                'AssociateManagedNotificationAccountContact',
                'AssociateManagedNotificationAdditionalChannel',
                'AssociateOrganizationalUnit',
                'CreateEventRule',
                'CreateNotificationConfiguration',
                'DeleteEventRule',
                'DeleteNotificationConfiguration',
                'DeregisterNotificationHub',
                'DisassociateChannel',
                'DisassociateManagedNotificationAccountContact',
                'DisassociateManagedNotificationAdditionalChannel',
                'DisassociateOrganizationalUnit',
                'PutFeatureOptInStatus',
                'RegisterNotificationHub',
                'UpdateEventRule',
                'UpdateNotificationConfiguration'
            ],
            'Permissions management': [
                'DisableNotificationsAccessForOrganization',
                'EnableNotificationsAccessForOrganization'
            ],
            Read: [
                'GetEventRule',
                'GetFeatureOptInStatus',
                'GetManagedNotificationChildEvent',
                'GetManagedNotificationConfiguration',
                'GetManagedNotificationEvent',
                'GetNotificationConfiguration',
                'GetNotificationEvent',
                'GetNotificationsAccessForOrganization'
            ],
            List: [
                'ListChannels',
                'ListEventRules',
                'ListManagedNotificationChannelAssociations',
                'ListManagedNotificationChildEvents',
                'ListManagedNotificationConfigurations',
                'ListManagedNotificationEvents',
                'ListMemberAccounts',
                'ListNotificationConfigurations',
                'ListNotificationEvents',
                'ListNotificationHubs',
                'ListOrganizationalUnits',
                'ListTagsForResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Notifications = Notifications;
//# sourceMappingURL=data:application/json;base64,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