"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Verifiedpermissions = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [verifiedpermissions](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonverifiedpermissions.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Verifiedpermissions extends shared_1.PolicyStatement {
    /**
     * Grants permission to create a reference to an external identity provider (IdP) that is compatible with OpenID Connect (OIDC) authentication protocol, such as Amazon Cognito
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_CreateIdentitySource.html
     */
    toCreateIdentitySource() {
        return this.to('CreateIdentitySource');
    }
    /**
     * Grants permission to create a Cedar policy and save it in the specified policy store
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_CreatePolicy.html
     */
    toCreatePolicy() {
        return this.to('CreatePolicy');
    }
    /**
     * Grants permission to create a Cedar policy and save it in the specified policy store
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_CreatePolicyStore.html
     */
    toCreatePolicyStore() {
        return this.to('CreatePolicyStore');
    }
    /**
     * Grants permission to create a policy template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_CreatePolicyTemplate.html
     */
    toCreatePolicyTemplate() {
        return this.to('CreatePolicyTemplate');
    }
    /**
     * Grants permission to delete an identity source that references an identity provider (IdP) such as Amazon Cognito
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_DeleteIdentitySource.html
     */
    toDeleteIdentitySource() {
        return this.to('DeleteIdentitySource');
    }
    /**
     * Grants permission to delete the specified policy from the policy store
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_DeletePolicy.html
     */
    toDeletePolicy() {
        return this.to('DeletePolicy');
    }
    /**
     * Grants permission to delete the specified policy store
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_DeletePolicyStore.html
     */
    toDeletePolicyStore() {
        return this.to('DeletePolicyStore');
    }
    /**
     * Grants permission to delete the specified policy template from the policy store
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_DeletePolicyTemplate.html
     */
    toDeletePolicyTemplate() {
        return this.to('DeletePolicyTemplate');
    }
    /**
     * Grants permission to retrieve the details about the specified identity source
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_GetIdentitySource.html
     */
    toGetIdentitySource() {
        return this.to('GetIdentitySource');
    }
    /**
     * Grants permission to retrieve information about the specified policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_GetPolicy.html
     */
    toGetPolicy() {
        return this.to('GetPolicy');
    }
    /**
     * Grants permission to retrieve details about a policy store
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - verifiedpermissions:ListTagsForResource
     *
     * https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_GetPolicyStore.html
     */
    toGetPolicyStore() {
        return this.to('GetPolicyStore');
    }
    /**
     * Grants permission to retrieve the details for the specified policy template in the specified policy store
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_GetPolicyTemplate.html
     */
    toGetPolicyTemplate() {
        return this.to('GetPolicyTemplate');
    }
    /**
     * Grants permission to retrieve the details for the specified schema in the specified policy store
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_GetSchema.html
     */
    toGetSchema() {
        return this.to('GetSchema');
    }
    /**
     * Grants permission to make an authorization decision about a service request described in the parameters
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_IsAuthorized.html
     */
    toIsAuthorized() {
        return this.to('IsAuthorized');
    }
    /**
     * Grants permission to make an authorization decision about a service request described in the parameters. The principal in this request comes from an external identity source
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_IsAuthorizedWithToken.html
     */
    toIsAuthorizedWithToken() {
        return this.to('IsAuthorizedWithToken');
    }
    /**
     * Grants permission to return a paginated list of all of the identity sources defined in the specified policy store
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_ListIdentitySources.html
     */
    toListIdentitySources() {
        return this.to('ListIdentitySources');
    }
    /**
     * Grants permission to return a paginated list of all policies stored in the specified policy store
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_ListPolicies.html
     */
    toListPolicies() {
        return this.to('ListPolicies');
    }
    /**
     * Grants permission to return a paginated list of all policy stores in the calling Amazon Web Services account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_ListPolicyStores.html
     */
    toListPolicyStores() {
        return this.to('ListPolicyStores');
    }
    /**
     * Grants permission to return a paginated list of all policy templates in the specified policy store
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_ListPolicyTemplates.html
     */
    toListPolicyTemplates() {
        return this.to('ListPolicyTemplates');
    }
    /**
     * Grants permission to view a list of resource tags for the specified policy store
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to create or update the policy schema in the specified policy store
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_PutSchema.html
     */
    toPutSchema() {
        return this.to('PutSchema');
    }
    /**
     * Grants permission to add tags to the specified policy store
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from the specified policy store
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update the specified identity source to use a new identity provider (IdP) source, or to change the mapping of identities from the IdP to a different principal entity type
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_UpdateIdentitySource.html
     */
    toUpdateIdentitySource() {
        return this.to('UpdateIdentitySource');
    }
    /**
     * Grants permission to modify the specified Cedar static policy in the specified policy store
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_UpdatePolicy.html
     */
    toUpdatePolicy() {
        return this.to('UpdatePolicy');
    }
    /**
     * Grants permission to modify the validation setting for a policy store
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_UpdatePolicyStore.html
     */
    toUpdatePolicyStore() {
        return this.to('UpdatePolicyStore');
    }
    /**
     * Grants permission to update the specified policy template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/API_UpdatePolicyTemplate.html
     */
    toUpdatePolicyTemplate() {
        return this.to('UpdatePolicyTemplate');
    }
    /**
     * Adds a resource of type policy-store to the statement
     *
     * https://docs.aws.amazon.com/verifiedpermissions/latest/apireference/policy-stores.html
     *
     * @param policyStoreId - Identifier for the policyStoreId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPolicyStore(policyStoreId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:verifiedpermissions::${account ?? this.defaultAccount}:policy-store/${policyStoreId}`);
    }
    /**
     * Filters access by a tag key and value pair that is allowed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreatePolicyStore()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a tag key and value pair of a resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toCreatePolicyStore()
     *
     * Applies to resource types:
     * - policy-store
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a list of tag keys that are allowed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreatePolicyStore()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [verifiedpermissions](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonverifiedpermissions.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'verifiedpermissions';
        this.accessLevelList = {
            Write: [
                'CreateIdentitySource',
                'CreatePolicy',
                'CreatePolicyStore',
                'CreatePolicyTemplate',
                'DeleteIdentitySource',
                'DeletePolicy',
                'DeletePolicyStore',
                'DeletePolicyTemplate',
                'PutSchema',
                'UpdateIdentitySource',
                'UpdatePolicy',
                'UpdatePolicyStore',
                'UpdatePolicyTemplate'
            ],
            Read: [
                'GetIdentitySource',
                'GetPolicy',
                'GetPolicyStore',
                'GetPolicyTemplate',
                'GetSchema',
                'IsAuthorized',
                'IsAuthorizedWithToken',
                'ListTagsForResource'
            ],
            List: [
                'ListIdentitySources',
                'ListPolicies',
                'ListPolicyStores',
                'ListPolicyTemplates'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Verifiedpermissions = Verifiedpermissions;
//# sourceMappingURL=data:application/json;base64,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