"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Workspaces = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [workspaces](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonworkspaces.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Workspaces extends shared_1.PolicyStatement {
    /**
     * Grants permission to accept invitations from other AWS accounts to share the same configuration for WorkSpaces BYOL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_AcceptAccountLinkInvitation.html
     */
    toAcceptAccountLinkInvitation() {
        return this.to('AcceptAccountLinkInvitation');
    }
    /**
     * Grants permission to associate connection aliases with directories
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_AssociateConnectionAlias.html
     */
    toAssociateConnectionAlias() {
        return this.to('AssociateConnectionAlias');
    }
    /**
     * Grants permission to associate IP access control groups with directories
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_AssociateIpGroups.html
     */
    toAssociateIpGroups() {
        return this.to('AssociateIpGroups');
    }
    /**
     * Grants permission to associate a workspace application with a WorkSpace
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_AssociateWorkspaceApplication.html
     */
    toAssociateWorkspaceApplication() {
        return this.to('AssociateWorkspaceApplication');
    }
    /**
     * Grants permission to add rules to IP access control groups
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - workspaces:UpdateRulesOfIpGroup
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_AuthorizeIpRules.html
     */
    toAuthorizeIpRules() {
        return this.to('AuthorizeIpRules');
    }
    /**
     * Grants permission to copy a WorkSpace image
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - workspaces:DescribeWorkspaceImages
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_CopyWorkspaceImage.html
     */
    toCopyWorkspaceImage() {
        return this.to('CopyWorkspaceImage');
    }
    /**
     * Grants permission to invite other AWS accounts to share the same configuration for WorkSpaces BYOL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_CreateAccountLinkInvitation.html
     */
    toCreateAccountLinkInvitation() {
        return this.to('CreateAccountLinkInvitation');
    }
    /**
     * Grants permission to create an Amazon Connect client add-in within a directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_CreateConnectClientAddIn.html
     */
    toCreateConnectClientAddIn() {
        return this.to('CreateConnectClientAddIn');
    }
    /**
     * Grants permission to create connection aliases for use with cross-Region redirection
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_CreateConnectionAlias.html
     */
    toCreateConnectionAlias() {
        return this.to('CreateConnectionAlias');
    }
    /**
     * Grants permission to create IP access control groups
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_CreateIpGroup.html
     */
    toCreateIpGroup() {
        return this.to('CreateIpGroup');
    }
    /**
     * Grants permission to create a root client certificate
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/adminguide/wsp-console-permissions-ref.html
     */
    toCreateRootClientCertificate() {
        return this.to('CreateRootClientCertificate');
    }
    /**
     * Grants permission to create one or more Standby WorkSpaces
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_CreateStandbyWorkspaces.html
     */
    toCreateStandbyWorkspaces() {
        return this.to('CreateStandbyWorkspaces');
    }
    /**
     * Grants permission to create tags for WorkSpaces resources
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_CreateTags.html
     */
    toCreateTags() {
        return this.to('CreateTags');
    }
    /**
     * Grants permission to create an updated WorkSpace image
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_CreateUpdatedWorkspaceImage.html
     */
    toCreateUpdatedWorkspaceImage() {
        return this.to('CreateUpdatedWorkspaceImage');
    }
    /**
     * Grants permission to create a WorkSpace bundle
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - workspaces:CreateTags
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_CreateWorkspaceBundle.html
     */
    toCreateWorkspaceBundle() {
        return this.to('CreateWorkspaceBundle');
    }
    /**
     * Grants permission to create a new WorkSpace image
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_CreateWorkspaceImage.html
     */
    toCreateWorkspaceImage() {
        return this.to('CreateWorkspaceImage');
    }
    /**
     * Grants permission to create one or more WorkSpaces
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_CreateWorkspaces.html
     */
    toCreateWorkspaces() {
        return this.to('CreateWorkspaces');
    }
    /**
     * Grants permission to create a WorkSpaces Pool
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_CreateWorkspacesPool.html
     */
    toCreateWorkspacesPool() {
        return this.to('CreateWorkspacesPool');
    }
    /**
     * Grants permission to delete invitations to other AWS accounts to share the same configuration for WorkSpaces BYOL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DeleteAccountLinkInvitation.html
     */
    toDeleteAccountLinkInvitation() {
        return this.to('DeleteAccountLinkInvitation');
    }
    /**
     * Grants permission to delete AWS WorkSpaces Client branding data within a directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DeleteClientBranding.html
     */
    toDeleteClientBranding() {
        return this.to('DeleteClientBranding');
    }
    /**
     * Grants permission to delete an Amazon Connect client add-in that is configured within a directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DeleteConnectClientAddIn.html
     */
    toDeleteConnectClientAddIn() {
        return this.to('DeleteConnectClientAddIn');
    }
    /**
     * Grants permission to delete connection aliases
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DeleteConnectionAlias.html
     */
    toDeleteConnectionAlias() {
        return this.to('DeleteConnectionAlias');
    }
    /**
     * Grants permission to delete IP access control groups
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DeleteIpGroup.html
     */
    toDeleteIpGroup() {
        return this.to('DeleteIpGroup');
    }
    /**
     * Grants permission to delete root client certificate
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/adminguide/wsp-console-permissions-ref.html
     */
    toDeleteRootClientCertificate() {
        return this.to('DeleteRootClientCertificate');
    }
    /**
     * Grants permission to delete tags from WorkSpaces resources
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DeleteTags.html
     */
    toDeleteTags() {
        return this.to('DeleteTags');
    }
    /**
     * Grants permission to delete WorkSpace bundles
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DeleteWorkspaceBundle.html
     */
    toDeleteWorkspaceBundle() {
        return this.to('DeleteWorkspaceBundle');
    }
    /**
     * Grants permission to delete WorkSpace images
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DeleteWorkspaceImage.html
     */
    toDeleteWorkspaceImage() {
        return this.to('DeleteWorkspaceImage');
    }
    /**
     * Grants permission to deploy all pending workspace applications on a WorkSpace
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DeployWorkspaceApplications.html
     */
    toDeployWorkspaceApplications() {
        return this.to('DeployWorkspaceApplications');
    }
    /**
     * Grants permission to deregister directories from use with Amazon WorkSpaces
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DeregisterWorkspaceDirectory.html
     */
    toDeregisterWorkspaceDirectory() {
        return this.to('DeregisterWorkspaceDirectory');
    }
    /**
     * Grants permission to retrieve the configuration of Bring Your Own License (BYOL) for WorkSpaces accounts
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeAccount.html
     */
    toDescribeAccount() {
        return this.to('DescribeAccount');
    }
    /**
     * Grants permission to retrieve modifications to the configuration of Bring Your Own License (BYOL) for WorkSpaces accounts
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeAccountModifications.html
     */
    toDescribeAccountModifications() {
        return this.to('DescribeAccountModifications');
    }
    /**
     * Grants permission to retrieve information about resources associated with a WorkSpace application
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeApplicationAssociations.html
     */
    toDescribeApplicationAssociations() {
        return this.to('DescribeApplicationAssociations');
    }
    /**
     * Grants permission to obtain information about WorkSpace applications
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeApplications.html
     */
    toDescribeApplications() {
        return this.to('DescribeApplications');
    }
    /**
     * Grants permission to retrieve information about resources associated with a WorkSpace bundle
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeBundleAssociations.html
     */
    toDescribeBundleAssociations() {
        return this.to('DescribeBundleAssociations');
    }
    /**
     * Grants permission to retrieve AWS WorkSpaces Client branding data within a directory
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeClientBranding.html
     */
    toDescribeClientBranding() {
        return this.to('DescribeClientBranding');
    }
    /**
     * Grants permission to retrieve information about WorkSpaces clients
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeClientProperties.html
     */
    toDescribeClientProperties() {
        return this.to('DescribeClientProperties');
    }
    /**
     * Grants permission to retrieve a list of Amazon Connect client add-ins that have been created
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeConnectClientAddIns.html
     */
    toDescribeConnectClientAddIns() {
        return this.to('DescribeConnectClientAddIns');
    }
    /**
     * Grants permission to retrieve the permissions that the owners of connection aliases have granted to other AWS accounts for connection aliases
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeConnectionAliasPermissions.html
     */
    toDescribeConnectionAliasPermissions() {
        return this.to('DescribeConnectionAliasPermissions');
    }
    /**
     * Grants permission to retrieve a list that describes the connection aliases used for cross-Region redirection
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeConnectionAliases.html
     */
    toDescribeConnectionAliases() {
        return this.to('DescribeConnectionAliases');
    }
    /**
     * Grants permission to retrieve information about consent agreement to BYOL minimum requirements
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces/latest/adminguide/wsp-console-permissions-ref.html
     */
    toDescribeConsent() {
        return this.to('DescribeConsent');
    }
    /**
     * Grants permission to retrieve information about WorkSpace BYOL image import task
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeCustomWorkspaceImageImport.html
     */
    toDescribeCustomWorkspaceImageImport() {
        return this.to('DescribeCustomWorkspaceImageImport');
    }
    /**
     * Grants permission to retrieve information about resources associated with a WorkSpace image
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeImageAssociations.html
     */
    toDescribeImageAssociations() {
        return this.to('DescribeImageAssociations');
    }
    /**
     * Grants permission to retrieve information about IP access control groups
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeIpGroups.html
     */
    toDescribeIpGroups() {
        return this.to('DescribeIpGroups');
    }
    /**
     * Grants permission to describe the tags for WorkSpaces resources
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeTags.html
     */
    toDescribeTags() {
        return this.to('DescribeTags');
    }
    /**
     * Grants permission to retrieve information about resources associated with a WorkSpace
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaceAssociations.html
     */
    toDescribeWorkspaceAssociations() {
        return this.to('DescribeWorkspaceAssociations');
    }
    /**
     * Grants permission to obtain information about WorkSpace bundles
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaceBundles.html
     */
    toDescribeWorkspaceBundles() {
        return this.to('DescribeWorkspaceBundles');
    }
    /**
     * Grants permission to retrieve information about directories that are registered with WorkSpaces
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaceDirectories.html
     */
    toDescribeWorkspaceDirectories() {
        return this.to('DescribeWorkspaceDirectories');
    }
    /**
     * Grants permission to retrieve information about WorkSpace image permissions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaceImagePermissions.html
     */
    toDescribeWorkspaceImagePermissions() {
        return this.to('DescribeWorkspaceImagePermissions');
    }
    /**
     * Grants permission to retrieve information about WorkSpace images
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaceImages.html
     */
    toDescribeWorkspaceImages() {
        return this.to('DescribeWorkspaceImages');
    }
    /**
     * Grants permission to retrieve information about WorkSpace snapshots
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaceSnapshots.html
     */
    toDescribeWorkspaceSnapshots() {
        return this.to('DescribeWorkspaceSnapshots');
    }
    /**
     * Grants permission to obtain information about WorkSpaces
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspaces.html
     */
    toDescribeWorkspaces() {
        return this.to('DescribeWorkspaces');
    }
    /**
     * Grants permission to obtain the connection status of WorkSpaces
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspacesConnectionStatus.html
     */
    toDescribeWorkspacesConnectionStatus() {
        return this.to('DescribeWorkspacesConnectionStatus');
    }
    /**
     * Grants permission to retrieve information about the sessions of a WorkSpaces Pool
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspacesPoolSessions.html
     */
    toDescribeWorkspacesPoolSessions() {
        return this.to('DescribeWorkspacesPoolSessions');
    }
    /**
     * Grants permission to retrieve information about WorkSpaces Pools
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DescribeWorkspacesPools.html
     */
    toDescribeWorkspacesPools() {
        return this.to('DescribeWorkspacesPools');
    }
    /**
     * Grants permission to directory management actions while managing and provisioning workspaces
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/adminguide/wsp-console-permissions-ref.html
     */
    toDirectoryAccessManagement() {
        return this.to('DirectoryAccessManagement');
    }
    /**
     * Grants permission to disassociate connection aliases from directories
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DisassociateConnectionAlias.html
     */
    toDisassociateConnectionAlias() {
        return this.to('DisassociateConnectionAlias');
    }
    /**
     * Grants permission to disassociate IP access control groups from directories
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DisassociateIpGroups.html
     */
    toDisassociateIpGroups() {
        return this.to('DisassociateIpGroups');
    }
    /**
     * Grants permission to disassociate a workspace application from a WorkSpace
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_DisassociateWorkspaceApplication.html
     */
    toDisassociateWorkspaceApplication() {
        return this.to('DisassociateWorkspaceApplication');
    }
    /**
     * Grants permission to retrieve a link with another AWS Account for sharing configuration for WorkSpaces BYOL
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_GetAccountLink.html
     */
    toGetAccountLink() {
        return this.to('GetAccountLink');
    }
    /**
     * Grants permission to import AWS WorkSpaces Client branding data within a directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ImportClientBranding.html
     */
    toImportClientBranding() {
        return this.to('ImportClientBranding');
    }
    /**
     * Grants permission to import Bring Your Own License (BYOL) images into Amazon WorkSpaces
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ImportCustomWorkspaceImage.html
     */
    toImportCustomWorkspaceImage() {
        return this.to('ImportCustomWorkspaceImage');
    }
    /**
     * Grants permission to import Bring Your Own License (BYOL) images into Amazon WorkSpaces
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:DescribeImages
     * - ec2:ModifyImageAttribute
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ImportWorkspaceImage.html
     */
    toImportWorkspaceImage() {
        return this.to('ImportWorkspaceImage');
    }
    /**
     * Grants permission to retrieve links with the AWS Account(s) that share your configuration for WorkSpaces BYOL
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ListAccountLinks.html
     */
    toListAccountLinks() {
        return this.to('ListAccountLinks');
    }
    /**
     * Grants permission to list the available CIDR ranges for enabling Bring Your Own License (BYOL) for WorkSpaces accounts
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ListAvailableManagementCidrRanges.html
     */
    toListAvailableManagementCidrRanges() {
        return this.to('ListAvailableManagementCidrRanges');
    }
    /**
     * Grants permission to migrate WorkSpaces
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_MigrateWorkspace.html
     */
    toMigrateWorkspace() {
        return this.to('MigrateWorkspace');
    }
    /**
     * Grants permission to modify the configuration of Bring Your Own License (BYOL) for WorkSpaces accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ModifyAccount.html
     */
    toModifyAccount() {
        return this.to('ModifyAccount');
    }
    /**
     * Grants permission to modify the certificate-based authorization properties of a directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ModifyCertificateBasedAuthProperties.html
     */
    toModifyCertificateBasedAuthProperties() {
        return this.to('ModifyCertificateBasedAuthProperties');
    }
    /**
     * Grants permission to modify the properties of WorkSpaces clients
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ModifyClientProperties.html
     */
    toModifyClientProperties() {
        return this.to('ModifyClientProperties');
    }
    /**
     * Grants permission to configure the specified directory between Standard TLS and FIPS 140-2 validated mode
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ModifyEndpointEncryptionMode.html
     */
    toModifyEndpointEncryptionMode() {
        return this.to('ModifyEndpointEncryptionMode');
    }
    /**
     * Grants permission to modify the SAML properties of a directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ModifySamlProperties.html
     */
    toModifySamlProperties() {
        return this.to('ModifySamlProperties');
    }
    /**
     * Grants permission to modify the self-service WorkSpace management capabilities for your users
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ModifySelfservicePermissions.html
     */
    toModifySelfservicePermissions() {
        return this.to('ModifySelfservicePermissions');
    }
    /**
     * Grants permission to modify the streaming properties
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ModifyStreamingProperties.html
     */
    toModifyStreamingProperties() {
        return this.to('ModifyStreamingProperties');
    }
    /**
     * Grants permission to specify which devices and operating systems users can use to access their WorkSpaces
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ModifyWorkspaceAccessProperties.html
     */
    toModifyWorkspaceAccessProperties() {
        return this.to('ModifyWorkspaceAccessProperties');
    }
    /**
     * Grants permission to modify the default properties used to create WorkSpaces
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ModifyWorkspaceCreationProperties.html
     */
    toModifyWorkspaceCreationProperties() {
        return this.to('ModifyWorkspaceCreationProperties');
    }
    /**
     * Grants permission to modify WorkSpace properties, including the running mode and the AutoStop period
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ModifyWorkspaceProperties.html
     */
    toModifyWorkspaceProperties() {
        return this.to('ModifyWorkspaceProperties');
    }
    /**
     * Grants permission to modify the state of WorkSpaces
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_ModifyWorkspaceState.html
     */
    toModifyWorkspaceState() {
        return this.to('ModifyWorkspaceState');
    }
    /**
     * Grants permission to reboot WorkSpaces
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_RebootWorkspaces.html
     */
    toRebootWorkspaces() {
        return this.to('RebootWorkspaces');
    }
    /**
     * Grants permission to rebuild WorkSpaces
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_RebuildWorkspaces.html
     */
    toRebuildWorkspaces() {
        return this.to('RebuildWorkspaces');
    }
    /**
     * Grants permission to register directories for use with Amazon WorkSpaces
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_RegisterWorkspaceDirectory.html
     */
    toRegisterWorkspaceDirectory() {
        return this.to('RegisterWorkspaceDirectory');
    }
    /**
     * Grants permission to reject invitations from other AWS accounts to share the same configuration for WorkSpaces BYOL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_RejectAccountLinkInvitation.html
     */
    toRejectAccountLinkInvitation() {
        return this.to('RejectAccountLinkInvitation');
    }
    /**
     * Grants permission to restore WorkSpaces
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_RestoreWorkspace.html
     */
    toRestoreWorkspace() {
        return this.to('RestoreWorkspace');
    }
    /**
     * Grants permission to remove rules from IP access control groups
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - workspaces:UpdateRulesOfIpGroup
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_RevokeIpRules.html
     */
    toRevokeIpRules() {
        return this.to('RevokeIpRules');
    }
    /**
     * Grants permission to start AutoStop WorkSpaces
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_StartWorkspaces.html
     */
    toStartWorkspaces() {
        return this.to('StartWorkspaces');
    }
    /**
     * Grants permission to start a WorkSpaces Pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_StartWorkspacesPool.html
     */
    toStartWorkspacesPool() {
        return this.to('StartWorkspacesPool');
    }
    /**
     * Grants permission to stop AutoStop WorkSpaces
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_StopWorkspaces.html
     */
    toStopWorkspaces() {
        return this.to('StopWorkspaces');
    }
    /**
     * Grants permission to stop a WorkSpaces Pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_StopWorkspacesPool.html
     */
    toStopWorkspacesPool() {
        return this.to('StopWorkspacesPool');
    }
    /**
     * Grants permission to federated users to sign in by using their existing credentials and stream their workspace
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUserId()
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_Stream.html
     */
    toStream() {
        return this.to('Stream');
    }
    /**
     * Grants permission to terminate WorkSpaces
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_TerminateWorkspaces.html
     */
    toTerminateWorkspaces() {
        return this.to('TerminateWorkspaces');
    }
    /**
     * Grants permission to terminate a WorkSpaces Pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_TerminateWorkspacesPool.html
     */
    toTerminateWorkspacesPool() {
        return this.to('TerminateWorkspacesPool');
    }
    /**
     * Grants permission to terminate a WorkSpaces Pool session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_TerminateWorkspacesPoolSession.html
     */
    toTerminateWorkspacesPoolSession() {
        return this.to('TerminateWorkspacesPoolSession');
    }
    /**
     * Grants permission to update an Amazon Connect client add-in. Use this action to update the name and endpoint URL of an Amazon Connect client add-in
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_UpdateConnectClientAddIn.html
     */
    toUpdateConnectClientAddIn() {
        return this.to('UpdateConnectClientAddIn');
    }
    /**
     * Grants permission to share or unshare connection aliases with other accounts
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_UpdateConnectionAliasPermission.html
     */
    toUpdateConnectionAliasPermission() {
        return this.to('UpdateConnectionAliasPermission');
    }
    /**
     * Grants permission to update the consent agreement to BYOL minimum requirements
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/adminguide/wsp-console-permissions-ref.html
     */
    toUpdateConsent() {
        return this.to('UpdateConsent');
    }
    /**
     * Grants permission to update a root client certificate
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/adminguide/wsp-console-permissions-ref.html
     */
    toUpdateRootClientCertificate() {
        return this.to('UpdateRootClientCertificate');
    }
    /**
     * Grants permission to replace rules for IP access control groups
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - workspaces:AuthorizeIpRules
     * - workspaces:RevokeIpRules
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_UpdateRulesOfIpGroup.html
     */
    toUpdateRulesOfIpGroup() {
        return this.to('UpdateRulesOfIpGroup');
    }
    /**
     * Grants permission to update the WorkSpace images used in WorkSpace bundles
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_UpdateWorkspaceBundle.html
     */
    toUpdateWorkspaceBundle() {
        return this.to('UpdateWorkspaceBundle');
    }
    /**
     * Grants permission to share or unshare WorkSpace images with other accounts by specifying whether other accounts have permission to copy the image
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_UpdateWorkspaceImagePermission.html
     */
    toUpdateWorkspaceImagePermission() {
        return this.to('UpdateWorkspaceImagePermission');
    }
    /**
     * Grants permission to update the WorkSpaces pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces/latest/api/API_UpdateWorkspacesPool.html
     */
    toUpdateWorkspacesPool() {
        return this.to('UpdateWorkspacesPool');
    }
    /**
     * Adds a resource of type certificateid to the statement
     *
     * https://docs.aws.amazon.com/workspaces/latest/adminguide/trusted-devices.html
     *
     * @param certificateId - Identifier for the certificateId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onCertificateid(certificateId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:workspaces:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:workspacecertificate/${certificateId}`);
    }
    /**
     * Adds a resource of type directoryid to the statement
     *
     * https://docs.aws.amazon.com/workspaces/latest/adminguide/manage-workspaces-directory.html
     *
     * @param directoryId - Identifier for the directoryId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDirectoryid(directoryId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:workspaces:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:directory/${directoryId}`);
    }
    /**
     * Adds a resource of type workspacebundle to the statement
     *
     * https://docs.aws.amazon.com/workspaces/latest/adminguide/bundles.html
     *
     * @param bundleId - Identifier for the bundleId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWorkspacebundle(bundleId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:workspaces:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:workspacebundle/${bundleId}`);
    }
    /**
     * Adds a resource of type workspaceid to the statement
     *
     * https://docs.aws.amazon.com/workspaces/latest/adminguide/wsp_workspace_management.html
     *
     * @param workspaceId - Identifier for the workspaceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWorkspaceid(workspaceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:workspaces:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:workspace/${workspaceId}`);
    }
    /**
     * Adds a resource of type workspaceimage to the statement
     *
     * https://docs.aws.amazon.com/workspaces/latest/adminguide/bundles.html
     *
     * @param imageId - Identifier for the imageId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWorkspaceimage(imageId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:workspaces:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:workspaceimage/${imageId}`);
    }
    /**
     * Adds a resource of type workspaceipgroup to the statement
     *
     * https://docs.aws.amazon.com/workspaces/latest/adminguide/amazon-workspaces-ip-access-control-groups.html
     *
     * @param groupId - Identifier for the groupId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWorkspaceipgroup(groupId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:workspaces:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:workspaceipgroup/${groupId}`);
    }
    /**
     * Adds a resource of type workspacespoolid to the statement
     *
     * https://docs.aws.amazon.com/workspaces/latest/adminguide/amazon-workspaces-pool.html
     *
     * @param poolId - Identifier for the poolId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWorkspacespoolid(poolId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:workspaces:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:workspacespool/${poolId}`);
    }
    /**
     * Adds a resource of type connectionalias to the statement
     *
     * https://docs.aws.amazon.com/workspaces/latest/adminguide/cross-region-redirection.html
     *
     * @param connectionAliasId - Identifier for the connectionAliasId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConnectionalias(connectionAliasId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:workspaces:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:connectionalias/${connectionAliasId}`);
    }
    /**
     * Adds a resource of type workspaceapplication to the statement
     *
     * https://docs.aws.amazon.com/workspaces/latest/adminguide/application-bundle-management.html
     *
     * @param workSpaceApplicationId - Identifier for the workSpaceApplicationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWorkspaceapplication(workSpaceApplicationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:workspaces:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:workspaceapplication/${workSpaceApplicationId}`);
    }
    /**
     * Filters access based on the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCopyWorkspaceImage()
     * - .toCreateConnectionAlias()
     * - .toCreateIpGroup()
     * - .toCreateStandbyWorkspaces()
     * - .toCreateTags()
     * - .toCreateUpdatedWorkspaceImage()
     * - .toCreateWorkspaceBundle()
     * - .toCreateWorkspaceImage()
     * - .toCreateWorkspaces()
     * - .toCreateWorkspacesPool()
     * - .toDeleteTags()
     * - .toRegisterWorkspaceDirectory()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toAssociateWorkspaceApplication()
     * - .toDeployWorkspaceApplications()
     * - .toDescribeApplicationAssociations()
     * - .toDescribeBundleAssociations()
     * - .toDescribeImageAssociations()
     * - .toDescribeWorkspaceAssociations()
     * - .toDisassociateWorkspaceApplication()
     *
     * Applies to resource types:
     * - directoryid
     * - workspacebundle
     * - workspaceid
     * - workspaceimage
     * - workspaceipgroup
     * - workspacespoolid
     * - connectionalias
     * - workspaceapplication
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCopyWorkspaceImage()
     * - .toCreateConnectionAlias()
     * - .toCreateIpGroup()
     * - .toCreateStandbyWorkspaces()
     * - .toCreateTags()
     * - .toCreateUpdatedWorkspaceImage()
     * - .toCreateWorkspaceBundle()
     * - .toCreateWorkspaceImage()
     * - .toCreateWorkspaces()
     * - .toCreateWorkspacesPool()
     * - .toDeleteTags()
     * - .toRegisterWorkspaceDirectory()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the ID of the Workspaces user
     *
     * https://docs.aws.amazon.com/workspaces/latest/adminguide/external-identity-providers-setting-up-saml.html#external-identity-providers-embed-inline-policy-for-IAM-role
     *
     * Applies to actions:
     * - .toStream()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifUserId(value, operator) {
        return this.if(`userId`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [workspaces](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonworkspaces.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'workspaces';
        this.accessLevelList = {
            Write: [
                'AcceptAccountLinkInvitation',
                'AssociateConnectionAlias',
                'AssociateIpGroups',
                'AssociateWorkspaceApplication',
                'AuthorizeIpRules',
                'CopyWorkspaceImage',
                'CreateAccountLinkInvitation',
                'CreateConnectClientAddIn',
                'CreateConnectionAlias',
                'CreateIpGroup',
                'CreateRootClientCertificate',
                'CreateStandbyWorkspaces',
                'CreateUpdatedWorkspaceImage',
                'CreateWorkspaceBundle',
                'CreateWorkspaceImage',
                'CreateWorkspaces',
                'CreateWorkspacesPool',
                'DeleteAccountLinkInvitation',
                'DeleteClientBranding',
                'DeleteConnectClientAddIn',
                'DeleteConnectionAlias',
                'DeleteIpGroup',
                'DeleteRootClientCertificate',
                'DeleteWorkspaceBundle',
                'DeleteWorkspaceImage',
                'DeployWorkspaceApplications',
                'DeregisterWorkspaceDirectory',
                'DisassociateConnectionAlias',
                'DisassociateIpGroups',
                'DisassociateWorkspaceApplication',
                'ImportClientBranding',
                'ImportCustomWorkspaceImage',
                'ImportWorkspaceImage',
                'MigrateWorkspace',
                'ModifyAccount',
                'ModifyCertificateBasedAuthProperties',
                'ModifyClientProperties',
                'ModifyEndpointEncryptionMode',
                'ModifySamlProperties',
                'ModifyStreamingProperties',
                'ModifyWorkspaceAccessProperties',
                'ModifyWorkspaceCreationProperties',
                'ModifyWorkspaceProperties',
                'ModifyWorkspaceState',
                'RebootWorkspaces',
                'RebuildWorkspaces',
                'RegisterWorkspaceDirectory',
                'RejectAccountLinkInvitation',
                'RestoreWorkspace',
                'RevokeIpRules',
                'StartWorkspaces',
                'StartWorkspacesPool',
                'StopWorkspaces',
                'StopWorkspacesPool',
                'Stream',
                'TerminateWorkspaces',
                'TerminateWorkspacesPool',
                'TerminateWorkspacesPoolSession',
                'UpdateConnectClientAddIn',
                'UpdateConsent',
                'UpdateRootClientCertificate',
                'UpdateRulesOfIpGroup',
                'UpdateWorkspaceBundle',
                'UpdateWorkspacesPool'
            ],
            Tagging: [
                'CreateTags',
                'DeleteTags'
            ],
            Read: [
                'DescribeAccount',
                'DescribeAccountModifications',
                'DescribeClientBranding',
                'DescribeConnectionAliasPermissions',
                'DescribeConnectionAliases',
                'DescribeConsent',
                'DescribeCustomWorkspaceImageImport',
                'DescribeIpGroups',
                'DescribeTags',
                'DescribeWorkspaceDirectories',
                'DescribeWorkspaceImagePermissions',
                'DescribeWorkspacesConnectionStatus',
                'GetAccountLink'
            ],
            List: [
                'DescribeApplicationAssociations',
                'DescribeApplications',
                'DescribeBundleAssociations',
                'DescribeClientProperties',
                'DescribeConnectClientAddIns',
                'DescribeImageAssociations',
                'DescribeWorkspaceAssociations',
                'DescribeWorkspaceBundles',
                'DescribeWorkspaceImages',
                'DescribeWorkspaceSnapshots',
                'DescribeWorkspaces',
                'DescribeWorkspacesPoolSessions',
                'DescribeWorkspacesPools',
                'DirectoryAccessManagement',
                'ListAccountLinks',
                'ListAvailableManagementCidrRanges'
            ],
            'Permissions management': [
                'ModifySelfservicePermissions',
                'UpdateConnectionAliasPermission',
                'UpdateWorkspaceImagePermission'
            ]
        };
    }
}
exports.Workspaces = Workspaces;
//# sourceMappingURL=data:application/json;base64,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