"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.WorkspacesWeb = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [workspaces-web](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonworkspacessecurebrowser.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class WorkspacesWeb extends shared_1.PolicyStatement {
    /**
     * Grants permission to associate browser settings to web portals
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_AssociateBrowserSettings.html
     */
    toAssociateBrowserSettings() {
        return this.to('AssociateBrowserSettings');
    }
    /**
     * Grants permission to associate data protection settings with web portals
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_AssociateDataProtectionSettings.html
     */
    toAssociateDataProtectionSettings() {
        return this.to('AssociateDataProtectionSettings');
    }
    /**
     * Grants permission to associate ip access settings with web portals
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_AssociateIpAccessSettings.html
     */
    toAssociateIpAccessSettings() {
        return this.to('AssociateIpAccessSettings');
    }
    /**
     * Grants permission to associate network settings to web portals
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:CreateNetworkInterfacePermission
     * - ec2:CreateTags
     * - ec2:DeleteNetworkInterface
     * - ec2:DeleteNetworkInterfacePermission
     * - ec2:ModifyNetworkInterfaceAttribute
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_AssociateNetworkSettings.html
     */
    toAssociateNetworkSettings() {
        return this.to('AssociateNetworkSettings');
    }
    /**
     * Grants permission to associate session logger with web portals
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_AssociateSessionLogger.html
     */
    toAssociateSessionLogger() {
        return this.to('AssociateSessionLogger');
    }
    /**
     * Grants permission to associate trust stores with web portals
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_AssociateTrustStore.html
     */
    toAssociateTrustStore() {
        return this.to('AssociateTrustStore');
    }
    /**
     * Grants permission to associate user access logging settings with web portals
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kinesis:PutRecord
     * - kinesis:PutRecords
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_AssociateUserAccessLoggingSettings.html
     */
    toAssociateUserAccessLoggingSettings() {
        return this.to('AssociateUserAccessLoggingSettings');
    }
    /**
     * Grants permission to associate user settings with web portals
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_AssociateUserSettings.html
     */
    toAssociateUserSettings() {
        return this.to('AssociateUserSettings');
    }
    /**
     * Grants permission to create browser settings
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * Dependent actions:
     * - kms:CreateGrant
     * - kms:Decrypt
     * - kms:DescribeKey
     * - kms:GenerateDataKey
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_CreateBrowserSettings.html
     */
    toCreateBrowserSettings() {
        return this.to('CreateBrowserSettings');
    }
    /**
     * Grants permission to create data protection settings
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_CreateDataProtectionSettings.html
     */
    toCreateDataProtectionSettings() {
        return this.to('CreateDataProtectionSettings');
    }
    /**
     * Grants permission to create identity providers
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_CreateIdentityProvider.html
     */
    toCreateIdentityProvider() {
        return this.to('CreateIdentityProvider');
    }
    /**
     * Grants permission to create ip access settings
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_CreateIpAccessSettings.html
     */
    toCreateIpAccessSettings() {
        return this.to('CreateIpAccessSettings');
    }
    /**
     * Grants permission to create network settings
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_CreateNetworkSettings.html
     */
    toCreateNetworkSettings() {
        return this.to('CreateNetworkSettings');
    }
    /**
     * Grants permission to create web portals
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - kms:CreateGrant
     * - kms:Decrypt
     * - kms:DescribeKey
     * - kms:GenerateDataKey
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_CreatePortal.html
     */
    toCreatePortal() {
        return this.to('CreatePortal');
    }
    /**
     * Grants permission to create session logger
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_CreateSessionLogger.html
     */
    toCreateSessionLogger() {
        return this.to('CreateSessionLogger');
    }
    /**
     * Grants permission to create trust stores
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_CreateTrustStore.html
     */
    toCreateTrustStore() {
        return this.to('CreateTrustStore');
    }
    /**
     * Grants permission to create user access logging settings
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_CreateUserAccessLoggingSettings.html
     */
    toCreateUserAccessLoggingSettings() {
        return this.to('CreateUserAccessLoggingSettings');
    }
    /**
     * Grants permission to create user settings
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_CreateUserSettings.html
     */
    toCreateUserSettings() {
        return this.to('CreateUserSettings');
    }
    /**
     * Grants permission to delete browser settings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_DeleteBrowserSettings.html
     */
    toDeleteBrowserSettings() {
        return this.to('DeleteBrowserSettings');
    }
    /**
     * Grants permission to delete data protection settings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_DeleteDataProtectionSettings.html
     */
    toDeleteDataProtectionSettings() {
        return this.to('DeleteDataProtectionSettings');
    }
    /**
     * Grants permission to delete identity providers
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_DeleteIdentityProvider.html
     */
    toDeleteIdentityProvider() {
        return this.to('DeleteIdentityProvider');
    }
    /**
     * Grants permission to delete ip access settings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_DeleteIpAccessSettings.html
     */
    toDeleteIpAccessSettings() {
        return this.to('DeleteIpAccessSettings');
    }
    /**
     * Grants permission to delete network settings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_DeleteNetworkSettings.html
     */
    toDeleteNetworkSettings() {
        return this.to('DeleteNetworkSettings');
    }
    /**
     * Grants permission to delete web portals
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_DeletePortal.html
     */
    toDeletePortal() {
        return this.to('DeletePortal');
    }
    /**
     * Grants permission to delete session logger
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_DeleteSessionLogger.html
     */
    toDeleteSessionLogger() {
        return this.to('DeleteSessionLogger');
    }
    /**
     * Grants permission to delete trust stores
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_DeleteTrustStore.html
     */
    toDeleteTrustStore() {
        return this.to('DeleteTrustStore');
    }
    /**
     * Grants permission to delete user access logging settings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_DeleteUserAccessLoggingSettings.html
     */
    toDeleteUserAccessLoggingSettings() {
        return this.to('DeleteUserAccessLoggingSettings');
    }
    /**
     * Grants permission to delete user settings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_DeleteUserSettings.html
     */
    toDeleteUserSettings() {
        return this.to('DeleteUserSettings');
    }
    /**
     * Grants permission to disassociate browser settings from web portals
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_DisassociateBrowserSettings.html
     */
    toDisassociateBrowserSettings() {
        return this.to('DisassociateBrowserSettings');
    }
    /**
     * Grants permission to disassociate data protection logging from web portals
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_DisassociateDataProtectionSettings.html
     */
    toDisassociateDataProtectionSettings() {
        return this.to('DisassociateDataProtectionSettings');
    }
    /**
     * Grants permission to disassociate ip access logging from web portals
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_DisassociateIpAccessSettings.html
     */
    toDisassociateIpAccessSettings() {
        return this.to('DisassociateIpAccessSettings');
    }
    /**
     * Grants permission to disassociate network settings from web portals
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_DisassociateNetworkSettings.html
     */
    toDisassociateNetworkSettings() {
        return this.to('DisassociateNetworkSettings');
    }
    /**
     * Grants permission to disassociate session logger from web portals
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_DisassociateSessionLogger.html
     */
    toDisassociateSessionLogger() {
        return this.to('DisassociateSessionLogger');
    }
    /**
     * Grants permission to disassociate trust stores from web portals
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_DisassociateTrustStore.html
     */
    toDisassociateTrustStore() {
        return this.to('DisassociateTrustStore');
    }
    /**
     * Grants permission to disassociate user access logging settings from web portals
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_DisassociateUserAccessLoggingSettings.html
     */
    toDisassociateUserAccessLoggingSettings() {
        return this.to('DisassociateUserAccessLoggingSettings');
    }
    /**
     * Grants permission to disassociate user settings from web portals
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_DisassociateUserSettings.html
     */
    toDisassociateUserSettings() {
        return this.to('DisassociateUserSettings');
    }
    /**
     * Grants permission to expire a session from a specific portal
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_ExpireSession.html
     */
    toExpireSession() {
        return this.to('ExpireSession');
    }
    /**
     * Grants permission to get details on browser settings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_GetBrowserSettings.html
     */
    toGetBrowserSettings() {
        return this.to('GetBrowserSettings');
    }
    /**
     * Grants permission to get details on data protection settings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_GetDataProtectionSettings.html
     */
    toGetDataProtectionSettings() {
        return this.to('GetDataProtectionSettings');
    }
    /**
     * Grants permission to get details on identity providers
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_GetIdentityProvider.html
     */
    toGetIdentityProvider() {
        return this.to('GetIdentityProvider');
    }
    /**
     * Grants permission to get details on ip access settings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_GetIpAccessSettings.html
     */
    toGetIpAccessSettings() {
        return this.to('GetIpAccessSettings');
    }
    /**
     * Grants permission to get details on network settings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_GetNetworkSettings.html
     */
    toGetNetworkSettings() {
        return this.to('GetNetworkSettings');
    }
    /**
     * Grants permission to get details on web portals
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_GetPortal.html
     */
    toGetPortal() {
        return this.to('GetPortal');
    }
    /**
     * Grants permission to get service provider metadata information for web portals
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_GetPortalServiceProviderMetadata.html
     */
    toGetPortalServiceProviderMetadata() {
        return this.to('GetPortalServiceProviderMetadata');
    }
    /**
     * Grants permission to get information about a particular session for a portal
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_GetSession.html
     */
    toGetSession() {
        return this.to('GetSession');
    }
    /**
     * Grants permission to get details on session logger
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_GetSessionLogger.html
     */
    toGetSessionLogger() {
        return this.to('GetSessionLogger');
    }
    /**
     * Grants permission to get details on trust stores
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_GetTrustStore.html
     */
    toGetTrustStore() {
        return this.to('GetTrustStore');
    }
    /**
     * Grants permission to get certificates from trust stores
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_GetTrustStoreCertificate.html
     */
    toGetTrustStoreCertificate() {
        return this.to('GetTrustStoreCertificate');
    }
    /**
     * Grants permission to get details on user access logging settings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_GetUserAccessLoggingSettings.html
     */
    toGetUserAccessLoggingSettings() {
        return this.to('GetUserAccessLoggingSettings');
    }
    /**
     * Grants permission to get details on user settings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_GetUserSettings.html
     */
    toGetUserSettings() {
        return this.to('GetUserSettings');
    }
    /**
     * Grants permission to list browser settings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_ListBrowserSettings.html
     */
    toListBrowserSettings() {
        return this.to('ListBrowserSettings');
    }
    /**
     * Grants permission to list data protection settings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_ListDataProtectionSettings.html
     */
    toListDataProtectionSettings() {
        return this.to('ListDataProtectionSettings');
    }
    /**
     * Grants permission to list identity providers
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_ListIdentityProviders.html
     */
    toListIdentityProviders() {
        return this.to('ListIdentityProviders');
    }
    /**
     * Grants permission to list ip access settings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_ListIpAccessSettings.html
     */
    toListIpAccessSettings() {
        return this.to('ListIpAccessSettings');
    }
    /**
     * Grants permission to list network settings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_ListNetworkSettings.html
     */
    toListNetworkSettings() {
        return this.to('ListNetworkSettings');
    }
    /**
     * Grants permission to list web portals
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_ListPortals.html
     */
    toListPortals() {
        return this.to('ListPortals');
    }
    /**
     * Grants permission to list session loggers
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_ListSessionLoggers.html
     */
    toListSessionLoggers() {
        return this.to('ListSessionLoggers');
    }
    /**
     * Grants permission to list sessions for a Portal using optional filters
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_ListSessions.html
     */
    toListSessions() {
        return this.to('ListSessions');
    }
    /**
     * Grants permission to list tags for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list certificates in a trust store
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_ListTrustStoreCertificates.html
     */
    toListTrustStoreCertificates() {
        return this.to('ListTrustStoreCertificates');
    }
    /**
     * Grants permission to list trust stores
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_ListTrustStores.html
     */
    toListTrustStores() {
        return this.to('ListTrustStores');
    }
    /**
     * Grants permission to list user access logging settings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_ListUserAccessLoggingSettings.html
     */
    toListUserAccessLoggingSettings() {
        return this.to('ListUserAccessLoggingSettings');
    }
    /**
     * Grants permission to list user settings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_ListUserSettings.html
     */
    toListUserSettings() {
        return this.to('ListUserSettings');
    }
    /**
     * Grants permission to add one or more tags to a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove one or more tags from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update browser settings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_UpdateBrowserSettings.html
     */
    toUpdateBrowserSettings() {
        return this.to('UpdateBrowserSettings');
    }
    /**
     * Grants permission to update data protection settings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_UpdateDataProtectionSettings.html
     */
    toUpdateDataProtectionSettings() {
        return this.to('UpdateDataProtectionSettings');
    }
    /**
     * Grants permission to update identity provider
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_UpdateIdentityProvider.html
     */
    toUpdateIdentityProvider() {
        return this.to('UpdateIdentityProvider');
    }
    /**
     * Grants permission to update ip access settings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_UpdateIpAccessSettings.html
     */
    toUpdateIpAccessSettings() {
        return this.to('UpdateIpAccessSettings');
    }
    /**
     * Grants permission to update network settings
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:CreateNetworkInterfacePermission
     * - ec2:CreateTags
     * - ec2:DeleteNetworkInterface
     * - ec2:DeleteNetworkInterfacePermission
     * - ec2:ModifyNetworkInterfaceAttribute
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_UpdateNetworkSettings.html
     */
    toUpdateNetworkSettings() {
        return this.to('UpdateNetworkSettings');
    }
    /**
     * Grants permission to update web portals
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_UpdatePortal.html
     */
    toUpdatePortal() {
        return this.to('UpdatePortal');
    }
    /**
     * Grants permission to update session logger
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_UpdateSessionLogger.html
     */
    toUpdateSessionLogger() {
        return this.to('UpdateSessionLogger');
    }
    /**
     * Grants permission to update trust stores
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_UpdateTrustStore.html
     */
    toUpdateTrustStore() {
        return this.to('UpdateTrustStore');
    }
    /**
     * Grants permission to update user access logging settings
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kinesis:PutRecord
     * - kinesis:PutRecords
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_UpdateUserAccessLoggingSettings.html
     */
    toUpdateUserAccessLoggingSettings() {
        return this.to('UpdateUserAccessLoggingSettings');
    }
    /**
     * Grants permission to update user settings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_UpdateUserSettings.html
     */
    toUpdateUserSettings() {
        return this.to('UpdateUserSettings');
    }
    /**
     * Adds a resource of type browserSettings to the statement
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_CreateBrowserSettings.html
     *
     * @param browserSettingsId - Identifier for the browserSettingsId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBrowserSettings(browserSettingsId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:workspaces-web:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:browserSettings/${browserSettingsId}`);
    }
    /**
     * Adds a resource of type identityProvider to the statement
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_CreateIdentityProvider.html
     *
     * @param portalId - Identifier for the portalId.
     * @param identityProviderId - Identifier for the identityProviderId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIdentityProvider(portalId, identityProviderId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:workspaces-web:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:identityProvider/${portalId}/${identityProviderId}`);
    }
    /**
     * Adds a resource of type networkSettings to the statement
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_CreateNetworkSettings.html
     *
     * @param networkSettingsId - Identifier for the networkSettingsId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onNetworkSettings(networkSettingsId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:workspaces-web:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:networkSettings/${networkSettingsId}`);
    }
    /**
     * Adds a resource of type portal to the statement
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_CreatePortal.html
     *
     * @param portalId - Identifier for the portalId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPortal(portalId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:workspaces-web:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:portal/${portalId}`);
    }
    /**
     * Adds a resource of type trustStore to the statement
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_CreateTrustStore.html
     *
     * @param trustStoreId - Identifier for the trustStoreId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTrustStore(trustStoreId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:workspaces-web:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:trustStore/${trustStoreId}`);
    }
    /**
     * Adds a resource of type userSettings to the statement
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_CreateUserSettings.html
     *
     * @param userSettingsId - Identifier for the userSettingsId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onUserSettings(userSettingsId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:workspaces-web:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:userSettings/${userSettingsId}`);
    }
    /**
     * Adds a resource of type userAccessLoggingSettings to the statement
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_CreateUserAccessLoggingSettings.html
     *
     * @param userAccessLoggingSettingsId - Identifier for the userAccessLoggingSettingsId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onUserAccessLoggingSettings(userAccessLoggingSettingsId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:workspaces-web:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:userAccessLoggingSettings/${userAccessLoggingSettingsId}`);
    }
    /**
     * Adds a resource of type ipAccessSettings to the statement
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_CreateIpAccessSettings.html
     *
     * @param ipAccessSettingsId - Identifier for the ipAccessSettingsId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIpAccessSettings(ipAccessSettingsId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:workspaces-web:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:ipAccessSettings/${ipAccessSettingsId}`);
    }
    /**
     * Adds a resource of type dataProtectionSettings to the statement
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_CreateDataProtectionSettings.html
     *
     * @param dataProtectionSettingsId - Identifier for the dataProtectionSettingsId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDataProtectionSettings(dataProtectionSettingsId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:workspaces-web:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:dataProtectionSettings/${dataProtectionSettingsId}`);
    }
    /**
     * Adds a resource of type sessionLogger to the statement
     *
     * https://docs.aws.amazon.com/workspaces-web/latest/APIReference/API_CreateSessionLogger.html
     *
     * @param sessionLoggerId - Identifier for the sessionLoggerId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSessionLogger(sessionLoggerId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:workspaces-web:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:sessionLogger/${sessionLoggerId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateBrowserSettings()
     * - .toCreateDataProtectionSettings()
     * - .toCreateIdentityProvider()
     * - .toCreateIpAccessSettings()
     * - .toCreateNetworkSettings()
     * - .toCreatePortal()
     * - .toCreateSessionLogger()
     * - .toCreateTrustStore()
     * - .toCreateUserAccessLoggingSettings()
     * - .toCreateUserSettings()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - browserSettings
     * - identityProvider
     * - networkSettings
     * - portal
     * - trustStore
     * - userSettings
     * - userAccessLoggingSettings
     * - ipAccessSettings
     * - dataProtectionSettings
     * - sessionLogger
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateBrowserSettings()
     * - .toCreateDataProtectionSettings()
     * - .toCreateIdentityProvider()
     * - .toCreateIpAccessSettings()
     * - .toCreateNetworkSettings()
     * - .toCreatePortal()
     * - .toCreateSessionLogger()
     * - .toCreateTrustStore()
     * - .toCreateUserAccessLoggingSettings()
     * - .toCreateUserSettings()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [workspaces-web](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonworkspacessecurebrowser.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'workspaces-web';
        this.accessLevelList = {
            Write: [
                'AssociateBrowserSettings',
                'AssociateDataProtectionSettings',
                'AssociateIpAccessSettings',
                'AssociateNetworkSettings',
                'AssociateSessionLogger',
                'AssociateTrustStore',
                'AssociateUserAccessLoggingSettings',
                'AssociateUserSettings',
                'CreateBrowserSettings',
                'CreateDataProtectionSettings',
                'CreateIdentityProvider',
                'CreateIpAccessSettings',
                'CreateNetworkSettings',
                'CreatePortal',
                'CreateSessionLogger',
                'CreateTrustStore',
                'CreateUserAccessLoggingSettings',
                'CreateUserSettings',
                'DeleteBrowserSettings',
                'DeleteDataProtectionSettings',
                'DeleteIdentityProvider',
                'DeleteIpAccessSettings',
                'DeleteNetworkSettings',
                'DeletePortal',
                'DeleteSessionLogger',
                'DeleteTrustStore',
                'DeleteUserAccessLoggingSettings',
                'DeleteUserSettings',
                'DisassociateBrowserSettings',
                'DisassociateDataProtectionSettings',
                'DisassociateIpAccessSettings',
                'DisassociateNetworkSettings',
                'DisassociateSessionLogger',
                'DisassociateTrustStore',
                'DisassociateUserAccessLoggingSettings',
                'DisassociateUserSettings',
                'ExpireSession',
                'UpdateBrowserSettings',
                'UpdateDataProtectionSettings',
                'UpdateIdentityProvider',
                'UpdateIpAccessSettings',
                'UpdateNetworkSettings',
                'UpdatePortal',
                'UpdateSessionLogger',
                'UpdateTrustStore',
                'UpdateUserAccessLoggingSettings',
                'UpdateUserSettings'
            ],
            Read: [
                'GetBrowserSettings',
                'GetDataProtectionSettings',
                'GetIdentityProvider',
                'GetIpAccessSettings',
                'GetNetworkSettings',
                'GetPortal',
                'GetPortalServiceProviderMetadata',
                'GetSession',
                'GetSessionLogger',
                'GetTrustStore',
                'GetTrustStoreCertificate',
                'GetUserAccessLoggingSettings',
                'GetUserSettings',
                'ListBrowserSettings',
                'ListDataProtectionSettings',
                'ListIdentityProviders',
                'ListIpAccessSettings',
                'ListNetworkSettings',
                'ListPortals',
                'ListSessionLoggers',
                'ListSessions',
                'ListTagsForResource',
                'ListTrustStoreCertificates',
                'ListTrustStores',
                'ListUserAccessLoggingSettings',
                'ListUserSettings'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.WorkspacesWeb = WorkspacesWeb;
//# sourceMappingURL=data:application/json;base64,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