"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Xray = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [xray](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsx-ray.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Xray extends shared_1.PolicyStatement {
    /**
     * Grants permission to retrieve metadata for a list of traces specified by ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/devguide/security_iam_id-based-policy-examples.html#security_iam_id-based-policy-examples-console
     */
    toBatchGetTraceSummaryById() {
        return this.to('BatchGetTraceSummaryById');
    }
    /**
     * Grants permission to retrieve a list of traces specified by ID. Each trace is a collection of segment documents that originates from a single request. Use GetTraceSummaries to get a list of trace IDs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_BatchGetTraces.html
     */
    toBatchGetTraces() {
        return this.to('BatchGetTraces');
    }
    /**
     * Grants permission to cancel an ongoing trace retrieval job initiated by StartTraceRetrieval using the provided RetrievalToken. A successful cancellation will return an HTTP 200 response
     *
     * Access Level: Read
     */
    toCancelTraceRetrieval() {
        return this.to('CancelTraceRetrieval');
    }
    /**
     * Grants permission to create a group resource with a name and a filter expression
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_CreateGroup.html
     */
    toCreateGroup() {
        return this.to('CreateGroup');
    }
    /**
     * Grants permission to create a rule to control sampling behavior for instrumented applications
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_CreateSamplingRule.html
     */
    toCreateSamplingRule() {
        return this.to('CreateSamplingRule');
    }
    /**
     * Grants permission to delete a group resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_DeleteGroup.html
     */
    toDeleteGroup() {
        return this.to('DeleteGroup');
    }
    /**
     * Grants permission to delete resource policies
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifResourcePolicyName()
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_DeleteResourcePolicy.html
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to delete a sampling rule
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_DeleteSamplingRule.html
     */
    toDeleteSamplingRule() {
        return this.to('DeleteSamplingRule');
    }
    /**
     * Grants permission to retrieve distinct service graphs for one or more specific trace IDs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/devguide/security_iam_id-based-policy-examples.html#security_iam_id-based-policy-examples-console
     */
    toGetDistinctTraceGraphs() {
        return this.to('GetDistinctTraceGraphs');
    }
    /**
     * Grants permission to retrieve the current encryption configuration for X-Ray data
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetEncryptionConfig.html
     */
    toGetEncryptionConfig() {
        return this.to('GetEncryptionConfig');
    }
    /**
     * Grants permission to retrieve group resource details
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetGroup.html
     */
    toGetGroup() {
        return this.to('GetGroup');
    }
    /**
     * Grants permission to retrieve all active group details
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetGroups.html
     */
    toGetGroups() {
        return this.to('GetGroups');
    }
    /**
     * Grants permission to retrieve all indexing rules. Indexing rules are used to determine the server-side sampling rate for spans ingested through the CloudWatchLogs destination and indexed by X-Ray
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetIndexingRules.html
     */
    toGetIndexingRules() {
        return this.to('GetIndexingRules');
    }
    /**
     * Grants permission to retrieve the details of a specific insight
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetInsight.html
     */
    toGetInsight() {
        return this.to('GetInsight');
    }
    /**
     * Grants permission to retrieve the events of a specific insight
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetInsightEvents.html
     */
    toGetInsightEvents() {
        return this.to('GetInsightEvents');
    }
    /**
     * Grants permission to retrieve the part of the service graph which is impacted for a specific insight
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetInsightImpactGraph.html
     */
    toGetInsightImpactGraph() {
        return this.to('GetInsightImpactGraph');
    }
    /**
     * Grants permission to retrieve the summary of all insights for a group and time range with optional filters
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetInsightSummaries.html
     */
    toGetInsightSummaries() {
        return this.to('GetInsightSummaries');
    }
    /**
     * Grants permission to retrieve a service graph for traces based on the specified RetrievalToken from the Transaction Search CloudWatch log group
     *
     * Access Level: Read
     */
    toGetRetrievedTracesGraph() {
        return this.to('GetRetrievedTracesGraph');
    }
    /**
     * Grants permission to retrieve all sampling rules
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetSamplingRules.html
     */
    toGetSamplingRules() {
        return this.to('GetSamplingRules');
    }
    /**
     * Grants permission to retrieve information about recent sampling results for all sampling rules
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetSamplingStatisticSummaries.html
     */
    toGetSamplingStatisticSummaries() {
        return this.to('GetSamplingStatisticSummaries');
    }
    /**
     * Grants permission to request a sampling quota for rules that the service is using to sample requests
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetSamplingTargets.html
     */
    toGetSamplingTargets() {
        return this.to('GetSamplingTargets');
    }
    /**
     * Grants permission to retrieve a document that describes services that process incoming requests, and downstream services that they call as a result
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetServiceGraph.html
     */
    toGetServiceGraph() {
        return this.to('GetServiceGraph');
    }
    /**
     * Grants permission to retrieve an aggregation of service statistics defined by a specific time range bucketed into time intervals
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetTimeSeriesServiceStatistics.html
     */
    toGetTimeSeriesServiceStatistics() {
        return this.to('GetTimeSeriesServiceStatistics');
    }
    /**
     * Grants permission to retrieve a service graph for one or more specific trace IDs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetTraceGraph.html
     */
    toGetTraceGraph() {
        return this.to('GetTraceGraph');
    }
    /**
     * Grants permission to retrieve the current destination of data sent to PutTraceSegments and OpenTelemetry API
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetTraceSegmentDestination.html
     */
    toGetTraceSegmentDestination() {
        return this.to('GetTraceSegmentDestination');
    }
    /**
     * Grants permission to retrieve IDs and metadata for traces available for a specified time frame using an optional filter. To get the full traces, pass the trace IDs to BatchGetTraces
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetTraceSummaries.html
     */
    toGetTraceSummaries() {
        return this.to('GetTraceSummaries');
    }
    /**
     * Grants permission to share X-Ray resources with a monitoring account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-Unified-Cross-Account-Setup.html#CloudWatch-Unified-Cross-Account-Setup-permissions
     */
    toLink() {
        return this.to('Link');
    }
    /**
     * Grants permission to list resource policies
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_ListResourcePolicies.html
     */
    toListResourcePolicies() {
        return this.to('ListResourcePolicies');
    }
    /**
     * Grants permission to retrieve a list of traces for a given RetrievalToken from the Transaction Search CloudWatch log group
     *
     * Access Level: List
     */
    toListRetrievedTraces() {
        return this.to('ListRetrievedTraces');
    }
    /**
     * Grants permission to list tags for an X-Ray resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to update the encryption configuration for X-Ray data
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_PutEncryptionConfig.html
     */
    toPutEncryptionConfig() {
        return this.to('PutEncryptionConfig');
    }
    /**
     * Grants permission to create or update resource policies
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifResourcePolicyName()
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_PutResourcePolicy.html
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to upload OpenTelemetry spans to AWS X-Ray
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-OTLPEndpoint.html
     */
    toPutSpans() {
        return this.to('PutSpans');
    }
    /**
     * Grants permission to upload spans to AWS X-Ray to be indexed
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/xray/latest/devguide/security_iam_id-based-policy-examples.html#security_iam_id-based-policy-examples-console
     */
    toPutSpansForIndexing() {
        return this.to('PutSpansForIndexing');
    }
    /**
     * Grants permission to send AWS X-Ray daemon telemetry to the service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_PutTelemetryRecords.html
     */
    toPutTelemetryRecords() {
        return this.to('PutTelemetryRecords');
    }
    /**
     * Grants permission to upload segment documents to AWS X-Ray. The X-Ray SDK generates segment documents and sends them to the X-Ray daemon, which uploads them in batches
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_PutTraceSegments.html
     */
    toPutTraceSegments() {
        return this.to('PutTraceSegments');
    }
    /**
     * Grants permission to initiate a trace retrieval process using the specified time range and for the given trace IDs on the Transaction Search CloudWatch log group
     *
     * Access Level: Read
     */
    toStartTraceRetrieval() {
        return this.to('StartTraceRetrieval');
    }
    /**
     * Grants permission to add tags to an X-Ray resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from an X-Ray resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a group resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_UpdateGroup.html
     */
    toUpdateGroup() {
        return this.to('UpdateGroup');
    }
    /**
     * Grants permission to modify an indexing rule's configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_UpdateIndexingRule.html
     */
    toUpdateIndexingRule() {
        return this.to('UpdateIndexingRule');
    }
    /**
     * Grants permission to modify a sampling rule's configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_UpdateSamplingRule.html
     */
    toUpdateSamplingRule() {
        return this.to('UpdateSamplingRule');
    }
    /**
     * Grants permission to modify the destination of data sent to PutTraceSegments and OpenTelemetry API
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTraceSegmentDestination()
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_UpdateTraceSegmentDestination.html
     */
    toUpdateTraceSegmentDestination() {
        return this.to('UpdateTraceSegmentDestination');
    }
    /**
     * Adds a resource of type group to the statement
     *
     * https://docs.aws.amazon.com/xray/latest/devguide/xray-concepts.html#xray-concepts-groups
     *
     * @param groupName - Identifier for the groupName.
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onGroup(groupName, id, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:xray:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:group/${groupName}/${id}`);
    }
    /**
     * Adds a resource of type sampling-rule to the statement
     *
     * https://docs.aws.amazon.com/xray/latest/devguide/xray-concepts.html#xray-concepts-sampling
     *
     * @param samplingRuleName - Identifier for the samplingRuleName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSamplingRule(samplingRuleName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:xray:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:sampling-rule/${samplingRuleName}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateGroup()
     * - .toCreateSamplingRule()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toDeleteGroup()
     * - .toDeleteSamplingRule()
     * - .toGetGroup()
     * - .toUpdateGroup()
     * - .toUpdateSamplingRule()
     *
     * Applies to resource types:
     * - group
     * - sampling-rule
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateGroup()
     * - .toCreateSamplingRule()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by PolicyName in the request
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsx-ray.html#awsx-ray-actions-as-permissions
     *
     * Applies to actions:
     * - .toDeleteResourcePolicy()
     * - .toPutResourcePolicy()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourcePolicyName(value, operator) {
        return this.if(`ResourcePolicyName`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by TraceSegmentDestination type in the request
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsx-ray.html#awsx-ray-actions-as-permissions
     *
     * Applies to actions:
     * - .toUpdateTraceSegmentDestination()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTraceSegmentDestination(value, operator) {
        return this.if(`TraceSegmentDestination`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [xray](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsx-ray.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'xray';
        this.accessLevelList = {
            Read: [
                'BatchGetTraceSummaryById',
                'CancelTraceRetrieval',
                'GetDistinctTraceGraphs',
                'GetEncryptionConfig',
                'GetGroup',
                'GetGroups',
                'GetIndexingRules',
                'GetInsight',
                'GetInsightEvents',
                'GetInsightImpactGraph',
                'GetInsightSummaries',
                'GetRetrievedTracesGraph',
                'GetSamplingRules',
                'GetSamplingStatisticSummaries',
                'GetSamplingTargets',
                'GetServiceGraph',
                'GetTimeSeriesServiceStatistics',
                'GetTraceGraph',
                'GetTraceSegmentDestination',
                'GetTraceSummaries',
                'StartTraceRetrieval'
            ],
            List: [
                'BatchGetTraces',
                'ListResourcePolicies',
                'ListRetrievedTraces',
                'ListTagsForResource'
            ],
            Write: [
                'CreateGroup',
                'CreateSamplingRule',
                'DeleteGroup',
                'DeleteResourcePolicy',
                'DeleteSamplingRule',
                'Link',
                'PutResourcePolicy',
                'PutSpans',
                'PutSpansForIndexing',
                'PutTelemetryRecords',
                'PutTraceSegments',
                'UpdateGroup',
                'UpdateIndexingRule',
                'UpdateSamplingRule',
                'UpdateTraceSegmentDestination'
            ],
            'Permissions management': [
                'PutEncryptionConfig'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Xray = Xray;
//# sourceMappingURL=data:application/json;base64,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