"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeploymentStrategy = exports.PercentOrAbsolute = exports.Deployment = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const container = require("./container");
const k8s = require("./imports/k8s");
const service = require("./service");
const workload = require("./workload");
/**
*
* A Deployment provides declarative updates for Pods and ReplicaSets.
*
* You describe a desired state in a Deployment, and the Deployment Controller changes the actual
* state to the desired state at a controlled rate. You can define Deployments to create new ReplicaSets, or to remove
* existing Deployments and adopt all their resources with new Deployments.
*
* > Note: Do not manage ReplicaSets owned by a Deployment. Consider opening an issue in the main Kubernetes repository if your use case is not covered below.
*
* Use Case
*
* The following are typical use cases for Deployments:
*
* - Create a Deployment to rollout a ReplicaSet. The ReplicaSet creates Pods in the background.
*   Check the status of the rollout to see if it succeeds or not.
* - Declare the new state of the Pods by updating the PodTemplateSpec of the Deployment.
*   A new ReplicaSet is created and the Deployment manages moving the Pods from the old ReplicaSet to the new one at a controlled rate.
*   Each new ReplicaSet updates the revision of the Deployment.
* - Rollback to an earlier Deployment revision if the current state of the Deployment is not stable.
*   Each rollback updates the revision of the Deployment.
* - Scale up the Deployment to facilitate more load.
* - Pause the Deployment to apply multiple fixes to its PodTemplateSpec and then resume it to start a new rollout.
* - Use the status of the Deployment as an indicator that a rollout has stuck.
* - Clean up older ReplicaSets that you don't need anymore.
*
**/
class Deployment extends workload.Workload {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.resourceType = 'deployments';
        this.apiObject = new k8s.KubeDeployment(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        this.minReady = props.minReady ?? cdk8s_1.Duration.seconds(0);
        this.progressDeadline = props.progressDeadline ?? cdk8s_1.Duration.seconds(600);
        if (this.progressDeadline.toSeconds() <= this.minReady.toSeconds()) {
            throw new Error(`'progressDeadline' (${this.progressDeadline.toSeconds()}s) must be greater than 'minReady' (${this.minReady.toSeconds()}s)`);
        }
        this.replicas = props.replicas ?? 2;
        this.strategy = props.strategy ?? DeploymentStrategy.rollingUpdate();
    }
    /**
     * Expose a deployment via a service.
     *
     * This is equivalent to running `kubectl expose deployment <deployment-name>`.
     *
     * @param options Options to determine details of the service and port exposed.
     */
    exposeViaService(options = {}) {
        const myPorts = container.extractContainerPorts(this);
        const myPortNumbers = myPorts.map(p => p.number);
        const ports = options.ports ?? myPorts.map(p => ({ port: p.number, targetPort: p.number, protocol: p.protocol }));
        if (ports.length === 0) {
            throw new Error(`Unable to expose deployment ${this.name} via a service: `
                + 'Deployment port cannot be determined.'
                + 'Either pass \'ports\', or configure ports on the containers of the deployment');
        }
        // validate the ports are owned by our containers
        for (const port of ports) {
            const targetPort = port.targetPort ?? port.port;
            if (!myPortNumbers.includes(targetPort)) {
                throw new Error(`Unable to expose deployment ${this.name} via a service: Port ${targetPort} is not exposed by any container`);
            }
        }
        const metadata = { namespace: this.metadata.namespace };
        if (options.name) {
            metadata.name = options.name;
        }
        return new service.Service(this, `${options.name ?? ''}Service`, {
            selector: this,
            ports,
            metadata,
            type: options.serviceType ?? service.ServiceType.CLUSTER_IP,
        });
    }
    /**
     * Expose a deployment via an ingress.
     *
     * This will first expose the deployment with a service, and then expose the service via an ingress.
     *
     * @param path The ingress path to register under.
     * @param options Additional options.
     */
    exposeViaIngress(path, options = {}) {
        const ser = this.exposeViaService(options);
        return ser.exposeViaIngress(path, options);
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            replicas: this.replicas,
            minReadySeconds: this.minReady.toSeconds(),
            progressDeadlineSeconds: this.progressDeadline.toSeconds(),
            template: {
                metadata: this.podMetadata.toJson(),
                spec: this._toPodSpec(),
            },
            selector: this._toLabelSelector(),
            strategy: this.strategy._toKube(),
        };
    }
}
exports.Deployment = Deployment;
_a = JSII_RTTI_SYMBOL_1;
Deployment[_a] = { fqn: "cdk8s-plus-23.Deployment", version: "2.0.0-rc.139" };
/**
 * Union like class repsenting either a ration in
 * percents or an absolute number.
 */
class PercentOrAbsolute {
    constructor(value) {
        this.value = value;
    }
    /**
     * Percent ratio.
     */
    static percent(percent) {
        return new PercentOrAbsolute(`${percent}%`);
    }
    /**
     * Absolute number.
     */
    static absolute(num) {
        return new PercentOrAbsolute(num);
    }
    isZero() {
        return this.value === PercentOrAbsolute.absolute(0).value || this.value === PercentOrAbsolute.percent(0).value;
    }
}
exports.PercentOrAbsolute = PercentOrAbsolute;
_b = JSII_RTTI_SYMBOL_1;
PercentOrAbsolute[_b] = { fqn: "cdk8s-plus-23.PercentOrAbsolute", version: "2.0.0-rc.139" };
/**
 * Deployment strategies.
 */
class DeploymentStrategy {
    constructor(strategy) {
        this.strategy = strategy;
    }
    /**
     * All existing Pods are killed before new ones are created.
     *
     * @see https://kubernetes.io/docs/concepts/workloads/controllers/deployment/#recreate-deployment
     */
    static recreate() {
        return new DeploymentStrategy({
            type: k8s.IoK8SApiAppsV1DeploymentStrategyType.RECREATE,
        });
    }
    static rollingUpdate(options = {}) {
        const maxSurge = options.maxSurge ?? PercentOrAbsolute.percent(25);
        const maxUnavailable = options.maxUnavailable ?? PercentOrAbsolute.percent(25);
        if (maxSurge.isZero() && maxUnavailable.isZero()) {
            throw new Error('\'maxSurge\' and \'maxUnavailable\' cannot be both zero');
        }
        return new DeploymentStrategy({
            type: k8s.IoK8SApiAppsV1DeploymentStrategyType.ROLLING_UPDATE,
            rollingUpdate: { maxSurge, maxUnavailable },
        });
    }
    /**
     * @internal
     */
    _toKube() {
        return this.strategy;
    }
}
exports.DeploymentStrategy = DeploymentStrategy;
_c = JSII_RTTI_SYMBOL_1;
DeploymentStrategy[_c] = { fqn: "cdk8s-plus-23.DeploymentStrategy", version: "2.0.0-rc.139" };
//# sourceMappingURL=data:application/json;base64,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