"""Base DTO for file parsing"""

from abc import ABC
from dataclasses import dataclass, fields, field
from typing import ClassVar, Optional

__all__ = ["FileDTO"]


@dataclass
class FileDTO(ABC):
    """
    Base class for file DTOs.

    Usage:
        @dataclass
        class EmailPasswordDTO(FileDTO):
            email: str = ''
            password: str = ''

        parser = FileParser(
            dto_class=EmailPasswordDTO,
            file_path='emails.txt',
            delimiter=':'
        )

        items = parser.parse()
    """

    # Line metadata
    line_number: int = field(default=0, repr=False)
    raw_line: str = field(default='', repr=False)

    def validate(self) -> bool:
        """
        Validate the DTO after parsing.
        Override this method for custom validation.

        Returns:
            bool: True if valid, False otherwise

        Example:
            def validate(self) -> bool:
                return '@' in self.email and len(self.password) > 6
        """
        return True

    def post_parse(self) -> None:
        """
        Hook called after parsing.
        Use for data transformation, cleanup, etc.

        Example:
            def post_parse(self) -> None:
                self.email = self.email.lower()
                self.password = self.password.strip()
        """
        pass