# proxy_parser.py
from __future__ import annotations
from dataclasses import dataclass, field
from typing import Literal, List, Optional

from ..dto import FileDTO
from ..parser import FileParser

# ——— Наш готовый DTO, пользователь его не указывает — просто получает List[ProxyDTO]
@dataclass
class ProxyDTO(FileDTO):
    protocol: str = ''
    ip: str = ''
    port: str = ''
    login: str = ''
    password: str = ''
    url: str = field(init=False)

    def __post_init__(self) -> None:
        self._rebuild_url()

    def _rebuild_url(self) -> None:
        scheme = self.protocol or "http"
        if self.login:
            auth = self.login if not self.password else f"{self.login}:{self.password}"
            auth += "@"
        else:
            auth = ""
        port_part = f":{self.port}" if self.port else ""
        self.url = f"{scheme}://{auth}{self.ip}{port_part}"


# ——— Шаблоны (подсказки в IDE показывают СТРУКТУРУ)
ProxyTemplate = Literal[
    "protocol,login,password,ip,port",  # protocol://login:pass@ip:port
    "ip,port,login,password",           # ip:port:login:password
    "login,password,ip,port",           # login:password:ip:port
    "ip,port",                          # ip:port  (протокол подставим default_protocol)
]

# ——— Конфиг для каждого шаблона: чем резать и порядок маппинга в ProxyDTO
# Используем наш «stateless» FileParser.parse(...)
# - для мульти-разделителей: 'delimiters' + field_order
# - для одного разделителя:  'delimiter'  + field_order
_STRUCTURE_CONFIG = {
    "protocol,login,password,ip,port": {
        "delimiters": ["://", ":", "@", ":"],
        "field_order": ["protocol", "login", "password", "ip", "port"],
    },
    "ip,port,login,password": {
        "delimiter": ":",
        "field_order": ["ip", "port", "login", "password"],
    },
    "login,password,ip,port": {
        "delimiter": ":",
        "field_order": ["login", "password", "ip", "port"],
    },
    "ip,port": {
        "delimiter": ":",
        "field_order": ["ip", "port"],  # protocol добавим пост-обработкой
    },
}


class ProxyParser:
    """
    Простой фасад над FileParser:
      - parse(file_path=..., structure=..., default_protocol='http') -> List[ProxyDTO]
      - DTO не указываем: он фиксированный (ProxyDTO)
    """

    def __init__(
        self,
        *,
        encoding: str = 'utf-8',
        skip_invalid: bool = True,
        strip_whitespace: bool = True,
    ) -> None:
        self._fp = FileParser(
            encoding=encoding,
            skip_invalid=skip_invalid,
            strip_whitespace=strip_whitespace,
        )

    def parse(
        self,
        file_path: str,
        structure: ProxyTemplate,
        default_protocol: Optional[str] = "http",
        *,
        max_lines: Optional[int] = None,
        skip_lines: int = 0,
    ) -> List[ProxyDTO]:
        cfg = _STRUCTURE_CONFIG[structure]
        field_order: List[str] = cfg["field_order"]

        # выбираем режим резки
        delimiter = cfg.get("delimiter")
        delimiters = cfg.get("delimiters")

        items = self._fp.parse(
            ProxyDTO,
            file_path=file_path,
            delimiter=delimiter,
            delimiters=delimiters,
            field_order=field_order,
            max_lines=max_lines,
            skip_lines=skip_lines,
        )

        # если в шаблоне нет protocol — докинем default_protocol
        if default_protocol:
            for it in items:
                if not it.protocol:
                    it.protocol = default_protocol

        return items

