"""Reusable parsing templates (presets)"""
import re
from dataclasses import dataclass
from typing import Callable, Dict, List, Optional, Pattern

@dataclass
class LinePattern:
    """
    Универсальная схема парсинга одной строки.
    Варианты:
      - delimiter: один разделитель для N полей
      - delimiters: список разделителей в фиксированном порядке (даёт len(delimiters)+1 полей)
      - regex: регулярка с ИМЕНОВАННЫМИ группами (groupdict())
    Поля (fields) описывают порядок маппинга в DTO.
    Можно добавить post_map для доп. обработки.
    """
    fields: List[str]
    delimiter: Optional[str] = None
    delimiters: Optional[List[str]] = None
    regex: Optional[Pattern] = None
    post_map: Optional[Callable[[Dict[str, str]], Dict[str, str]]] = None


# ---------- Готовые пресеты под прокси ----------

# 1) http://login:pass@ip:port
PROXY_HTTP_URL = LinePattern(
    fields=["protocol", "login", "password", "ip", "port"],
    delimiters=["://", ":", "@", ":"]
)

# 2) ip:port:login:pass
PROXY_IPPORT_USERPASS = LinePattern(
    fields=["ip", "port", "login", "password"],
    delimiter=":"
)

# 3) login:pass:ip:port
PROXY_USERPASS_IPPORT = LinePattern(
    fields=["login", "password", "ip", "port"],
    delimiter=":"
)

# 4) Только ip:port (протокол по умолчанию можно добавить в post_map)
def _add_default_protocol_http(data: Dict[str, str]) -> Dict[str, str]:
    if "protocol" not in data or not data["protocol"]:
        data["protocol"] = "http"
    return data

PROXY_IPPORT = LinePattern(
    fields=["ip", "port"],
    delimiter=":",
    post_map=_add_default_protocol_http
)

# ---------- Пресет для извлечения домена из URL ----------
# Вариант с regex и именованными группами. Достаём hostname, а домен сведём к "корневому" (last two labels).
URL_HOST_REGEX = re.compile(
    r"^(?P<scheme>[a-zA-Z][a-zA-Z0-9+\-.]*):\/\/(?P<host>[^\/\?#:]+)(?::(?P<port>\d+))?(?:[\/\?#].*)?$"
)

def _to_root_domain(data: Dict[str, str]) -> Dict[str, str]:
    host = data.get("host", "")
    # Очень простой хелпер для root-домена: последние 2 лейбла.
    # Для сложных TLD (co.uk и т.п.) лучше использовать tldextract (но это внешняя зависимость).
    parts = host.split(".")
    root = host if len(parts) < 2 else ".".join(parts[-2:])
    data["domain"] = root
    return data

URL_DOMAIN = LinePattern(
    fields=["scheme", "host", "port"],   # временные поля из именованных групп
    regex=URL_HOST_REGEX,
    post_map=_to_root_domain            # добавит data["domain"]
)

# Регистр для удобного доступа по строковому имени
TEMPLATES: Dict[str, LinePattern] = {
    "PROXY_HTTP_URL": PROXY_HTTP_URL,
    "PROXY_IPPORT_USERPASS": PROXY_IPPORT_USERPASS,
    "PROXY_USERPASS_IPPORT": PROXY_USERPASS_IPPORT,
    "PROXY_IPPORT": PROXY_IPPORT,
    "URL_DOMAIN": URL_DOMAIN,
}
