"use strict";
const assert_1 = require("@aws-cdk/assert");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const s3 = require("../lib");
module.exports = {
    'default properties'(test) {
        const stack = new core_1.Stack();
        const myBucket = new s3.Bucket(stack, 'MyBucket');
        const myBucketPolicy = new s3.BucketPolicy(stack, 'MyBucketPolicy', {
            bucket: myBucket,
        });
        myBucketPolicy.document.addStatements(new aws_iam_1.PolicyStatement({
            resources: [myBucket.bucketArn],
            actions: ['s3:GetObject*'],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::BucketPolicy', {
            Bucket: {
                'Ref': 'MyBucketF68F3FF0',
            },
            PolicyDocument: {
                'Version': '2012-10-17',
                'Statement': [
                    {
                        'Action': 's3:GetObject*',
                        'Effect': 'Allow',
                        'Resource': { 'Fn::GetAtt': ['MyBucketF68F3FF0', 'Arn'] },
                    },
                ],
            },
        }));
        test.done();
    },
    'when specifying a removalPolicy at creation'(test) {
        const stack = new core_1.Stack();
        const myBucket = new s3.Bucket(stack, 'MyBucket');
        const myBucketPolicy = new s3.BucketPolicy(stack, 'MyBucketPolicy', {
            bucket: myBucket,
            removalPolicy: core_1.RemovalPolicy.RETAIN,
        });
        myBucketPolicy.document.addStatements(new aws_iam_1.PolicyStatement({
            resources: [myBucket.bucketArn],
            actions: ['s3:GetObject*'],
        }));
        assert_1.expect(stack).toMatch({
            'Resources': {
                'MyBucketF68F3FF0': {
                    'Type': 'AWS::S3::Bucket',
                    'DeletionPolicy': 'Retain',
                    'UpdateReplacePolicy': 'Retain',
                },
                'MyBucketPolicy0AFEFDBE': {
                    'Type': 'AWS::S3::BucketPolicy',
                    'Properties': {
                        'Bucket': {
                            'Ref': 'MyBucketF68F3FF0',
                        },
                        'PolicyDocument': {
                            'Statement': [
                                {
                                    'Action': 's3:GetObject*',
                                    'Effect': 'Allow',
                                    'Resource': { 'Fn::GetAtt': ['MyBucketF68F3FF0', 'Arn'] },
                                },
                            ],
                            'Version': '2012-10-17',
                        },
                    },
                    'DeletionPolicy': 'Retain',
                    'UpdateReplacePolicy': 'Retain',
                },
            },
        });
        test.done();
    },
    'when specifying a removalPolicy after creation'(test) {
        var _a;
        const stack = new core_1.Stack();
        const myBucket = new s3.Bucket(stack, 'MyBucket');
        myBucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            resources: [myBucket.bucketArn],
            actions: ['s3:GetObject*'],
        }));
        (_a = myBucket.policy) === null || _a === void 0 ? void 0 : _a.applyRemovalPolicy(core_1.RemovalPolicy.RETAIN);
        assert_1.expect(stack).toMatch({
            'Resources': {
                'MyBucketF68F3FF0': {
                    'Type': 'AWS::S3::Bucket',
                    'DeletionPolicy': 'Retain',
                    'UpdateReplacePolicy': 'Retain',
                },
                'MyBucketPolicyE7FBAC7B': {
                    'Type': 'AWS::S3::BucketPolicy',
                    'Properties': {
                        'Bucket': {
                            'Ref': 'MyBucketF68F3FF0',
                        },
                        'PolicyDocument': {
                            'Statement': [
                                {
                                    'Action': 's3:GetObject*',
                                    'Effect': 'Allow',
                                    'Resource': { 'Fn::GetAtt': ['MyBucketF68F3FF0', 'Arn'] },
                                },
                            ],
                            'Version': '2012-10-17',
                        },
                    },
                    'DeletionPolicy': 'Retain',
                    'UpdateReplacePolicy': 'Retain',
                },
            },
        });
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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