"""Signal dispatcher for Gentoo Build Publisher"""

from typing import Any, Callable, ParamSpec, TypeAlias

from blinker import Signal, signal

P = ParamSpec("P")
PyDispatchHandler: TypeAlias = Callable[..., Any]
BlinkerHandler: TypeAlias = Callable[[Any, P.kwargs], Any]

CORE_EVENTS = ["predelete", "postdelete", "published", "prepull", "postpull"]


class DoesNotExistError(LookupError):
    """Binding or emitting a nonexistent signal"""

    def __init__(self, name: str):
        self.name = name

    def __str__(self) -> str:
        return f'Event "{self.name}" not registered'


class EventExistsError(Exception):
    """Registering an event that already exists in the registry"""

    def __init__(self, name: str):
        self.name = name

    def __str__(self) -> str:
        return f'"{self.name}" already exists'


class PyDispatcherAdapter:
    """Adapter to run python-dispatcher handlers through blinker"""

    def __init__(self) -> None:
        self._registry: dict[str, Signal] = {}
        self._handlers: dict[PyDispatchHandler, BlinkerHandler[Any]] = {}

        event: str
        for event in getattr(self, "_events_", []):
            self._registry[event] = signal(event)

    def register_event(self, *events: str) -> None:
        """Add the given event(s) to to the event registry"""
        for event in events:
            if event in self._registry:
                raise EventExistsError(event)

            self._registry[event] = signal(event)

    def bind(self, **kwargs: PyDispatchHandler) -> None:
        """Bind the python-dispatcher style handlers to the given signal"""
        for signal_name, callback in kwargs.items():
            try:
                sig = self._registry[signal_name]
            except KeyError as error:
                raise DoesNotExistError(signal_name) from error

            self._handlers[callback] = blinker_adapter(callback)
            sig.connect(self._handlers[callback])

    def emit(self, signal_name: str, *args: Any, **kwargs: Any) -> None:
        """Emit the given signal"""
        try:
            sig = self._registry[signal_name]
        except KeyError as error:
            raise DoesNotExistError(signal_name) from error

        sig.send(*args, **kwargs)

    def unbind(self, callback: PyDispatchHandler) -> None:
        """Unbind the given signal handler"""
        # remove the reference
        self._handlers.pop(callback)

    def get_dispatcher_event(self, name: str) -> Signal:
        """Return the given registered event (Signal)

        Raise DoesNotExistError if the given event is not registered.
        """
        try:
            return self._registry[name]
        except KeyError as error:
            raise DoesNotExistError(name) from error


def blinker_adapter(callback: PyDispatchHandler) -> BlinkerHandler:  # type: ignore[type-arg]
    """Covert a python-dispatch style handler to a blinker handler"""

    def adapted_callback(_sender: Any, *_args: Any, **kwargs: Any) -> Any:
        return callback(**kwargs)

    return adapted_callback


class PublisherDispatcher(PyDispatcherAdapter):
    """GBP event dispatcher"""

    _events_ = CORE_EVENTS


dispatcher = PublisherDispatcher()


__all__ = ("dispatcher", "DoesNotExistError", "EventExistsError")
