# -*- coding: utf-8 -*-
import argparse

from poppy.core.command import Command
from poppy.core.conf import settings
from poppy.pop.tools import paths
import os


class Master(Command):
    """
    The master command, with some global options useful for the other commands.
    """

    __command__ = "master"
    __parent__ = None
    __help__ = "Pipeline manager"

    def parser(self, subparser, parents):
        """
        Create the parser for the root command.
        """
        # create a new parser to add the parameters inside it and be able to
        # reuse them for other commands
        parent = argparse.ArgumentParser(add_help=False, allow_abbrev=False)

        # redefine the pipeline arguments here to include the help messages to the main parser
        for argument in Command.manager.pipeline_arguments.values():
            parent.add_argument(*argument[0], **argument[1])

        # argument to read the path to the configuration file to load
        parent.add_argument(
            "-c",
            "--config",
            help="""
            The absolute path to the configuration file.
            """,
            type=str,
            default=os.path.join(settings.ROOT_DIRECTORY, "config.json"),
        )

        # argument to read the path to the configuration file to load
        parent.add_argument(
            "-C",
            "--config-schema",
            help="""
            The absolute path to the schema for configuration file.
            """,
            type=str,
            default=paths.from_json_schemas("config-schema.json"),
        )

        # argument to read the path to the descriptor file to load
        parent.add_argument(
            "-d",
            "--descriptor",
            help="""
            The absolute path to the descriptor file.
            """,
            type=str,
            default=os.path.join(settings.ROOT_DIRECTORY, "descriptor.json"),
        )

        # argument to read the path to the configuration file to load
        parent.add_argument(
            "-D",
            "--pipeline-descriptor-schema",
            help="""
            The absolute path to the schema for descriptor file.
            """,
            type=str,
            default=paths.from_json_schemas("pipeline-descriptor-schema.json"),
        )

        # argument to set a dry run or not
        parent.add_argument(
            "--dry-run",
            help="""
            Set this flag to do a dry run, i.e. tasks/jobs are not registered
            into the database, etc.
            """,
            action="store_true",
            default=False,
        )

        # the name of the instance to use
        parent.add_argument(
            "--provider",
            help="""
            The name of the instance to use. If not provided, the name
            defined in the configuration file in field pipeline.provider is
            used.
            """,
            default=None,
        )

        # specify the pipeline version to use
        parent.add_argument(
            "-o",
            "--output-dir",
            help="""
            Output directory path for files generated by the pipeline. If not
            provided, use the value defined in the configuration file in
            pipeline.output_path.
            """,
            type=str,
            default=None,
            dest="output",
        )

        parent.add_argument(
            "-O",
            "--overwrite",
            action="store_true",
            help="Overwrite existing output files",
            default=False,
        )

        # add the parser to the collection
        self.add_parent_parser("base", parent)

        # the parser for the master command
        parser = argparse.ArgumentParser(parents=[parent], allow_abbrev=False)

        return parser
