"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GCEPersistentDiskPersistentVolume = exports.AzureDiskPersistentVolume = exports.AwsElasticBlockStorePersistentVolume = exports.PersistentVolumeReclaimPolicy = exports.PersistentVolume = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const constructs_1 = require("constructs");
const base = require("./base");
const k8s = require("./imports/k8s");
const pvc = require("./pvc");
const volume = require("./volume");
class ImportedPersistentVolume extends constructs_1.Construct {
    constructor(scope, id, name) {
        super(scope, id);
        this.resourceType = 'persistentvolumes';
        this._name = name;
    }
    get name() {
        return this._name;
    }
    get apiVersion() {
        return k8s.KubePersistentVolume.GVK.apiVersion;
    }
    get apiGroup() {
        return '';
    }
    get kind() {
        return k8s.KubePersistentVolume.GVK.kind;
    }
    get resourceName() {
        return this.name;
    }
}
/**
 * A PersistentVolume (PV) is a piece of storage in the cluster that has been
 * provisioned by an administrator or dynamically provisioned using Storage Classes.
 * It is a resource in the cluster just like a node is a cluster resource.
 * PVs are volume plugins like Volumes, but have a lifecycle independent of any
 * individual Pod that uses the PV. This API object captures the details of the
 * implementation of the storage, be that NFS, iSCSI, or a
 * cloud-provider-specific storage system.
 */
class PersistentVolume extends base.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.resourceType = 'persistentvolumes';
        this.storage = props.storage;
        this.mode = props.volumeMode ?? pvc.PersistentVolumeMode.FILE_SYSTEM;
        this.storageClassName = props.storageClassName;
        this._accessModes = props.accessModes;
        this.mountOptions = props.mountOptions;
        this.reclaimPolicy = props.reclaimPolicy ?? PersistentVolumeReclaimPolicy.RETAIN;
        if (props.claim) {
            this.bind(props.claim);
        }
        this.apiObject = new k8s.KubePersistentVolume(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
    }
    /**
     * Imports a pv from the cluster as a reference.
     */
    static fromPersistentVolumeName(scope, id, volumeName) {
        return new ImportedPersistentVolume(scope, id, volumeName);
    }
    /**
     * Access modes requirement of this claim.
     */
    get accessModes() {
        return this._accessModes ? [...this._accessModes] : undefined;
    }
    /**
     * PVC this volume is bound to. Undefined means this volume is not yet
     * claimed by any PVC.
     */
    get claim() {
        return this._claim;
    }
    /**
     * Reserve a `PersistentVolume` by creating a `PersistentVolumeClaim`
     * that is wired to claim this volume.
     *
     * Note that this method will throw in case the volume is already claimed.
     *
     * @see https://kubernetes.io/docs/concepts/storage/persistent-volumes/#reserving-a-persistentvolume
     */
    reserve() {
        const claim = new pvc.PersistentVolumeClaim(this, `${this.name}PVC`, {
            metadata: { name: `pvc-${this.name}`, namespace: this.metadata.namespace },
            // the storage classes must match, otherwise the claim
            // will use the default class (or no class at all), which may be different than the class
            // of this volume. note that other requirements are not needed since
            // when they are not defined, any volume satisifies them.
            storageClassName: this.storageClassName,
        });
        this.bind(claim);
        claim.bind(this);
        return claim;
    }
    /**
     * Bind a volume to a specific claim.
     * Note that you must also bind the claim to the volume.
     *
     * @see https://kubernetes.io/docs/concepts/storage/persistent-volumes/#binding
     *
     * @param claim The PVC to bind to.
     */
    bind(claim) {
        if (this._claim && this._claim.name !== claim.name) {
            throw new Error(`Cannot bind volume '${this.name}' to claim '${claim.name}' since it is already bound to claim '${this._claim.name}'`);
        }
        this._claim = claim;
    }
    asVolume() {
        const claim = this.reserve();
        return volume.Volume.fromPersistentVolumeClaim(this, 'Volume', claim);
    }
    /**
     * @internal
     */
    _toKube() {
        const storage = this.storage ? k8s.Quantity.fromString(this.storage.toGibibytes() + 'Gi') : undefined;
        return {
            claimRef: this._claim ? { name: this._claim?.name } : undefined,
            accessModes: this.accessModes?.map(a => a.toString()),
            capacity: storage ? { storage } : undefined,
            mountOptions: this.mountOptions?.map(o => o),
            storageClassName: this.storageClassName,
            persistentVolumeReclaimPolicy: this._reclaimPolicyToKube(this.reclaimPolicy),
            volumeMode: this.mode,
        };
    }
    _reclaimPolicyToKube(reclaimPolicy) {
        switch (reclaimPolicy) {
            case PersistentVolumeReclaimPolicy.DELETE:
                return k8s.IoK8SApiCoreV1PersistentVolumeSpecPersistentVolumeReclaimPolicy.DELETE;
            case PersistentVolumeReclaimPolicy.RETAIN:
                return k8s.IoK8SApiCoreV1PersistentVolumeSpecPersistentVolumeReclaimPolicy.RETAIN;
            default:
                throw new Error(`Unsupported reclaim policy: ${reclaimPolicy}`);
        }
    }
}
exports.PersistentVolume = PersistentVolume;
_a = JSII_RTTI_SYMBOL_1;
PersistentVolume[_a] = { fqn: "cdk8s-plus-23.PersistentVolume", version: "2.4.27" };
/**
 * Reclaim Policies.
 */
var PersistentVolumeReclaimPolicy;
(function (PersistentVolumeReclaimPolicy) {
    /**
     * The Retain reclaim policy allows for manual reclamation of the resource.
     * When the PersistentVolumeClaim is deleted, the PersistentVolume still exists and the
     * volume is considered "released". But it is not yet available for another claim
     * because the previous claimant's data remains on the volume.
     * An administrator can manually reclaim the volume with the following steps:
     *
     * 1. Delete the PersistentVolume. The associated storage asset in external
     *    infrastructure (such as an AWS EBS, GCE PD, Azure Disk, or Cinder volume) still exists after the PV is deleted.
     * 2. Manually clean up the data on the associated storage asset accordingly.
     * 3. Manually delete the associated storage asset.
     *
     * If you want to reuse the same storage asset, create a new PersistentVolume
     * with the same storage asset definition.
     */
    PersistentVolumeReclaimPolicy["RETAIN"] = "Retain";
    /**
     * For volume plugins that support the Delete reclaim policy, deletion removes both the
     * PersistentVolume object from Kubernetes, as well as the associated storage asset in
     * the external infrastructure, such as an AWS EBS, GCE PD, Azure Disk, or Cinder volume.
     * Volumes that were dynamically provisioned inherit the reclaim policy of their StorageClass, which defaults to Delete.
     * The administrator should configure the StorageClass according to users' expectations; otherwise,
     * the PV must be edited or patched after it is created
     */
    PersistentVolumeReclaimPolicy["DELETE"] = "Delete";
})(PersistentVolumeReclaimPolicy = exports.PersistentVolumeReclaimPolicy || (exports.PersistentVolumeReclaimPolicy = {}));
/**
 * Represents an AWS Disk resource that is attached to a kubelet's host machine and
 * then exposed to the pod.
 *
 * @see https://kubernetes.io/docs/concepts/storage/volumes#awselasticblockstore
 */
class AwsElasticBlockStorePersistentVolume extends PersistentVolume {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.volumeId = props.volumeId;
        this.readOnly = props.readOnly ?? false;
        this.fsType = props.fsType ?? 'ext4';
        this.partition = props.partition;
    }
    /**
     * @internal
     */
    _toKube() {
        const spec = super._toKube();
        return {
            ...spec,
            awsElasticBlockStore: {
                volumeId: this.volumeId,
                fsType: this.fsType,
                partition: this.partition,
                readOnly: this.readOnly,
            },
        };
    }
}
exports.AwsElasticBlockStorePersistentVolume = AwsElasticBlockStorePersistentVolume;
_b = JSII_RTTI_SYMBOL_1;
AwsElasticBlockStorePersistentVolume[_b] = { fqn: "cdk8s-plus-23.AwsElasticBlockStorePersistentVolume", version: "2.4.27" };
/**
 * AzureDisk represents an Azure Data Disk mount on the host and bind mount to the pod.
 */
class AzureDiskPersistentVolume extends PersistentVolume {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.diskName = props.diskName;
        this.diskUri = props.diskUri;
        this.cachingMode = props.cachingMode ?? volume.AzureDiskPersistentVolumeCachingMode.NONE;
        this.fsType = props.fsType ?? 'ext4';
        this.azureKind = props.kind ?? volume.AzureDiskPersistentVolumeKind.SHARED;
        this.readOnly = props.readOnly ?? false;
    }
    /**
     * @internal
     *
     * @see https://github.com/kubernetes/examples/blob/master/staging/volumes/azure_disk/README.md
     */
    _toKube() {
        const spec = super._toKube();
        return {
            ...spec,
            azureDisk: {
                diskName: this.diskName,
                diskUri: this.diskUri,
                cachingMode: this.cachingMode,
                fsType: this.fsType,
                kind: this.azureKind,
                readOnly: this.readOnly,
            },
        };
    }
}
exports.AzureDiskPersistentVolume = AzureDiskPersistentVolume;
_c = JSII_RTTI_SYMBOL_1;
AzureDiskPersistentVolume[_c] = { fqn: "cdk8s-plus-23.AzureDiskPersistentVolume", version: "2.4.27" };
/**
 * GCEPersistentDisk represents a GCE Disk resource that is attached to a kubelet's host machine
 * and then exposed to the pod. Provisioned by an admin.
 *
 * @see https://kubernetes.io/docs/concepts/storage/volumes#gcepersistentdisk
 */
class GCEPersistentDiskPersistentVolume extends PersistentVolume {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.pdName = props.pdName;
        this.readOnly = props.readOnly ?? false;
        this.fsType = props.fsType ?? 'ext4';
        this.partition = props.partition;
    }
    /**
     * @internal
     *
     * @see https://github.com/kubernetes/examples/blob/master/staging/volumes/azure_disk/README.md
     */
    _toKube() {
        const spec = super._toKube();
        return {
            ...spec,
            gcePersistentDisk: {
                pdName: this.pdName,
                fsType: this.fsType,
                partition: this.partition,
                readOnly: this.readOnly,
            },
        };
    }
}
exports.GCEPersistentDiskPersistentVolume = GCEPersistentDiskPersistentVolume;
_d = JSII_RTTI_SYMBOL_1;
GCEPersistentDiskPersistentVolume[_d] = { fqn: "cdk8s-plus-23.GCEPersistentDiskPersistentVolume", version: "2.4.27" };
//# sourceMappingURL=data:application/json;base64,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